#!/usr/bin/env python
import posixpath

from ddbclient import utils
import ddbclient.base


class AcquisitionClient(ddbclient.base.BaseClient):
    """
    A class used to connect with DispimDb specifically for interacting with
    acquisitions.

    ...

    Attributes
    ----------
    base_url : str
        Base url of the api (e.g. http://bigkahuna.corp.alleninstitute.org/api)

    Methods
    -------
    post(data)
        Post a new acquisition object to DispimDb
    get_all(specimen_id)
        Get all acquisitions from a given specimen
    get(acquisition_id)
        Get a single acquisition
    query(query_dict)
        Use a custom query to get some desired subset of acquisitions
    put_data_location(acquisition_id, data_key, data_location)
        Add a new data location to an acquisition object
    patch_status(acquisition_id, data_key, state)
        Change the status of a specified data location for an acquisition
    delete(acquisition_id)
        Delete a single acquisition
    """
    def post(self, data):
        """Post a new acquisition object to DispimDb

        Parameters
        ----------
        data : dict
            Dict of the acquisition object to post

        Returns
        -------
        str
            The acquisition_id generated by DispimDb
        """
        url = posixpath.join(
            self.base_url,
            'new_acquisition')

        response_json = utils.post_json(url, data)

        return response_json['acquisition_id']

    def get_all(self, specimen_id):
        """Get all acquisitions from a given specimen

        Parameters
        ----------
        specimen_id : str
            The specimen_id from which to get all acquisitions

        Returns
        -------
        list
            List of acquisition ids from the given specimen
        """
        url = posixpath.join(
            self.base_url,
            specimen_id,
            'acquisitions')

        response = utils.get_json(url)
        return response

    def get(self, acquisition_id):
        """Get a single acquisition

        Parameters
        ----------
        acquisition_id : str
            Acquisition_id of the desired acquisition

        Returns
        -------
        dict
            Dict of the desired acquisition object
        """
        url = posixpath.join(
            self.base_url,
            'acquisition',
            acquisition_id)

        query = {'acquisition_id': acquisition_id}

        response_json = utils.get_json(url, query)
        return response_json

    def query(self, query_dict):
        """Use a custom query to get some desired subset of acquisitions

        Parameters
        ----------
        query_dict : dict
            Generic MongoDB query for DispimDb

        Returns
        -------
        list
            Results of the query
        """
        url = posixpath.join(
            self.base_url,
            'acquisition',
            'query')

        response_json = utils.put_json(url, query_dict)
        return response_json

    def put_data_location(self, acquisition_id, data_key, data_location):
        """Add a new data location to an acquisition object

        Parameters
        ----------
        acquisition_id : str
            acquisition_id for the acquisition object
        data_key : str
            the key for the data location
        data_location : str
            the uri of the data location

        Returns
        -------
        dict
            The output acquisition with the data location
        """
        url = posixpath.join(
            self.base_url,
            'acquisition',
            acquisition_id,
            'data_location',
            data_key)

        response_json = utils.put_json(url, data_location)
        return response_json

    def patch_status(self, acquisition_id, data_key, state):
        """Change the status of a specified data location for an acquisition

        Parameters
        ----------
        acquisition_id : str
            acquisition_id for the acquisition object
        data_key : str
            the key for the data location
        state : str
            updated status of acquisition for the data location

        Returns
        -------
        dict
            JSON response of the HTTP request
        """
        url = posixpath.join(
            self.base_url,
            'acquisition',
            acquisition_id,
            'data_location',
            data_key,
            'status',
            state)

        response_json = utils.patch_json(url)
        return response_json

    def delete(self, acquisition_id):
        """Delete a single acquisition

        Parameters
        ----------
        acquisition_id : str
            acquisition_id for the acquisition object

        Returns
        -------
        dict
            None if successful, else error code
        """
        url = posixpath.join(
            self.base_url,
            'acquisition',
            acquisition_id)

        response_json = utils.delete_json(url)
        return response_json
