"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const sns = require("@aws-cdk/aws-sns");
const subs = require("@aws-cdk/aws-sns-subscriptions");
const sqs = require("@aws-cdk/aws-sqs");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const publish_1 = require("../../lib/sns/publish");
/*
 * Creates a state machine with a task state to publish to an SNS topic.
 * The SNS topic has an SQS queue added as a subscriber.
 * When the state machine is executed, it will publish a message to our
 * topic, which can subsequently be consumed from the SQS queue.
 *
 * Stack verification steps:
 * The generated State Machine can be executed from the CLI (or Step Functions console)
 * and runs with an execution status of `Succeeded`.
 *
 * -- aws stepfunctions start-execution --state-machine-arn <state-machine-arn-from-output> provides execution arn
 * -- aws stepfunctions describe-execution --execution-arn <from previous command> returns a status of `Succeeded`
 * -- aws sqs receive-message --queue-url <queue-url-from-output> has a message of 'sending message over'
 */
const app = new cdk.App();
const stack = new cdk.Stack(app, 'aws-stepfunctions-tasks-sns-publish-integ');
const topic = new sns.Topic(stack, 'cool-topic');
const queue = new sqs.Queue(stack, 'show-me-the-messages');
topic.addSubscription(new subs.SqsSubscription(queue));
const publishTask = new publish_1.SnsPublish(stack, 'publish to SNS', {
    topic,
    message: sfn.TaskInput.fromText('sending message over'),
});
const finalStatus = new sfn.Pass(stack, 'Final step');
const chain = sfn.Chain.start(publishTask)
    .next(finalStatus);
const sm = new sfn.StateMachine(stack, 'StateMachine', {
    definition: chain,
    timeout: cdk.Duration.seconds(30),
});
new cdk.CfnOutput(stack, 'stateMachineArn', {
    value: sm.stateMachineArn,
});
new cdk.CfnOutput(stack, 'queueUrl', {
    value: queue.queueUrl,
});
app.synth();
//# sourceMappingURL=data:application/json;base64,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