"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const tasks = require("../../lib");
let stack;
let vpc;
let cluster;
beforeEach(() => {
    // GIVEN
    stack = new core_1.Stack();
    vpc = new ec2.Vpc(stack, 'Vpc');
    cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
    cluster.addCapacity('Capacity', {
        instanceType: new ec2.InstanceType('t3.medium'),
    });
});
test('Cannot create a Fargate task with a fargate-incompatible task definition', () => {
    const taskDefinition = new ecs.TaskDefinition(stack, 'TD', {
        memoryMiB: '512',
        cpu: '256',
        compatibility: ecs.Compatibility.EC2,
    });
    taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('foo/bar'),
        memoryLimitMiB: 256,
    });
    expect(() => new tasks.RunEcsFargateTask({ cluster, taskDefinition })).toThrowError(/not configured for compatibility with Fargate/);
});
test('Cannot create a Fargate task without a default container', () => {
    const taskDefinition = new ecs.TaskDefinition(stack, 'TD', {
        memoryMiB: '512',
        cpu: '256',
        compatibility: ecs.Compatibility.FARGATE,
    });
    expect(() => new tasks.RunEcsFargateTask({ cluster, taskDefinition })).toThrowError(/must have at least one essential container/);
});
test('Running a Fargate Task', () => {
    const taskDefinition = new ecs.TaskDefinition(stack, 'TD', {
        memoryMiB: '512',
        cpu: '256',
        compatibility: ecs.Compatibility.FARGATE,
    });
    const containerDefinition = taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('foo/bar'),
        memoryLimitMiB: 256,
    });
    // WHEN
    const runTask = new sfn.Task(stack, 'RunFargate', { task: new tasks.RunEcsFargateTask({
            integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
            cluster,
            taskDefinition,
            containerOverrides: [
                {
                    containerDefinition,
                    environment: [
                        { name: 'SOME_KEY', value: sfn.JsonPath.stringAt('$.SomeKey') },
                    ],
                },
            ],
        }) });
    new sfn.StateMachine(stack, 'SM', {
        definition: runTask,
    });
    // THEN
    expect(stack.resolve(runTask.toStateJson())).toEqual({
        End: true,
        Parameters: {
            Cluster: { 'Fn::GetAtt': ['ClusterEB0386A7', 'Arn'] },
            LaunchType: 'FARGATE',
            NetworkConfiguration: {
                AwsvpcConfiguration: {
                    SecurityGroups: [{ 'Fn::GetAtt': ['RunFargateSecurityGroup709740F2', 'GroupId'] }],
                    Subnets: [{ Ref: 'VpcPrivateSubnet1Subnet536B997A' }, { Ref: 'VpcPrivateSubnet2Subnet3788AAA1' }],
                },
            },
            TaskDefinition: { Ref: 'TD49C78F36' },
            Overrides: {
                ContainerOverrides: [
                    {
                        Environment: [
                            {
                                'Name': 'SOME_KEY',
                                'Value.$': '$.SomeKey',
                            },
                        ],
                        Name: 'TheContainer',
                    },
                ],
            },
        },
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::ecs:runTask.sync',
                ],
            ],
        },
        Type: 'Task',
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'ecs:RunTask',
                    Effect: 'Allow',
                    Resource: { Ref: 'TD49C78F36' },
                },
                {
                    Action: ['ecs:StopTask', 'ecs:DescribeTasks'],
                    Effect: 'Allow',
                    Resource: '*',
                },
                {
                    Action: 'iam:PassRole',
                    Effect: 'Allow',
                    Resource: [{ 'Fn::GetAtt': ['TDTaskRoleC497AFFC', 'Arn'] }],
                },
                {
                    Action: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                { Ref: 'AWS::Partition' },
                                ':events:',
                                { Ref: 'AWS::Region' },
                                ':',
                                { Ref: 'AWS::AccountId' },
                                ':rule/StepFunctionsGetEventsForECSTaskRule',
                            ],
                        ],
                    },
                },
            ],
        },
    });
});
test('Running an EC2 Task with bridge network', () => {
    const taskDefinition = new ecs.TaskDefinition(stack, 'TD', {
        compatibility: ecs.Compatibility.EC2,
    });
    const containerDefinition = taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('foo/bar'),
        memoryLimitMiB: 256,
    });
    // WHEN
    const runTask = new sfn.Task(stack, 'Run', { task: new tasks.RunEcsEc2Task({
            integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
            cluster,
            taskDefinition,
            containerOverrides: [
                {
                    containerDefinition,
                    environment: [
                        { name: 'SOME_KEY', value: sfn.JsonPath.stringAt('$.SomeKey') },
                    ],
                },
            ],
        }) });
    new sfn.StateMachine(stack, 'SM', {
        definition: runTask,
    });
    // THEN
    expect(stack.resolve(runTask.toStateJson())).toEqual({
        End: true,
        Parameters: {
            Cluster: { 'Fn::GetAtt': ['ClusterEB0386A7', 'Arn'] },
            LaunchType: 'EC2',
            TaskDefinition: { Ref: 'TD49C78F36' },
            Overrides: {
                ContainerOverrides: [
                    {
                        Environment: [
                            {
                                'Name': 'SOME_KEY',
                                'Value.$': '$.SomeKey',
                            },
                        ],
                        Name: 'TheContainer',
                    },
                ],
            },
        },
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::ecs:runTask.sync',
                ],
            ],
        },
        Type: 'Task',
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'ecs:RunTask',
                    Effect: 'Allow',
                    Resource: { Ref: 'TD49C78F36' },
                },
                {
                    Action: ['ecs:StopTask', 'ecs:DescribeTasks'],
                    Effect: 'Allow',
                    Resource: '*',
                },
                {
                    Action: 'iam:PassRole',
                    Effect: 'Allow',
                    Resource: [{ 'Fn::GetAtt': ['TDTaskRoleC497AFFC', 'Arn'] }],
                },
                {
                    Action: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                { Ref: 'AWS::Partition' },
                                ':events:',
                                { Ref: 'AWS::Region' },
                                ':',
                                { Ref: 'AWS::AccountId' },
                                ':rule/StepFunctionsGetEventsForECSTaskRule',
                            ],
                        ],
                    },
                },
            ],
        },
    });
});
test('Running an EC2 Task with placement strategies', () => {
    const taskDefinition = new ecs.TaskDefinition(stack, 'TD', {
        compatibility: ecs.Compatibility.EC2,
    });
    taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('foo/bar'),
        memoryLimitMiB: 256,
    });
    const ec2Task = new tasks.RunEcsEc2Task({
        integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
        cluster,
        taskDefinition,
        placementStrategies: [ecs.PlacementStrategy.spreadAcrossInstances(), ecs.PlacementStrategy.packedByCpu(), ecs.PlacementStrategy.randomly()],
        placementConstraints: [ecs.PlacementConstraint.memberOf('blieptuut')],
    });
    // WHEN
    const runTask = new sfn.Task(stack, 'Run', { task: ec2Task });
    new sfn.StateMachine(stack, 'SM', {
        definition: runTask,
    });
    // THEN
    expect(stack.resolve(runTask.toStateJson())).toEqual({
        End: true,
        Parameters: {
            Cluster: { 'Fn::GetAtt': ['ClusterEB0386A7', 'Arn'] },
            LaunchType: 'EC2',
            TaskDefinition: { Ref: 'TD49C78F36' },
            PlacementConstraints: [{ Type: 'memberOf', Expression: 'blieptuut' }],
            PlacementStrategy: [{ Field: 'instanceId', Type: 'spread' }, { Field: 'cpu', Type: 'binpack' }, { Type: 'random' }],
        },
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::ecs:runTask.sync',
                ],
            ],
        },
        Type: 'Task',
    });
});
test('Running an EC2 Task with overridden number values', () => {
    const taskDefinition = new ecs.TaskDefinition(stack, 'TD', {
        compatibility: ecs.Compatibility.EC2,
    });
    const containerDefinition = taskDefinition.addContainer('TheContainer', {
        image: ecs.ContainerImage.fromRegistry('foo/bar'),
        memoryLimitMiB: 256,
    });
    const ec2Task = new tasks.RunEcsEc2Task({
        integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
        cluster,
        taskDefinition,
        containerOverrides: [
            {
                containerDefinition,
                command: sfn.JsonPath.listAt('$.TheCommand'),
                cpu: 5,
                memoryLimit: sfn.JsonPath.numberAt('$.MemoryLimit'),
            },
        ],
    });
    // WHEN
    const runTask = new sfn.Task(stack, 'Run', { task: ec2Task });
    // THEN
    expect(stack.resolve(runTask.toStateJson())).toEqual({
        End: true,
        Parameters: {
            Cluster: { 'Fn::GetAtt': ['ClusterEB0386A7', 'Arn'] },
            LaunchType: 'EC2',
            TaskDefinition: { Ref: 'TD49C78F36' },
            Overrides: {
                ContainerOverrides: [
                    {
                        'Command.$': '$.TheCommand',
                        'Cpu': 5,
                        'Memory.$': '$.MemoryLimit',
                        'Name': 'TheContainer',
                    },
                ],
            },
        },
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::ecs:runTask.sync',
                ],
            ],
        },
        Type: 'Task',
    });
});
//# sourceMappingURL=data:application/json;base64,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