"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const batch = require("@aws-cdk/aws-batch");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const path = require("path");
const lib_1 = require("../../lib");
let stack;
let batchJobDefinition;
let batchJobQueue;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
    batchJobDefinition = new batch.JobDefinition(stack, 'JobDefinition', {
        container: {
            image: ecs.ContainerImage.fromAsset(path.join(__dirname, 'batchjob-image')),
        },
    });
    batchJobQueue = new batch.JobQueue(stack, 'JobQueue', {
        computeEnvironments: [
            {
                order: 1,
                computeEnvironment: new batch.ComputeEnvironment(stack, 'ComputeEnv', {
                    computeResources: { vpc: new ec2.Vpc(stack, 'vpc') },
                }),
            },
        ],
    });
});
test('Task with only the required parameters', () => {
    // WHEN
    const task = new lib_1.BatchSubmitJob(stack, 'Task', {
        jobDefinition: batchJobDefinition,
        jobName: 'JobName',
        jobQueue: batchJobQueue,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::batch:submitJob.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            JobDefinition: { Ref: 'JobDefinition24FFE3ED' },
            JobName: 'JobName',
            JobQueue: { Ref: 'JobQueueEE3AD499' },
        },
    });
});
test('Task with all the parameters', () => {
    // WHEN
    const task = new lib_1.BatchSubmitJob(stack, 'Task', {
        jobDefinition: batchJobDefinition,
        jobName: 'JobName',
        jobQueue: batchJobQueue,
        arraySize: 15,
        containerOverrides: {
            command: ['sudo', 'rm'],
            environment: { key: 'value' },
            instanceType: new ec2.InstanceType('MULTI'),
            memory: cdk.Size.mebibytes(1024),
            gpuCount: 1,
            vcpus: 10,
        },
        dependsOn: [{ jobId: '1234', type: 'some_type' }],
        payload: sfn.TaskInput.fromObject({
            foo: sfn.JsonPath.stringAt('$.bar'),
        }),
        attempts: 3,
        timeout: cdk.Duration.seconds(60),
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::batch:submitJob',
                ],
            ],
        },
        End: true,
        Parameters: {
            JobDefinition: { Ref: 'JobDefinition24FFE3ED' },
            JobName: 'JobName',
            JobQueue: { Ref: 'JobQueueEE3AD499' },
            ArrayProperties: { Size: 15 },
            ContainerOverrides: {
                Command: ['sudo', 'rm'],
                Environment: [{ Name: 'key', Value: 'value' }],
                InstanceType: 'MULTI',
                Memory: 1024,
                ResourceRequirements: [{ Type: 'GPU', Value: '1' }],
                Vcpus: 10,
            },
            DependsOn: [{ JobId: '1234', Type: 'some_type' }],
            Parameters: { 'foo.$': '$.bar' },
            RetryStrategy: { Attempts: 3 },
            Timeout: { AttemptDurationSeconds: 60 },
        },
    });
});
test('supports tokens', () => {
    // WHEN
    const task = new lib_1.BatchSubmitJob(stack, 'Task', {
        jobDefinition: batchJobDefinition,
        jobName: sfn.JsonPath.stringAt('$.jobName'),
        jobQueue: batchJobQueue,
        arraySize: sfn.JsonPath.numberAt('$.arraySize'),
        timeout: cdk.Duration.seconds(sfn.JsonPath.numberAt('$.timeout')),
        attempts: sfn.JsonPath.numberAt('$.attempts'),
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::batch:submitJob.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            'JobDefinition': { Ref: 'JobDefinition24FFE3ED' },
            'JobName.$': '$.jobName',
            'JobQueue': { Ref: 'JobQueueEE3AD499' },
            'ArrayProperties': {
                'Size.$': '$.arraySize',
            },
            'RetryStrategy': {
                'Attempts.$': '$.attempts',
            },
            'Timeout': {
                'AttemptDurationSeconds.$': '$.timeout',
            },
        },
    });
});
test('supports passing task input into payload', () => {
    // WHEN
    const task = new lib_1.BatchSubmitJob(stack, 'Task', {
        jobDefinition: batchJobDefinition,
        jobName: sfn.JsonPath.stringAt('$.jobName'),
        jobQueue: batchJobQueue,
        payload: sfn.TaskInput.fromDataAt('$.foo'),
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::batch:submitJob.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            'JobDefinition': { Ref: 'JobDefinition24FFE3ED' },
            'JobName.$': '$.jobName',
            'JobQueue': { Ref: 'JobQueueEE3AD499' },
            'Parameters.$': '$.foo',
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied as service integration pattern', () => {
    expect(() => {
        new lib_1.BatchSubmitJob(stack, 'Task', {
            jobDefinition: batchJobDefinition,
            jobName: 'JobName',
            jobQueue: batchJobQueue,
            integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
        });
    }).toThrow(/Unsupported service integration pattern. Supported Patterns: REQUEST_RESPONSE,RUN_JOB. Received: WAIT_FOR_TASK_TOKEN/);
});
test('Task throws if environment in containerOverrides contain env with name starting with AWS_BATCH', () => {
    expect(() => {
        new lib_1.BatchSubmitJob(stack, 'Task', {
            jobDefinition: batchJobDefinition,
            jobName: 'JobName',
            jobQueue: batchJobQueue,
            containerOverrides: {
                environment: { AWS_BATCH_MY_NAME: 'MY_VALUE' },
            },
        });
    }).toThrow(/Invalid environment variable name: AWS_BATCH_MY_NAME. Environment variable names starting with 'AWS_BATCH' are reserved./);
});
test('Task throws if arraySize is out of limits 2-10000', () => {
    expect(() => {
        new lib_1.BatchSubmitJob(stack, 'Task', {
            jobDefinition: batchJobDefinition,
            jobName: 'JobName',
            jobQueue: batchJobQueue,
            arraySize: 1,
        });
    }).toThrow(/arraySize must be between 2 and 10,000/);
    expect(() => {
        new lib_1.BatchSubmitJob(stack, 'Task2', {
            jobDefinition: batchJobDefinition,
            jobName: 'JobName',
            jobQueue: batchJobQueue,
            arraySize: 10001,
        });
    }).toThrow(/arraySize must be between 2 and 10,000/);
});
test('Task throws if dependencies exceeds 20', () => {
    expect(() => {
        new lib_1.BatchSubmitJob(stack, 'Task', {
            jobDefinition: batchJobDefinition,
            jobName: 'JobName',
            jobQueue: batchJobQueue,
            dependsOn: [...Array(21).keys()].map(i => ({
                jobId: `${i}`,
                type: `some_type-${i}`,
            })),
        });
    }).toThrow(/dependencies must be 20 or less/);
});
test('Task throws if attempts is out of limits 1-10', () => {
    expect(() => {
        new lib_1.BatchSubmitJob(stack, 'Task', {
            jobDefinition: batchJobDefinition,
            jobName: 'JobName',
            jobQueue: batchJobQueue,
            attempts: 0,
        });
    }).toThrow(/attempts must be between 1 and 10/);
    expect(() => {
        new lib_1.BatchSubmitJob(stack, 'Task2', {
            jobDefinition: batchJobDefinition,
            jobName: 'JobName',
            jobQueue: batchJobQueue,
            attempts: 11,
        });
    }).toThrow(/attempts must be between 1 and 10/);
});
test('Task throws if attempt duration is less than 60 sec', () => {
    expect(() => {
        new lib_1.BatchSubmitJob(stack, 'Task', {
            jobDefinition: batchJobDefinition,
            jobName: 'JobName',
            jobQueue: batchJobQueue,
            timeout: cdk.Duration.seconds(59),
        });
    }).toThrow(/attempt duration must be greater than 60 seconds./);
});
//# sourceMappingURL=data:application/json;base64,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