"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EvaluateExpression = void 0;
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const sfn = require("@aws-cdk/aws-stepfunctions");
const path = require("path");
/**
 * A Step Functions Task to evaluate an expression
 *
 * OUTPUT: the output of this task is the evaluated expression.
 *
 * @experimental
 */
class EvaluateExpression extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        this.evalFn = createEvalFn(this.props.runtime || lambda.Runtime.NODEJS_10_X, this);
        this.taskPolicies = [
            new iam.PolicyStatement({
                resources: [this.evalFn.functionArn],
                actions: ['lambda:InvokeFunction'],
            }),
        ];
    }
    /**
     * @internal
     */
    _renderTask() {
        const matches = this.props.expression.match(/\$[.\[][.a-zA-Z[\]0-9]+/g);
        let expressionAttributeValues = {};
        if (matches) {
            expressionAttributeValues = matches.reduce((acc, m) => ({
                ...acc,
                [m]: sfn.JsonPath.stringAt(m),
            }), {});
        }
        const parameters = {
            expression: this.props.expression,
            expressionAttributeValues,
        };
        return {
            Resource: this.evalFn.functionArn,
            Parameters: sfn.FieldUtils.renderObject(parameters),
        };
    }
}
exports.EvaluateExpression = EvaluateExpression;
function createEvalFn(runtime, scope) {
    const code = lambda.Code.asset(path.join(__dirname, `eval-${runtime.name}-handler`));
    const lambdaPurpose = 'Eval';
    switch (runtime) {
        case lambda.Runtime.NODEJS_10_X:
            return new lambda.SingletonFunction(scope, 'EvalFunction', {
                runtime,
                handler: 'index.handler',
                uuid: 'a0d2ce44-871b-4e74-87a1-f5e63d7c3bdc',
                lambdaPurpose,
                code,
            });
        // TODO: implement other runtimes
        default:
            throw new Error(`The runtime ${runtime.name} is currently not supported.`);
    }
}
//# sourceMappingURL=data:application/json;base64,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