"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmrCreateCluster = void 0;
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
const cluster_utils_1 = require("./private/cluster-utils");
/**
 * A Step Functions Task to create an EMR Cluster.
 *
 * The ClusterConfiguration is defined as Parameters in the state machine definition.
 *
 * OUTPUT: the ClusterId.
 *
 * @experimental
 */
class EmrCreateCluster extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id, props);
        this.props = props;
        this.visibleToAllUsers = this.props.visibleToAllUsers !== undefined ? this.props.visibleToAllUsers : true;
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.RUN_JOB;
        task_utils_1.validatePatternSupported(this.integrationPattern, EmrCreateCluster.SUPPORTED_INTEGRATION_PATTERNS);
        this._autoScalingRole = this.props.autoScalingRole;
        // If the Roles are undefined then they weren't provided, so create them
        this._serviceRole = (_a = this.props.serviceRole) !== null && _a !== void 0 ? _a : this.createServiceRole();
        this._clusterRole = (_b = this.props.clusterRole) !== null && _b !== void 0 ? _b : this.createClusterRole();
        // AutoScaling roles are not valid with InstanceFleet clusters.
        // Attempt to create only if .instances.instanceFleets is undefined or empty
        if (this.props.instances.instanceFleets === undefined || this.props.instances.instanceFleets.length === 0) {
            this._autoScalingRole = this._autoScalingRole || this.createAutoScalingRole();
            // If InstanceFleets are used and an AutoScaling Role is specified, throw an error
        }
        else if (this._autoScalingRole !== undefined) {
            throw new Error('Auto Scaling roles can not be specified with instance fleets.');
        }
        this.taskPolicies = this.createPolicyStatements(this._serviceRole, this._clusterRole, this._autoScalingRole);
    }
    /**
     * The service role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     */
    get serviceRole() {
        if (this._serviceRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._serviceRole;
    }
    /**
     * The instance role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     */
    get clusterRole() {
        if (this._clusterRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._clusterRole;
    }
    /**
     * The autoscaling role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     */
    get autoScalingRole() {
        if (this._autoScalingRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._autoScalingRole;
    }
    /**
     * @internal
     */
    _renderTask() {
        var _a, _b, _c;
        return {
            Resource: task_utils_1.integrationResourceArn('elasticmapreduce', 'createCluster', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                Instances: cluster_utils_1.InstancesConfigPropertyToJson(this.props.instances),
                JobFlowRole: cdk.stringToCloudFormation(this._clusterRole.roleName),
                Name: cdk.stringToCloudFormation(this.props.name),
                ServiceRole: cdk.stringToCloudFormation(this._serviceRole.roleName),
                AdditionalInfo: cdk.stringToCloudFormation(this.props.additionalInfo),
                Applications: cdk.listMapper(cluster_utils_1.ApplicationConfigPropertyToJson)(this.props.applications),
                AutoScalingRole: cdk.stringToCloudFormation((_a = this._autoScalingRole) === null || _a === void 0 ? void 0 : _a.roleName),
                BootstrapActions: cdk.listMapper(cluster_utils_1.BootstrapActionConfigToJson)(this.props.bootstrapActions),
                Configurations: cdk.listMapper(cluster_utils_1.ConfigurationPropertyToJson)(this.props.configurations),
                CustomAmiId: cdk.stringToCloudFormation(this.props.customAmiId),
                EbsRootVolumeSize: (_b = this.props.ebsRootVolumeSize) === null || _b === void 0 ? void 0 : _b.toGibibytes(),
                KerberosAttributes: this.props.kerberosAttributes ? cluster_utils_1.KerberosAttributesPropertyToJson(this.props.kerberosAttributes) : undefined,
                LogUri: cdk.stringToCloudFormation(this.props.logUri),
                ReleaseLabel: cdk.stringToCloudFormation(this.props.releaseLabel),
                ScaleDownBehavior: cdk.stringToCloudFormation((_c = this.props.scaleDownBehavior) === null || _c === void 0 ? void 0 : _c.valueOf()),
                SecurityConfiguration: cdk.stringToCloudFormation(this.props.securityConfiguration),
                ...(this.props.tags ? this.renderTags(this.props.tags) : undefined),
                VisibleToAllUsers: cdk.booleanToCloudFormation(this.visibleToAllUsers),
            }),
        };
    }
    renderTags(tags) {
        return tags ? { Tags: Object.keys(tags).map((key) => ({ Key: key, Value: tags[key] })) } : {};
    }
    /**
     * This generates the PolicyStatements required by the Task to call CreateCluster.
     */
    createPolicyStatements(serviceRole, clusterRole, autoScalingRole) {
        const stack = cdk.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['elasticmapreduce:RunJobFlow', 'elasticmapreduce:DescribeCluster', 'elasticmapreduce:TerminateJobFlows'],
                resources: ['*'],
            }),
        ];
        // Allow the StateMachine to PassRole to Cluster roles
        policyStatements.push(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [serviceRole.roleArn, clusterRole.roleArn],
        }));
        if (autoScalingRole !== undefined) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [autoScalingRole.roleArn],
            }));
        }
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventForEMRRunJobFlowRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
    /**
     * Generate the Role used by the EMR Service
     */
    createServiceRole() {
        return new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonElasticMapReduceRole')],
        });
    }
    /**
     * Generate the Role and Instance Profile used by the EC2 instances
     *
     * Data access permissions will need to be updated by the user
     */
    createClusterRole() {
        const role = new iam.Role(this, 'InstanceRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [role.roleName],
            instanceProfileName: role.roleName,
        });
        return role;
    }
    /**
     * Generate the Role used to AutoScale the Cluster
     */
    createAutoScalingRole() {
        var _a;
        const role = new iam.Role(this, 'AutoScalingRole', {
            assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonElasticMapReduceforAutoScalingRole')],
        });
        (_a = role.assumeRolePolicy) === null || _a === void 0 ? void 0 : _a.addStatements(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            principals: [new iam.ServicePrincipal('application-autoscaling.amazonaws.com')],
            actions: ['sts:AssumeRole'],
        }));
        return role;
    }
}
exports.EmrCreateCluster = EmrCreateCluster;
EmrCreateCluster.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
(function (EmrCreateCluster) {
    /**
     * Valid valus for the Cluster ScaleDownBehavior
     *
     * @experimental
     */
    let EmrClusterScaleDownBehavior;
    (function (EmrClusterScaleDownBehavior) {
        /**
         * Indicates that Amazon EMR terminates nodes at the instance-hour boundary, regardless of when the request to terminate the instance was
         * submitted. This option is only available with Amazon EMR 5.1.0 and later and is the default for clusters created using that version
         */
        EmrClusterScaleDownBehavior["TERMINATE_AT_INSTANCE_HOUR"] = "TERMINATE_AT_INSTANCE_HOUR";
        /**
         * Indicates that Amazon EMR blacklists and drains tasks from nodes before terminating the Amazon EC2 instances, regardless of the
         * instance-hour boundary.
         */
        EmrClusterScaleDownBehavior["TERMINATE_AT_TASK_COMPLETION"] = "TERMINATE_AT_TASK_COMPLETION";
    })(EmrClusterScaleDownBehavior = EmrCreateCluster.EmrClusterScaleDownBehavior || (EmrCreateCluster.EmrClusterScaleDownBehavior = {}));
    /**
     * Instance Role Types
     *
     * @experimental
     */
    let InstanceRoleType;
    (function (InstanceRoleType) {
        /**
         * Master Node
         */
        InstanceRoleType["MASTER"] = "MASTER";
        /**
         * Core Node
         */
        InstanceRoleType["CORE"] = "CORE";
        /**
         * Task Node
         */
        InstanceRoleType["TASK"] = "TASK";
    })(InstanceRoleType = EmrCreateCluster.InstanceRoleType || (EmrCreateCluster.InstanceRoleType = {}));
    /**
     * EBS Volume Types
     *
     * @experimental
     */
    let EbsBlockDeviceVolumeType;
    (function (EbsBlockDeviceVolumeType) {
        /**
         * gp2 Volume Type
         */
        EbsBlockDeviceVolumeType["GP2"] = "gp2";
        /**
         * io1 Volume Type
         */
        EbsBlockDeviceVolumeType["IO1"] = "io1";
        /**
         * Standard Volume Type
         */
        EbsBlockDeviceVolumeType["STANDARD"] = "standard";
    })(EbsBlockDeviceVolumeType = EmrCreateCluster.EbsBlockDeviceVolumeType || (EmrCreateCluster.EbsBlockDeviceVolumeType = {}));
    /**
     * Spot Timeout Actions
     *
     * @experimental
     */
    let SpotTimeoutAction;
    (function (SpotTimeoutAction) {
        /**\
         * SWITCH_TO_ON_DEMAND
         */
        SpotTimeoutAction["SWITCH_TO_ON_DEMAND"] = "SWITCH_TO_ON_DEMAND";
        /**
         * TERMINATE_CLUSTER
         */
        SpotTimeoutAction["TERMINATE_CLUSTER"] = "TERMINATE_CLUSTER";
    })(SpotTimeoutAction = EmrCreateCluster.SpotTimeoutAction || (EmrCreateCluster.SpotTimeoutAction = {}));
    /**
     * CloudWatch Alarm Comparison Operators
     *
     * @experimental
     */
    let CloudWatchAlarmComparisonOperator;
    (function (CloudWatchAlarmComparisonOperator) {
        /**
         * GREATER_THAN_OR_EQUAL
         */
        CloudWatchAlarmComparisonOperator["GREATER_THAN_OR_EQUAL"] = "GREATER_THAN_OR_EQUAL";
        /**
         * GREATER_THAN
         */
        CloudWatchAlarmComparisonOperator["GREATER_THAN"] = "GREATER_THAN";
        /**
         * LESS_THAN
         */
        CloudWatchAlarmComparisonOperator["LESS_THAN"] = "LESS_THAN";
        /**
         * LESS_THAN_OR_EQUAL
         */
        CloudWatchAlarmComparisonOperator["LESS_THAN_OR_EQUAL"] = "LESS_THAN_OR_EQUAL";
    })(CloudWatchAlarmComparisonOperator = EmrCreateCluster.CloudWatchAlarmComparisonOperator || (EmrCreateCluster.CloudWatchAlarmComparisonOperator = {}));
    /**
     * CloudWatch Alarm Statistics
     *
     * @experimental
     */
    let CloudWatchAlarmStatistic;
    (function (CloudWatchAlarmStatistic) {
        /**
         * SAMPLE_COUNT
         */
        CloudWatchAlarmStatistic["SAMPLE_COUNT"] = "SAMPLE_COUNT";
        /**
         * AVERAGE
         */
        CloudWatchAlarmStatistic["AVERAGE"] = "AVERAGE";
        /**
         * SUM
         */
        CloudWatchAlarmStatistic["SUM"] = "SUM";
        /**
         * MINIMUM
         */
        CloudWatchAlarmStatistic["MINIMUM"] = "MINIMUM";
        /**
         * MAXIMUM
         */
        CloudWatchAlarmStatistic["MAXIMUM"] = "MAXIMUM";
    })(CloudWatchAlarmStatistic = EmrCreateCluster.CloudWatchAlarmStatistic || (EmrCreateCluster.CloudWatchAlarmStatistic = {}));
    /**
     * CloudWatch Alarm Units
     *
     * @experimental
     */
    let CloudWatchAlarmUnit;
    (function (CloudWatchAlarmUnit) {
        /**
         * NONE
         */
        CloudWatchAlarmUnit["NONE"] = "NONE";
        /**
         * SECONDS
         */
        CloudWatchAlarmUnit["SECONDS"] = "SECONDS";
        /**
         * MICRO_SECONDS
         */
        CloudWatchAlarmUnit["MICRO_SECONDS"] = "MICRO_SECONDS";
        /**
         * MILLI_SECONDS
         */
        CloudWatchAlarmUnit["MILLI_SECONDS"] = "MILLI_SECONDS";
        /**
         * BYTES
         */
        CloudWatchAlarmUnit["BYTES"] = "BYTES";
        /**
         * KILO_BYTES
         */
        CloudWatchAlarmUnit["KILO_BYTES"] = "KILO_BYTES";
        /**
         * MEGA_BYTES
         */
        CloudWatchAlarmUnit["MEGA_BYTES"] = "MEGA_BYTES";
        /**
         * GIGA_BYTES
         */
        CloudWatchAlarmUnit["GIGA_BYTES"] = "GIGA_BYTES";
        /**
         * TERA_BYTES
         */
        CloudWatchAlarmUnit["TERA_BYTES"] = "TERA_BYTES";
        /**
         * BITS
         */
        CloudWatchAlarmUnit["BITS"] = "BITS";
        /**
         * KILO_BITS
         */
        CloudWatchAlarmUnit["KILO_BITS"] = "KILO_BITS";
        /**
         * MEGA_BITS
         */
        CloudWatchAlarmUnit["MEGA_BITS"] = "MEGA_BITS";
        /**
         * GIGA_BITS
         */
        CloudWatchAlarmUnit["GIGA_BITS"] = "GIGA_BITS";
        /**
         * TERA_BITS
         */
        CloudWatchAlarmUnit["TERA_BITS"] = "TERA_BITS";
        /**
         * PERCENT
         */
        CloudWatchAlarmUnit["PERCENT"] = "PERCENT";
        /**
         * COUNT
         */
        CloudWatchAlarmUnit["COUNT"] = "COUNT";
        /**
         * BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["BYTES_PER_SECOND"] = "BYTES_PER_SECOND";
        /**
         * KILO_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["KILO_BYTES_PER_SECOND"] = "KILO_BYTES_PER_SECOND";
        /**
         * MEGA_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["MEGA_BYTES_PER_SECOND"] = "MEGA_BYTES_PER_SECOND";
        /**
         * GIGA_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["GIGA_BYTES_PER_SECOND"] = "GIGA_BYTES_PER_SECOND";
        /**
         * TERA_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["TERA_BYTES_PER_SECOND"] = "TERA_BYTES_PER_SECOND";
        /**
         * BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["BITS_PER_SECOND"] = "BITS_PER_SECOND";
        /**
         * KILO_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["KILO_BITS_PER_SECOND"] = "KILO_BITS_PER_SECOND";
        /**
         * MEGA_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["MEGA_BITS_PER_SECOND"] = "MEGA_BITS_PER_SECOND";
        /**
         * GIGA_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["GIGA_BITS_PER_SECOND"] = "GIGA_BITS_PER_SECOND";
        /**
         * TERA_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["TERA_BITS_PER_SECOND"] = "TERA_BITS_PER_SECOND";
        /**
         * COUNT_PER_SECOND
         */
        CloudWatchAlarmUnit["COUNT_PER_SECOND"] = "COUNT_PER_SECOND";
    })(CloudWatchAlarmUnit = EmrCreateCluster.CloudWatchAlarmUnit || (EmrCreateCluster.CloudWatchAlarmUnit = {}));
    /**
     * EC2 Instance Market
     *
     * @experimental
     */
    let InstanceMarket;
    (function (InstanceMarket) {
        /**
         * On Demand Instance
         */
        InstanceMarket["ON_DEMAND"] = "ON_DEMAND";
        /**
         * Spot Instance
         */
        InstanceMarket["SPOT"] = "SPOT";
    })(InstanceMarket = EmrCreateCluster.InstanceMarket || (EmrCreateCluster.InstanceMarket = {}));
    /**
     * AutoScaling Adjustment Type
     *
     * @experimental
     */
    let ScalingAdjustmentType;
    (function (ScalingAdjustmentType) {
        /**
         * CHANGE_IN_CAPACITY
         */
        ScalingAdjustmentType["CHANGE_IN_CAPACITY"] = "CHANGE_IN_CAPACITY";
        /**
         * PERCENT_CHANGE_IN_CAPACITY
         */
        ScalingAdjustmentType["PERCENT_CHANGE_IN_CAPACITY"] = "PERCENT_CHANGE_IN_CAPACITY";
        /**
         * EXACT_CAPACITY
         */
        ScalingAdjustmentType["EXACT_CAPACITY"] = "EXACT_CAPACITY";
    })(ScalingAdjustmentType = EmrCreateCluster.ScalingAdjustmentType || (EmrCreateCluster.ScalingAdjustmentType = {}));
})(EmrCreateCluster = exports.EmrCreateCluster || (exports.EmrCreateCluster = {}));
//# sourceMappingURL=data:application/json;base64,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