"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('User Pool Client', () => {
    test('custom domain name', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        const certificate = aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'cert', 'arn:aws:acm:eu-west-1:0123456789:certificate/7ec3e4ac-808a-4649-b805-66ae02346ad8');
        new lib_1.UserPoolDomain(stack, 'Domain', {
            userPool: pool,
            customDomain: {
                domainName: 'test-domain.example.com',
                certificate,
            },
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolDomain', {
            UserPoolId: stack.resolve(pool.userPoolId),
            Domain: 'test-domain.example.com',
            CustomDomainConfig: {
                CertificateArn: 'arn:aws:acm:eu-west-1:0123456789:certificate/7ec3e4ac-808a-4649-b805-66ae02346ad8',
            },
        });
    });
    test('cognito domain prefix', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        new lib_1.UserPoolDomain(stack, 'Domain', {
            userPool: pool,
            cognitoDomain: {
                domainPrefix: 'cognito-domain-prefix',
            },
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolDomain', {
            UserPoolId: stack.resolve(pool.userPoolId),
            Domain: 'cognito-domain-prefix',
        });
    });
    test('fails when neither cognitoDomain nor customDomain are specified', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        const certificate = aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'cert', 'arn:aws:acm:eu-west-1:0123456789:certificate/7ec3e4ac-808a-4649-b805-66ae02346ad8');
        expect(() => new lib_1.UserPoolDomain(stack, 'Domain', {
            userPool: pool,
            cognitoDomain: {
                domainPrefix: 'cognito-domain-prefix',
            },
            customDomain: {
                domainName: 'mydomain.com',
                certificate,
            },
        })).toThrow(/cognitoDomain or customDomain must be specified/);
    });
    test('fails when both cognitoDomain and customDomain are specified', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        expect(() => new lib_1.UserPoolDomain(stack, 'Domain', {
            userPool: pool,
        })).toThrow(/cognitoDomain or customDomain must be specified/);
    });
    test('fails when domainPrefix has characters outside the allowed charset', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        expect(() => pool.addDomain('Domain1', {
            cognitoDomain: { domainPrefix: 'domain.prefix' },
        })).toThrow(/lowercase alphabets, numbers and hyphens/);
        expect(() => pool.addDomain('Domain2', {
            cognitoDomain: { domainPrefix: 'Domain-Prefix' },
        })).toThrow(/lowercase alphabets, numbers and hyphens/);
        expect(() => pool.addDomain('Domain3', {
            cognitoDomain: { domainPrefix: 'dómäin-prefix' },
        })).toThrow(/lowercase alphabets, numbers and hyphens/);
    });
    test('custom resource is added when cloudFrontDistribution method is called', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        const domain = pool.addDomain('Domain', {
            cognitoDomain: {
                domainPrefix: 'cognito-domain-prefix',
            },
        });
        // WHEN
        const cfDomainName = domain.cloudFrontDomainName;
        // THEN
        expect(stack.resolve(cfDomainName)).toEqual({
            'Fn::GetAtt': [
                'PoolDomainCloudFrontDomainName340BF87E',
                'DomainDescription.CloudFrontDistribution',
            ],
        });
        expect(stack).toHaveResource('Custom::UserPoolCloudFrontDomainName');
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [{
                        Action: 'cognito-idp:DescribeUserPoolDomain',
                        Effect: 'Allow',
                        Resource: '*',
                    }],
                Version: '2012-10-17',
            },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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