"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const cognito_generated_1 = require("./cognito.generated");
/**
 * OAuth scopes that are allowed with this client.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-idp-settings.html
 */
class OAuthScope {
    constructor(scopeName) {
        this.scopeName = scopeName;
    }
    /**
     * Custom scope is one that you define for your own resource server in the Resource Servers.
     * The format is 'resource-server-identifier/scope'.
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-define-resource-servers.html
     */
    static custom(name) {
        return new OAuthScope(name);
    }
}
exports.OAuthScope = OAuthScope;
/**
 * Grants access to the 'phone_number' and 'phone_number_verified' claims.
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.PHONE = new OAuthScope('phone');
/**
 * Grants access to the 'email' and 'email_verified' claims.
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.EMAIL = new OAuthScope('email');
/**
 * Returns all user attributes in the ID token that are readable by the client
 */
OAuthScope.OPENID = new OAuthScope('openid');
/**
 * Grants access to all user attributes that are readable by the client
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.PROFILE = new OAuthScope('profile');
/**
 * Grants access to Amazon Cognito User Pool API operations that require access tokens,
 * such as UpdateUserAttributes and VerifyUserAttribute.
 */
OAuthScope.COGNITO_ADMIN = new OAuthScope('aws.cognito.signin.user.admin');
/**
 * Define a UserPool App Client
 */
class UserPoolClient extends core_1.Resource {
    /*
     * Note to implementers: Two CloudFormation return values Name and ClientSecret are part of the spec.
     * However, they have been explicity not implemented here. They are not documented in CloudFormation, and
     * CloudFormation returns the following the string when these two attributes are 'GetAtt' - "attribute not supported
     * at this time, please use the CLI or Console to retrieve this value".
     * Awaiting updates from CloudFormation.
     */
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        const resource = new cognito_generated_1.CfnUserPoolClient(this, 'Resource', {
            clientName: props.userPoolClientName,
            generateSecret: props.generateSecret,
            userPoolId: props.userPool.userPoolId,
            explicitAuthFlows: this.configureAuthFlows(props),
            allowedOAuthFlows: this.configureOAuthFlows(props.oAuth),
            allowedOAuthScopes: this.configureOAuthScopes(props.oAuth),
            callbackUrLs: (((_a = props.oAuth) === null || _a === void 0 ? void 0 : _a.callbackUrls) && ((_b = props.oAuth) === null || _b === void 0 ? void 0 : _b.callbackUrls.length) > 0) ? (_c = props.oAuth) === null || _c === void 0 ? void 0 : _c.callbackUrls : undefined,
            allowedOAuthFlowsUserPoolClient: props.oAuth ? true : undefined,
        });
        this.userPoolClientId = resource.ref;
        this._userPoolClientName = props.userPoolClientName;
    }
    /**
     * Import a user pool client given its id.
     */
    static fromUserPoolClientId(scope, id, userPoolClientId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.userPoolClientId = userPoolClientId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * The client name that was specified via the `userPoolClientName` property during initialization,
     * throws an error otherwise.
     */
    get userPoolClientName() {
        if (this._userPoolClientName === undefined) {
            throw new Error('userPoolClientName is available only if specified on the UserPoolClient during initialization');
        }
        return this._userPoolClientName;
    }
    configureAuthFlows(props) {
        var _a, _b, _c, _d, _e;
        const authFlows = [];
        if ((_a = props.authFlows) === null || _a === void 0 ? void 0 : _a.userPassword) {
            authFlows.push('ALLOW_USER_PASSWORD_AUTH');
        }
        if ((_b = props.authFlows) === null || _b === void 0 ? void 0 : _b.adminUserPassword) {
            authFlows.push('ALLOW_ADMIN_USER_PASSWORD_AUTH');
        }
        if ((_c = props.authFlows) === null || _c === void 0 ? void 0 : _c.custom) {
            authFlows.push('ALLOW_CUSTOM_AUTH');
        }
        if ((_d = props.authFlows) === null || _d === void 0 ? void 0 : _d.userSrp) {
            authFlows.push('ALLOW_USER_SRP_AUTH');
        }
        if ((_e = props.authFlows) === null || _e === void 0 ? void 0 : _e.refreshToken) {
            authFlows.push('ALLOW_REFRESH_TOKEN_AUTH');
        }
        if (authFlows.length === 0) {
            return undefined;
        }
        return authFlows;
    }
    configureOAuthFlows(oAuth) {
        if ((oAuth === null || oAuth === void 0 ? void 0 : oAuth.flows.authorizationCodeGrant) || (oAuth === null || oAuth === void 0 ? void 0 : oAuth.flows.implicitCodeGrant)) {
            if ((oAuth === null || oAuth === void 0 ? void 0 : oAuth.callbackUrls) === undefined || (oAuth === null || oAuth === void 0 ? void 0 : oAuth.callbackUrls.length) === 0) {
                throw new Error('callbackUrl must be specified when codeGrant or implicitGrant OAuth flows are enabled.');
            }
            if (oAuth === null || oAuth === void 0 ? void 0 : oAuth.flows.clientCredentials) {
                throw new Error('clientCredentials OAuth flow cannot be selected along with codeGrant or implicitGrant.');
            }
        }
        const oAuthFlows = [];
        if (oAuth === null || oAuth === void 0 ? void 0 : oAuth.flows.clientCredentials) {
            oAuthFlows.push('client_credentials');
        }
        if (oAuth === null || oAuth === void 0 ? void 0 : oAuth.flows.implicitCodeGrant) {
            oAuthFlows.push('implicit');
        }
        if (oAuth === null || oAuth === void 0 ? void 0 : oAuth.flows.authorizationCodeGrant) {
            oAuthFlows.push('code');
        }
        if (oAuthFlows.length === 0) {
            return undefined;
        }
        return oAuthFlows;
    }
    configureOAuthScopes(oAuth) {
        const oAuthScopes = new Set(oAuth === null || oAuth === void 0 ? void 0 : oAuth.scopes.map((x) => x.scopeName));
        const autoOpenIdScopes = [OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.PROFILE];
        if (autoOpenIdScopes.reduce((agg, s) => agg || oAuthScopes.has(s.scopeName), false)) {
            oAuthScopes.add(OAuthScope.OPENID.scopeName);
        }
        if (oAuthScopes.size > 0) {
            return Array.from(oAuthScopes);
        }
        return undefined;
    }
}
exports.UserPoolClient = UserPoolClient;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidXNlci1wb29sLWNsaWVudC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInVzZXItcG9vbC1jbGllbnQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSx3Q0FBK0Q7QUFDL0QsMkRBQXdEO0FBMEZ4RDs7O0dBR0c7QUFDSCxNQUFhLFVBQVU7SUErQ3JCLFlBQW9CLFNBQWlCO1FBQ25DLElBQUksQ0FBQyxTQUFTLEdBQUcsU0FBUyxDQUFDO0lBQzdCLENBQUM7SUFuQkQ7Ozs7T0FJRztJQUNJLE1BQU0sQ0FBQyxNQUFNLENBQUMsSUFBWTtRQUMvQixPQUFPLElBQUksVUFBVSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzlCLENBQUM7O0FBckNILGdDQWtEQztBQWpEQzs7O0dBR0c7QUFDb0IsZ0JBQUssR0FBRyxJQUFJLFVBQVUsQ0FBQyxPQUFPLENBQUMsQ0FBQztBQUV2RDs7O0dBR0c7QUFDb0IsZ0JBQUssR0FBRyxJQUFJLFVBQVUsQ0FBQyxPQUFPLENBQUMsQ0FBQztBQUV2RDs7R0FFRztBQUNvQixpQkFBTSxHQUFHLElBQUksVUFBVSxDQUFDLFFBQVEsQ0FBQyxDQUFDO0FBRXpEOzs7R0FHRztBQUNvQixrQkFBTyxHQUFHLElBQUksVUFBVSxDQUFDLFNBQVMsQ0FBQyxDQUFDO0FBRTNEOzs7R0FHRztBQUNvQix3QkFBYSxHQUFHLElBQUksVUFBVSxDQUFDLCtCQUErQixDQUFDLENBQUM7QUEyRXpGOztHQUVHO0FBQ0gsTUFBYSxjQUFlLFNBQVEsZUFBUTtJQWUxQzs7Ozs7O09BTUc7SUFFSCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQTBCOztRQUNsRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLE1BQU0sUUFBUSxHQUFHLElBQUkscUNBQWlCLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUN2RCxVQUFVLEVBQUUsS0FBSyxDQUFDLGtCQUFrQjtZQUNwQyxjQUFjLEVBQUUsS0FBSyxDQUFDLGNBQWM7WUFDcEMsVUFBVSxFQUFFLEtBQUssQ0FBQyxRQUFRLENBQUMsVUFBVTtZQUNyQyxpQkFBaUIsRUFBRSxJQUFJLENBQUMsa0JBQWtCLENBQUMsS0FBSyxDQUFDO1lBQ2pELGlCQUFpQixFQUFFLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDO1lBQ3hELGtCQUFrQixFQUFFLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDO1lBQzFELFlBQVksRUFBRSxDQUFDLE9BQUEsS0FBSyxDQUFDLEtBQUssMENBQUUsWUFBWSxLQUFJLE9BQUEsS0FBSyxDQUFDLEtBQUssMENBQUUsWUFBWSxDQUFDLE1BQU0sSUFBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLE9BQUMsS0FBSyxDQUFDLEtBQUssMENBQUUsWUFBWSxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQ3pILCtCQUErQixFQUFFLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsU0FBUztTQUNoRSxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsUUFBUSxDQUFDLEdBQUcsQ0FBQztRQUNyQyxJQUFJLENBQUMsbUJBQW1CLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixDQUFDO0lBQ3RELENBQUM7SUF0Q0Q7O09BRUc7SUFDSSxNQUFNLENBQUMsb0JBQW9CLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsZ0JBQXdCO1FBQ3ZGLE1BQU0sTUFBTyxTQUFRLGVBQVE7WUFBN0I7O2dCQUNrQixxQkFBZ0IsR0FBRyxnQkFBZ0IsQ0FBQztZQUN0RCxDQUFDO1NBQUE7UUFFRCxPQUFPLElBQUksTUFBTSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztJQUMvQixDQUFDO0lBK0JEOzs7T0FHRztJQUNILElBQVcsa0JBQWtCO1FBQzNCLElBQUksSUFBSSxDQUFDLG1CQUFtQixLQUFLLFNBQVMsRUFBRTtZQUMxQyxNQUFNLElBQUksS0FBSyxDQUFDLCtGQUErRixDQUFDLENBQUM7U0FDbEg7UUFDRCxPQUFPLElBQUksQ0FBQyxtQkFBbUIsQ0FBQztJQUNsQyxDQUFDO0lBRU8sa0JBQWtCLENBQUMsS0FBMEI7O1FBQ25ELE1BQU0sU0FBUyxHQUFhLEVBQUUsQ0FBQztRQUMvQixVQUFJLEtBQUssQ0FBQyxTQUFTLDBDQUFFLFlBQVksRUFBRTtZQUFFLFNBQVMsQ0FBQyxJQUFJLENBQUMsMEJBQTBCLENBQUMsQ0FBQztTQUFFO1FBQ2xGLFVBQUksS0FBSyxDQUFDLFNBQVMsMENBQUUsaUJBQWlCLEVBQUU7WUFBRSxTQUFTLENBQUMsSUFBSSxDQUFDLGdDQUFnQyxDQUFDLENBQUM7U0FBRTtRQUM3RixVQUFJLEtBQUssQ0FBQyxTQUFTLDBDQUFFLE1BQU0sRUFBRTtZQUFFLFNBQVMsQ0FBQyxJQUFJLENBQUMsbUJBQW1CLENBQUMsQ0FBQztTQUFFO1FBQ3JFLFVBQUksS0FBSyxDQUFDLFNBQVMsMENBQUUsT0FBTyxFQUFFO1lBQUUsU0FBUyxDQUFDLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1NBQUU7UUFDeEUsVUFBSSxLQUFLLENBQUMsU0FBUywwQ0FBRSxZQUFZLEVBQUU7WUFBRSxTQUFTLENBQUMsSUFBSSxDQUFDLDBCQUEwQixDQUFDLENBQUM7U0FBRTtRQUVsRixJQUFJLFNBQVMsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQzFCLE9BQU8sU0FBUyxDQUFDO1NBQ2xCO1FBQ0QsT0FBTyxTQUFTLENBQUM7SUFDbkIsQ0FBQztJQUVPLG1CQUFtQixDQUFDLEtBQXFCO1FBQy9DLElBQUksQ0FBQSxLQUFLLGFBQUwsS0FBSyx1QkFBTCxLQUFLLENBQUUsS0FBSyxDQUFDLHNCQUFzQixNQUFJLEtBQUssYUFBTCxLQUFLLHVCQUFMLEtBQUssQ0FBRSxLQUFLLENBQUMsaUJBQWlCLENBQUEsRUFBRTtZQUN6RSxJQUFJLENBQUEsS0FBSyxhQUFMLEtBQUssdUJBQUwsS0FBSyxDQUFFLFlBQVksTUFBSyxTQUFTLElBQUksQ0FBQSxLQUFLLGFBQUwsS0FBSyx1QkFBTCxLQUFLLENBQUUsWUFBWSxDQUFDLE1BQU0sTUFBSyxDQUFDLEVBQUU7Z0JBQ3pFLE1BQU0sSUFBSSxLQUFLLENBQUMsd0ZBQXdGLENBQUMsQ0FBQzthQUMzRztZQUNELElBQUksS0FBSyxhQUFMLEtBQUssdUJBQUwsS0FBSyxDQUFFLEtBQUssQ0FBQyxpQkFBaUIsRUFBRTtnQkFDbEMsTUFBTSxJQUFJLEtBQUssQ0FBQyx3RkFBd0YsQ0FBQyxDQUFDO2FBQzNHO1NBQ0Y7UUFFRCxNQUFNLFVBQVUsR0FBYSxFQUFFLENBQUM7UUFDaEMsSUFBSSxLQUFLLGFBQUwsS0FBSyx1QkFBTCxLQUFLLENBQUUsS0FBSyxDQUFDLGlCQUFpQixFQUFFO1lBQUUsVUFBVSxDQUFDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO1NBQUU7UUFDOUUsSUFBSSxLQUFLLGFBQUwsS0FBSyx1QkFBTCxLQUFLLENBQUUsS0FBSyxDQUFDLGlCQUFpQixFQUFFO1lBQUUsVUFBVSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQztTQUFFO1FBQ3BFLElBQUksS0FBSyxhQUFMLEtBQUssdUJBQUwsS0FBSyxDQUFFLEtBQUssQ0FBQyxzQkFBc0IsRUFBRTtZQUFFLFVBQVUsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7U0FBRTtRQUVyRSxJQUFJLFVBQVUsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQzNCLE9BQU8sU0FBUyxDQUFDO1NBQ2xCO1FBQ0QsT0FBTyxVQUFVLENBQUM7SUFDcEIsQ0FBQztJQUVPLG9CQUFvQixDQUFDLEtBQXFCO1FBQ2hELE1BQU0sV0FBVyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssYUFBTCxLQUFLLHVCQUFMLEtBQUssQ0FBRSxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUyxFQUFFLENBQUM7UUFDbkUsTUFBTSxnQkFBZ0IsR0FBRyxDQUFFLFVBQVUsQ0FBQyxLQUFLLEVBQUUsVUFBVSxDQUFDLEtBQUssRUFBRSxVQUFVLENBQUMsT0FBTyxDQUFFLENBQUM7UUFDcEYsSUFBSSxnQkFBZ0IsQ0FBQyxNQUFNLENBQUMsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxHQUFHLElBQUksV0FBVyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLEVBQUUsS0FBSyxDQUFDLEVBQUU7WUFDbkYsV0FBVyxDQUFDLEdBQUcsQ0FBQyxVQUFVLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1NBQzlDO1FBQ0QsSUFBSSxXQUFXLENBQUMsSUFBSSxHQUFHLENBQUMsRUFBRTtZQUN4QixPQUFPLEtBQUssQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLENBQUM7U0FDaEM7UUFDRCxPQUFPLFNBQVMsQ0FBQztJQUNuQixDQUFDO0NBQ0Y7QUFsR0Qsd0NBa0dDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQ29uc3RydWN0LCBJUmVzb3VyY2UsIFJlc291cmNlIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBDZm5Vc2VyUG9vbENsaWVudCB9IGZyb20gJy4vY29nbml0by5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgSVVzZXJQb29sIH0gZnJvbSAnLi91c2VyLXBvb2wnO1xuXG4vKipcbiAqIFR5cGVzIG9mIGF1dGhlbnRpY2F0aW9uIGZsb3dcbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZ25pdG8vbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2FtYXpvbi1jb2duaXRvLXVzZXItcG9vbHMtYXV0aGVudGljYXRpb24tZmxvdy5odG1sXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQXV0aEZsb3cge1xuICAvKipcbiAgICogRW5hYmxlIGFkbWluIGJhc2VkIHVzZXIgcGFzc3dvcmQgYXV0aGVudGljYXRpb24gZmxvd1xuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgYWRtaW5Vc2VyUGFzc3dvcmQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBFbmFibGUgY3VzdG9tIGF1dGhlbnRpY2F0aW9uIGZsb3dcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGN1c3RvbT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEVuYWJsZSBhdXRoIHVzaW5nIHVzZXJuYW1lICYgcGFzc3dvcmRcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHVzZXJQYXNzd29yZD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEVuYWJsZSBTUlAgYmFzZWQgYXV0aGVudGljYXRpb25cbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHVzZXJTcnA/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBFbmFibGUgYXV0aGZsb3cgdG8gcmVmcmVzaCB0b2tlbnNcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHJlZnJlc2hUb2tlbj86IGJvb2xlYW47XG59XG5cbi8qKlxuICogT0F1dGggc2V0dGluZ3MgdG8gY29uZmlndXJlIHRoZSBpbnRlcmFjdGlvbiBiZXR3ZWVuIHRoZSBhcHAgYW5kIHRoaXMgY2xpZW50LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIE9BdXRoU2V0dGluZ3Mge1xuXG4gIC8qKlxuICAgKiBPQXV0aCBmbG93cyB0aGF0IGFyZSBhbGxvd2VkIHdpdGggdGhpcyBjbGllbnQuXG4gICAqIEBzZWUgLSB0aGUgJ0FsbG93ZWQgT0F1dGggRmxvd3MnIHNlY3Rpb24gYXQgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZ25pdG8vbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2NvZ25pdG8tdXNlci1wb29scy1hcHAtaWRwLXNldHRpbmdzLmh0bWxcbiAgICogQGRlZmF1bHQgLSBhbGwgT0F1dGggZmxvd3MgZGlzYWJsZWRcbiAgICovXG4gIHJlYWRvbmx5IGZsb3dzOiBPQXV0aEZsb3dzO1xuXG4gIC8qKlxuICAgKiBMaXN0IG9mIGFsbG93ZWQgcmVkaXJlY3QgVVJMcyBmb3IgdGhlIGlkZW50aXR5IHByb3ZpZGVycy5cbiAgICogQGRlZmF1bHQgLSBubyBjYWxsYmFjayBVUkxzXG4gICAqL1xuICByZWFkb25seSBjYWxsYmFja1VybHM/OiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogT0F1dGggc2NvcGVzIHRoYXQgYXJlIGFsbG93ZWQgd2l0aCB0aGlzIGNsaWVudC5cbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29nbml0by9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvY29nbml0by11c2VyLXBvb2xzLWFwcC1pZHAtc2V0dGluZ3MuaHRtbFxuICAgKiBAZGVmYXVsdCAtIG5vIE9BdXRoIHNjb3BlcyBhcmUgY29uZmlndXJlZC5cbiAgICovXG4gIHJlYWRvbmx5IHNjb3BlczogT0F1dGhTY29wZVtdO1xufVxuXG4vKipcbiAqIFR5cGVzIG9mIE9BdXRoIGdyYW50IGZsb3dzXG4gKiBAc2VlIC0gdGhlICdBbGxvd2VkIE9BdXRoIEZsb3dzJyBzZWN0aW9uIGF0IGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2duaXRvL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9jb2duaXRvLXVzZXItcG9vbHMtYXBwLWlkcC1zZXR0aW5ncy5odG1sXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgT0F1dGhGbG93cyB7XG4gIC8qKlxuICAgKiBJbml0aWF0ZSBhbiBhdXRob3JpemF0aW9uIGNvZGUgZ3JhbnQgZmxvdywgd2hpY2ggcHJvdmlkZXMgYW4gYXV0aG9yaXphdGlvbiBjb2RlIGFzIHRoZSByZXNwb25zZS5cbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGF1dGhvcml6YXRpb25Db2RlR3JhbnQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgY2xpZW50IHNob3VsZCBnZXQgdGhlIGFjY2VzcyB0b2tlbiBhbmQgSUQgdG9rZW4gZGlyZWN0bHkuXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBpbXBsaWNpdENvZGVHcmFudD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIENsaWVudCBzaG91bGQgZ2V0IHRoZSBhY2Nlc3MgdG9rZW4gYW5kIElEIHRva2VuIGZyb20gdGhlIHRva2VuIGVuZHBvaW50XG4gICAqIHVzaW5nIGEgY29tYmluYXRpb24gb2YgY2xpZW50IGFuZCBjbGllbnRfc2VjcmV0LlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgY2xpZW50Q3JlZGVudGlhbHM/OiBib29sZWFuO1xufVxuXG4vKipcbiAqIE9BdXRoIHNjb3BlcyB0aGF0IGFyZSBhbGxvd2VkIHdpdGggdGhpcyBjbGllbnQuXG4gKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2duaXRvL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9jb2duaXRvLXVzZXItcG9vbHMtYXBwLWlkcC1zZXR0aW5ncy5odG1sXG4gKi9cbmV4cG9ydCBjbGFzcyBPQXV0aFNjb3BlIHtcbiAgLyoqXG4gICAqIEdyYW50cyBhY2Nlc3MgdG8gdGhlICdwaG9uZV9udW1iZXInIGFuZCAncGhvbmVfbnVtYmVyX3ZlcmlmaWVkJyBjbGFpbXMuXG4gICAqIEF1dG9tYXRpY2FsbHkgaW5jbHVkZXMgYWNjZXNzIHRvIGBPQXV0aFNjb3BlLk9QRU5JRGAuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFBIT05FID0gbmV3IE9BdXRoU2NvcGUoJ3Bob25lJyk7XG5cbiAgLyoqXG4gICAqIEdyYW50cyBhY2Nlc3MgdG8gdGhlICdlbWFpbCcgYW5kICdlbWFpbF92ZXJpZmllZCcgY2xhaW1zLlxuICAgKiBBdXRvbWF0aWNhbGx5IGluY2x1ZGVzIGFjY2VzcyB0byBgT0F1dGhTY29wZS5PUEVOSURgLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBFTUFJTCA9IG5ldyBPQXV0aFNjb3BlKCdlbWFpbCcpO1xuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGFsbCB1c2VyIGF0dHJpYnV0ZXMgaW4gdGhlIElEIHRva2VuIHRoYXQgYXJlIHJlYWRhYmxlIGJ5IHRoZSBjbGllbnRcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgT1BFTklEID0gbmV3IE9BdXRoU2NvcGUoJ29wZW5pZCcpO1xuXG4gIC8qKlxuICAgKiBHcmFudHMgYWNjZXNzIHRvIGFsbCB1c2VyIGF0dHJpYnV0ZXMgdGhhdCBhcmUgcmVhZGFibGUgYnkgdGhlIGNsaWVudFxuICAgKiBBdXRvbWF0aWNhbGx5IGluY2x1ZGVzIGFjY2VzcyB0byBgT0F1dGhTY29wZS5PUEVOSURgLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBQUk9GSUxFID0gbmV3IE9BdXRoU2NvcGUoJ3Byb2ZpbGUnKTtcblxuICAvKipcbiAgICogR3JhbnRzIGFjY2VzcyB0byBBbWF6b24gQ29nbml0byBVc2VyIFBvb2wgQVBJIG9wZXJhdGlvbnMgdGhhdCByZXF1aXJlIGFjY2VzcyB0b2tlbnMsXG4gICAqIHN1Y2ggYXMgVXBkYXRlVXNlckF0dHJpYnV0ZXMgYW5kIFZlcmlmeVVzZXJBdHRyaWJ1dGUuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IENPR05JVE9fQURNSU4gPSBuZXcgT0F1dGhTY29wZSgnYXdzLmNvZ25pdG8uc2lnbmluLnVzZXIuYWRtaW4nKTtcblxuICAvKipcbiAgICogQ3VzdG9tIHNjb3BlIGlzIG9uZSB0aGF0IHlvdSBkZWZpbmUgZm9yIHlvdXIgb3duIHJlc291cmNlIHNlcnZlciBpbiB0aGUgUmVzb3VyY2UgU2VydmVycy5cbiAgICogVGhlIGZvcm1hdCBpcyAncmVzb3VyY2Utc2VydmVyLWlkZW50aWZpZXIvc2NvcGUnLlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2duaXRvL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9jb2duaXRvLXVzZXItcG9vbHMtZGVmaW5lLXJlc291cmNlLXNlcnZlcnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBjdXN0b20obmFtZTogc3RyaW5nKSB7XG4gICAgcmV0dXJuIG5ldyBPQXV0aFNjb3BlKG5hbWUpO1xuICB9XG5cbiAgLy8gdHNsaW50OmRpc2FibGU6bWF4LWxpbmUtbGVuZ3RoXG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGlzIHNjb3BlIGFzIHJlY29nbml6ZWQgYnkgQ2xvdWRGb3JtYXRpb24uXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXJlc291cmNlLWNvZ25pdG8tdXNlcnBvb2xjbGllbnQuaHRtbCNjZm4tY29nbml0by11c2VycG9vbGNsaWVudC1hbGxvd2Vkb2F1dGhzY29wZXNcbiAgICovXG4gIC8vIHRzbGludDplbmFibGU6bWF4LWxpbmUtbGVuZ3RoXG4gIHB1YmxpYyByZWFkb25seSBzY29wZU5hbWU6IHN0cmluZztcblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKHNjb3BlTmFtZTogc3RyaW5nKSB7XG4gICAgdGhpcy5zY29wZU5hbWUgPSBzY29wZU5hbWU7XG4gIH1cbn1cblxuLyoqXG4gKiBPcHRpb25zIHRvIGNyZWF0ZSBhIFVzZXJQb29sQ2xpZW50XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVXNlclBvb2xDbGllbnRPcHRpb25zIHtcbiAgLyoqXG4gICAqIE5hbWUgb2YgdGhlIGFwcGxpY2F0aW9uIGNsaWVudFxuICAgKiBAZGVmYXVsdCAtIGNsb3VkZm9ybWF0aW9uIGdlbmVyYXRlZCBuYW1lXG4gICAqL1xuICByZWFkb25seSB1c2VyUG9vbENsaWVudE5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgdG8gZ2VuZXJhdGUgYSBjbGllbnQgc2VjcmV0XG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBnZW5lcmF0ZVNlY3JldD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBzZXQgb2YgT0F1dGggYXV0aGVudGljYXRpb24gZmxvd3MgdG8gZW5hYmxlIG9uIHRoZSBjbGllbnRcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29nbml0by9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvYW1hem9uLWNvZ25pdG8tdXNlci1wb29scy1hdXRoZW50aWNhdGlvbi1mbG93Lmh0bWxcbiAgICogQGRlZmF1bHQgLSBhbGwgYXV0aCBmbG93cyBkaXNhYmxlZFxuICAgKi9cbiAgcmVhZG9ubHkgYXV0aEZsb3dzPzogQXV0aEZsb3c7XG5cbiAgLyoqXG4gICAqIE9BdXRoIHNldHRpbmdzIGZvciB0aGlzIHRvIGNsaWVudCB0byBpbnRlcmFjdCB3aXRoIHRoZSBhcHAuXG4gICAqIEBkZWZhdWx0IC0gc2VlIGRlZmF1bHRzIGluIGBPQXV0aFNldHRpbmdzYFxuICAgKi9cbiAgcmVhZG9ubHkgb0F1dGg/OiBPQXV0aFNldHRpbmdzO1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIHRoZSBVc2VyUG9vbENsaWVudCBjb25zdHJ1Y3RcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBVc2VyUG9vbENsaWVudFByb3BzIGV4dGVuZHMgVXNlclBvb2xDbGllbnRPcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSBVc2VyUG9vbCByZXNvdXJjZSB0aGlzIGNsaWVudCB3aWxsIGhhdmUgYWNjZXNzIHRvXG4gICAqL1xuICByZWFkb25seSB1c2VyUG9vbDogSVVzZXJQb29sO1xufVxuXG4vKipcbiAqIFJlcHJlc2VudHMgYSBDb2duaXRvIHVzZXIgcG9vbCBjbGllbnQuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSVVzZXJQb29sQ2xpZW50IGV4dGVuZHMgSVJlc291cmNlIHtcbiAgLyoqXG4gICAqIE5hbWUgb2YgdGhlIGFwcGxpY2F0aW9uIGNsaWVudFxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICByZWFkb25seSB1c2VyUG9vbENsaWVudElkOiBzdHJpbmc7XG59XG5cbi8qKlxuICogRGVmaW5lIGEgVXNlclBvb2wgQXBwIENsaWVudFxuICovXG5leHBvcnQgY2xhc3MgVXNlclBvb2xDbGllbnQgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElVc2VyUG9vbENsaWVudCB7XG4gIC8qKlxuICAgKiBJbXBvcnQgYSB1c2VyIHBvb2wgY2xpZW50IGdpdmVuIGl0cyBpZC5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbVVzZXJQb29sQ2xpZW50SWQoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgdXNlclBvb2xDbGllbnRJZDogc3RyaW5nKTogSVVzZXJQb29sQ2xpZW50IHtcbiAgICBjbGFzcyBJbXBvcnQgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElVc2VyUG9vbENsaWVudCB7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgdXNlclBvb2xDbGllbnRJZCA9IHVzZXJQb29sQ2xpZW50SWQ7XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBJbXBvcnQoc2NvcGUsIGlkKTtcbiAgfVxuXG4gIHB1YmxpYyByZWFkb25seSB1c2VyUG9vbENsaWVudElkOiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgX3VzZXJQb29sQ2xpZW50TmFtZT86IHN0cmluZztcblxuICAvKlxuICAgKiBOb3RlIHRvIGltcGxlbWVudGVyczogVHdvIENsb3VkRm9ybWF0aW9uIHJldHVybiB2YWx1ZXMgTmFtZSBhbmQgQ2xpZW50U2VjcmV0IGFyZSBwYXJ0IG9mIHRoZSBzcGVjLlxuICAgKiBIb3dldmVyLCB0aGV5IGhhdmUgYmVlbiBleHBsaWNpdHkgbm90IGltcGxlbWVudGVkIGhlcmUuIFRoZXkgYXJlIG5vdCBkb2N1bWVudGVkIGluIENsb3VkRm9ybWF0aW9uLCBhbmRcbiAgICogQ2xvdWRGb3JtYXRpb24gcmV0dXJucyB0aGUgZm9sbG93aW5nIHRoZSBzdHJpbmcgd2hlbiB0aGVzZSB0d28gYXR0cmlidXRlcyBhcmUgJ0dldEF0dCcgLSBcImF0dHJpYnV0ZSBub3Qgc3VwcG9ydGVkXG4gICAqIGF0IHRoaXMgdGltZSwgcGxlYXNlIHVzZSB0aGUgQ0xJIG9yIENvbnNvbGUgdG8gcmV0cmlldmUgdGhpcyB2YWx1ZVwiLlxuICAgKiBBd2FpdGluZyB1cGRhdGVzIGZyb20gQ2xvdWRGb3JtYXRpb24uXG4gICAqL1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBVc2VyUG9vbENsaWVudFByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGNvbnN0IHJlc291cmNlID0gbmV3IENmblVzZXJQb29sQ2xpZW50KHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIGNsaWVudE5hbWU6IHByb3BzLnVzZXJQb29sQ2xpZW50TmFtZSxcbiAgICAgIGdlbmVyYXRlU2VjcmV0OiBwcm9wcy5nZW5lcmF0ZVNlY3JldCxcbiAgICAgIHVzZXJQb29sSWQ6IHByb3BzLnVzZXJQb29sLnVzZXJQb29sSWQsXG4gICAgICBleHBsaWNpdEF1dGhGbG93czogdGhpcy5jb25maWd1cmVBdXRoRmxvd3MocHJvcHMpLFxuICAgICAgYWxsb3dlZE9BdXRoRmxvd3M6IHRoaXMuY29uZmlndXJlT0F1dGhGbG93cyhwcm9wcy5vQXV0aCksXG4gICAgICBhbGxvd2VkT0F1dGhTY29wZXM6IHRoaXMuY29uZmlndXJlT0F1dGhTY29wZXMocHJvcHMub0F1dGgpLFxuICAgICAgY2FsbGJhY2tVckxzOiAocHJvcHMub0F1dGg/LmNhbGxiYWNrVXJscyAmJiBwcm9wcy5vQXV0aD8uY2FsbGJhY2tVcmxzLmxlbmd0aCA+IDApID8gcHJvcHMub0F1dGg/LmNhbGxiYWNrVXJscyA6IHVuZGVmaW5lZCxcbiAgICAgIGFsbG93ZWRPQXV0aEZsb3dzVXNlclBvb2xDbGllbnQ6IHByb3BzLm9BdXRoID8gdHJ1ZSA6IHVuZGVmaW5lZCxcbiAgICB9KTtcblxuICAgIHRoaXMudXNlclBvb2xDbGllbnRJZCA9IHJlc291cmNlLnJlZjtcbiAgICB0aGlzLl91c2VyUG9vbENsaWVudE5hbWUgPSBwcm9wcy51c2VyUG9vbENsaWVudE5hbWU7XG4gIH1cblxuICAvKipcbiAgICogVGhlIGNsaWVudCBuYW1lIHRoYXQgd2FzIHNwZWNpZmllZCB2aWEgdGhlIGB1c2VyUG9vbENsaWVudE5hbWVgIHByb3BlcnR5IGR1cmluZyBpbml0aWFsaXphdGlvbixcbiAgICogdGhyb3dzIGFuIGVycm9yIG90aGVyd2lzZS5cbiAgICovXG4gIHB1YmxpYyBnZXQgdXNlclBvb2xDbGllbnROYW1lKCk6IHN0cmluZyB7XG4gICAgaWYgKHRoaXMuX3VzZXJQb29sQ2xpZW50TmFtZSA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ3VzZXJQb29sQ2xpZW50TmFtZSBpcyBhdmFpbGFibGUgb25seSBpZiBzcGVjaWZpZWQgb24gdGhlIFVzZXJQb29sQ2xpZW50IGR1cmluZyBpbml0aWFsaXphdGlvbicpO1xuICAgIH1cbiAgICByZXR1cm4gdGhpcy5fdXNlclBvb2xDbGllbnROYW1lO1xuICB9XG5cbiAgcHJpdmF0ZSBjb25maWd1cmVBdXRoRmxvd3MocHJvcHM6IFVzZXJQb29sQ2xpZW50UHJvcHMpOiBzdHJpbmdbXSB8IHVuZGVmaW5lZCB7XG4gICAgY29uc3QgYXV0aEZsb3dzOiBzdHJpbmdbXSA9IFtdO1xuICAgIGlmIChwcm9wcy5hdXRoRmxvd3M/LnVzZXJQYXNzd29yZCkgeyBhdXRoRmxvd3MucHVzaCgnQUxMT1dfVVNFUl9QQVNTV09SRF9BVVRIJyk7IH1cbiAgICBpZiAocHJvcHMuYXV0aEZsb3dzPy5hZG1pblVzZXJQYXNzd29yZCkgeyBhdXRoRmxvd3MucHVzaCgnQUxMT1dfQURNSU5fVVNFUl9QQVNTV09SRF9BVVRIJyk7IH1cbiAgICBpZiAocHJvcHMuYXV0aEZsb3dzPy5jdXN0b20pIHsgYXV0aEZsb3dzLnB1c2goJ0FMTE9XX0NVU1RPTV9BVVRIJyk7IH1cbiAgICBpZiAocHJvcHMuYXV0aEZsb3dzPy51c2VyU3JwKSB7IGF1dGhGbG93cy5wdXNoKCdBTExPV19VU0VSX1NSUF9BVVRIJyk7IH1cbiAgICBpZiAocHJvcHMuYXV0aEZsb3dzPy5yZWZyZXNoVG9rZW4pIHsgYXV0aEZsb3dzLnB1c2goJ0FMTE9XX1JFRlJFU0hfVE9LRU5fQVVUSCcpOyB9XG5cbiAgICBpZiAoYXV0aEZsb3dzLmxlbmd0aCA9PT0gMCkge1xuICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICB9XG4gICAgcmV0dXJuIGF1dGhGbG93cztcbiAgfVxuXG4gIHByaXZhdGUgY29uZmlndXJlT0F1dGhGbG93cyhvQXV0aD86IE9BdXRoU2V0dGluZ3MpOiBzdHJpbmdbXSB8IHVuZGVmaW5lZCB7XG4gICAgaWYgKG9BdXRoPy5mbG93cy5hdXRob3JpemF0aW9uQ29kZUdyYW50IHx8IG9BdXRoPy5mbG93cy5pbXBsaWNpdENvZGVHcmFudCkge1xuICAgICAgaWYgKG9BdXRoPy5jYWxsYmFja1VybHMgPT09IHVuZGVmaW5lZCB8fCBvQXV0aD8uY2FsbGJhY2tVcmxzLmxlbmd0aCA9PT0gMCkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ2NhbGxiYWNrVXJsIG11c3QgYmUgc3BlY2lmaWVkIHdoZW4gY29kZUdyYW50IG9yIGltcGxpY2l0R3JhbnQgT0F1dGggZmxvd3MgYXJlIGVuYWJsZWQuJyk7XG4gICAgICB9XG4gICAgICBpZiAob0F1dGg/LmZsb3dzLmNsaWVudENyZWRlbnRpYWxzKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignY2xpZW50Q3JlZGVudGlhbHMgT0F1dGggZmxvdyBjYW5ub3QgYmUgc2VsZWN0ZWQgYWxvbmcgd2l0aCBjb2RlR3JhbnQgb3IgaW1wbGljaXRHcmFudC4nKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBjb25zdCBvQXV0aEZsb3dzOiBzdHJpbmdbXSA9IFtdO1xuICAgIGlmIChvQXV0aD8uZmxvd3MuY2xpZW50Q3JlZGVudGlhbHMpIHsgb0F1dGhGbG93cy5wdXNoKCdjbGllbnRfY3JlZGVudGlhbHMnKTsgfVxuICAgIGlmIChvQXV0aD8uZmxvd3MuaW1wbGljaXRDb2RlR3JhbnQpIHsgb0F1dGhGbG93cy5wdXNoKCdpbXBsaWNpdCcpOyB9XG4gICAgaWYgKG9BdXRoPy5mbG93cy5hdXRob3JpemF0aW9uQ29kZUdyYW50KSB7IG9BdXRoRmxvd3MucHVzaCgnY29kZScpOyB9XG5cbiAgICBpZiAob0F1dGhGbG93cy5sZW5ndGggPT09IDApIHtcbiAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgfVxuICAgIHJldHVybiBvQXV0aEZsb3dzO1xuICB9XG5cbiAgcHJpdmF0ZSBjb25maWd1cmVPQXV0aFNjb3BlcyhvQXV0aD86IE9BdXRoU2V0dGluZ3MpOiBzdHJpbmdbXSB8IHVuZGVmaW5lZCB7XG4gICAgY29uc3Qgb0F1dGhTY29wZXMgPSBuZXcgU2V0KG9BdXRoPy5zY29wZXMubWFwKCh4KSA9PiB4LnNjb3BlTmFtZSkpO1xuICAgIGNvbnN0IGF1dG9PcGVuSWRTY29wZXMgPSBbIE9BdXRoU2NvcGUuUEhPTkUsIE9BdXRoU2NvcGUuRU1BSUwsIE9BdXRoU2NvcGUuUFJPRklMRSBdO1xuICAgIGlmIChhdXRvT3BlbklkU2NvcGVzLnJlZHVjZSgoYWdnLCBzKSA9PiBhZ2cgfHwgb0F1dGhTY29wZXMuaGFzKHMuc2NvcGVOYW1lKSwgZmFsc2UpKSB7XG4gICAgICBvQXV0aFNjb3Blcy5hZGQoT0F1dGhTY29wZS5PUEVOSUQuc2NvcGVOYW1lKTtcbiAgICB9XG4gICAgaWYgKG9BdXRoU2NvcGVzLnNpemUgPiAwKSB7XG4gICAgICByZXR1cm4gQXJyYXkuZnJvbShvQXV0aFNjb3Blcyk7XG4gICAgfVxuICAgIHJldHVybiB1bmRlZmluZWQ7XG4gIH1cbn1cbiJdfQ==