__author__ = 'cltanuki'
from django import forms
from django.contrib.auth.forms import ReadOnlyPasswordHashField

from .models import CorpUser, CorpUnit


class UserCreationForm(forms.ModelForm):
    """A form for creating new users. Includes all the required
    fields, plus a repeated password."""
    password1 = forms.CharField(label='Password', widget=forms.PasswordInput)
    password2 = forms.CharField(label='Password confirmation', widget=forms.PasswordInput)

    class Meta:
        model = CorpUser
        fields = ('email', 'date_of_birth', 'username')

    def clean_password2(self):
        # Check that the two password entries match
        password1 = self.cleaned_data.get("password1")
        password2 = self.cleaned_data.get("password2")
        if password1 and password2 and password1 != password2:
            raise forms.ValidationError("Passwords don't match")
        return password2

    def save(self, commit=True):
        user = super(UserCreationForm, self).save(commit=False)
        user.set_password(self.cleaned_data["password1"])
        if commit:
            user.save()
        return user


class UserChangeForm(forms.ModelForm):
    """A form for updating users. Includes all the fields on
    the user, but replaces the password field with admin's
    password hash display field.
    """
    password = ReadOnlyPasswordHashField()

    class Meta:
        model = CorpUser
        exclude = ['date_joined', 'is_active']

    def clean_password(self):
        # Regardless of what the user provides, return the initial value.
        # This is done here, rather than on the field, because the
        # field does not have access to the initial value
        return self.initial["password"]


class SignInForm(forms.Form):
    email = forms.EmailField(widget=forms.TextInput(attrs={'humanReadable': 'E-mail'}), label='E-mail')
    password = forms.CharField(min_length=6, max_length=32,
                               widget=forms.PasswordInput(attrs={'humanReadable': 'Пароль'}), label='Пароль')

    def clean(self):
        if not self._errors:
            cleaned_data = super(SignInForm, self).clean()
            email = cleaned_data.get('email')
            password = cleaned_data.get('password')
            try:
                user = CorpUser.objects.get(email=email)
                if not user.check_password(password):
                    raise forms.ValidationError(u'Неверное сочетание e-mail \ Пароль.')
                elif not user.is_active:
                    raise forms.ValidationError(u'Пользователь с таким e-mail заблокирован.')
            except CorpUser.DoesNotExist:
                raise forms.ValidationError(u'Пользователь с таким e-mail не существует.')
            return cleaned_data


class UnitCreateForm(forms.ModelForm):

    class Meta:
        model = CorpUnit
        fields = '__all__'
