# PyPI Automated Publishing Setup Guide

This document will guide you through setting up the automated PyPI publishing workflow for VibeSurf.

## 🔑 Step 1: Get PyPI API Token

1. Log in to your [PyPI account](https://pypi.org/)
2. Go to Account Settings → API tokens
3. Click "Add API token"
4. Select "Entire account" or create a token for a specific project
5. Copy the generated token (format: `pypi-...`)

## 🔧 Step 2: Configure GitHub Secrets

In your GitHub repository:

1. Go to Settings → Secrets and variables → Actions
2. Click "New repository secret"
3. Create a secret named `PYPI_API_TOKEN`
4. Paste your PyPI API token into the value field

## 🚀 Step 3: Create Release to Trigger Automated Publishing

### Method 1: Via GitHub Web Interface

1. On your GitHub repository page, click "Releases" on the right side
2. Click "Create a new release"
3. Create a new tag (e.g., `v0.1.0`)
4. Fill in the Release title and description
5. Click "Publish release"

### Method 2: Via Command Line

```bash
# Make sure you're on the main/master branch
git checkout main
git pull origin main

# Create and push tag
git tag v0.1.0
git push origin v0.1.0

# Or use annotated tag
git tag -a v0.1.0 -m "Release version 0.1.0"
git push origin v0.1.0
```

Then create a release from this tag on GitHub.

## 📦 Version Management

This project uses `setuptools-scm` for dynamic version management:

- Version numbers are automatically generated from Git tags
- Development versions automatically get `+dev` suffix
- No need to manually update version numbers

### Version Number Convention

Recommended to use semantic versioning:
- `v1.0.0` - Major version
- `v1.1.0` - Feature update
- `v1.1.1` - Bug fix
- `v1.2.0-alpha.1` - Pre-release version

## 🔄 Automated Workflow Explanation

When you create a release, GitHub Actions will automatically:

1. **Test Phase**:
   - Run tests on Python 3.11 and 3.12
   - Verify package import and CLI functionality

2. **Build Phase**:
   - Use `setuptools-scm` to get version from git tag
   - Build wheel and source distribution
   - Validate the built package

3. **Publish Phase**:
   - Automatically upload to PyPI
   - Use your configured API token

## 🛠️ Local Build Testing

Before creating a release, you can test the build locally:

```bash
# Install build dependencies
pip install build setuptools-scm[toml]

# Build package
python -m build

# Check build results
pip install twine
twine check dist/*

# Local install test
pip install dist/*.whl
vibesurf  # Test CLI command
```

## 🔍 Troubleshooting

### Common Issues

1. **Build Failure**:
   - Check `pyproject.toml` configuration
   - Ensure all dependencies are correctly listed
   - Verify `MANIFEST.in` includes necessary files

2. **Version Number Issues**:
   - Ensure git tag exists
   - Check `setuptools-scm` configuration
   - Verify `_version.py` file generation

3. **PyPI Upload Failure**:
   - Verify API token is correct
   - Check if package name already exists
   - Ensure version number is unique

### Manual Publishing (Emergency)

If automated workflow fails, you can publish manually:

```bash
# Build package
python -m build

# Upload to PyPI
twine upload dist/*
```

## 📋 Pre-Release Checklist

Before publishing, confirm:

- [ ] Code merged to main branch
- [ ] All tests pass
- [ ] README and documentation updated
- [ ] PyPI API token configured
- [ ] Git tag version number is correct
- [ ] Release notes prepared

## 🔄 Continuous Improvement

Regularly recommended:

- Update dependency versions
- Check for security vulnerabilities
- Optimize build process
- Monitor download statistics

## 📞 Getting Help

If you encounter issues:

1. Check GitHub Actions logs
2. Review PyPI project page
3. Reference [PyPI official documentation](https://packaging.python.org/)
4. Check [setuptools-scm documentation](https://setuptools-scm.readthedocs.io/)

## 💡 Additional Notes

### Dynamic Versioning Explained

The `_version.py` file is automatically generated by `setuptools-scm`:

- **When generated**: During `python -m build`, `pip install .`, or CI/CD builds
- **Content example**:
  ```python
  # coding: utf-8
  # file generated by setuptools_scm
  # don't change, don't track in version control
  __version__ = version = '0.1.0'
  __version_tuple__ = version_tuple = (0, 1, 0)
  ```
- **Version rules**:
  - With git tag `v0.1.0` → version `0.1.0`
  - After tag with new commits → version `0.1.0.dev1+g1234567`
  - No tags → version `0.0.0+unknown`

### Development vs Production

**Development Environment**:
- `_version.py` may not exist (if never built)
- Uses fallback version `"0.0.0+dev"`
- Can manually generate: `python -m setuptools_scm`

**Production Environment**:
- `_version.py` automatically generated during packaging
- Contains correct version information
- Users get proper version after installation

**Important**: Don't add `_version.py` to version control. Add to `.gitignore`:
```
vibe_surf/_version.py