"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeBuildRunner = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_stepfunctions_1 = require("aws-cdk-lib/aws-stepfunctions");
const common_1 = require("./common");
const codebuild_1 = require("./image-builders/codebuild");
/**
 * GitHub Actions runner provider using CodeBuild to execute jobs.
 *
 * Creates a project that gets started for each job.
 *
 * This construct is not meant to be used by itself. It should be passed in the providers property for GitHubRunners.
 */
class CodeBuildRunner extends common_1.BaseProvider {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.labels = this.labelsFromProperties('codebuild', props?.label, props?.labels);
        this.vpc = props?.vpc;
        if (props?.securityGroup) {
            this.securityGroups = [props.securityGroup];
        }
        else {
            if (props?.securityGroups) {
                this.securityGroups = props.securityGroups;
            }
            else {
                if (this.vpc) {
                    this.securityGroups = [new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, 'SG', { vpc: this.vpc })];
                }
            }
        }
        this.dind = props?.dockerInDocker ?? true;
        let buildSpec = {
            version: '0.2',
            env: {
                variables: {
                    RUNNER_TOKEN: 'unspecified',
                    RUNNER_NAME: 'unspecified',
                    RUNNER_LABEL: 'unspecified',
                    OWNER: 'unspecified',
                    REPO: 'unspecified',
                    GITHUB_DOMAIN: 'github.com',
                },
            },
            phases: {
                install: {
                    commands: [
                        this.dind ? 'nohup /usr/local/bin/dockerd --host=unix:///var/run/docker.sock --host=tcp://127.0.0.1:2375 --storage-driver=overlay2 &' : '',
                        this.dind ? 'timeout 15 sh -c "until docker info; do echo .; sleep 1; done"' : '',
                        'if [ "${RUNNER_VERSION}" = "latest" ]; then RUNNER_FLAGS=""; else RUNNER_FLAGS="--disableupdate"; fi',
                        'sudo -Hu runner /home/runner/config.sh --unattended --url "https://${GITHUB_DOMAIN}/${OWNER}/${REPO}" --token "${RUNNER_TOKEN}" --ephemeral --work _work --labels "${RUNNER_LABEL}" ${RUNNER_FLAGS} --name "${RUNNER_NAME}"',
                    ],
                },
                build: {
                    commands: [
                        'sudo --preserve-env=AWS_CONTAINER_CREDENTIALS_RELATIVE_URI,AWS_DEFAULT_REGION,AWS_REGION -Hu runner /home/runner/run.sh',
                    ],
                },
            },
        };
        const imageBuilder = props?.imageBuilder ?? new codebuild_1.CodeBuildImageBuilder(this, 'Image Builder', {
            dockerfilePath: CodeBuildRunner.LINUX_X64_DOCKERFILE_PATH,
        });
        const image = this.image = imageBuilder.bind();
        if (image.os.is(common_1.Os.WINDOWS)) {
            buildSpec.phases.install.commands = [
                'cd \\actions',
                'if (${Env:RUNNER_VERSION} -eq "latest") { $RunnerFlags = "" } else { $RunnerFlags = "--disableupdate" }',
                './config.cmd --unattended --url "https://${Env:GITHUB_DOMAIN}/${Env:OWNER}/${Env:REPO}" --token "${Env:RUNNER_TOKEN}" --ephemeral --work _work --labels "${Env:RUNNER_LABEL}" ${RunnerFlags} --name "${Env:RUNNER_NAME}"',
            ];
            buildSpec.phases.build.commands = [
                'cd \\actions',
                './run.cmd',
            ];
        }
        // choose build image
        let buildImage;
        if (image.os.is(common_1.Os.LINUX)) {
            if (image.architecture.is(common_1.Architecture.X86_64)) {
                buildImage = aws_cdk_lib_1.aws_codebuild.LinuxBuildImage.fromEcrRepository(image.imageRepository, image.imageTag);
            }
            else if (image.architecture.is(common_1.Architecture.ARM64)) {
                buildImage = aws_cdk_lib_1.aws_codebuild.LinuxArmBuildImage.fromEcrRepository(image.imageRepository, image.imageTag);
            }
        }
        if (image.os.is(common_1.Os.WINDOWS)) {
            if (image.architecture.is(common_1.Architecture.X86_64)) {
                buildImage = aws_cdk_lib_1.aws_codebuild.WindowsBuildImage.fromEcrRepository(image.imageRepository, image.imageTag, aws_cdk_lib_1.aws_codebuild.WindowsImageType.SERVER_2019);
            }
        }
        if (buildImage === undefined) {
            throw new Error(`Unable to find supported CodeBuild image for ${image.os.name}/${image.architecture.name}`);
        }
        // create project
        this.project = new aws_cdk_lib_1.aws_codebuild.Project(this, 'CodeBuild', {
            description: `GitHub Actions self-hosted runner for labels ${this.labels}`,
            buildSpec: aws_cdk_lib_1.aws_codebuild.BuildSpec.fromObject(buildSpec),
            vpc: this.vpc,
            securityGroups: this.securityGroups,
            subnetSelection: props?.subnetSelection,
            timeout: props?.timeout ?? aws_cdk_lib_1.Duration.hours(1),
            environment: {
                buildImage,
                computeType: props?.computeType ?? aws_codebuild_1.ComputeType.SMALL,
                privileged: this.dind ? image.os.is(common_1.Os.LINUX) : false,
            },
            logging: {
                cloudWatch: {
                    logGroup: new aws_cdk_lib_1.aws_logs.LogGroup(this, 'Logs', {
                        retention: props?.logRetention ?? aws_logs_1.RetentionDays.ONE_MONTH,
                        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                    }),
                },
            },
        });
        this.grantPrincipal = this.project.grantPrincipal;
    }
    /**
     * Generate step function task(s) to start a new runner.
     *
     * Called by GithubRunners and shouldn't be called manually.
     *
     * @param parameters workflow job details
     */
    getStepFunctionTask(parameters) {
        const step = new aws_cdk_lib_1.aws_stepfunctions_tasks.CodeBuildStartBuild(this, this.labels.join(', '), {
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.RUN_JOB,
            project: this.project,
            environmentVariablesOverride: {
                RUNNER_TOKEN: {
                    type: aws_cdk_lib_1.aws_codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                    value: parameters.runnerTokenPath,
                },
                RUNNER_NAME: {
                    type: aws_cdk_lib_1.aws_codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                    value: parameters.runnerNamePath,
                },
                RUNNER_LABEL: {
                    type: aws_cdk_lib_1.aws_codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                    value: this.labels.join(','),
                },
                GITHUB_DOMAIN: {
                    type: aws_cdk_lib_1.aws_codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                    value: parameters.githubDomainPath,
                },
                OWNER: {
                    type: aws_cdk_lib_1.aws_codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                    value: parameters.ownerPath,
                },
                REPO: {
                    type: aws_cdk_lib_1.aws_codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                    value: parameters.repoPath,
                },
            },
        });
        this.addRetry(step, ['CodeBuild.CodeBuildException', 'CodeBuild.AccountLimitExceededException']);
        return step;
    }
    grantStateMachine(_) {
    }
    status(statusFunctionRole) {
        this.image.imageRepository.grant(statusFunctionRole, 'ecr:DescribeImages');
        return {
            type: this.constructor.name,
            labels: this.labels,
            vpcArn: this.vpc?.vpcArn,
            securityGroups: this.securityGroups?.map(sg => sg.securityGroupId),
            roleArn: this.project.role?.roleArn,
            image: {
                imageRepository: this.image.imageRepository.repositoryUri,
                imageTag: this.image.imageTag,
                imageBuilderLogGroup: this.image.logGroup?.logGroupName,
            },
        };
    }
    /**
     * The network connections associated with this resource.
     */
    get connections() {
        return this.project.connections;
    }
}
exports.CodeBuildRunner = CodeBuildRunner;
_a = JSII_RTTI_SYMBOL_1;
CodeBuildRunner[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.CodeBuildRunner", version: "0.7.4" };
/**
 * Path to Dockerfile for Linux x64 with all the requirements for CodeBuild runner. Use this Dockerfile unless you need to customize it further than allowed by hooks.
 *
 * Available build arguments that can be set in the image builder:
 * * `BASE_IMAGE` sets the `FROM` line. This should be an Ubuntu compatible image.
 * * `EXTRA_PACKAGES` can be used to install additional packages.
 * * `DOCKER_CHANNEL` overrides the channel from which Docker will be downloaded. Defaults to `"stable"`.
 * * `DIND_COMMIT` overrides the commit where dind is found.
 * * `DOCKER_VERSION` overrides the installed Docker version.
 * * `DOCKER_COMPOSE_VERSION` overrides the installed docker-compose version.
 */
CodeBuildRunner.LINUX_X64_DOCKERFILE_PATH = path.join(__dirname, 'docker-images', 'codebuild', 'linux-x64');
/**
 * Path to Dockerfile for Linux ARM64 with all the requirements for CodeBuild runner. Use this Dockerfile unless you need to customize it further than allowed by hooks.
 *
 * Available build arguments that can be set in the image builder:
 * * `BASE_IMAGE` sets the `FROM` line. This should be an Ubuntu compatible image.
 * * `EXTRA_PACKAGES` can be used to install additional packages.
 * * `DOCKER_CHANNEL` overrides the channel from which Docker will be downloaded. Defaults to `"stable"`.
 * * `DIND_COMMIT` overrides the commit where dind is found.
 * * `DOCKER_VERSION` overrides the installed Docker version.
 * * `DOCKER_COMPOSE_VERSION` overrides the installed docker-compose version.
 */
CodeBuildRunner.LINUX_ARM64_DOCKERFILE_PATH = path.join(__dirname, 'docker-images', 'codebuild', 'linux-arm64');
//# sourceMappingURL=data:application/json;base64,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