"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontWebDistribution = exports.GeoRestriction = exports.ViewerCertificate = exports.LambdaEdgeEventType = exports.CloudFrontAllowedCachedMethods = exports.CloudFrontAllowedMethods = exports.OriginProtocolPolicy = exports.OriginSslPolicy = exports.SecurityPolicyProtocol = exports.SSLMethod = exports.ViewerProtocolPolicy = exports.PriceClass = exports.HttpVersion = void 0;
const certificatemanager = require("@aws-cdk/aws-certificatemanager");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const cloudfront_generated_1 = require("./cloudfront.generated");
var HttpVersion;
(function (HttpVersion) {
    HttpVersion["HTTP1_1"] = "http1.1";
    HttpVersion["HTTP2"] = "http2";
})(HttpVersion = exports.HttpVersion || (exports.HttpVersion = {}));
/**
 * The price class determines how many edge locations CloudFront will use for your distribution.
 */
var PriceClass;
(function (PriceClass) {
    PriceClass["PRICE_CLASS_100"] = "PriceClass_100";
    PriceClass["PRICE_CLASS_200"] = "PriceClass_200";
    PriceClass["PRICE_CLASS_ALL"] = "PriceClass_All";
})(PriceClass = exports.PriceClass || (exports.PriceClass = {}));
/**
 * How HTTPs should be handled with your distribution.
 */
var ViewerProtocolPolicy;
(function (ViewerProtocolPolicy) {
    ViewerProtocolPolicy["HTTPS_ONLY"] = "https-only";
    ViewerProtocolPolicy["REDIRECT_TO_HTTPS"] = "redirect-to-https";
    ViewerProtocolPolicy["ALLOW_ALL"] = "allow-all";
})(ViewerProtocolPolicy = exports.ViewerProtocolPolicy || (exports.ViewerProtocolPolicy = {}));
/**
 * The SSL method CloudFront will use for your distribution.
 *
 * Server Name Indication (SNI) - is an extension to the TLS computer networking protocol by which a client indicates
 *  which hostname it is attempting to connect to at the start of the handshaking process. This allows a server to present
 *  multiple certificates on the same IP address and TCP port number and hence allows multiple secure (HTTPS) websites
 * (or any other service over TLS) to be served by the same IP address without requiring all those sites to use the same certificate.
 *
 * CloudFront can use SNI to host multiple distributions on the same IP - which a large majority of clients will support.
 *
 * If your clients cannot support SNI however - CloudFront can use dedicated IPs for your distribution - but there is a prorated monthly charge for
 * using this feature. By default, we use SNI - but you can optionally enable dedicated IPs (VIP).
 *
 * See the CloudFront SSL for more details about pricing : https://aws.amazon.com/cloudfront/custom-ssl-domains/
 *
 */
var SSLMethod;
(function (SSLMethod) {
    SSLMethod["SNI"] = "sni-only";
    SSLMethod["VIP"] = "vip";
})(SSLMethod = exports.SSLMethod || (exports.SSLMethod = {}));
/**
 * The minimum version of the SSL protocol that you want CloudFront to use for HTTPS connections.
 * CloudFront serves your objects only to browsers or devices that support at least the SSL version that you specify.
 */
var SecurityPolicyProtocol;
(function (SecurityPolicyProtocol) {
    SecurityPolicyProtocol["SSL_V3"] = "SSLv3";
    SecurityPolicyProtocol["TLS_V1"] = "TLSv1";
    SecurityPolicyProtocol["TLS_V1_2016"] = "TLSv1_2016";
    SecurityPolicyProtocol["TLS_V1_1_2016"] = "TLSv1.1_2016";
    SecurityPolicyProtocol["TLS_V1_2_2018"] = "TLSv1.2_2018";
})(SecurityPolicyProtocol = exports.SecurityPolicyProtocol || (exports.SecurityPolicyProtocol = {}));
var OriginSslPolicy;
(function (OriginSslPolicy) {
    OriginSslPolicy["SSL_V3"] = "SSLv3";
    OriginSslPolicy["TLS_V1"] = "TLSv1";
    OriginSslPolicy["TLS_V1_1"] = "TLSv1.1";
    OriginSslPolicy["TLS_V1_2"] = "TLSv1.2";
})(OriginSslPolicy = exports.OriginSslPolicy || (exports.OriginSslPolicy = {}));
var OriginProtocolPolicy;
(function (OriginProtocolPolicy) {
    OriginProtocolPolicy["HTTP_ONLY"] = "http-only";
    OriginProtocolPolicy["MATCH_VIEWER"] = "match-viewer";
    OriginProtocolPolicy["HTTPS_ONLY"] = "https-only";
})(OriginProtocolPolicy = exports.OriginProtocolPolicy || (exports.OriginProtocolPolicy = {}));
/**
 * An enum for the supported methods to a CloudFront distribution.
 */
var CloudFrontAllowedMethods;
(function (CloudFrontAllowedMethods) {
    CloudFrontAllowedMethods["GET_HEAD"] = "GH";
    CloudFrontAllowedMethods["GET_HEAD_OPTIONS"] = "GHO";
    CloudFrontAllowedMethods["ALL"] = "ALL";
})(CloudFrontAllowedMethods = exports.CloudFrontAllowedMethods || (exports.CloudFrontAllowedMethods = {}));
/**
 * Enums for the methods CloudFront can cache.
 */
var CloudFrontAllowedCachedMethods;
(function (CloudFrontAllowedCachedMethods) {
    CloudFrontAllowedCachedMethods["GET_HEAD"] = "GH";
    CloudFrontAllowedCachedMethods["GET_HEAD_OPTIONS"] = "GHO";
})(CloudFrontAllowedCachedMethods = exports.CloudFrontAllowedCachedMethods || (exports.CloudFrontAllowedCachedMethods = {}));
var LambdaEdgeEventType;
(function (LambdaEdgeEventType) {
    /**
     * The origin-request specifies the request to the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_REQUEST"] = "origin-request";
    /**
     * The origin-response specifies the response from the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_RESPONSE"] = "origin-response";
    /**
     * The viewer-request specifies the incoming request
     */
    LambdaEdgeEventType["VIEWER_REQUEST"] = "viewer-request";
    /**
     * The viewer-response specifies the outgoing reponse
     */
    LambdaEdgeEventType["VIEWER_RESPONSE"] = "viewer-response";
})(LambdaEdgeEventType = exports.LambdaEdgeEventType || (exports.LambdaEdgeEventType = {}));
/**
 * Viewer certificate configuration class
 */
class ViewerCertificate {
    constructor(props, aliases = []) {
        this.props = props;
        this.aliases = aliases;
    }
    /**
     * Generate an AWS Certificate Manager (ACM) viewer certificate configuration
     *
     * @param certificate AWS Certificate Manager (ACM) certificate.
     *                    Your certificate must be located in the us-east-1 (US East (N. Virginia)) region to be accessed by CloudFront
     * @param options certificate configuration options
     */
    static fromAcmCertificate(certificate, options = {}) {
        const { sslMethod: sslSupportMethod = SSLMethod.SNI, securityPolicy: minimumProtocolVersion, aliases, } = options;
        return new ViewerCertificate({
            acmCertificateArn: certificate.certificateArn, sslSupportMethod, minimumProtocolVersion,
        }, aliases);
    }
    /**
     * Generate an IAM viewer certificate configuration
     *
     * @param iamCertificateId Identifier of the IAM certificate
     * @param options certificate configuration options
     */
    static fromIamCertificate(iamCertificateId, options = {}) {
        const { sslMethod: sslSupportMethod = SSLMethod.SNI, securityPolicy: minimumProtocolVersion, aliases, } = options;
        return new ViewerCertificate({
            iamCertificateId, sslSupportMethod, minimumProtocolVersion,
        }, aliases);
    }
    /**
     * Generate a viewer certifcate configuration using
     * the CloudFront default certificate (e.g. d111111abcdef8.cloudfront.net)
     * and a {@link SecurityPolicyProtocol.TLS_V1} security policy.
     *
     * @param aliases Alternative CNAME aliases
     *                You also must create a CNAME record with your DNS service to route queries
     */
    static fromCloudFrontDefaultCertificate(...aliases) {
        return new ViewerCertificate({ cloudFrontDefaultCertificate: true }, aliases);
    }
}
exports.ViewerCertificate = ViewerCertificate;
/**
 * Controls the countries in which your content is distributed.
 */
class GeoRestriction {
    /**
     * Creates an instance of GeoRestriction for internal use
     *
     * @param restrictionType Specifies the restriction type to impose (whitelist or blacklist)
     * @param locations Two-letter, uppercase country code for a country
     * that you want to whitelist/blacklist. Include one element for each country.
     * See ISO 3166-1-alpha-2 code on the *International Organization for Standardization* website
     */
    constructor(restrictionType, locations) {
        this.restrictionType = restrictionType;
        this.locations = locations;
    }
    /**
     * Whitelist specific countries which you want CloudFront to distribute your content.
     *
     * @param locations Two-letter, uppercase country code for a country
     * that you want to whitelist. Include one element for each country.
     * See ISO 3166-1-alpha-2 code on the *International Organization for Standardization* website
     */
    static whitelist(...locations) {
        return new GeoRestriction('whitelist', GeoRestriction.validateLocations(locations));
    }
    /**
     * Blacklist specific countries which you don't want CloudFront to distribute your content.
     *
     * @param locations Two-letter, uppercase country code for a country
     * that you want to blacklist. Include one element for each country.
     * See ISO 3166-1-alpha-2 code on the *International Organization for Standardization* website
     */
    static blacklist(...locations) {
        return new GeoRestriction('blacklist', GeoRestriction.validateLocations(locations));
    }
    static validateLocations(locations) {
        if (locations.length === 0) {
            throw new Error('Should provide at least 1 location');
        }
        locations.forEach(location => {
            if (!GeoRestriction.LOCATION_REGEX.test(location)) {
                throw new Error(`Invalid location format for location: ${location}, location should be two-letter and uppercase country ISO 3166-1-alpha-2 code`);
            }
        });
        return locations;
    }
}
exports.GeoRestriction = GeoRestriction;
GeoRestriction.LOCATION_REGEX = /^[A-Z]{2}$/;
/**
 * Amazon CloudFront is a global content delivery network (CDN) service that securely delivers data, videos,
 * applications, and APIs to your viewers with low latency and high transfer speeds.
 * CloudFront fronts user provided content and caches it at edge locations across the world.
 *
 * Here's how you can use this construct:
 *
 * ```ts
 * import { CloudFrontWebDistribution } from '@aws-cdk/aws-cloudfront'
 *
 * const sourceBucket = new Bucket(this, 'Bucket');
 *
 * const distribution = new CloudFrontWebDistribution(this, 'MyDistribution', {
 *  originConfigs: [
 *    {
 *      s3OriginSource: {
 *      s3BucketSource: sourceBucket
 *      },
 *      behaviors : [ {isDefaultBehavior: true}]
 *    }
 *  ]
 * });
 * ```
 *
 * This will create a CloudFront distribution that uses your S3Bucket as it's origin.
 *
 * You can customize the distribution using additional properties from the CloudFrontWebDistributionProps interface.
 *
 *
 */
class CloudFrontWebDistribution extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Maps our methods to the string arrays they are
         */
        this.METHOD_LOOKUP_MAP = {
            GH: ['GET', 'HEAD'],
            GHO: ['GET', 'HEAD', 'OPTIONS'],
            ALL: ['DELETE', 'GET', 'HEAD', 'OPTIONS', 'PATCH', 'POST', 'PUT'],
        };
        /**
         * Maps for which SecurityPolicyProtocol are available to which SSLMethods
         */
        this.VALID_SSL_PROTOCOLS = {
            [SSLMethod.SNI]: [
                SecurityPolicyProtocol.TLS_V1, SecurityPolicyProtocol.TLS_V1_1_2016,
                SecurityPolicyProtocol.TLS_V1_2016, SecurityPolicyProtocol.TLS_V1_2_2018,
            ],
            [SSLMethod.VIP]: [SecurityPolicyProtocol.SSL_V3, SecurityPolicyProtocol.TLS_V1],
        };
        let distributionConfig = {
            comment: props.comment,
            enabled: true,
            defaultRootObject: props.defaultRootObject !== undefined ? props.defaultRootObject : 'index.html',
            httpVersion: props.httpVersion || HttpVersion.HTTP2,
            priceClass: props.priceClass || PriceClass.PRICE_CLASS_100,
            ipv6Enabled: (props.enableIpV6 !== undefined) ? props.enableIpV6 : true,
            // tslint:disable-next-line:max-line-length
            customErrorResponses: props.errorConfigurations,
            webAclId: props.webACLId,
        };
        const behaviors = [];
        const origins = [];
        let originIndex = 1;
        for (const originConfig of props.originConfigs) {
            const originId = `origin${originIndex}`;
            if (!originConfig.s3OriginSource && !originConfig.customOriginSource) {
                throw new Error('There must be at least one origin source - either an s3OriginSource or a customOriginSource');
            }
            if (originConfig.customOriginSource && originConfig.s3OriginSource) {
                throw new Error('There cannot be both an s3OriginSource and a customOriginSource in the same SourceConfiguration.');
            }
            const originHeaders = [];
            if (originConfig.originHeaders) {
                Object.keys(originConfig.originHeaders).forEach(key => {
                    const oHeader = {
                        headerName: key,
                        headerValue: originConfig.originHeaders[key],
                    };
                    originHeaders.push(oHeader);
                });
            }
            let s3OriginConfig;
            if (originConfig.s3OriginSource) {
                // first case for backwards compatibility
                if (originConfig.s3OriginSource.originAccessIdentity) {
                    // grant CloudFront OriginAccessIdentity read access to S3 bucket
                    originConfig.s3OriginSource.s3BucketSource.grantRead(originConfig.s3OriginSource.originAccessIdentity);
                    s3OriginConfig = {
                        originAccessIdentity: `origin-access-identity/cloudfront/${originConfig.s3OriginSource.originAccessIdentity.originAccessIdentityName}`,
                    };
                }
                else {
                    s3OriginConfig = {};
                }
            }
            const originProperty = {
                id: originId,
                domainName: originConfig.s3OriginSource
                    ? originConfig.s3OriginSource.s3BucketSource.bucketRegionalDomainName
                    : originConfig.customOriginSource.domainName,
                originPath: originConfig.originPath,
                originCustomHeaders: originHeaders.length > 0 ? originHeaders : undefined,
                s3OriginConfig,
                customOriginConfig: originConfig.customOriginSource
                    ? {
                        httpPort: originConfig.customOriginSource.httpPort || 80,
                        httpsPort: originConfig.customOriginSource.httpsPort || 443,
                        originKeepaliveTimeout: originConfig.customOriginSource.originKeepaliveTimeout
                            && originConfig.customOriginSource.originKeepaliveTimeout.toSeconds() || 5,
                        originReadTimeout: originConfig.customOriginSource.originReadTimeout
                            && originConfig.customOriginSource.originReadTimeout.toSeconds() || 30,
                        originProtocolPolicy: originConfig.customOriginSource.originProtocolPolicy || OriginProtocolPolicy.HTTPS_ONLY,
                        originSslProtocols: originConfig.customOriginSource.allowedOriginSSLVersions || [OriginSslPolicy.TLS_V1_2],
                    }
                    : undefined,
            };
            for (const behavior of originConfig.behaviors) {
                behaviors.push({ ...behavior, targetOriginId: originId });
            }
            origins.push(originProperty);
            originIndex++;
        }
        origins.forEach(origin => {
            if (!origin.s3OriginConfig && !origin.customOriginConfig) {
                throw new Error(`Origin ${origin.domainName} is missing either S3OriginConfig or CustomOriginConfig. At least 1 must be specified.`);
            }
        });
        distributionConfig = {
            ...distributionConfig,
            origins,
        };
        const defaultBehaviors = behaviors.filter(behavior => behavior.isDefaultBehavior);
        if (defaultBehaviors.length !== 1) {
            throw new Error('There can only be one default behavior across all sources. [ One default behavior per distribution ].');
        }
        distributionConfig = { ...distributionConfig, defaultCacheBehavior: this.toBehavior(defaultBehaviors[0], props.viewerProtocolPolicy) };
        const otherBehaviors = [];
        for (const behavior of behaviors.filter(b => !b.isDefaultBehavior)) {
            if (!behavior.pathPattern) {
                throw new Error('pathPattern is required for all non-default behaviors');
            }
            otherBehaviors.push(this.toBehavior(behavior, props.viewerProtocolPolicy));
        }
        distributionConfig = { ...distributionConfig, cacheBehaviors: otherBehaviors.length > 0 ? otherBehaviors : undefined };
        if (props.aliasConfiguration && props.viewerCertificate) {
            throw new Error([
                'You cannot set both aliasConfiguration and viewerCertificate properties.',
                'Please only use viewerCertificate, as aliasConfiguration is deprecated.',
            ].join(' '));
        }
        let _viewerCertificate = props.viewerCertificate;
        if (props.aliasConfiguration) {
            const { acmCertRef, securityPolicy, sslMethod, names: aliases } = props.aliasConfiguration;
            _viewerCertificate = ViewerCertificate.fromAcmCertificate(certificatemanager.Certificate.fromCertificateArn(this, 'AliasConfigurationCert', acmCertRef), { securityPolicy, sslMethod, aliases });
        }
        if (_viewerCertificate) {
            const { props: viewerCertificate, aliases } = _viewerCertificate;
            Object.assign(distributionConfig, { aliases, viewerCertificate });
            const { minimumProtocolVersion, sslSupportMethod } = viewerCertificate;
            if (minimumProtocolVersion != null && sslSupportMethod != null) {
                const validProtocols = this.VALID_SSL_PROTOCOLS[sslSupportMethod];
                if (validProtocols.indexOf(minimumProtocolVersion.toString()) === -1) {
                    // tslint:disable-next-line:max-line-length
                    throw new Error(`${minimumProtocolVersion} is not compabtible with sslMethod ${sslSupportMethod}.\n\tValid Protocols are: ${validProtocols.join(', ')}`);
                }
            }
        }
        else {
            distributionConfig = { ...distributionConfig,
                viewerCertificate: { cloudFrontDefaultCertificate: true },
            };
        }
        if (props.loggingConfig) {
            this.loggingBucket = props.loggingConfig.bucket || new s3.Bucket(this, 'LoggingBucket');
            distributionConfig = {
                ...distributionConfig,
                logging: {
                    bucket: this.loggingBucket.bucketRegionalDomainName,
                    includeCookies: props.loggingConfig.includeCookies || false,
                    prefix: props.loggingConfig.prefix,
                },
            };
        }
        if (props.geoRestriction) {
            distributionConfig = {
                ...distributionConfig,
                restrictions: {
                    geoRestriction: {
                        restrictionType: props.geoRestriction.restrictionType,
                        locations: props.geoRestriction.locations,
                    },
                },
            };
        }
        const distribution = new cloudfront_generated_1.CfnDistribution(this, 'CFDistribution', { distributionConfig });
        this.node.defaultChild = distribution;
        this.domainName = distribution.attrDomainName;
        this.distributionId = distribution.ref;
    }
    toBehavior(input, protoPolicy) {
        let toReturn = {
            allowedMethods: this.METHOD_LOOKUP_MAP[input.allowedMethods || CloudFrontAllowedMethods.GET_HEAD],
            cachedMethods: this.METHOD_LOOKUP_MAP[input.cachedMethods || CloudFrontAllowedCachedMethods.GET_HEAD],
            compress: input.compress !== false,
            defaultTtl: input.defaultTtl && input.defaultTtl.toSeconds(),
            forwardedValues: input.forwardedValues || { queryString: false, cookies: { forward: 'none' } },
            maxTtl: input.maxTtl && input.maxTtl.toSeconds(),
            minTtl: input.minTtl && input.minTtl.toSeconds(),
            trustedSigners: input.trustedSigners,
            targetOriginId: input.targetOriginId,
            viewerProtocolPolicy: protoPolicy || ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        };
        if (!input.isDefaultBehavior) {
            toReturn = Object.assign(toReturn, { pathPattern: input.pathPattern });
        }
        if (input.lambdaFunctionAssociations) {
            toReturn = Object.assign(toReturn, {
                lambdaFunctionAssociations: input.lambdaFunctionAssociations
                    .map(fna => ({
                    eventType: fna.eventType,
                    lambdaFunctionArn: fna.lambdaFunction && fna.lambdaFunction.functionArn,
                })),
            });
            // allow edgelambda.amazonaws.com to assume the functions' execution role.
            for (const a of input.lambdaFunctionAssociations) {
                if (a.lambdaFunction.role && a.lambdaFunction.role instanceof iam.Role && a.lambdaFunction.role.assumeRolePolicy) {
                    a.lambdaFunction.role.assumeRolePolicy.addStatements(new iam.PolicyStatement({
                        actions: ['sts:AssumeRole'],
                        principals: [new iam.ServicePrincipal('edgelambda.amazonaws.com')],
                    }));
                }
            }
        }
        return toReturn;
    }
}
exports.CloudFrontWebDistribution = CloudFrontWebDistribution;
//# sourceMappingURL=data:application/json;base64,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