![](https://raw.githubusercontent.com/jrasband-dev/pandas-geojson/master/static/pandasgeojson.png)

---
![PyPI - Version](https://img.shields.io/pypi/v/pandas-geojson)
![PyPI - Downloads](https://img.shields.io/pypi/dm/pandas-geojson)
![PyPI - License](https://img.shields.io/pypi/l/pandas-geojson)
![Libraries.io SourceRank](https://img.shields.io/librariesio/sourcerank/pypi/pandas-geojson)


# Getting Started
Install the latest version of pandas-geojson using pip.
```
pip install pandas-geojson
```



## Open GeoJSON Files
Using the `read_geojson` function, you can easily read in GeoJSON data int the GeoJSON object. 



```python
import pandas_geojson as pdg
geojson = pdg.read_geojson('datasets/National_Obesity_By_State.geojson')
type(geojson)
```




    pandas_geojson.core.GeoJSON



## List Properties
Pandas-GeoJSON utilizes properties to filter large geojson files. This functionallity helps you identify all of the properties in the geojson file so that you can choose a filter criteria. For more on filtering see [filter_geojson](#filter-geojson)


```python
geojson.get_properties()
```




    ['FID', 'NAME', 'Obesity', 'SHAPE_Length', 'SHAPE_Area']



## DataFrame to GeoJSON

Pandas-GeoJSON makes it easy to convert a dataframe with coordinates into a GeoJSON object. Your DataFrame **must** contain at least two columns with the following information:
* Geometry Type
* Coodinates

If you have properties in your DataFrame that you want to include, you can add those as well. If you don't have you own file, feel free to use the example csv file in the repo.


```python
import pandas as pd
import json
csv = pd.read_csv('datasets/ObesityByState.csv')
csv.head()
```




<div>
<table border="1" class="dataframe">
  <thead>
    <tr style="text-align: right;">
      <th></th>
      <th>type</th>
      <th>coordinates</th>
      <th>FID</th>
      <th>NAME</th>
      <th>Obesity</th>
      <th>SHAPE_Length</th>
      <th>SHAPE_Area</th>
    </tr>
  </thead>
  <tbody>
    <tr>
      <th>0</th>
      <td>MultiPolygon</td>
      <td>[[[[-109.05997287099996, 38.49999297200003], [...</td>
      <td>26</td>
      <td>Colorado</td>
      <td>20.2</td>
      <td>22.017327</td>
      <td>28.036526</td>
    </tr>
    <tr>
      <th>1</th>
      <td>MultiPolygon</td>
      <td>[[[[-114.05248377599997, 37.60478152100006], [...</td>
      <td>28</td>
      <td>Utah</td>
      <td>24.5</td>
      <td>19.997476</td>
      <td>22.977382</td>
    </tr>
    <tr>
      <th>2</th>
      <td>MultiPolygon</td>
      <td>[[[[-114.79969447199994, 32.593625864000046], ...</td>
      <td>43</td>
      <td>Arizona</td>
      <td>28.4</td>
      <td>23.022491</td>
      <td>28.921300</td>
    </tr>
  </tbody>
</table>
</div>




```python
import pandas_geojson as pdg
geojson = pdg.GeoJSON.from_dataframe(csv
                                     ,geometry_type_col='type'
                                     ,coordinate_col='coordinates'
                                     ,property_col_list=['FID','NAME','Obesity']
                                     )
geojson
```




    {
        "type": "FeatureCollection",
        "features": [
            {
                "type": "Feature",
                "geometry": {
                    "type": "MultiPolygon",
                    "coordinates": "[[[[-109.05997287099996, 38.49999297200003], [-109.05997284299997, 38.49999500400003], [-109.05152293999998, 39.12610105400006], [-109.05123479899999, 39.36668363300004], [-109.0510778299999, 39.49774491000005], [-109.05088292899995, 39.66047789800007], [-109.05062602899994, 39.87497614700004], [-109.05098406799999, 40.18085518500004], [-109.05074318499999, 40.22266144900004], [-109.04826012699999, 40.65360724400006], [-109.04827087899997, 40.66260806200006], [-109.04846614999997, 40.82608726400008], [-109.05008717399994, 41.000665285000025], [-108.25065999899999, 41.000120346000074], [-107.91842921699998, 41.00123344200006], [-107.36745380399998, 41.00307941300008], [-107.31780655299997, 41.002848898000025], [-106.86038863899995, 41.00072508000005], [-106.32117978399998, 40.998221502000035], [-106.21758354499997, 40.99774049900003], [-106.19056461699995, 40.997753114000034], [-105.27714832799995, 40.99817956800007], [-105.27687032299997, 40.998179486000026], [-104.94338125099995, 40.998080696000045], [-104.85528323099999, 40.998054599000056], [-104.497068147, 41.00181162500007], [-104.05325904199998, 41.00141265800005], [-103.57453192899999, 41.00172769300008], [-103.57378393399995, 41.00172849100005], [-103.38250153499996, 41.00193285200004], [-103.07654580899998, 41.00225972800007], [-102.65347384399996, 41.002232082000035], [-102.62104272399995, 41.00222996300005], [-102.55679868399994, 41.00222576500005], [-102.05172707699995, 41.00238376800007], [-102.05162355999994, 41.002383801000065], [-102.05130152999999, 40.74959776400004], [-102.05130152999999, 40.74959276000004], [-102.05130367499999, 40.69755359000004], [-102.05131429099998, 40.44001471400003], [-102.05131803399996, 40.349228064000044], [-102.05131848099995, 40.33838770600005], [-102.05175344199995, 40.003084657000045], [-102.05126342199998, 39.81899863000007], [-102.05000152899999, 39.57406269300003], [-102.04997125099999, 39.56818557400004], [-102.04896937099994, 39.37371856500005], [-102.04721006699998, 39.13315323900008], [-102.04658033599998, 39.047044517000074], [-102.04572220199998, 38.69757223500005], [-102.04551986499996, 38.61517104200004], [-102.04495326599994, 38.384425417000045], [-102.04465961399995, 38.26875562200007], [-102.04464352499997, 38.26241796700003], [-102.04426423899997, 38.11301737600007], [-102.04197485999998, 37.738546892000045], [-102.04188519999997, 37.723881316000075], [-102.04190006299996, 37.64428398900003], [-102.04194769199995, 37.38919632000005], [-102.04197215599999, 37.25817024400004], [-102.04199214199997, 37.10655722000007], [-102.04192913499998, 37.035089209000034], [-102.04224913199994, 36.99308920200008], [-102.35529720599999, 36.99451218400003], [-102.69815128799996, 36.99515516400004], [-102.84199832199994, 36.99960415600003], [-103.00220835999994, 37.00011014700004], [-103.08611423799994, 36.99987049200007], [-103.73325653099994, 36.99802210300004], [-104.00786444499994, 36.99599014100005], [-104.33884267099995, 36.993541066000034], [-104.73204076099995, 36.993453042000056], [-105.12080984999994, 36.99543401900007], [-105.15505260599997, 36.995480462000046], [-105.22062233699995, 36.99556939400003], [-105.25130587999996, 36.99561101000006], [-105.41931991699994, 36.995862001000035], [-105.53393194299997, 36.99588099400006], [-105.71648098399999, 36.995854982000026], [-105.71847966899998, 36.995851909000066], [-105.99748204699995, 36.995422965000046], [-106.00664419599997, 36.995391502000075], [-106.34314912399998, 36.994235943000035], [-106.47624924899998, 36.99378001700006], [-106.86980623999995, 36.99243191000005], [-106.87730224299997, 37.00014491100006], [-107.42092336199994, 37.00001087800007], [-107.42092536199999, 37.00001087800007], [-107.48174737499994, 37.00001045400006], [-108.00063348699996, 37.00000684100007], [-108.37926787299995, 36.99957055600004], [-108.62031961999998, 36.99929280200007], [-109.04523371, 36.99908977500007], [-109.04379147899999, 37.484825072000035], [-109.04261490099998, 37.88109071000008], [-109.04180744399997, 38.153038233000075], [-109.04177282999996, 38.16469593000005], [-109.06007284499998, 38.27549494300007], [-109.05997287099996, 38.49999297200003]]]]"
                },
                "properties": {
                    "FID": 26,
                    "NAME": "Colorado",
                    "Obesity": 20.2
                }
            },
            {
                "type": "Feature",
                "geometry": {
                    "type": "MultiPolygon",
                    "coordinates": "[[[[-114.05248377599997, 37.60478152100006], [-114.05248377199996, 37.60478216300004], [-114.05173979099999, 37.74600253700004], [-114.04966980499995, 37.88137355300006], [-114.04991483599997, 38.14860658200007], [-114.04991521099998, 38.14876861000005], [-114.05014984699994, 38.24996559400006], [-114.05013186399998, 38.40454161200006], [-114.05016588499996, 38.57292563000004], [-114.05016550499994, 38.57298084300004], [-114.04944860199998, 38.677368099000034], [-114.04806591999994, 38.87869866500006], [-114.04911593599996, 39.005514678000054], [-114.04709099599995, 39.49994873300005], [-114.04719341399999, 39.54274934000006], [-114.04779503299994, 39.79416576500006], [-114.04728020099998, 39.906103890000054], [-114.04639807299998, 40.09790179700008], [-114.04638489199999, 40.11694163200008], [-114.04619011299997, 40.39831882900006], [-114.04558912499999, 40.49580683900007], [-114.04351715699994, 40.72629786300007], [-114.04215734099995, 40.99990246900006], [-114.04215719399997, 40.99993189200006], [-114.041459224, 41.20775791400007], [-114.04024326399997, 41.491695942000035], [-114.03991330199995, 41.75378696900003], [-114.04173533899996, 41.993725993000055], [-113.817976295, 41.98858601200004], [-113.49656023199998, 41.99331103900005], [-113.24917118399998, 41.99620906000007], [-113.00083520299995, 41.99822906400004], [-113.000833135, 41.998229080000044], [-112.64803106499994, 42.000313110000036], [-112.26494798799996, 42.000997142000074], [-112.159189417, 41.99868808000008], [-112.10954395499999, 41.997604154000044], [-112.10952400999997, 41.99760425000005], [-111.75078988299998, 41.99933618500006], [-111.50785684799996, 41.99969182500007], [-111.50781802799997, 41.999691883000025], [-111.47139282499995, 41.999745208000036], [-111.37414080499997, 42.00089921500006], [-111.04670073699998, 42.00157324200006], [-111.04583014099995, 41.57985114400003], [-111.04580067399996, 41.56557719400007], [-111.04661164499998, 41.36069817100008], [-111.04664861399999, 41.25163311500006], [-111.04673459499998, 40.99796513000007], [-110.53983048999999, 40.99635216900003], [-110.12165040199994, 40.99710720200005], [-110.04849438199994, 40.997303500000044], [-110.00072811299998, 40.99743167100007], [-109.71542031599995, 40.99819723400003], [-109.25074621699997, 41.00101527000004], [-109.05008717399994, 41.000665285000025], [-109.04846614999997, 40.82608726400008], [-109.04827087899997, 40.66260806200006], [-109.04826012699999, 40.65360724400006], [-109.05074318499999, 40.22266144900004], [-109.05098406799999, 40.18085518500004], [-109.05062602899994, 39.87497614700004], [-109.05088292899995, 39.66047789800007], [-109.0510778299999, 39.49774491000005], [-109.05123479899999, 39.36668363300004], [-109.05152293999998, 39.12610105400006], [-109.05997284299997, 38.49999500400003], [-109.05997287099996, 38.49999297200003], [-109.06007284499998, 38.27549494300007], [-109.04177282999996, 38.16469593000005], [-109.04180744399997, 38.153038233000075], [-109.04261490099998, 37.88109071000008], [-109.04379147899999, 37.484825072000035], [-109.04523371, 36.99908977500007], [-109.49534880499999, 36.99911074600004], [-110.00068790999995, 36.99797371400007], [-110.00069272799999, 36.99797371500006], [-110.47020100599997, 36.99800268400003], [-110.75069754299994, 37.003202602000044], [-110.75070106399994, 37.00320266600005], [-111.06650712799996, 37.00239464600003], [-111.27829717099996, 37.00047063200003], [-111.40588019699999, 37.00148662400005], [-111.41279518599998, 37.001483310000026], [-112.35770138499998, 37.00103056200004], [-112.53858242099994, 37.00074967900008], [-112.54510542199995, 37.00073954900006], [-112.82951342199999, 37.00039193300006], [-112.89920238899998, 37.000306755000054], [-112.96648250499999, 37.00022452200005], [-113.96591869599996, 37.00003045600005], [-114.05061171199998, 37.00040145100007], [-114.051760721, 37.088439460000075], [-114.05198574099995, 37.28451648300006], [-114.05271276399998, 37.492019508000055], [-114.05248377599997, 37.60478152100006]]]]"
                },
                "properties": {
                    "FID": 28,
                    "NAME": "Utah",
                    "Obesity": 24.5
                }
            },
            {
                "type": "Feature",
                "geometry": {
                    "type": "MultiPolygon",
                    "coordinates": "[[[[-114.79969447199994, 32.593625864000046], [-114.80940447699999, 32.61712386800008], [-114.76496147099999, 32.64939587400005], [-114.71964446699997, 32.718767884000044], [-114.70572846599998, 32.74158588800003], [-114.66750445899999, 32.73423089000005], [-114.61739797399997, 32.741057666000074], [-114.57068644099996, 32.74742189600005], [-114.53175743699995, 32.78250790200008], [-114.46898242999998, 32.84515991300003], [-114.46313843299998, 32.90188892100008], [-114.47665143599994, 32.92363292300007], [-114.48132644099996, 32.972068930000034], [-114.51135445099999, 33.02345993500006], [-114.51707824199997, 33.02463369800006], [-114.57517246299994, 33.03654693200008], [-114.62830447299996, 33.031056929000044], [-114.67081448099998, 33.03798892800006], [-114.70618649199997, 33.105339935000075], [-114.67937049099999, 33.15952394300007], [-114.678108496, 33.23030495200004], [-114.67450249699999, 33.25560195600008], [-114.72327050899997, 33.288083957000026], [-114.70797350899994, 33.323425963000034], [-114.70735951199998, 33.37663297000006], [-114.72529351799994, 33.40505297200008], [-114.67391250899999, 33.41830397700005], [-114.63519450299998, 33.42273097900005], [-114.62915801999998, 33.433549981000056], [-114.59729450099996, 33.490657990000045], [-114.52461049199997, 33.55223600100004], [-114.52919749699998, 33.606655008000075], [-114.52521249999995, 33.66158801500006], [-114.50500449999998, 33.693027021000034], [-114.49657649999995, 33.719160024000075], [-114.50487450399999, 33.76047002800004], [-114.52047651299995, 33.827783037000074], [-114.50564951199999, 33.86428104200007], [-114.50871951599999, 33.90064504600008], [-114.53499852299996, 33.928504048000036], [-114.50957952099998, 33.95726905300006], [-114.45481851499994, 34.01097306300005], [-114.43551551399997, 34.042620068000076], [-114.43010265399994, 34.07893647900005], [-114.42803751599996, 34.092792075000034], [-114.40595251399998, 34.11154507800006], [-114.34806350499997, 34.13446308500005], [-114.29281749699999, 34.166730091000034], [-114.229726487, 34.186933097000065], [-114.17806148199998, 34.23997410700008], [-114.13906647599998, 34.259543112000074], [-114.14082863199997, 34.30313263100004], [-114.14094148099997, 34.305924118000064], [-114.17285648899997, 34.34498412100004], [-114.26432851099997, 34.40133412200004], [-114.33538352799997, 34.45004312400005], [-114.37886353699997, 34.45038112200007], [-114.37823454199997, 34.516526130000045], [-114.42239355599997, 34.580716136000035], [-114.46525757299997, 34.69120714700006], [-114.49098258099997, 34.72485315000006], [-114.57646360399997, 34.81530515600008], [-114.63439361999997, 34.87289516000004], [-114.62978062499997, 34.943045168000026], [-114.63349863099995, 35.00186217500004], [-114.62508063499996, 35.06848318400006], [-114.61991663899994, 35.12163719100005], [-114.59913163499999, 35.12105519100004], [-114.57275863099994, 35.13873019500005], [-114.58714064499998, 35.26238121000006], [-114.62714866499999, 35.40950922500008], [-114.66451167599996, 35.44950222700004], [-114.66311668299994, 35.52449623700005], [-114.65341768899998, 35.61079424800005], [-114.68941869899999, 35.651417250000065], [-114.69732070799995, 35.73369125900007], [-114.70372171699995, 35.814590269000064], [-114.66969871499998, 35.86508927700004], [-114.70028272499997, 35.901777279000044], [-114.73117073399999, 35.943921282000076], [-114.74279074299994, 36.00996828900003], [-114.74331099399996, 36.065940469000054], [-114.743353749, 36.07054029600005], [-114.73617675099996, 36.10437230100007], [-114.66654973899995, 36.11734830700004], [-114.62786673399995, 36.14101731200003], [-114.57204272499996, 36.15161531700005], [-114.51173271399995, 36.15096132000008], [-114.48704570599995, 36.129401320000056], [-114.44866569899995, 36.12641532200007], [-114.41696169499994, 36.14576632600006], [-114.37211768699996, 36.14311932800007], [-114.33728467699996, 36.108025327000064], [-114.31612066899999, 36.063114323000036], [-114.27065665699996, 36.035725322000076], [-114.21370164499996, 36.01561832300007], [-114.15173663399997, 36.02456832800004], [-114.13821463499994, 36.053166332000046], [-114.09988163299994, 36.12165934300003], [-114.04684963099999, 36.194074355000055], [-114.04823763899998, 36.26887936400004], [-114.04759564299997, 36.32557837100006], [-114.04950467199996, 36.60406440400004], [-114.05017247099994, 36.843146972000056], [-114.05061171199998, 37.00040145100007], [-113.96591869599996, 37.00003045600005], [-112.96648250499999, 37.00022452200005], [-112.89920238899998, 37.000306755000054], [-112.82951342199999, 37.00039193300006], [-112.54510542199995, 37.00073954900006], [-112.53858242099994, 37.00074967900008], [-112.35770138499998, 37.00103056200004], [-111.41279518599998, 37.001483310000026], [-111.40588019699999, 37.00148662400005], [-111.27829717099996, 37.00047063200003], [-111.06650712799996, 37.00239464600003], [-110.75070106399994, 37.00320266600005], [-110.75069754299994, 37.003202602000044], [-110.47020100599997, 36.99800268400003], [-110.00069272799999, 36.99797371500006], [-110.00068790999995, 36.99797371400007], [-109.49534880499999, 36.99911074600004], [-109.04523371, 36.99908977500007], [-109.04544369799999, 36.87459475800006], [-109.04573962599994, 36.11703365500006], [-109.04588223499997, 36.002344074000064], [-109.04603460399994, 35.87980562100006], [-109.04630658099995, 35.61425658400003], [-109.04680656, 35.36361155000003], [-109.04636450999999, 35.174689181000076], [-109.04586152499996, 34.95972349200008], [-109.04614942899997, 34.57929653200006], [-109.04619248899996, 34.52239843000007], [-109.04643643899999, 33.875057336000054], [-109.04661789, 33.77823769300005], [-109.04730840499997, 33.40978826700007], [-109.04724738999994, 33.20897023800006], [-109.04712736, 32.77757517400005], [-109.04712736199997, 32.77757417400005], [-109.04762233699995, 32.42638212000003], [-109.04830631499999, 32.08409806900005], [-109.04920529699996, 31.79655602400004], [-109.05005426899999, 31.332506953000063], [-109.82969942999995, 31.334071920000042], [-110.46018316799996, 31.333146271000032], [-111.07483567999998, 31.332243865000066], [-111.36698011499999, 31.424829153000076], [-112.36505395799998, 31.74113486600004], [-113.33377913599998, 32.04025394200005], [-113.75076724999997, 32.169009860000074], [-114.81362446799994, 32.49428185100004], [-114.81154747, 32.52283885500003], [-114.79564646899996, 32.55096085900004], [-114.81419647399997, 32.56479286000007], [-114.79969447199994, 32.593625864000046]]]]"
                },
                "properties": {
                    "FID": 43,
                    "NAME": "Arizona",
                    "Obesity": 28.4
                }
            }
        ]
    }



## GeoJSON to DataFrame

You can convert your GeoJSON object to a dataframe using the `to_dataframe` function. This will return a normalized dataframe of your GeoJSON data. From here you can manipulate, filter, and save your dataframe. 


```python
import pandas_geojson as pdg

geojson = pdg.read_geojson('datasets/National_Obesity_By_State.geojson')
df = geojson.to_dataframe()
df.head()
```




<div>
<table border="1" class="dataframe">
  <thead>
    <tr style="text-align: right;">
      <th></th>
      <th>type</th>
      <th>geometry.type</th>
      <th>geometry.coordinates</th>
      <th>properties.FID</th>
      <th>properties.NAME</th>
      <th>properties.Obesity</th>
      <th>properties.SHAPE_Length</th>
      <th>properties.SHAPE_Area</th>
      <th>geometry</th>
    </tr>
  </thead>
  <tbody>
    <tr>
      <th>0</th>
      <td>Feature</td>
      <td>MultiPolygon</td>
      <td>[[[[-106.62345478999998, 31.914039152000043], ...</td>
      <td>1</td>
      <td>Texas</td>
      <td>32.4</td>
      <td>45.047446</td>
      <td>65.778624</td>
      <td>NaN</td>
    </tr>
    <tr>
      <th>1</th>
      <td>Feature</td>
      <td>MultiPolygon</td>
      <td>[[[[-118.59398119899998, 33.467202772000064], ...</td>
      <td>2</td>
      <td>California</td>
      <td>24.2</td>
      <td>40.755927</td>
      <td>41.730164</td>
      <td>NaN</td>
    </tr>
    <tr>
      <th>2</th>
      <td>Feature</td>
      <td>MultiPolygon</td>
      <td>[[[[-89.54443984499994, 36.574516791000065], [...</td>
      <td>3</td>
      <td>Kentucky</td>
      <td>34.6</td>
      <td>20.348345</td>
      <td>10.665636</td>
      <td>NaN</td>
    </tr>
    <tr>
      <th>3</th>
      <td>Feature</td>
      <td>MultiPolygon</td>
      <td>[[[[-85.60516962299994, 34.98468464700005], [-...</td>
      <td>4</td>
      <td>Georgia</td>
      <td>30.7</td>
      <td>17.288448</td>
      <td>14.682554</td>
      <td>NaN</td>
    </tr>
    <tr>
      <th>4</th>
      <td>Feature</td>
      <td>MultiPolygon</td>
      <td>[[[[-86.93428193399996, 45.42115747400004], [-...</td>
      <td>5</td>
      <td>Wisconsin</td>
      <td>30.7</td>
      <td>22.428845</td>
      <td>16.512937</td>
      <td>NaN</td>
    </tr>
  </tbody>
</table>
</div>



## Filter GeoJSON
GeoJSON files can be large and complex. Pandas-GeoJSON gives you the ability to filter geojson data. 

You can filter based on properties in the file. The `filter_geojson` function asks for a property key and a list of property values to filter on. The function will return a new GeoJSON object with only the values passed into the list. You can only use one property key at a time to filter. 



```python
import pandas_geojson as pdg
geojson = pdg.read_geojson('datasets/National_Obesity_By_State.geojson')
properties = geojson.get_properties()
properties

```




    ['FID', 'NAME', 'Obesity', 'SHAPE_Length', 'SHAPE_Area']




```python
new_geojson = geojson.filter_geojson(property_values=['Arizona'],property_key='NAME')
new_geojson
```




    {
        "type": "FeatureCollection",
        "features": [
            {
                "type": "Feature",
                "geometry": {
                    "type": "MultiPolygon",
                    "coordinates": [
                        [
                            [
                                [
                                    -114.79969447199994,
                                    32.593625864000046
                                ],
                                [
                                    -114.80940447699999,
                                    32.61712386800008
                                ],
                                [
                                    -114.76496147099999,
                                    32.64939587400005
                                ],
                                [
                                    -114.71964446699997,
                                    32.718767884000044
                                ],
                                [
                                    -114.70572846599998,
                                    32.74158588800003
                                ],
                                [
                                    -114.66750445899999,
                                    32.73423089000005
                                ],
                                [
                                    -114.61739797399997,
                                    32.741057666000074
                                ],
                                [
                                    -114.57068644099996,
                                    32.74742189600005
                                ],
                                [
                                    -114.53175743699995,
                                    32.78250790200008
                                ],
                                [
                                    -114.46898242999998,
                                    32.84515991300003
                                ],
                                [
                                    -114.46313843299998,
                                    32.90188892100008
                                ],
                                [
                                    -114.47665143599994,
                                    32.92363292300007
                                ],
                                [
                                    -114.48132644099996,
                                    32.972068930000034
                                ],
                                [
                                    -114.51135445099999,
                                    33.02345993500006
                                ],
                                [
                                    -114.51707824199997,
                                    33.02463369800006
                                ],
                                [
                                    -114.57517246299994,
                                    33.03654693200008
                                ],
                                [
                                    -114.62830447299996,
                                    33.031056929000044
                                ],
                                [
                                    -114.67081448099998,
                                    33.03798892800006
                                ],
                                [
                                    -114.70618649199997,
                                    33.105339935000075
                                ],
                                [
                                    -114.67937049099999,
                                    33.15952394300007
                                ],
                                [
                                    -114.678108496,
                                    33.23030495200004
                                ],
                                [
                                    -114.67450249699999,
                                    33.25560195600008
                                ],
                                [
                                    -114.72327050899997,
                                    33.288083957000026
                                ],
                                [
                                    -114.70797350899994,
                                    33.323425963000034
                                ],
                                [
                                    -114.70735951199998,
                                    33.37663297000006
                                ],
                                [
                                    -114.72529351799994,
                                    33.40505297200008
                                ],
                                [
                                    -114.67391250899999,
                                    33.41830397700005
                                ],
                                [
                                    -114.63519450299998,
                                    33.42273097900005
                                ],
                                [
                                    -114.62915801999998,
                                    33.433549981000056
                                ],
                                [
                                    -114.59729450099996,
                                    33.490657990000045
                                ],
                                [
                                    -114.52461049199997,
                                    33.55223600100004
                                ],
                                [
                                    -114.52919749699998,
                                    33.606655008000075
                                ],
                                [
                                    -114.52521249999995,
                                    33.66158801500006
                                ],
                                [
                                    -114.50500449999998,
                                    33.693027021000034
                                ],
                                [
                                    -114.49657649999995,
                                    33.719160024000075
                                ],
                                [
                                    -114.50487450399999,
                                    33.76047002800004
                                ],
                                [
                                    -114.52047651299995,
                                    33.827783037000074
                                ],
                                [
                                    -114.50564951199999,
                                    33.86428104200007
                                ],
                                [
                                    -114.50871951599999,
                                    33.90064504600008
                                ],
                                [
                                    -114.53499852299996,
                                    33.928504048000036
                                ],
                                [
                                    -114.50957952099998,
                                    33.95726905300006
                                ],
                                [
                                    -114.45481851499994,
                                    34.01097306300005
                                ],
                                [
                                    -114.43551551399997,
                                    34.042620068000076
                                ],
                                [
                                    -114.43010265399994,
                                    34.07893647900005
                                ],
                                [
                                    -114.42803751599996,
                                    34.092792075000034
                                ],
                                [
                                    -114.40595251399998,
                                    34.11154507800006
                                ],
                                [
                                    -114.34806350499997,
                                    34.13446308500005
                                ],
                                [
                                    -114.29281749699999,
                                    34.166730091000034
                                ],
                                [
                                    -114.229726487,
                                    34.186933097000065
                                ],
                                [
                                    -114.17806148199998,
                                    34.23997410700008
                                ],
                                [
                                    -114.13906647599998,
                                    34.259543112000074
                                ],
                                [
                                    -114.14082863199997,
                                    34.30313263100004
                                ],
                                [
                                    -114.14094148099997,
                                    34.305924118000064
                                ],
                                [
                                    -114.17285648899997,
                                    34.34498412100004
                                ],
                                [
                                    -114.26432851099997,
                                    34.40133412200004
                                ],
                                [
                                    -114.33538352799997,
                                    34.45004312400005
                                ],
                                [
                                    -114.37886353699997,
                                    34.45038112200007
                                ],
                                [
                                    -114.37823454199997,
                                    34.516526130000045
                                ],
                                [
                                    -114.42239355599997,
                                    34.580716136000035
                                ],
                                [
                                    -114.46525757299997,
                                    34.69120714700006
                                ],
                                [
                                    -114.49098258099997,
                                    34.72485315000006
                                ],
                                [
                                    -114.57646360399997,
                                    34.81530515600008
                                ],
                                [
                                    -114.63439361999997,
                                    34.87289516000004
                                ],
                                [
                                    -114.62978062499997,
                                    34.943045168000026
                                ],
                                [
                                    -114.63349863099995,
                                    35.00186217500004
                                ],
                                [
                                    -114.62508063499996,
                                    35.06848318400006
                                ],
                                [
                                    -114.61991663899994,
                                    35.12163719100005
                                ],
                                [
                                    -114.59913163499999,
                                    35.12105519100004
                                ],
                                [
                                    -114.57275863099994,
                                    35.13873019500005
                                ],
                                [
                                    -114.58714064499998,
                                    35.26238121000006
                                ],
                                [
                                    -114.62714866499999,
                                    35.40950922500008
                                ],
                                [
                                    -114.66451167599996,
                                    35.44950222700004
                                ],
                                [
                                    -114.66311668299994,
                                    35.52449623700005
                                ],
                                [
                                    -114.65341768899998,
                                    35.61079424800005
                                ],
                                [
                                    -114.68941869899999,
                                    35.651417250000065
                                ],
                                [
                                    -114.69732070799995,
                                    35.73369125900007
                                ],
                                [
                                    -114.70372171699995,
                                    35.814590269000064
                                ],
                                [
                                    -114.66969871499998,
                                    35.86508927700004
                                ],
                                [
                                    -114.70028272499997,
                                    35.901777279000044
                                ],
                                [
                                    -114.73117073399999,
                                    35.943921282000076
                                ],
                                [
                                    -114.74279074299994,
                                    36.00996828900003
                                ],
                                [
                                    -114.74331099399996,
                                    36.065940469000054
                                ],
                                [
                                    -114.743353749,
                                    36.07054029600005
                                ],
                                [
                                    -114.73617675099996,
                                    36.10437230100007
                                ],
                                [
                                    -114.66654973899995,
                                    36.11734830700004
                                ],
                                [
                                    -114.62786673399995,
                                    36.14101731200003
                                ],
                                [
                                    -114.57204272499996,
                                    36.15161531700005
                                ],
                                [
                                    -114.51173271399995,
                                    36.15096132000008
                                ],
                                [
                                    -114.48704570599995,
                                    36.129401320000056
                                ],
                                [
                                    -114.44866569899995,
                                    36.12641532200007
                                ],
                                [
                                    -114.41696169499994,
                                    36.14576632600006
                                ],
                                [
                                    -114.37211768699996,
                                    36.14311932800007
                                ],
                                [
                                    -114.33728467699996,
                                    36.108025327000064
                                ],
                                [
                                    -114.31612066899999,
                                    36.063114323000036
                                ],
                                [
                                    -114.27065665699996,
                                    36.035725322000076
                                ],
                                [
                                    -114.21370164499996,
                                    36.01561832300007
                                ],
                                [
                                    -114.15173663399997,
                                    36.02456832800004
                                ],
                                [
                                    -114.13821463499994,
                                    36.053166332000046
                                ],
                                [
                                    -114.09988163299994,
                                    36.12165934300003
                                ],
                                [
                                    -114.04684963099999,
                                    36.194074355000055
                                ],
                                [
                                    -114.04823763899998,
                                    36.26887936400004
                                ],
                                [
                                    -114.04759564299997,
                                    36.32557837100006
                                ],
                                [
                                    -114.04950467199996,
                                    36.60406440400004
                                ],
                                [
                                    -114.05017247099994,
                                    36.843146972000056
                                ],
                                [
                                    -114.05061171199998,
                                    37.00040145100007
                                ],
                                [
                                    -113.96591869599996,
                                    37.00003045600005
                                ],
                                [
                                    -112.96648250499999,
                                    37.00022452200005
                                ],
                                [
                                    -112.89920238899998,
                                    37.000306755000054
                                ],
                                [
                                    -112.82951342199999,
                                    37.00039193300006
                                ],
                                [
                                    -112.54510542199995,
                                    37.00073954900006
                                ],
                                [
                                    -112.53858242099994,
                                    37.00074967900008
                                ],
                                [
                                    -112.35770138499998,
                                    37.00103056200004
                                ],
                                [
                                    -111.41279518599998,
                                    37.001483310000026
                                ],
                                [
                                    -111.40588019699999,
                                    37.00148662400005
                                ],
                                [
                                    -111.27829717099996,
                                    37.00047063200003
                                ],
                                [
                                    -111.06650712799996,
                                    37.00239464600003
                                ],
                                [
                                    -110.75070106399994,
                                    37.00320266600005
                                ],
                                [
                                    -110.75069754299994,
                                    37.003202602000044
                                ],
                                [
                                    -110.47020100599997,
                                    36.99800268400003
                                ],
                                [
                                    -110.00069272799999,
                                    36.99797371500006
                                ],
                                [
                                    -110.00068790999995,
                                    36.99797371400007
                                ],
                                [
                                    -109.49534880499999,
                                    36.99911074600004
                                ],
                                [
                                    -109.04523371,
                                    36.99908977500007
                                ],
                                [
                                    -109.04544369799999,
                                    36.87459475800006
                                ],
                                [
                                    -109.04573962599994,
                                    36.11703365500006
                                ],
                                [
                                    -109.04588223499997,
                                    36.002344074000064
                                ],
                                [
                                    -109.04603460399994,
                                    35.87980562100006
                                ],
                                [
                                    -109.04630658099995,
                                    35.61425658400003
                                ],
                                [
                                    -109.04680656,
                                    35.36361155000003
                                ],
                                [
                                    -109.04636450999999,
                                    35.174689181000076
                                ],
                                [
                                    -109.04586152499996,
                                    34.95972349200008
                                ],
                                [
                                    -109.04614942899997,
                                    34.57929653200006
                                ],
                                [
                                    -109.04619248899996,
                                    34.52239843000007
                                ],
                                [
                                    -109.04643643899999,
                                    33.875057336000054
                                ],
                                [
                                    -109.04661789,
                                    33.77823769300005
                                ],
                                [
                                    -109.04730840499997,
                                    33.40978826700007
                                ],
                                [
                                    -109.04724738999994,
                                    33.20897023800006
                                ],
                                [
                                    -109.04712736,
                                    32.77757517400005
                                ],
                                [
                                    -109.04712736199997,
                                    32.77757417400005
                                ],
                                [
                                    -109.04762233699995,
                                    32.42638212000003
                                ],
                                [
                                    -109.04830631499999,
                                    32.08409806900005
                                ],
                                [
                                    -109.04920529699996,
                                    31.79655602400004
                                ],
                                [
                                    -109.05005426899999,
                                    31.332506953000063
                                ],
                                [
                                    -109.82969942999995,
                                    31.334071920000042
                                ],
                                [
                                    -110.46018316799996,
                                    31.333146271000032
                                ],
                                [
                                    -111.07483567999998,
                                    31.332243865000066
                                ],
                                [
                                    -111.36698011499999,
                                    31.424829153000076
                                ],
                                [
                                    -112.36505395799998,
                                    31.74113486600004
                                ],
                                [
                                    -113.33377913599998,
                                    32.04025394200005
                                ],
                                [
                                    -113.75076724999997,
                                    32.169009860000074
                                ],
                                [
                                    -114.81362446799994,
                                    32.49428185100004
                                ],
                                [
                                    -114.81154747,
                                    32.52283885500003
                                ],
                                [
                                    -114.79564646899996,
                                    32.55096085900004
                                ],
                                [
                                    -114.81419647399997,
                                    32.56479286000007
                                ],
                                [
                                    -114.79969447199994,
                                    32.593625864000046
                                ]
                            ]
                        ]
                    ]
                },
                "properties": {
                    "FID": 43,
                    "NAME": "Arizona",
                    "Obesity": 28.4,
                    "SHAPE_Length": 23.022490599978838,
                    "SHAPE_Area": 28.92130004720625
                }
            }
        ]
    }



## Export GeoJSON

Once you've filtered your GeoJSON object you can easily export it as a new GeoJSON file using the `save_geojson` function. 


```python
pdg.save_geojson(new_geojson,'Filtered.geojson',indent=4)
```

## Create Empty GeoJSON

How about creating a GeoJSON object from scratch? You can do this programmically through the package. Below is an example of how to create an empty GeoJSON object.


```python
import pandas_geojson as pdg
geojson = pdg.GeoJSON()
geojson
```




    {
        "type": "FeatureCollection",
        "features": []
    }



## Adding Features

Building on our blank GeoJSON object, we can add features to it using the `add_feature` function. Simply pass the Feature Type, and the coordinates. Optionally you can pass properties as a dictionary.


```python
geojson.add_feature(feature_type='Point',coordinates=[-105.63012379499997, 32.971263161000024],properties={'ID':1})
geojson
```




    {
        "type": "FeatureCollection",
        "features": [
            {
                "type": "Feature",
                "geometry": {
                    "type": "Point",
                    "coordinates": [
                        -105.63012379499997,
                        32.971263161000024
                    ]
                },
                "properties": {
                    "ID": 1
                }
            }
        ]
    }


