"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CrowApi = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const lambda = require("aws-cdk-lib/aws-lambda");
const apigateway = require("aws-cdk-lib/aws-apigateway");
const logs = require("aws-cdk-lib/aws-logs");
/**
 * For copying shared code to all paths
 */
const fse = require("fs-extra");
const DEFAULT_LAMBDA_CODE = `
exports.handler = async function (event, context, callback) {
  try {
    const data = {
      statusCode: 201,
    };
    return data;
  } catch (uncaughtError) {
    console.error(uncaughtError);
    throw uncaughtError;
  }
}
`;
/**
 * @experimental
 */
class CrowApi extends constructs_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Pulling out props
        const { sourceDirectory = 'src', sharedDirectory = 'shared', useAuthorizerLambda = false, authorizerDirectory = 'authorizer', authorizerLambdaConfiguration = {}, tokenAuthorizerConfiguration = {}, createApiKey = false, logRetention = logs.RetentionDays.ONE_WEEK, apiGatewayConfiguration = {}, apiGatewayName = 'crow-api', lambdaConfigurations = {}, methodConfigurations = {}, } = props;
        // Initializing constants
        const LAMBDA_RUNTIME = lambda.Runtime.NODEJS_14_X;
        const SPECIAL_DIRECTORIES = [
            sharedDirectory,
            authorizerDirectory,
        ];
        // Helpers functions for constructor
        // Prepares default Lambda props and overrides them with user input
        function bundleLambdaProps(codePath, userConfiguration, sharedLayer) {
            let layers;
            if (sharedLayer) {
                const { layers: userLayers = [], } = userConfiguration;
                layers = [sharedLayer, ...userLayers];
            }
            const defaultProps = {
                runtime: LAMBDA_RUNTIME,
                code: lambda.Code.fromAsset(codePath),
                handler: 'index.handler',
                logRetention,
            };
            const lambdaProps = {
                ...defaultProps,
                ...userConfiguration,
                layers,
            };
            return lambdaProps;
        }
        // Returns child directories given the path of a parent
        function getDirectoryChildren(parentDirectory) {
            try {
                const directories = fse.readdirSync(parentDirectory, { withFileTypes: true })
                    .filter((dirent) => dirent.isDirectory())
                    .map((dirent) => dirent.name);
                return directories;
            }
            catch {
                /**
                 * The only time I have run into this was when the src/ directory
                 * was empty.
                 * If it is empty, let CDK tree validation tell user that the
                 * REST API does not have any methods.
                 */
            }
            return [];
        }
        // A default Lambda function is needed for the API Gateway
        const defaultLambda = new lambda.Function(this, 'default-crow-lambda', {
            runtime: LAMBDA_RUNTIME,
            code: new lambda.InlineCode(DEFAULT_LAMBDA_CODE),
            handler: 'index.handler',
            logRetention,
        });
        // API Gateway log group
        const gatewayLogGroup = new logs.LogGroup(this, 'api-access-logs', {
            retention: logs.RetentionDays.ONE_WEEK,
        });
        // The API Gateway itself
        const gateway = new apigateway.LambdaRestApi(this, apiGatewayName, {
            handler: defaultLambda,
            proxy: false,
            deploy: true,
            deployOptions: {
                loggingLevel: apigateway.MethodLoggingLevel.ERROR,
                accessLogDestination: new apigateway.LogGroupLogDestination(gatewayLogGroup),
            },
            apiKeySourceType: createApiKey ? apigateway.ApiKeySourceType.HEADER : undefined,
            ...apiGatewayConfiguration,
        });
        // Create API key if desired
        if (createApiKey) {
            const apiKey = gateway.addApiKey('api-key');
            const usagePlan = new apigateway.UsagePlan(this, 'usage-plan', {
                throttle: {
                    burstLimit: 5000,
                    rateLimit: 10000,
                },
                apiStages: [
                    {
                        api: gateway,
                        stage: gateway.deploymentStage,
                    },
                ],
            });
            usagePlan.addApiKey(apiKey);
        }
        // Create Lambda layer out of shared directory if it exists
        const sourceSharedDirectory = `${sourceDirectory}/${sharedDirectory}`;
        let sharedLayer;
        if (fse.existsSync(sourceSharedDirectory)) {
            sharedLayer = new lambda.LayerVersion(this, 'shared-layer', {
                code: lambda.Code.fromAsset(sourceSharedDirectory),
                compatibleRuntimes: [LAMBDA_RUNTIME],
                removalPolicy: cdk.RemovalPolicy.DESTROY,
            });
            this.lambdaLayer = sharedLayer;
        }
        // Create Lambda authorizer to be used in subsequent Methods
        let tokenAuthorizer;
        if (useAuthorizerLambda) {
            const fullAuthorizerDirectory = `${sourceDirectory}/${authorizerDirectory}`;
            const authorizerLambdaProps = bundleLambdaProps(fullAuthorizerDirectory, authorizerLambdaConfiguration, sharedLayer);
            const authorizerLambda = new lambda.Function(this, 'authorizer-lambda', authorizerLambdaProps);
            this.authorizerLambda = authorizerLambda;
            const bundledTokenAuthConfig = {
                handler: authorizerLambda,
                resultsCacheTtl: cdk.Duration.seconds(3600),
                ...tokenAuthorizerConfiguration,
            };
            tokenAuthorizer = new apigateway.TokenAuthorizer(this, 'token-authorizer', bundledTokenAuthConfig);
        }
        // Time to start walking the directories
        const root = sourceDirectory;
        const verbs = ['get', 'post', 'put', 'delete'];
        const graph = {};
        const lambdasByPath = {};
        // Initialize with root
        graph['/'] = {
            resource: gateway.root,
            path: root,
            paths: [],
            verbs: [],
        };
        // First element in tuple is directory path, second is API path
        const nodes = [[root, '/']];
        // BFS that creates API Gateway structure using addMethod
        while (nodes.length) {
            // The `|| ['type', 'script']` piece is needed or TS throws a fit
            const [directoryPath, apiPath] = nodes.shift() || ['type', 'script'];
            const children = getDirectoryChildren(directoryPath);
            // For debugging purposes
            // console.log(`${apiPath}'s children are: ${children}`);
            // Don't have to worry about previously visited nodes
            // since this is a file structure
            // ...unless there are symlinks? Haven't run into that
            children.forEach((child) => {
                const newDirectoryPath = `${directoryPath}/${child}`;
                // If we're on the root path, don't separate with a slash (/)
                //   because it ends up looking like //child-path
                const newApiPath = apiPath === '/' ? `/${child}` : `${apiPath}/${child}`;
                if (verbs.includes(child)) {
                    // If directory is a verb, we don't traverse it anymore
                    //   and need to create an API Gateway method and Lambda
                    const userLambdaConfiguration = lambdaConfigurations[newApiPath]
                        || {};
                    const lambdaProps = bundleLambdaProps(newDirectoryPath, userLambdaConfiguration, sharedLayer);
                    const newLambda = new lambda.Function(this, newDirectoryPath, lambdaProps);
                    // Pull out useAuthorizerLambda value
                    const { useAuthorizerLambda: authorizerLambdaConfigured, ...userMethodConfiguration } = methodConfigurations[newApiPath] || {};
                    let methodConfiguration = userMethodConfiguration;
                    // If this method should be behind an authorizer Lambda
                    //   construct the methodConfiguration object as such
                    if (authorizerLambdaConfigured && useAuthorizerLambda) {
                        methodConfiguration = {
                            ...userMethodConfiguration,
                            authorizationType: apigateway.AuthorizationType.CUSTOM,
                            authorizer: tokenAuthorizer,
                        };
                    }
                    graph[apiPath].resource.addMethod(child.toUpperCase(), new apigateway.LambdaIntegration(newLambda), methodConfiguration);
                    graph[apiPath].verbs.push(child);
                    lambdasByPath[newApiPath] = newLambda;
                }
                else if (SPECIAL_DIRECTORIES.includes(child)) {
                    // The special directories should not result in an API path
                    // This means the API also cannot have a resource with the
                    //   same name
                }
                else {
                    // If directory is not a verb, create new API Gateway resource
                    //   for use by verb directory later
                    const newResource = graph[apiPath].resource
                        .resourceForPath(child);
                    nodes.push([newDirectoryPath, newApiPath]);
                    // Add child to parent's paths
                    graph[apiPath].paths.push(child);
                    // Initialize graph node to include child
                    graph[newApiPath] = {
                        resource: newResource,
                        path: newDirectoryPath,
                        paths: [],
                        verbs: [],
                    };
                }
            });
        }
        // For debugging purposes
        // console.log(graph);
        // Expose API Gateway
        this.gateway = gateway;
        this.lambdaFunctions = lambdasByPath;
    }
}
exports.CrowApi = CrowApi;
_a = JSII_RTTI_SYMBOL_1;
CrowApi[_a] = { fqn: "crow-api.CrowApi", version: "2.1.2" };
//# sourceMappingURL=data:application/json;base64,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