"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const cdk = require("aws-cdk-lib");
const kinesis = require("aws-cdk-lib/aws-kinesis");
const defaults = require("@aws-solutions-constructs/core");
require("@aws-cdk/assert/jest");
const assertions_1 = require("aws-cdk-lib/assertions");
test('Default construct has all expected properties', () => {
    const stack = new cdk.Stack();
    const construct = new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
    });
    expect(construct.vpc).toBeDefined();
    expect(construct.service).toBeDefined();
    expect(construct.container).toBeDefined();
    expect(construct.kinesisStream).toBeDefined();
    expect(construct.cloudwatchAlarms).toBeDefined();
});
test('Construct deploys Fargate Service in isolated subnets when publicApi is set to false', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
    });
    const template = assertions_1.Template.fromStack(stack);
    // The default isolated VPC should have two subnets, 2 route tables, and no nat/internet gateway, or routes
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toCountResources('AWS::EC2::Subnet', 2);
    expect(stack).toCountResources('AWS::EC2::RouteTable', 2);
    expect(stack).toCountResources('AWS::EC2::Route', 0);
    expect(stack).toCountResources('AWS::EC2::NatGateway', 0);
    expect(stack).toCountResources('AWS::EC2::InternetGateway', 0);
    // Get the first isolated subnet
    const subnet1Resource = template.findResources('AWS::EC2::Subnet', {
        Properties: {
            MapPublicIpOnLaunch: false,
            Tags: [
                {
                    Key: assertions_1.Match.exact("Name"),
                    Value: assertions_1.Match.exact("Default/Vpc/IsolatedSubnet1")
                }
            ]
        }
    });
    const [subnet1] = Object.keys(subnet1Resource);
    // Get the second isolated subnet
    const subnet2Resource = template.findResources('AWS::EC2::Subnet', {
        Properties: {
            MapPublicIpOnLaunch: false,
            Tags: [
                {
                    Key: assertions_1.Match.exact("Name"),
                    Value: assertions_1.Match.exact("Default/Vpc/IsolatedSubnet2")
                }
            ]
        }
    });
    const [subnet2] = Object.keys(subnet2Resource);
    // Verify the Fargate Service is deployed into the two isolated subnets
    expect(stack).toHaveResourceLike('AWS::ECS::Service', {
        NetworkConfiguration: {
            AwsvpcConfiguration: {
                AssignPublicIp: "DISABLED",
                Subnets: [
                    {
                        Ref: subnet1
                    },
                    {
                        Ref: subnet2
                    }
                ]
            },
        }
    });
});
test('Construct deploys Fargate Service in private subnets when publicApi is set to true', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
    });
    // The default public/private VPC should have 4 subnets (two public, two private)
    // 4 route tables, 4 routes, 2 NAT Gateways and 1 Internet Gateway
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toCountResources('AWS::EC2::Subnet', 4);
    expect(stack).toCountResources('AWS::EC2::RouteTable', 4);
    expect(stack).toCountResources('AWS::EC2::Route', 4);
    expect(stack).toCountResources('AWS::EC2::NatGateway', 2);
    expect(stack).toCountResources('AWS::EC2::InternetGateway', 1);
    const template = assertions_1.Template.fromStack(stack);
    // Get the first private subnet
    const privateSubnet1Resource = template.findResources('AWS::EC2::Subnet', {
        Properties: {
            MapPublicIpOnLaunch: false,
            Tags: [
                {
                    Key: assertions_1.Match.exact("Name"),
                    Value: assertions_1.Match.exact("Default/Vpc/PrivateSubnet1")
                }
            ]
        }
    });
    const [privateSubnet1] = Object.keys(privateSubnet1Resource);
    // Get the second private subnet
    const privateSubnet2Resource = template.findResources('AWS::EC2::Subnet', {
        Properties: {
            MapPublicIpOnLaunch: false,
            Tags: [
                {
                    Key: assertions_1.Match.exact("Name"),
                    Value: assertions_1.Match.exact("Default/Vpc/PrivateSubnet2")
                }
            ]
        }
    });
    const [privateSubnet2] = Object.keys(privateSubnet2Resource);
    // Verify the Fargate Service is deployed into the two private subnets
    expect(stack).toHaveResourceLike('AWS::ECS::Service', {
        NetworkConfiguration: {
            AwsvpcConfiguration: {
                AssignPublicIp: "DISABLED",
                Subnets: [
                    {
                        Ref: privateSubnet1
                    },
                    {
                        Ref: privateSubnet2
                    }
                ]
            },
        }
    });
});
test('Construct uses vpcProps when provided', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: {
            vpcName: 'my-vpc'
        }
    });
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toHaveResourceLike('AWS::EC2::VPC', {
        Tags: [
            {
                Key: 'Name',
                Value: 'my-vpc'
            }
        ]
    });
});
test('Construct uses existingVpc when provided', () => {
    const stack = new cdk.Stack();
    const existingVpc = defaults.getTestVpc(stack, false, {
        vpcName: 'my-vpc'
    });
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        existingVpc
    });
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toHaveResourceLike('AWS::EC2::VPC', {
        Tags: [
            {
                Key: 'Name',
                Value: 'my-vpc'
            }
        ]
    });
});
test('Construct creates VPC Interface Endpoints for ECR and Kinesis Streams', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn
    });
    expect(stack).toHaveResource('AWS::EC2::VPCEndpoint', {
        ServiceName: {
            'Fn::Join': [
                '',
                [
                    'com.amazonaws.',
                    {
                        Ref: 'AWS::Region'
                    },
                    '.kinesis-streams'
                ]
            ]
        },
    });
    expect(stack).toHaveResource('AWS::EC2::VPCEndpoint', {
        ServiceName: {
            'Fn::Join': [
                '',
                [
                    'com.amazonaws.',
                    {
                        Ref: 'AWS::Region'
                    },
                    '.ecr.api'
                ]
            ]
        },
    });
});
test('Container has default stream name environment variable', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn
    });
    const template = assertions_1.Template.fromStack(stack);
    // Get Kinesis Data Stream created by the construct
    const kinesisStream = template.findResources('AWS::Kinesis::Stream');
    const [kinesisStreamId] = Object.keys(kinesisStream);
    expect(stack).toHaveResourceLike('AWS::ECS::TaskDefinition', {
        ContainerDefinitions: [
            {
                Environment: [
                    {
                        Name: 'KINESIS_DATASTREAM_NAME',
                        Value: {
                            Ref: kinesisStreamId
                        }
                    }
                ],
            }
        ]
    });
});
test('Container stream name environment variable can be overridden', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        streamEnvironmentVariableName: 'my-stream-name'
    });
    const template = assertions_1.Template.fromStack(stack);
    // Get Kinesis Data Stream created by the construct
    const kinesisStream = template.findResources('AWS::Kinesis::Stream');
    const [kinesisStreamId] = Object.keys(kinesisStream);
    expect(stack).toHaveResourceLike('AWS::ECS::TaskDefinition', {
        ContainerDefinitions: [
            {
                Environment: [
                    {
                        Name: 'my-stream-name',
                        Value: {
                            Ref: kinesisStreamId
                        }
                    }
                ],
            }
        ]
    });
});
test('Kinesis Stream is encrypted with AWS-managed CMK by default', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
    });
    expect(stack).toHaveResourceLike('AWS::Kinesis::Stream', {
        StreamEncryption: {
            EncryptionType: 'KMS',
            KeyId: 'alias/aws/kinesis'
        }
    });
});
test('CloudWatch Alarms are created for Kinesis Stream by default', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
    });
    expect(stack).toHaveResourceLike('AWS::CloudWatch::Alarm', {
        Namespace: 'AWS/Kinesis',
        MetricName: 'GetRecords.IteratorAgeMilliseconds'
    });
    expect(stack).toHaveResourceLike('AWS::CloudWatch::Alarm', {
        Namespace: 'AWS/Kinesis',
        MetricName: 'ReadProvisionedThroughputExceeded'
    });
});
test('CloudWatch Alarms are not created when createCloudWatchAlarms property is set to false', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        createCloudWatchAlarms: false
    });
    expect(stack).toCountResources('AWS::CloudWatch::Alarm', 0);
});
test('Construct uses existingStreamObj when provided', () => {
    const stack = new cdk.Stack();
    const existingStreamObj = new kinesis.Stream(stack, 'test-stream', {
        streamName: 'my-stream',
    });
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        existingStreamObj
    });
    expect(stack).toHaveResourceLike('AWS::Kinesis::Stream', {
        Name: 'my-stream'
    });
    expect(stack).toCountResources('AWS::Kinesis::Stream', 1);
});
test('Construct uses kinesisStreamProps when provided', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        kinesisStreamProps: {
            streamName: 'my-stream',
            encryption: kinesis.StreamEncryption.UNENCRYPTED
        }
    });
    expect(stack).toHaveResourceLike('AWS::Kinesis::Stream', {
        Name: 'my-stream'
    });
    expect(stack).not.toHaveResourceLike('AWS::Kinesis::Stream', {
        StreamEncryption: {
            EncryptionType: 'KMS',
            KeyId: 'alias/aws/kinesis'
        }
    });
    expect(stack).toCountResources('AWS::Kinesis::Stream', 1);
});
test('Construct grants PutRecord permission for the Fargate Service to write to the Kinesis Stream', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
    });
    const template = assertions_1.Template.fromStack(stack);
    // Get Kinesis Data Stream created by the construct
    const kinesisStream = template.findResources('AWS::Kinesis::Stream');
    const [kinesisStreamId] = Object.keys(kinesisStream);
    expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'kinesis:ListShards',
                        'kinesis:PutRecord',
                        'kinesis:PutRecords'
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            kinesisStreamId,
                            'Arn'
                        ]
                    }
                }
            ]
        }
    });
});
test('Construct defaults to the latest version of the ECR image', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
    });
    expect(stack).toHaveResourceLike('AWS::ECS::TaskDefinition', {
        ContainerDefinitions: [
            {
                Image: {
                    'Fn::Join': [
                        '',
                        [
                            '123456789012.dkr.ecr.us-east-1.',
                            {
                                Ref: 'AWS::URLSuffix'
                            },
                            '/fake-repo:latest'
                        ]
                    ]
                },
            }
        ]
    });
});
test('Construct uses ecrImageVersion when provided', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        ecrImageVersion: 'my-version'
    });
    expect(stack).toHaveResourceLike('AWS::ECS::TaskDefinition', {
        ContainerDefinitions: [
            {
                Image: {
                    'Fn::Join': [
                        '',
                        [
                            '123456789012.dkr.ecr.us-east-1.',
                            {
                                Ref: 'AWS::URLSuffix'
                            },
                            '/fake-repo:my-version'
                        ]
                    ]
                },
            }
        ]
    });
});
test('Construct uses clusterProps when provided', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        clusterProps: {
            clusterName: 'my-cluster'
        }
    });
    expect(stack).toCountResources('AWS::ECS::Cluster', 1);
    expect(stack).toHaveResourceLike("AWS::ECS::Cluster", {
        ClusterName: 'my-cluster'
    });
});
test('Construct uses containerDefinitionProps when provided', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        containerDefinitionProps: {
            containerName: 'my-container'
        }
    });
    expect(stack).toCountResources('AWS::ECS::TaskDefinition', 1);
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        ContainerDefinitions: [
            {
                Name: 'my-container'
            }
        ]
    });
});
test('Construct uses fargateTaskDefinitionProps when provided', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        fargateTaskDefinitionProps: {
            family: 'my-family'
        }
    });
    expect(stack).toCountResources('AWS::ECS::TaskDefinition', 1);
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        Family: 'my-family'
    });
});
test('Construct uses fargateServiceProps when provided', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        fargateServiceProps: {
            serviceName: 'my-service',
            desiredCount: 7
        }
    });
    expect(stack).toCountResources('AWS::ECS::Service', 1);
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        ServiceName: 'my-service',
        DesiredCount: 7
    });
});
test('Construct uses existingFargateServiceObject when provided', () => {
    const stack = new cdk.Stack();
    const existingVpc = defaults.getTestVpc(stack);
    const createFargateServiceResponse = defaults.CreateFargateService(stack, 'test-existing-fargate-service', existingVpc, { clusterName: 'my-cluster' }, defaults.fakeEcrRepoArn, undefined, { family: 'my-family' }, { containerName: 'my-container' }, { serviceName: 'my-service' });
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        existingVpc,
        existingFargateServiceObject: createFargateServiceResponse.service,
        existingContainerDefinitionObject: createFargateServiceResponse.containerDefinition
    });
    expect(stack).toCountResources('AWS::ECS::Cluster', 1);
    expect(stack).toHaveResourceLike("AWS::ECS::Cluster", {
        ClusterName: 'my-cluster'
    });
    expect(stack).toCountResources('AWS::ECS::TaskDefinition', 1);
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        Family: 'my-family',
        ContainerDefinitions: [
            {
                Name: 'my-container'
            }
        ]
    });
    expect(stack).toCountResources('AWS::ECS::Service', 1);
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        ServiceName: 'my-service',
    });
});
//# sourceMappingURL=data:application/json;base64,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