"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FlowLog = exports.FlowLogDestination = exports.FlowLogResourceType = exports.FlowLogDestinationType = exports.FlowLogTrafficType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const ec2_generated_1 = require("./ec2.generated");
/**
 * The type of VPC traffic to log.
 *
 * @stability stable
 */
var FlowLogTrafficType;
(function (FlowLogTrafficType) {
    FlowLogTrafficType["ACCEPT"] = "ACCEPT";
    FlowLogTrafficType["ALL"] = "ALL";
    FlowLogTrafficType["REJECT"] = "REJECT";
})(FlowLogTrafficType = exports.FlowLogTrafficType || (exports.FlowLogTrafficType = {}));
/**
 * The available destination types for Flow Logs.
 *
 * @stability stable
 */
var FlowLogDestinationType;
(function (FlowLogDestinationType) {
    FlowLogDestinationType["CLOUD_WATCH_LOGS"] = "cloud-watch-logs";
    FlowLogDestinationType["S3"] = "s3";
})(FlowLogDestinationType = exports.FlowLogDestinationType || (exports.FlowLogDestinationType = {}));
/**
 * The type of resource to create the flow log for.
 *
 * @stability stable
 */
class FlowLogResourceType {
    /**
     * The subnet to attach the Flow Log to.
     *
     * @stability stable
     */
    static fromSubnet(subnet) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_ISubnet(subnet);
        return {
            resourceType: 'Subnet',
            resourceId: subnet.subnetId,
        };
    }
    /**
     * The VPC to attach the Flow Log to.
     *
     * @stability stable
     */
    static fromVpc(vpc) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_IVpc(vpc);
        return {
            resourceType: 'VPC',
            resourceId: vpc.vpcId,
        };
    }
    /**
     * The Network Interface to attach the Flow Log to.
     *
     * @stability stable
     */
    static fromNetworkInterfaceId(id) {
        return {
            resourceType: 'NetworkInterface',
            resourceId: id,
        };
    }
}
exports.FlowLogResourceType = FlowLogResourceType;
_a = JSII_RTTI_SYMBOL_1;
FlowLogResourceType[_a] = { fqn: "@aws-cdk/aws-ec2.FlowLogResourceType", version: "1.138.1" };
/**
 * The destination type for the flow log.
 *
 * @stability stable
 */
class FlowLogDestination {
    /**
     * Use CloudWatch logs as the destination.
     *
     * @stability stable
     */
    static toCloudWatchLogs(logGroup, iamRole) {
        return new CloudWatchLogsDestination({
            logDestinationType: FlowLogDestinationType.CLOUD_WATCH_LOGS,
            logGroup,
            iamRole,
        });
    }
    /**
     * Use S3 as the destination.
     *
     * @stability stable
     */
    static toS3(bucket, keyPrefix) {
        return new S3Destination({
            logDestinationType: FlowLogDestinationType.S3,
            s3Bucket: bucket,
            keyPrefix,
        });
    }
}
exports.FlowLogDestination = FlowLogDestination;
_b = JSII_RTTI_SYMBOL_1;
FlowLogDestination[_b] = { fqn: "@aws-cdk/aws-ec2.FlowLogDestination", version: "1.138.1" };
/**
 *
 */
class S3Destination extends FlowLogDestination {
    constructor(props) {
        super();
        this.props = props;
    }
    bind(scope, _flowLog) {
        let s3Bucket;
        if (this.props.s3Bucket === undefined) {
            s3Bucket = new s3.Bucket(scope, 'Bucket', {
                encryption: s3.BucketEncryption.UNENCRYPTED,
                removalPolicy: core_1.RemovalPolicy.RETAIN,
            });
        }
        else {
            s3Bucket = this.props.s3Bucket;
        }
        return {
            logDestinationType: FlowLogDestinationType.S3,
            s3Bucket,
            keyPrefix: this.props.keyPrefix,
        };
    }
}
/**
 *
 */
class CloudWatchLogsDestination extends FlowLogDestination {
    constructor(props) {
        super();
        this.props = props;
    }
    bind(scope, _flowLog) {
        let iamRole;
        let logGroup;
        if (this.props.iamRole === undefined) {
            iamRole = new iam.Role(scope, 'IAMRole', {
                roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
                assumedBy: new iam.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
            });
        }
        else {
            iamRole = this.props.iamRole;
        }
        if (this.props.logGroup === undefined) {
            logGroup = new logs.LogGroup(scope, 'LogGroup');
        }
        else {
            logGroup = this.props.logGroup;
        }
        iamRole.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: [
                'logs:CreateLogStream',
                'logs:PutLogEvents',
                'logs:DescribeLogStreams',
            ],
            effect: iam.Effect.ALLOW,
            resources: [logGroup.logGroupArn],
        }));
        iamRole.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            effect: iam.Effect.ALLOW,
            resources: [iamRole.roleArn],
        }));
        return {
            logDestinationType: FlowLogDestinationType.CLOUD_WATCH_LOGS,
            logGroup,
            iamRole,
        };
    }
}
/**
 * The base class for a Flow Log
 *
 *
 */
class FlowLogBase extends core_1.Resource {
}
/**
 * A VPC flow log.
 *
 * @stability stable
 * @resource AWS::EC2::FlowLog
 */
class FlowLog extends FlowLogBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.flowLogName,
        });
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_FlowLogProps(props);
        const destination = props.destination || FlowLogDestination.toCloudWatchLogs();
        const destinationConfig = destination.bind(this, this);
        this.logGroup = destinationConfig.logGroup;
        this.bucket = destinationConfig.s3Bucket;
        this.iamRole = destinationConfig.iamRole;
        this.keyPrefix = destinationConfig.keyPrefix;
        let logDestination = undefined;
        if (this.bucket) {
            logDestination = this.keyPrefix ? this.bucket.arnForObjects(this.keyPrefix) : this.bucket.bucketArn;
        }
        const flowLog = new ec2_generated_1.CfnFlowLog(this, 'FlowLog', {
            deliverLogsPermissionArn: this.iamRole ? this.iamRole.roleArn : undefined,
            logDestinationType: destinationConfig.logDestinationType,
            logGroupName: this.logGroup ? this.logGroup.logGroupName : undefined,
            resourceId: props.resourceType.resourceId,
            resourceType: props.resourceType.resourceType,
            trafficType: props.trafficType
                ? props.trafficType
                : FlowLogTrafficType.ALL,
            logDestination,
        });
        this.flowLogId = flowLog.ref;
    }
    /**
     * Import a Flow Log by it's Id.
     *
     * @stability stable
     */
    static fromFlowLogId(scope, id, flowLogId) {
        class Import extends FlowLogBase {
            constructor() {
                super(...arguments);
                this.flowLogId = flowLogId;
            }
        }
        return new Import(scope, id);
    }
}
exports.FlowLog = FlowLog;
_c = JSII_RTTI_SYMBOL_1;
FlowLog[_c] = { fqn: "@aws-cdk/aws-ec2.FlowLog", version: "1.138.1" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidnBjLWZsb3ctbG9ncy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInZwYy1mbG93LWxvZ3MudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7O0FBQUEsd0NBQXdDO0FBQ3hDLDBDQUEwQztBQUMxQyxzQ0FBc0M7QUFDdEMsd0NBQWlGO0FBRWpGLG1EQUE2Qzs7Ozs7O0FBYzdDLElBQVksa0JBU1g7QUFURCxXQUFZLGtCQUFrQjtJQUU1Qix1Q0FBaUIsQ0FBQTtJQUdqQixpQ0FBVyxDQUFBO0lBR1gsdUNBQWlCLENBQUE7QUFDbkIsQ0FBQyxFQVRXLGtCQUFrQixHQUFsQiwwQkFBa0IsS0FBbEIsMEJBQWtCLFFBUzdCOzs7Ozs7QUFHRCxJQUFZLHNCQU1YO0FBTkQsV0FBWSxzQkFBc0I7SUFFaEMsK0RBQXFDLENBQUE7SUFHckMsbUNBQVMsQ0FBQTtBQUNYLENBQUMsRUFOVyxzQkFBc0IsR0FBdEIsOEJBQXNCLEtBQXRCLDhCQUFzQixRQU1qQzs7Ozs7O0FBR0QsTUFBc0IsbUJBQW1COzs7Ozs7SUFFaEMsTUFBTSxDQUFDLFVBQVUsQ0FBQyxNQUFlOztRQUN0QyxPQUFPO1lBQ0wsWUFBWSxFQUFFLFFBQVE7WUFDdEIsVUFBVSxFQUFFLE1BQU0sQ0FBQyxRQUFRO1NBQzVCLENBQUM7S0FDSDs7Ozs7O0lBR00sTUFBTSxDQUFDLE9BQU8sQ0FBQyxHQUFTOztRQUM3QixPQUFPO1lBQ0wsWUFBWSxFQUFFLEtBQUs7WUFDbkIsVUFBVSxFQUFFLEdBQUcsQ0FBQyxLQUFLO1NBQ3RCLENBQUM7S0FDSDs7Ozs7O0lBR00sTUFBTSxDQUFDLHNCQUFzQixDQUFDLEVBQVU7UUFDN0MsT0FBTztZQUNMLFlBQVksRUFBRSxrQkFBa0I7WUFDaEMsVUFBVSxFQUFFLEVBQUU7U0FDZixDQUFDO0tBQ0g7O0FBdkJILGtEQThCQzs7Ozs7Ozs7QUFHRCxNQUFzQixrQkFBa0I7Ozs7OztJQUUvQixNQUFNLENBQUMsZ0JBQWdCLENBQUMsUUFBeUIsRUFBRSxPQUFtQjtRQUMzRSxPQUFPLElBQUkseUJBQXlCLENBQUM7WUFDbkMsa0JBQWtCLEVBQUUsc0JBQXNCLENBQUMsZ0JBQWdCO1lBQzNELFFBQVE7WUFDUixPQUFPO1NBQ1IsQ0FBQyxDQUFDO0tBQ0o7Ozs7OztJQUdNLE1BQU0sQ0FBQyxJQUFJLENBQUMsTUFBbUIsRUFBRSxTQUFrQjtRQUN4RCxPQUFPLElBQUksYUFBYSxDQUFDO1lBQ3ZCLGtCQUFrQixFQUFFLHNCQUFzQixDQUFDLEVBQUU7WUFDN0MsUUFBUSxFQUFFLE1BQU07WUFDaEIsU0FBUztTQUNWLENBQUMsQ0FBQztLQUNKOztBQWpCSCxnREFxQkM7OztBQW9CRDs7R0FFRztBQUNILE1BQU0sYUFBYyxTQUFRLGtCQUFrQjtJQUM1QyxZQUE2QixLQUErQjtRQUMxRCxLQUFLLEVBQUUsQ0FBQztRQURtQixVQUFLLEdBQUwsS0FBSyxDQUEwQjtLQUUzRDtJQUVNLElBQUksQ0FBQyxLQUFvQixFQUFFLFFBQWlCO1FBQ2pELElBQUksUUFBb0IsQ0FBQztRQUN6QixJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxLQUFLLFNBQVMsRUFBRTtZQUNyQyxRQUFRLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7Z0JBQ3hDLFVBQVUsRUFBRSxFQUFFLENBQUMsZ0JBQWdCLENBQUMsV0FBVztnQkFDM0MsYUFBYSxFQUFFLG9CQUFhLENBQUMsTUFBTTthQUNwQyxDQUFDLENBQUM7U0FDSjthQUFNO1lBQ0wsUUFBUSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDO1NBQ2hDO1FBQ0QsT0FBTztZQUNMLGtCQUFrQixFQUFFLHNCQUFzQixDQUFDLEVBQUU7WUFDN0MsUUFBUTtZQUNSLFNBQVMsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVM7U0FDaEMsQ0FBQztLQUNIO0NBQ0Y7QUFFRDs7R0FFRztBQUNILE1BQU0seUJBQTBCLFNBQVEsa0JBQWtCO0lBQ3hELFlBQTZCLEtBQStCO1FBQzFELEtBQUssRUFBRSxDQUFDO1FBRG1CLFVBQUssR0FBTCxLQUFLLENBQTBCO0tBRTNEO0lBRU0sSUFBSSxDQUFDLEtBQW9CLEVBQUUsUUFBaUI7UUFDakQsSUFBSSxPQUFrQixDQUFDO1FBQ3ZCLElBQUksUUFBd0IsQ0FBQztRQUM3QixJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxLQUFLLFNBQVMsRUFBRTtZQUNwQyxPQUFPLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUU7Z0JBQ3ZDLFFBQVEsRUFBRSxtQkFBWSxDQUFDLGtCQUFrQjtnQkFDekMsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLDZCQUE2QixDQUFDO2FBQ25FLENBQUMsQ0FBQztTQUNKO2FBQU07WUFDTCxPQUFPLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUM7U0FDOUI7UUFFRCxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxLQUFLLFNBQVMsRUFBRTtZQUNyQyxRQUFRLEdBQUcsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxVQUFVLENBQUMsQ0FBQztTQUNqRDthQUFNO1lBQ0wsUUFBUSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDO1NBQ2hDO1FBRUQsT0FBTyxDQUFDLG9CQUFvQixDQUMxQixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDdEIsT0FBTyxFQUFFO2dCQUNQLHNCQUFzQjtnQkFDdEIsbUJBQW1CO2dCQUNuQix5QkFBeUI7YUFDMUI7WUFDRCxNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO1lBQ3hCLFNBQVMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUM7U0FDbEMsQ0FBQyxDQUNILENBQUM7UUFFRixPQUFPLENBQUMsb0JBQW9CLENBQzFCLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUN0QixPQUFPLEVBQUUsQ0FBQyxjQUFjLENBQUM7WUFDekIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztZQUN4QixTQUFTLEVBQUUsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDO1NBQzdCLENBQUMsQ0FDSCxDQUFDO1FBRUYsT0FBTztZQUNMLGtCQUFrQixFQUFFLHNCQUFzQixDQUFDLGdCQUFnQjtZQUMzRCxRQUFRO1lBQ1IsT0FBTztTQUNSLENBQUM7S0FDSDtDQUNGO0FBb0JEOzs7O0dBSUc7QUFDSCxNQUFlLFdBQVksU0FBUSxlQUFRO0NBTzFDOzs7Ozs7O0FBR0QsTUFBYSxPQUFRLFNBQVEsV0FBVzs7OztJQXlCdEMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFtQjtRQUMzRCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUNmLFlBQVksRUFBRSxLQUFLLENBQUMsV0FBVztTQUNoQyxDQUFDLENBQUM7O1FBRUgsTUFBTSxXQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsSUFBSSxrQkFBa0IsQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDO1FBRS9FLE1BQU0saUJBQWlCLEdBQUcsV0FBVyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFDdkQsSUFBSSxDQUFDLFFBQVEsR0FBRyxpQkFBaUIsQ0FBQyxRQUFRLENBQUM7UUFDM0MsSUFBSSxDQUFDLE1BQU0sR0FBRyxpQkFBaUIsQ0FBQyxRQUFRLENBQUM7UUFDekMsSUFBSSxDQUFDLE9BQU8sR0FBRyxpQkFBaUIsQ0FBQyxPQUFPLENBQUM7UUFDekMsSUFBSSxDQUFDLFNBQVMsR0FBRyxpQkFBaUIsQ0FBQyxTQUFTLENBQUM7UUFFN0MsSUFBSSxjQUFjLEdBQXVCLFNBQVMsQ0FBQztRQUNuRCxJQUFJLElBQUksQ0FBQyxNQUFNLEVBQUU7WUFDZixjQUFjLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQztTQUNyRztRQUVELE1BQU0sT0FBTyxHQUFHLElBQUksMEJBQVUsQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFO1lBQzlDLHdCQUF3QixFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQ3pFLGtCQUFrQixFQUFFLGlCQUFpQixDQUFDLGtCQUFrQjtZQUN4RCxZQUFZLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDcEUsVUFBVSxFQUFFLEtBQUssQ0FBQyxZQUFZLENBQUMsVUFBVTtZQUN6QyxZQUFZLEVBQUUsS0FBSyxDQUFDLFlBQVksQ0FBQyxZQUFZO1lBQzdDLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVztnQkFDNUIsQ0FBQyxDQUFDLEtBQUssQ0FBQyxXQUFXO2dCQUNuQixDQUFDLENBQUMsa0JBQWtCLENBQUMsR0FBRztZQUMxQixjQUFjO1NBQ2YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFNBQVMsR0FBRyxPQUFPLENBQUMsR0FBRyxDQUFDO0tBQzlCOzs7Ozs7SUF0RE0sTUFBTSxDQUFDLGFBQWEsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxTQUFpQjtRQUN6RSxNQUFNLE1BQU8sU0FBUSxXQUFXO1lBQWhDOztnQkFDUyxjQUFTLEdBQUcsU0FBUyxDQUFDO1lBQy9CLENBQUM7U0FBQTtRQUVELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0tBQzlCOztBQVJILDBCQXlEQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGxvZ3MgZnJvbSAnQGF3cy1jZGsvYXdzLWxvZ3MnO1xuaW1wb3J0ICogYXMgczMgZnJvbSAnQGF3cy1jZGsvYXdzLXMzJztcbmltcG9ydCB7IElSZXNvdXJjZSwgUGh5c2ljYWxOYW1lLCBSZW1vdmFsUG9saWN5LCBSZXNvdXJjZSB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBDZm5GbG93TG9nIH0gZnJvbSAnLi9lYzIuZ2VuZXJhdGVkJztcbmltcG9ydCB7IElTdWJuZXQsIElWcGMgfSBmcm9tICcuL3ZwYyc7XG5cbi8vIHYyIC0ga2VlcCB0aGlzIGltcG9ydCBhcyBhIHNlcGFyYXRlIHNlY3Rpb24gdG8gcmVkdWNlIG1lcmdlIGNvbmZsaWN0IHdoZW4gZm9yd2FyZCBtZXJnaW5nIHdpdGggdGhlIHYyIGJyYW5jaC5cbi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZVxuaW1wb3J0IHsgQ29uc3RydWN0IGFzIENvcmVDb25zdHJ1Y3QgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgSUZsb3dMb2cgZXh0ZW5kcyBJUmVzb3VyY2Uge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBmbG93TG9nSWQ6IHN0cmluZztcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBlbnVtIEZsb3dMb2dUcmFmZmljVHlwZSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgQUNDRVBUID0gJ0FDQ0VQVCcsXG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBBTEwgPSAnQUxMJyxcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIFJFSkVDVCA9ICdSRUpFQ1QnXG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgZW51bSBGbG93TG9nRGVzdGluYXRpb25UeXBlIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIENMT1VEX1dBVENIX0xPR1MgPSAnY2xvdWQtd2F0Y2gtbG9ncycsXG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIFMzID0gJ3MzJ1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgRmxvd0xvZ1Jlc291cmNlVHlwZSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGZyb21TdWJuZXQoc3VibmV0OiBJU3VibmV0KTogRmxvd0xvZ1Jlc291cmNlVHlwZSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIHJlc291cmNlVHlwZTogJ1N1Ym5ldCcsXG4gICAgICByZXNvdXJjZUlkOiBzdWJuZXQuc3VibmV0SWQsXG4gICAgfTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGZyb21WcGModnBjOiBJVnBjKTogRmxvd0xvZ1Jlc291cmNlVHlwZSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIHJlc291cmNlVHlwZTogJ1ZQQycsXG4gICAgICByZXNvdXJjZUlkOiB2cGMudnBjSWQsXG4gICAgfTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgZnJvbU5ldHdvcmtJbnRlcmZhY2VJZChpZDogc3RyaW5nKTogRmxvd0xvZ1Jlc291cmNlVHlwZSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIHJlc291cmNlVHlwZTogJ05ldHdvcmtJbnRlcmZhY2UnLFxuICAgICAgcmVzb3VyY2VJZDogaWQsXG4gICAgfTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWJzdHJhY3QgcmVzb3VyY2VUeXBlOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhYnN0cmFjdCByZXNvdXJjZUlkOiBzdHJpbmc7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGFic3RyYWN0IGNsYXNzIEZsb3dMb2dEZXN0aW5hdGlvbiB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgdG9DbG91ZFdhdGNoTG9ncyhsb2dHcm91cD86IGxvZ3MuSUxvZ0dyb3VwLCBpYW1Sb2xlPzogaWFtLklSb2xlKTogRmxvd0xvZ0Rlc3RpbmF0aW9uIHtcbiAgICByZXR1cm4gbmV3IENsb3VkV2F0Y2hMb2dzRGVzdGluYXRpb24oe1xuICAgICAgbG9nRGVzdGluYXRpb25UeXBlOiBGbG93TG9nRGVzdGluYXRpb25UeXBlLkNMT1VEX1dBVENIX0xPR1MsXG4gICAgICBsb2dHcm91cCxcbiAgICAgIGlhbVJvbGUsXG4gICAgfSk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgdG9TMyhidWNrZXQ/OiBzMy5JQnVja2V0LCBrZXlQcmVmaXg/OiBzdHJpbmcpOiBGbG93TG9nRGVzdGluYXRpb24ge1xuICAgIHJldHVybiBuZXcgUzNEZXN0aW5hdGlvbih7XG4gICAgICBsb2dEZXN0aW5hdGlvblR5cGU6IEZsb3dMb2dEZXN0aW5hdGlvblR5cGUuUzMsXG4gICAgICBzM0J1Y2tldDogYnVja2V0LFxuICAgICAga2V5UHJlZml4LFxuICAgIH0pO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWJzdHJhY3QgYmluZChzY29wZTogQ29yZUNvbnN0cnVjdCwgZmxvd0xvZzogRmxvd0xvZyk6IEZsb3dMb2dEZXN0aW5hdGlvbkNvbmZpZztcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEZsb3dMb2dEZXN0aW5hdGlvbkNvbmZpZyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBsb2dEZXN0aW5hdGlvblR5cGU6IEZsb3dMb2dEZXN0aW5hdGlvblR5cGU7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBpYW1Sb2xlPzogaWFtLklSb2xlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbG9nR3JvdXA/OiBsb2dzLklMb2dHcm91cDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBzM0J1Y2tldD86IHMzLklCdWNrZXQ7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBrZXlQcmVmaXg/OiBzdHJpbmc7XG59XG5cbi8qKlxuICpcbiAqL1xuY2xhc3MgUzNEZXN0aW5hdGlvbiBleHRlbmRzIEZsb3dMb2dEZXN0aW5hdGlvbiB7XG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgcHJvcHM6IEZsb3dMb2dEZXN0aW5hdGlvbkNvbmZpZykge1xuICAgIHN1cGVyKCk7XG4gIH1cblxuICBwdWJsaWMgYmluZChzY29wZTogQ29yZUNvbnN0cnVjdCwgX2Zsb3dMb2c6IEZsb3dMb2cpOiBGbG93TG9nRGVzdGluYXRpb25Db25maWcge1xuICAgIGxldCBzM0J1Y2tldDogczMuSUJ1Y2tldDtcbiAgICBpZiAodGhpcy5wcm9wcy5zM0J1Y2tldCA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICBzM0J1Y2tldCA9IG5ldyBzMy5CdWNrZXQoc2NvcGUsICdCdWNrZXQnLCB7XG4gICAgICAgIGVuY3J5cHRpb246IHMzLkJ1Y2tldEVuY3J5cHRpb24uVU5FTkNSWVBURUQsXG4gICAgICAgIHJlbW92YWxQb2xpY3k6IFJlbW92YWxQb2xpY3kuUkVUQUlOLFxuICAgICAgfSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHMzQnVja2V0ID0gdGhpcy5wcm9wcy5zM0J1Y2tldDtcbiAgICB9XG4gICAgcmV0dXJuIHtcbiAgICAgIGxvZ0Rlc3RpbmF0aW9uVHlwZTogRmxvd0xvZ0Rlc3RpbmF0aW9uVHlwZS5TMyxcbiAgICAgIHMzQnVja2V0LFxuICAgICAga2V5UHJlZml4OiB0aGlzLnByb3BzLmtleVByZWZpeCxcbiAgICB9O1xuICB9XG59XG5cbi8qKlxuICpcbiAqL1xuY2xhc3MgQ2xvdWRXYXRjaExvZ3NEZXN0aW5hdGlvbiBleHRlbmRzIEZsb3dMb2dEZXN0aW5hdGlvbiB7XG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgcHJvcHM6IEZsb3dMb2dEZXN0aW5hdGlvbkNvbmZpZykge1xuICAgIHN1cGVyKCk7XG4gIH1cblxuICBwdWJsaWMgYmluZChzY29wZTogQ29yZUNvbnN0cnVjdCwgX2Zsb3dMb2c6IEZsb3dMb2cpOiBGbG93TG9nRGVzdGluYXRpb25Db25maWcge1xuICAgIGxldCBpYW1Sb2xlOiBpYW0uSVJvbGU7XG4gICAgbGV0IGxvZ0dyb3VwOiBsb2dzLklMb2dHcm91cDtcbiAgICBpZiAodGhpcy5wcm9wcy5pYW1Sb2xlID09PSB1bmRlZmluZWQpIHtcbiAgICAgIGlhbVJvbGUgPSBuZXcgaWFtLlJvbGUoc2NvcGUsICdJQU1Sb2xlJywge1xuICAgICAgICByb2xlTmFtZTogUGh5c2ljYWxOYW1lLkdFTkVSQVRFX0lGX05FRURFRCxcbiAgICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ3ZwYy1mbG93LWxvZ3MuYW1hem9uYXdzLmNvbScpLFxuICAgICAgfSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIGlhbVJvbGUgPSB0aGlzLnByb3BzLmlhbVJvbGU7XG4gICAgfVxuXG4gICAgaWYgKHRoaXMucHJvcHMubG9nR3JvdXAgPT09IHVuZGVmaW5lZCkge1xuICAgICAgbG9nR3JvdXAgPSBuZXcgbG9ncy5Mb2dHcm91cChzY29wZSwgJ0xvZ0dyb3VwJyk7XG4gICAgfSBlbHNlIHtcbiAgICAgIGxvZ0dyb3VwID0gdGhpcy5wcm9wcy5sb2dHcm91cDtcbiAgICB9XG5cbiAgICBpYW1Sb2xlLmFkZFRvUHJpbmNpcGFsUG9saWN5KFxuICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgJ2xvZ3M6Q3JlYXRlTG9nU3RyZWFtJyxcbiAgICAgICAgICAnbG9nczpQdXRMb2dFdmVudHMnLFxuICAgICAgICAgICdsb2dzOkRlc2NyaWJlTG9nU3RyZWFtcycsXG4gICAgICAgIF0sXG4gICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgcmVzb3VyY2VzOiBbbG9nR3JvdXAubG9nR3JvdXBBcm5dLFxuICAgICAgfSksXG4gICAgKTtcblxuICAgIGlhbVJvbGUuYWRkVG9QcmluY2lwYWxQb2xpY3koXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGFjdGlvbnM6IFsnaWFtOlBhc3NSb2xlJ10sXG4gICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgcmVzb3VyY2VzOiBbaWFtUm9sZS5yb2xlQXJuXSxcbiAgICAgIH0pLFxuICAgICk7XG5cbiAgICByZXR1cm4ge1xuICAgICAgbG9nRGVzdGluYXRpb25UeXBlOiBGbG93TG9nRGVzdGluYXRpb25UeXBlLkNMT1VEX1dBVENIX0xPR1MsXG4gICAgICBsb2dHcm91cCxcbiAgICAgIGlhbVJvbGUsXG4gICAgfTtcbiAgfVxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgRmxvd0xvZ09wdGlvbnMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHRyYWZmaWNUeXBlPzogRmxvd0xvZ1RyYWZmaWNUeXBlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGRlc3RpbmF0aW9uPzogRmxvd0xvZ0Rlc3RpbmF0aW9uO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgRmxvd0xvZ1Byb3BzIGV4dGVuZHMgRmxvd0xvZ09wdGlvbnMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBmbG93TG9nTmFtZT86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSByZXNvdXJjZVR5cGU6IEZsb3dMb2dSZXNvdXJjZVR5cGU7XG59XG5cbi8qKlxuICogVGhlIGJhc2UgY2xhc3MgZm9yIGEgRmxvdyBMb2dcbiAqXG4gKlxuICovXG5hYnN0cmFjdCBjbGFzcyBGbG93TG9nQmFzZSBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSUZsb3dMb2cge1xuICAvKipcbiAgICogVGhlIElkIG9mIHRoZSBWUEMgRmxvdyBMb2dcbiAgICpcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGZsb3dMb2dJZDogc3RyaW5nO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBGbG93TG9nIGV4dGVuZHMgRmxvd0xvZ0Jhc2Uge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUZsb3dMb2dJZChzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBmbG93TG9nSWQ6IHN0cmluZyk6IElGbG93TG9nIHtcbiAgICBjbGFzcyBJbXBvcnQgZXh0ZW5kcyBGbG93TG9nQmFzZSB7XG4gICAgICBwdWJsaWMgZmxvd0xvZ0lkID0gZmxvd0xvZ0lkO1xuICAgIH1cblxuICAgIHJldHVybiBuZXcgSW1wb3J0KHNjb3BlLCBpZCk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgZmxvd0xvZ0lkOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgYnVja2V0PzogczMuSUJ1Y2tldDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkga2V5UHJlZml4Pzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBpYW1Sb2xlPzogaWFtLklSb2xlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGxvZ0dyb3VwPzogbG9ncy5JTG9nR3JvdXA7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEZsb3dMb2dQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwge1xuICAgICAgcGh5c2ljYWxOYW1lOiBwcm9wcy5mbG93TG9nTmFtZSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGRlc3RpbmF0aW9uID0gcHJvcHMuZGVzdGluYXRpb24gfHwgRmxvd0xvZ0Rlc3RpbmF0aW9uLnRvQ2xvdWRXYXRjaExvZ3MoKTtcblxuICAgIGNvbnN0IGRlc3RpbmF0aW9uQ29uZmlnID0gZGVzdGluYXRpb24uYmluZCh0aGlzLCB0aGlzKTtcbiAgICB0aGlzLmxvZ0dyb3VwID0gZGVzdGluYXRpb25Db25maWcubG9nR3JvdXA7XG4gICAgdGhpcy5idWNrZXQgPSBkZXN0aW5hdGlvbkNvbmZpZy5zM0J1Y2tldDtcbiAgICB0aGlzLmlhbVJvbGUgPSBkZXN0aW5hdGlvbkNvbmZpZy5pYW1Sb2xlO1xuICAgIHRoaXMua2V5UHJlZml4ID0gZGVzdGluYXRpb25Db25maWcua2V5UHJlZml4O1xuXG4gICAgbGV0IGxvZ0Rlc3RpbmF0aW9uOiBzdHJpbmcgfCB1bmRlZmluZWQgPSB1bmRlZmluZWQ7XG4gICAgaWYgKHRoaXMuYnVja2V0KSB7XG4gICAgICBsb2dEZXN0aW5hdGlvbiA9IHRoaXMua2V5UHJlZml4ID8gdGhpcy5idWNrZXQuYXJuRm9yT2JqZWN0cyh0aGlzLmtleVByZWZpeCkgOiB0aGlzLmJ1Y2tldC5idWNrZXRBcm47XG4gICAgfVxuXG4gICAgY29uc3QgZmxvd0xvZyA9IG5ldyBDZm5GbG93TG9nKHRoaXMsICdGbG93TG9nJywge1xuICAgICAgZGVsaXZlckxvZ3NQZXJtaXNzaW9uQXJuOiB0aGlzLmlhbVJvbGUgPyB0aGlzLmlhbVJvbGUucm9sZUFybiA6IHVuZGVmaW5lZCxcbiAgICAgIGxvZ0Rlc3RpbmF0aW9uVHlwZTogZGVzdGluYXRpb25Db25maWcubG9nRGVzdGluYXRpb25UeXBlLFxuICAgICAgbG9nR3JvdXBOYW1lOiB0aGlzLmxvZ0dyb3VwID8gdGhpcy5sb2dHcm91cC5sb2dHcm91cE5hbWUgOiB1bmRlZmluZWQsXG4gICAgICByZXNvdXJjZUlkOiBwcm9wcy5yZXNvdXJjZVR5cGUucmVzb3VyY2VJZCxcbiAgICAgIHJlc291cmNlVHlwZTogcHJvcHMucmVzb3VyY2VUeXBlLnJlc291cmNlVHlwZSxcbiAgICAgIHRyYWZmaWNUeXBlOiBwcm9wcy50cmFmZmljVHlwZVxuICAgICAgICA/IHByb3BzLnRyYWZmaWNUeXBlXG4gICAgICAgIDogRmxvd0xvZ1RyYWZmaWNUeXBlLkFMTCxcbiAgICAgIGxvZ0Rlc3RpbmF0aW9uLFxuICAgIH0pO1xuXG4gICAgdGhpcy5mbG93TG9nSWQgPSBmbG93TG9nLnJlZjtcbiAgfVxufVxuIl19