from bson.objectid import ObjectId
from pymongo.collection import Collection

from .exceptions import KeyAlreadyExistsError, KeyDoesntExistError
from .key import Key


class KeyManager:
    """Establishes a connection to the collection holding the key data.

    Provides access to database CRUD operations.

    Parameters
    ----------
    collection :class:`Collection`:
        The collection the keys are stored in.
    """

    def __init__(self, collection: Collection) -> None:
        self._collection = collection

    @property
    def collection(self) -> Collection:
        return self._collection

    def add_to_collection(self, key: Key, *, ignore_exists: bool = False) -> None:
        """Adds a `Key` to the collection.

        The key is converted to a dictionary using the keys `to_database_data`,
        which includes everything aside from the `_id` field, as this is meant
        to be generated by mongodb. Once the key has been inserted, the keys
        `_id` will be updated to the inserted id.

        Parameters
        ----------
        key :class:`Key`:
            The key to add to the collection.

        ignore_exists :class:`bool`:
            Whether to ignore the fact that the key already exists.
            In the case of this happening, the key will be inserted regardless.
        """
        if not ignore_exists and self._collection.find_one({"key": key.key}):
            raise KeyAlreadyExistsError(
                f"{key.key} already exists in {self._collection.name}!"
            )

        result = self._collection.insert_one(key.to_database_data())
        key._id = result.inserted_id
        print(f"'{key.key}' has been inserted into {self._collection.name} at {key._id}!")

    def remove_from_collection(
        self, key: Key, *, ignore_nonexistent: bool = False
    ) -> None:
        """Removes a `Key` from the collection.

        The key will be found and deleted using the `_id` if available, otherwise
        they key-string will be used to find and delete it.

        Parameters
        ----------
        key :class:`Key`:
            The key to delete from the collection.

        ignore_nonexistent :class:`bool`:
            Whether to ignore the fact that the key does not exist within the database.
            In the case of this happening, the key will be inserted regardless
        """
        data: dict[str, str | ObjectId]
        if key._id is not None:
            data = {"_id": key._id}
        else:
            data = {"key": key.key}

        result = self._collection.delete_one(data)
        if not result.deleted_count and not ignore_nonexistent:
            raise KeyDoesntExistError(
                f"{key.key} does not exist in {self._collection.name}!"
            )
        print(f"'{key.key}' has been removed from {self._collection.name}!")

    def update(self, key: Key) -> None:
        """Updates a `Key` in the collection. This is useful when a new HWID
        has been added to a key, or the expiration date has changed."""
        update = {"$set": key.to_database_data()}
        self._collection.update_one({"_id": key._id}, update)
        print(f"'{key.key}' has been updated!")

    def exists(self, key: str) -> bool:
        """Checks whether a `Key` exists in the collection.
        
        This is done by calling `get_key` with the `Key` and catching
        the `LookupError`.        
        """
        try:
            self.get(key)
            return True
        except LookupError:
            return False
        
    def get(self, key: str) -> Key:
        """Returns a `Key` in the database, given it's key-string."""
        data = self._collection.find_one({"key": key})
        if not data:
            raise LookupError(f"Could not find key {key} in {self._collection.name}")
        return Key(**data)
    
    def get_all_keys(self) -> list[Key]:
        """Returns a list of all keys in the collection as `Key` objects."""
        results = self._collection.find()

        keys = []
        for key in results:
            keys.append(Key(**key))
        return keys




