import * as events from '@aws-cdk/aws-events';
import * as iam from '@aws-cdk/aws-iam';
import { IResource, RemovalPolicy, Resource } from '@aws-cdk/core';
import { IConstruct, Construct } from 'constructs';
import { LifecycleRule } from './lifecycle';
/**
 * Represents an ECR repository.
 *
 * @stability stable
 */
export interface IRepository extends IResource {
    /**
     * The name of the repository.
     *
     * @stability stable
     * @attribute true
     */
    readonly repositoryName: string;
    /**
     * The ARN of the repository.
     *
     * @stability stable
     * @attribute true
     */
    readonly repositoryArn: string;
    /**
     * The URI of this repository (represents the latest image):.
     *
     * ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY
     *
     * @stability stable
     * @attribute true
     */
    readonly repositoryUri: string;
    /**
     * Returns the URI of the repository for a certain tag. Can be used in `docker push/pull`.
     *
     * ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY[:TAG]
     *
     * @param tag Image tag to use (tools usually default to "latest" if omitted).
     * @stability stable
     */
    repositoryUriForTag(tag?: string): string;
    /**
     * Returns the URI of the repository for a certain tag. Can be used in `docker push/pull`.
     *
     * ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY[@DIGEST]
     *
     * @param digest Image digest to use (tools usually default to the image with the "latest" tag if omitted).
     * @stability stable
     */
    repositoryUriForDigest(digest?: string): string;
    /**
     * Add a policy statement to the repository's resource policy.
     *
     * @stability stable
     */
    addToResourcePolicy(statement: iam.PolicyStatement): iam.AddToResourcePolicyResult;
    /**
     * Grant the given principal identity permissions to perform the actions on this repository.
     *
     * @stability stable
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Grant the given identity permissions to pull images in this repository.
     *
     * @stability stable
     */
    grantPull(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant the given identity permissions to pull and push images to this repository.
     *
     * @stability stable
     */
    grantPullPush(grantee: iam.IGrantable): iam.Grant;
    /**
     * Define a CloudWatch event that triggers when something happens to this repository.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @stability stable
     */
    onCloudTrailEvent(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines an AWS CloudWatch event rule that can trigger a target when an image is pushed to this repository.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @stability stable
     */
    onCloudTrailImagePushed(id: string, options?: OnCloudTrailImagePushedOptions): events.Rule;
    /**
     * Defines an AWS CloudWatch event rule that can trigger a target when the image scan is completed.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @stability stable
     */
    onImageScanCompleted(id: string, options?: OnImageScanCompletedOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers for repository events.
     *
     * Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     *
     * @stability stable
     */
    onEvent(id: string, options?: events.OnEventOptions): events.Rule;
}
/**
 * Base class for ECR repository.
 *
 * Reused between imported repositories and owned repositories.
 *
 * @stability stable
 */
export declare abstract class RepositoryBase extends Resource implements IRepository {
    /**
     * The name of the repository.
     *
     * @stability stable
     */
    abstract readonly repositoryName: string;
    /**
     * The ARN of the repository.
     *
     * @stability stable
     */
    abstract readonly repositoryArn: string;
    /**
     * Add a policy statement to the repository's resource policy.
     *
     * @stability stable
     */
    abstract addToResourcePolicy(statement: iam.PolicyStatement): iam.AddToResourcePolicyResult;
    /**
     * The URI of this repository (represents the latest image):.
     *
     * ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY
     *
     * @stability stable
     */
    get repositoryUri(): string;
    /**
     * Returns the URL of the repository. Can be used in `docker push/pull`.
     *
     * ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY[:TAG]
     *
     * @param tag Optional image tag.
     * @stability stable
     */
    repositoryUriForTag(tag?: string): string;
    /**
     * Returns the URL of the repository. Can be used in `docker push/pull`.
     *
     * ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY[@DIGEST]
     *
     * @param digest Optional image digest.
     * @stability stable
     */
    repositoryUriForDigest(digest?: string): string;
    /**
     * Returns the repository URI, with an appended suffix, if provided.
     * @param suffix An image tag or an image digest.
     * @private
     */
    private repositoryUriWithSuffix;
    /**
     * Define a CloudWatch event that triggers when something happens to this repository.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @stability stable
     */
    onCloudTrailEvent(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines an AWS CloudWatch event rule that can trigger a target when an image is pushed to this repository.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @stability stable
     */
    onCloudTrailImagePushed(id: string, options?: OnCloudTrailImagePushedOptions): events.Rule;
    /**
     * Defines an AWS CloudWatch event rule that can trigger a target when an image scan is completed.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @stability stable
     */
    onImageScanCompleted(id: string, options?: OnImageScanCompletedOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule which triggers for repository events.
     *
     * Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     *
     * @stability stable
     */
    onEvent(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Grant the given principal identity permissions to perform the actions on this repository.
     *
     * @stability stable
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Grant the given identity permissions to use the images in this repository.
     *
     * @stability stable
     */
    grantPull(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant the given identity permissions to pull and push images to this repository.
     *
     * @stability stable
     */
    grantPullPush(grantee: iam.IGrantable): iam.Grant;
}
/**
 * Options for the onCloudTrailImagePushed method.
 *
 * @stability stable
 */
export interface OnCloudTrailImagePushedOptions extends events.OnEventOptions {
    /**
     * Only watch changes to this image tag.
     *
     * @default - Watch changes to all tags
     * @stability stable
     */
    readonly imageTag?: string;
}
/**
 * Options for the OnImageScanCompleted method.
 *
 * @stability stable
 */
export interface OnImageScanCompletedOptions extends events.OnEventOptions {
    /**
     * Only watch changes to the image tags spedified.
     *
     * Leave it undefined to watch the full repository.
     *
     * @default - Watch the changes to the repository with all image tags
     * @stability stable
     */
    readonly imageTags?: string[];
}
/**
 * @stability stable
 */
export interface RepositoryProps {
    /**
     * Name for this repository.
     *
     * @default Automatically generated name.
     * @stability stable
     */
    readonly repositoryName?: string;
    /**
     * Life cycle rules to apply to this registry.
     *
     * @default No life cycle rules
     * @stability stable
     */
    readonly lifecycleRules?: LifecycleRule[];
    /**
     * The AWS account ID associated with the registry that contains the repository.
     *
     * @default The default registry is assumed.
     * @see https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_PutLifecyclePolicy.html
     * @stability stable
     */
    readonly lifecycleRegistryId?: string;
    /**
     * Determine what happens to the repository when the resource/stack is deleted.
     *
     * @default RemovalPolicy.Retain
     * @stability stable
     */
    readonly removalPolicy?: RemovalPolicy;
    /**
     * Enable the scan on push when creating the repository.
     *
     * @default false
     * @stability stable
     */
    readonly imageScanOnPush?: boolean;
    /**
     * The tag mutability setting for the repository.
     *
     * If this parameter is omitted, the default setting of MUTABLE will be used which will allow image tags to be overwritten.
     *
     * @default TagMutability.MUTABLE
     * @stability stable
     */
    readonly imageTagMutability?: TagMutability;
}
/**
 * @stability stable
 */
export interface RepositoryAttributes {
    /**
     * @stability stable
     */
    readonly repositoryName: string;
    /**
     * @stability stable
     */
    readonly repositoryArn: string;
}
/**
 * Define an ECR repository.
 *
 * @stability stable
 */
export declare class Repository extends RepositoryBase {
    /**
     * Import a repository.
     *
     * @stability stable
     */
    static fromRepositoryAttributes(scope: Construct, id: string, attrs: RepositoryAttributes): IRepository;
    /**
     * @stability stable
     */
    static fromRepositoryArn(scope: Construct, id: string, repositoryArn: string): IRepository;
    /**
     * @stability stable
     */
    static fromRepositoryName(scope: Construct, id: string, repositoryName: string): IRepository;
    /**
     * Returns an ECR ARN for a repository that resides in the same account/region as the current stack.
     *
     * @stability stable
     */
    static arnForLocalRepository(repositoryName: string, scope: IConstruct, account?: string): string;
    private static validateRepositoryName;
    /**
     * The name of the repository.
     *
     * @stability stable
     */
    readonly repositoryName: string;
    /**
     * The ARN of the repository.
     *
     * @stability stable
     */
    readonly repositoryArn: string;
    private readonly lifecycleRules;
    private readonly registryId?;
    private policyDocument?;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props?: RepositoryProps);
    /**
     * Add a policy statement to the repository's resource policy.
     *
     * @stability stable
     */
    addToResourcePolicy(statement: iam.PolicyStatement): iam.AddToResourcePolicyResult;
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    protected validate(): string[];
    /**
     * Add a life cycle rule to the repository.
     *
     * Life cycle rules automatically expire images from the repository that match
     * certain conditions.
     *
     * @stability stable
     */
    addLifecycleRule(rule: LifecycleRule): void;
    /**
     * Render the life cycle policy object
     */
    private renderLifecyclePolicy;
    /**
     * Return life cycle rules with automatic ordering applied.
     *
     * Also applies validation of the 'any' rule.
     */
    private orderedLifecycleRules;
}
/**
 * The tag mutability setting for your repository.
 *
 * @stability stable
 */
export declare enum TagMutability {
    /**
     * allow image tags to be overwritten.
     *
     * @stability stable
     */
    MUTABLE = "MUTABLE",
    /**
     * all image tags within the repository will be immutable which will prevent them from being overwritten.
     *
     * @stability stable
     */
    IMMUTABLE = "IMMUTABLE"
}
