import logging
import os
import time
import ssl
import certifi
from urllib.parse import urlencode, urlparse

from requests import Request, Session, status_codes

from Tea.vendored import aiohttp
from Tea.exceptions import TeaException, RequiredArgumentException
from Tea.model import TeaModel
from Tea.request import TeaRequest
from Tea.response import TeaResponse
from Tea.stream import BaseStream

from typing import Dict, Any, Optional

DEFAULT_CONNECT_TIMEOUT = 5000
DEFAULT_READ_TIMEOUT = 10000

logger = logging.getLogger('alibabacloud-tea')
logger.setLevel(logging.DEBUG)
ch = logging.StreamHandler()
logger.addHandler(ch)


class TeaCore:
    @staticmethod
    def _prepare_http_debug(request, symbol):
        base = ''
        for key, value in request.headers.items():
            base += f'\n{symbol} {key} : {value}'
        return base

    @staticmethod
    def _do_http_debug(request, response):
        # logger the request
        url = urlparse(request.url)
        request_base = f'\n> {request.method.upper()} {url.path + url.query} HTTP/1.1'
        logger.debug(request_base + TeaCore._prepare_http_debug(request, '>'))

        # logger the response
        response_base = f'\n< HTTP/1.1 {response.status_code}' \
                        f' {status_codes._codes.get(response.status_code)[0].upper()}'
        logger.debug(response_base + TeaCore._prepare_http_debug(response, '<'))

    @staticmethod
    def compose_url(request):
        host = request.headers.get('host')
        if not host:
            raise RequiredArgumentException('endpoint')
        else:
            host = host.rstrip('/')
        protocol = f'{request.protocol.lower()}://'
        pathname = request.pathname

        if host.startswith(('http://', 'https://')):
            protocol = ''

        if request.port == 80:
            port = ''
        else:
            port = f':{request.port}'

        url = protocol + host + port + pathname

        if request.query:
            if "?" in url:
                if not url.endswith("&"):
                    url += "&"
            else:
                url += "?"

            encode_query = {}
            for key in request.query:
                value = request.query[key]
                if value is not None:
                    encode_query[key] = str(value)
            url += urlencode(encode_query)
        return url.rstrip("?&")

    @staticmethod
    async def async_do_action(
        request: TeaRequest,
        runtime_option=None
    ) -> TeaResponse:
        runtime_option = runtime_option or {}

        url = TeaCore.compose_url(request)
        verify = not runtime_option.get('ignoreSSL', False)
        connect_timeout = runtime_option.get('connectTimeout')
        connect_timeout = connect_timeout if connect_timeout else DEFAULT_CONNECT_TIMEOUT

        read_timeout = runtime_option.get('readTimeout')
        read_timeout = read_timeout if read_timeout else DEFAULT_READ_TIMEOUT

        connect_timeout, read_timeout = (int(connect_timeout) / 1000, int(read_timeout) / 1000)

        proxy = None
        if request.protocol.upper() == 'HTTP':
            proxy = runtime_option.get('httpProxy')
            if not proxy:
                proxy = os.environ.get('HTTP_PROXY') or os.environ.get('http_proxy')

        connector = None
        ca_cert = certifi.where()
        if ca_cert and request.protocol.upper() == 'HTTPS':
            ssl_context = ssl.create_default_context(ssl.Purpose.CLIENT_AUTH)
            ssl_context.load_verify_locations(ca_cert)
            connector = aiohttp.TCPConnector(
                ssl=ssl_context,
            )
        else:
            verify = False

        timeout = aiohttp.ClientTimeout(
            sock_read=read_timeout,
            sock_connect=connect_timeout
        )
        async with aiohttp.ClientSession(
            connector=connector
        ) as s:
            body = b''
            if isinstance(request.body, BaseStream):
                for content in request.body:
                    body += content
            else:
                body = request.body
            async with s.request(request.method, url,
                                 data=body,
                                 headers=request.headers,
                                 ssl=verify,
                                 proxy=proxy,
                                 timeout=timeout) as response:
                tea_resp = TeaResponse()
                tea_resp.body = await response.read()
                tea_resp.headers = {k.lower(): v for k, v in response.headers.items()}
                tea_resp.status_code = response.status
                tea_resp.status_message = response.reason
                tea_resp.response = response
        return tea_resp

    @staticmethod
    def do_action(
        request: TeaRequest,
        runtime_option=None
    ) -> TeaResponse:
        url = TeaCore.compose_url(request)

        runtime_option = runtime_option or {}

        verify = not runtime_option.get('ignoreSSL', False)

        connect_timeout = runtime_option.get('connectTimeout')
        connect_timeout = connect_timeout if connect_timeout else DEFAULT_CONNECT_TIMEOUT

        read_timeout = runtime_option.get('readTimeout')
        read_timeout = read_timeout if read_timeout else DEFAULT_READ_TIMEOUT

        timeout = (int(connect_timeout) / 1000, int(read_timeout) / 1000)
        with Session() as s:
            req = Request(method=request.method, url=url,
                          data=request.body, headers=request.headers)
            prepped = s.prepare_request(req)

            proxies = {}
            http_proxy = runtime_option.get('httpProxy')
            https_proxy = runtime_option.get('httpsProxy')
            no_proxy = runtime_option.get('noProxy')

            if not http_proxy:
                http_proxy = os.environ.get('HTTP_PROXY') or os.environ.get('http_proxy')
            if not https_proxy:
                https_proxy = os.environ.get('HTTPS_PROXY') or os.environ.get('https_proxy')

            if http_proxy:
                proxies['http'] = http_proxy
            if https_proxy:
                proxies['https'] = https_proxy
            if no_proxy:
                proxies['no_proxy'] = no_proxy

            resp = s.send(
                prepped,
                proxies=proxies,
                timeout=timeout,
                verify=verify,
            )

            debug = runtime_option.get('debug') or os.getenv('DEBUG')
            if debug and debug.lower() == 'sdk':
                TeaCore._do_http_debug(req, resp)

            response = TeaResponse()
            response.status_message = resp.reason
            response.status_code = resp.status_code
            response.headers = {k.lower(): v for k, v in resp.headers.items()}
            response.body = resp.content
            response.response = resp
            return response

    @staticmethod
    def get_response_body(resp) -> str:
        return resp.content.decode("utf-8")

    @staticmethod
    def allow_retry(dic, retry_times, now=None) -> bool:
        if dic is None or not dic.__contains__("maxAttempts"):
            return False
        else:
            retry = 0 if dic.get("maxAttempts") is None else int(
                dic.get("maxAttempts"))
        return retry >= retry_times

    @staticmethod
    def get_backoff_time(dic, retry_times) -> int:
        default_back_off_time = 0
        if dic is None or not dic.get("policy") or dic.get("policy") == "no":
            return default_back_off_time

        back_off_time = dic.get('period', default_back_off_time)
        if not isinstance(back_off_time, int) and \
                not (isinstance(back_off_time, str) and back_off_time.isdigit()):
            return default_back_off_time

        back_off_time = int(back_off_time)
        if back_off_time < 0:
            return retry_times

        return back_off_time

    @staticmethod
    def sleep(t):
        time.sleep(t)

    @staticmethod
    def is_retryable(ex) -> bool:
        return isinstance(ex, TeaException)

    @staticmethod
    def bytes_readable(body):
        return body

    @staticmethod
    def merge(*dic_list) -> dict:
        dic_result = {}
        for item in dic_list:
            if isinstance(item, dict):
                dic_result.update(item)
            elif isinstance(item, TeaModel):
                dic_result.update(item.to_map())
        return dic_result

    @staticmethod
    def to_map(model: Optional[TeaModel]) -> Dict[str, Any]:
        if isinstance(model, TeaModel):
            return model.to_map()
        else:
            return dict()

    @staticmethod
    def from_map(
            model: TeaModel,
            dic: Dict[str, Any]
    ) -> TeaModel:
        if isinstance(model, TeaModel):
            return model.from_map(dic)
        else:
            return model
