# IMPORTATION STANDARD
import logging
import os

# IMPORTATION THIRDPARTY
from typing import List, Optional, Tuple
import matplotlib.pyplot as plt
import pandas as pd

# IMPORTATION INTERNAL
import openbb_terminal.config_plot as cfp
from openbb_terminal.config_terminal import theme
from openbb_terminal.rich_config import console
from openbb_terminal.decorators import log_start_end
from openbb_terminal.helper_funcs import (
    is_valid_axes_count,
    plot_autoscale,
    print_rich_table,
    export_data,
)
from openbb_terminal.stocks.options.op_helpers import (
    calculate_max_pain,
    get_strikes,
    get_greeks,
)

logger = logging.getLogger(__name__)

# pylint: disable=C0302,R0913


def get_calls_and_puts(chain: pd.DataFrame) -> Tuple[pd.DataFrame, pd.DataFrame]:
    calls = chain[chain["optionType"] == "call"]
    puts = chain[chain["optionType"] == "put"]
    return calls, puts


def get_max_pain(calls: pd.DataFrame, puts: pd.DataFrame) -> float:
    call_oi = calls.set_index("strike")["openInterest"] / 1000
    put_oi = puts.set_index("strike")["openInterest"] / 1000
    df_opt = pd.merge(call_oi, put_oi, left_index=True, right_index=True)
    df_opt = df_opt.rename(
        columns={"openInterest_x": "OI_call", "openInterest_y": "OI_put"}
    )
    return calculate_max_pain(df_opt)


def print_raw(
    calls: pd.DataFrame,
    puts: pd.DataFrame,
    title: str,
    calls_only: bool = False,
    puts_only: bool = False,
):
    if not puts_only:
        print_rich_table(
            calls,
            headers=list(calls.columns),
            show_index=False,
            title=f"{title} - Calls",
        )
    if not calls_only:
        print_rich_table(
            puts,
            headers=list(puts.columns),
            show_index=False,
            title=f"{title} - Puts",
        )


@log_start_end(log=logger)
def plot_vol(
    chain: pd.DataFrame,
    current_price: float,
    symbol: str,
    expiry: str,
    min_sp: float = -1,
    max_sp: float = -1,
    calls_only: bool = False,
    puts_only: bool = False,
    raw: bool = False,
    export: str = "",
    sheet_name: str = None,
    external_axes: Optional[List[plt.Axes]] = None,
):
    """Plot volume

    Parameters
    ----------
    chain: pd.Dataframe
        Dataframe with options chain
    current_price: float
        Current price of selected symbol
    symbol: str
        Ticker symbol
    expiry: str
        expiration date for options
    min_sp: float
        Min strike to consider
    max_sp: float
        Max strike to consider
    calls_only: bool
        Show calls only
    puts_only: bool
        Show puts only
    export: str
        Format to export file
    sheet_name: str
        Optionally specify the name of the sheet to export to
    external_axes : Optional[List[plt.Axes]], optional
        External axes (1 axis is expected in the list), by default None


    Examples
    --------
    >>> from openbb_terminal.sdk import openbb
    >>> aapl_chain_data = openbb.stocks.options.chains("AAPL", expiration="2023-07-21", source="Nasdaq")
    >>> aapl_price = openbb.stocks.options.price("AAPL", source="Nasdaq")
    >>> openbb.stocks.options.vol(
            chain=aapl_chain_data,
            symbol="AAPL",
            current_price=aapl_price,
            expiry="2023-07-21",
        )
    """
    calls, puts = get_calls_and_puts(chain)

    min_strike, max_strike = get_strikes(min_sp, max_sp, current_price)

    if external_axes is None:
        _, ax = plt.subplots(figsize=plot_autoscale(), dpi=cfp.PLOT_DPI)
    elif is_valid_axes_count(external_axes, 1):
        (ax,) = external_axes
    else:
        return

    if not puts_only:
        ax.plot(
            calls.strike,
            calls["volume"] / 1000,
            ls="-",
            marker="o",
            label="Calls",
        )
    if not calls_only:
        ax.plot(
            puts.strike,
            puts["volume"] / 1000,
            ls="-",
            marker="o",
            label="Puts",
        )

    ax.axvline(current_price, lw=2, ls="--", label="Current Price", alpha=0.7)
    ax.set_xlabel("Strike Price")
    ax.set_ylabel("Volume (1k) ")
    ax.set_xlim(min_strike, max_strike)
    ax.legend(loc="best", fontsize="x-small")
    title = f"Volume for {symbol.upper()} expiring {expiry}"
    ax.set_title(title)

    theme.style_primary_axis(ax)
    if external_axes is None:
        theme.visualize_output()

    if raw:
        print_raw(calls, puts, title, calls_only, puts_only)

    export_data(
        export,
        os.path.dirname(os.path.abspath(__file__)),
        f"vol_{symbol}_{expiry}",
        chain,
        sheet_name,
    )


@log_start_end(log=logger)
def plot_oi(
    chain: pd.DataFrame,
    current_price: float,
    symbol: str,
    expiry: str,
    min_sp: float = -1,
    max_sp: float = -1,
    calls_only: bool = False,
    puts_only: bool = False,
    raw: bool = False,
    export: str = "",
    sheet_name: str = None,
    external_axes: Optional[List[plt.Axes]] = None,
):
    """Plot open interest

    Parameters
    ----------
    chain: pd.Dataframe
        Dataframe with options chain
    current_price: float
        Current price of selected symbol
    symbol: str
        Ticker symbol
    expiry: str
        expiration date for options
    min_sp: float
        Min strike to consider
    max_sp: float
        Max strike to consider
    calls_only: bool
        Show calls only
    puts_only: bool
        Show puts only
    export: str
        Format to export file
    sheet_name: str
        Optionally specify the name of the sheet to export to
    external_axes : Optional[List[plt.Axes]], optional
        External axes (1 axis is expected in the list), by default None

    Examples
    --------
    >>> from openbb_terminal.sdk import openbb
    >>> aapl_chain_data = openbb.stocks.options.chains("AAPL", expiration="2023-07-21", source="Nasdaq")
    >>> aapl_price = openbb.stocks.options.price("AAPL", source="Nasdaq")
    >>> openbb.stocks.options.oi(
            chain=aapl_chain_data,
            symbol="AAPL",
            current_price=aapl_price,
            expiry="2023-07-21",
        )
    """
    calls, puts = get_calls_and_puts(chain)

    min_strike, max_strike = get_strikes(min_sp, max_sp, current_price)
    max_pain = get_max_pain(calls, puts)

    if external_axes is None:
        _, ax = plt.subplots(figsize=plot_autoscale(), dpi=cfp.PLOT_DPI)
    elif is_valid_axes_count(external_axes, 1):
        (ax,) = external_axes
    else:
        return

    if not puts_only:
        ax.plot(
            calls.strike,
            calls["openInterest"] / 1000,
            ls="-",
            marker="o",
            label="Calls",
        )
    if not calls_only:
        ax.plot(
            puts.strike,
            puts["openInterest"] / 1000,
            ls="-",
            marker="o",
            label="Puts",
        )

    ax.axvline(current_price, lw=2, ls="--", label="Current Price", alpha=0.7)
    ax.axvline(max_pain, lw=3, label=f"Max Pain: {max_pain}", alpha=0.7)
    ax.set_xlabel("Strike Price")
    ax.set_ylabel("Open Interest (1k) ")
    ax.set_xlim(min_strike, max_strike)
    ax.legend(loc="best", fontsize="x-small")
    title = f"Open Interest for {symbol.upper()} expiring {expiry}"
    ax.set_title(title)

    theme.style_primary_axis(ax)
    if external_axes is None:
        theme.visualize_output()

    if raw:
        print_raw(calls, puts, title, calls_only, puts_only)

    export_data(
        export,
        os.path.dirname(os.path.abspath(__file__)),
        f"oi_{symbol}_{expiry}",
        chain,
        sheet_name,
    )


@log_start_end(log=logger)
def plot_voi(
    chain: pd.DataFrame,
    current_price: float,
    symbol: str,
    expiry: str,
    min_sp: float = -1,
    max_sp: float = -1,
    raw: bool = False,
    export: str = "",
    sheet_name: str = None,
    external_axes: Optional[List[plt.Axes]] = None,
):
    """Plot volume and open interest

    Parameters
    ----------
    chain: pd.Dataframe
        Dataframe with options chain
    current_price: float
        Current price of selected symbol
    symbol: str
        Stock ticker symbol
    expiry: str
        Option expiration
    min_sp: float
        Min strike price
    max_sp: float
        Max strike price
    export: str
        Format for exporting data
    sheet_name: str
        Optionally specify the name of the sheet to export to
    external_axes : Optional[List[plt.Axes]], optional
        External axes (1 axis is expected in the list), by default None

    Examples
    --------
    >>> from openbb_terminal.sdk import openbb
    >>> aapl_chain_data = openbb.stocks.options.chains("AAPL", expiration="2023-07-21", source="Nasdaq")
    >>> aapl_price = openbb.stocks.options.price("AAPL", source="Nasdaq")
    >>> openbb.stocks.options.voi(
            chain=aapl_chain_data,
            symbol="AAPL",
            current_price=aapl_price,
            expiry="2023-07-21",
        )
    """
    calls, puts = get_calls_and_puts(chain)

    min_strike, max_strike = get_strikes(min_sp, max_sp, current_price)
    max_pain = get_max_pain(calls, puts)

    option_chain = pd.merge(
        calls[["volume", "strike", "openInterest"]],
        puts[["volume", "strike", "openInterest"]],
        on="strike",
    )

    option_chain = option_chain.rename(
        columns={
            "volume_x": "volume_call",
            "volume_y": "volume_put",
            "openInterest_x": "openInterest_call",
            "openInterest_y": "openInterest_put",
        }
    )

    option_chain[["openInterest_put", "volume_put"]] = (
        option_chain[["openInterest_put", "volume_put"]] * -1 / 1000
    )
    option_chain[["openInterest_call", "volume_call"]] = (
        option_chain[["openInterest_call", "volume_call"]] / 1000
    )

    if external_axes is None:
        _, ax = plt.subplots(figsize=plot_autoscale(), dpi=cfp.PLOT_DPI)
    elif is_valid_axes_count(external_axes, 1):
        (ax,) = external_axes
    else:
        return

    ax.bar(
        option_chain.strike,
        option_chain.openInterest_call,
        color="green",
        label="Calls: OI",
    )
    ax.bar(
        option_chain.strike,
        option_chain.volume_call,
        color="lightgreen",
        label="Calls: Vol",
    )

    ax.bar(
        option_chain.strike,
        option_chain.openInterest_put,
        color="red",
        label="Puts: OI",
    )
    ax.bar(
        option_chain.strike,
        option_chain.volume_put,
        color="pink",
        label="Puts: Vol",
    )

    ax.axvline(
        current_price, lw=2, ls="--", label=f"Current Price: {current_price}", alpha=0.7
    )
    ax.axvline(max_pain, lw=2, ls="--", label=f"Max Pain: {max_pain:.2f}", alpha=0.7)
    ax.set_xlabel("Strike Price")
    ax.set_ylabel("Volume or OI (1k)")
    ax.set_yticklabels([f"{x:,.0f}".replace("-", "") for x in ax.get_yticks().tolist()])
    ax.set_xlim(min_strike, max_strike)
    ax.legend(loc="best", fontsize="xx-small")
    title = f"Volume and Open Interest for {symbol.upper()} expiring {expiry}"
    ax.set_title(title)

    theme.style_primary_axis(ax)
    if external_axes is None:
        theme.visualize_output()

    if raw:
        print_raw(calls, puts, title)

    export_data(
        export,
        os.path.dirname(os.path.abspath(__file__)),
        f"voi_{symbol}_{expiry}",
        chain,
        sheet_name,
    )


@log_start_end(log=logger)
def display_expiry_dates(expiry_dates: list):
    """Display expiry dates

    Parameters
    ----------
    expiry_dates: list
        The expiry dates of the chosen ticker.
    """
    expiry_dates_df = pd.DataFrame(expiry_dates, columns=["Date"])

    print_rich_table(
        expiry_dates_df,
        headers=list(expiry_dates_df.columns),
        title="Available expiry dates",
        show_index=True,
        index_name="Identifier",
    )


@log_start_end(log=logger)
def display_chains(
    chain: pd.DataFrame,
    expire: str,
    current_price: float = 0,
    calls_only: bool = False,
    puts_only: bool = False,
    min_sp: float = -1,
    max_sp: float = -1,
    export: str = "",
    sheet_name: str = None,
    to_display: list = None,
):
    """Display chains

    chain: pd.Dataframe
        Dataframe with options chain
    current_price: float
        Current price of selected symbol
    expire: str
        The date of expiration
    min_sp: float
        Min strike price
    max_sp: float
        Max strike price
    calls_only: bool
        Show calls only
    puts_only: bool
        Show puts only
    export: str
        Format for exporting data
    sheet_name: str
        Optionally specify the name of the sheet to export to
    to_display: list
        List of columns to display
    """
    if to_display:
        to_display += ["strike", "optionType"]
        to_display = list(set(to_display))
        chain = chain[to_display]

    min_strike, max_strike = get_strikes(
        min_sp=min_sp, max_sp=max_sp, current_price=current_price
    )

    chain = chain[chain["strike"] >= min_strike]
    chain = chain[chain["strike"] <= max_strike]
    calls, puts = get_calls_and_puts(chain)

    chain = get_greeks(
        current_price=current_price,
        calls=calls,
        expire=expire,
        puts=puts,
        show_all=True,
    )

    # if the greeks calculation went with no problems, otherwise keep the previous
    if not chain.empty:
        calls, puts = get_calls_and_puts(chain)
        console.print("Greeks calculated by OpenBB.")

    calls = calls.sort_index(axis=1)
    puts = puts.sort_index(axis=1)

    print_raw(calls, puts, "Option chain", calls_only, puts_only)

    export_data(
        export, os.path.dirname(os.path.abspath(__file__)), "chain", chain, sheet_name
    )
