import contextlib
import json
import os
import tempfile
from multiprocessing.managers import SharedMemoryManager

import gitlab
from filelock import FileLock
from flasket import endpoint
from flasket.clients.gitlab import HookEvents, webhook

FINAL_STATES = ["failed", "success", "canceled", "skipped", "manual"]


# TODO: Move to controller
def action_jobs_file(app, callback):
    # Get a unique ID from app.config
    key = app.config["UNIQUE_KEY"]
    filepath = os.path.join(tempfile.gettempdir(), f"gl-webhooks.jobs.{key}.json")
    lockfile = filepath + ".lock"

    with FileLock(lockfile):
        data = {}
        with contextlib.suppress(FileNotFoundError) as ctx:
            with open(filepath) as fd:
                data = json.load(fd)

        if callback:
            data = callback(data or {})
            with open(filepath, mode="w") as fd:
                json.dump(data, fd, indent=2, sort_keys=True)

        return data or {}


@endpoint
def monitor_purge(app, body):
    cfg = app.settings["gitlab"]

    gl = gitlab.Gitlab(cfg["url"], private_token=cfg["token"])

    def action1(data):
        # purge finished states
        data = {k: v for k, v in data.items() if v["status"] not in FINAL_STATES}
        return data

    def action2(data):
        for k, v in data.items():
            try:
                job = gl.projects.get(v["project_id"], lazy=True).jobs.get(v["build_id"])
                v["status"] = job.status
            except:
                pass
        return data

    data = action_jobs_file(app, action1)
    data = action_jobs_file(app, action2)
    data = action_jobs_file(app, action1)
    return data, 200


@webhook([HookEvents.JOB_HOOK])
def monitor(app, body):
    build_id = str(body["build_id"])
    build_status = body["build_status"]
    project_id = str(body["project_id"])

    cfg = app.settings["gitlab"]
    gl = gitlab.Gitlab(cfg["url"], private_token=cfg["token"])

    def action(data):
        if build_id not in data:
            data[build_id] = {}

        data[build_id]["build_id"] = int(build_id)
        data[build_id]["project_id"] = int(project_id)
        data[build_id]["status"] = build_status

        # Get a possible updated status
        if build_status not in FINAL_STATES:
            try:
                job = gl.projects.get(int(project_id), lazy=True).jobs.get(int(build_id))
                data[build_id]["status"] = job.status
            except:
                pass

        # Purge final states
        data = {k: v for k, v in data.items() if v["status"] not in FINAL_STATES}
        return data

    data = action_jobs_file(app, action)
    pending = len([True for v in data.values() if v["status"] == "pending"])
    running = len([True for v in data.values() if v["status"] == "running"])

    data["pending"] = pending
    data["running"] = running

    return data, 200
