# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/00_repr_str.ipynb.

# %% auto 0
__all__ = ['PRINT_OPTS', 'pretty_str', 'lovely']

# %% ../nbs/00_repr_str.ipynb 3
from typing import Optional, Union
import torch

# %% ../nbs/00_repr_str.ipynb 4
class __PrinterOptions(object):
    precision: int = 3
    threshold_max: int = 3 # .abs() larger than 1e3 -> Sci mode
    threshold_min: int = -4 # .abs() smaller that 1e-4 -> Sci mode
    sci_mode: Optional[bool] = None # None = auto. Otherwise, force sci mode.
    indent: int = 2 # Indent for .deeper()
    color: bool = True

PRINT_OPTS = __PrinterOptions()

# %% ../nbs/00_repr_str.ipynb 5
# |export


# %% ../nbs/00_repr_str.ipynb 6
# Do we want this float in decimal or scientific mode?
def sci_mode(f: float):
    return (abs(f) < 10**(PRINT_OPTS.threshold_min) or
            abs(f) > 10**PRINT_OPTS.threshold_max)

# %% ../nbs/00_repr_str.ipynb 9
# Convert a tensor or scalar into a string.
# This only looks good for small tensors, which is how it's intended to be used.
def pretty_str(t: Union[torch.Tensor, float, int]):
    """A slightly better way to print `float`-y values"""

    if isinstance(t, int):
        return '{}'.format(t)
    elif isinstance(t, float):
        if t == 0.:
            return "0."

        sci = (PRINT_OPTS.sci_mode or
                (PRINT_OPTS.sci_mode is None and sci_mode(t)))
        # The f-string will generate something like "{.4f}", which is used
        # to format the value.
        return f"{{:.{PRINT_OPTS.precision}{'e' if sci else 'f'}}}".format(t)
    elif t.dim() == 0:
            return pretty_str(t.item())
    else:
        slices = [pretty_str(t[i]) for i in range(0, t.size(0))]
        return '[' + ", ".join(slices) + ']'

# %% ../nbs/00_repr_str.ipynb 14
def space_join(lst):
    # Join non-empty list elements into a space-sepaeated string
    return " ".join( [ l for l in lst if l] )

# %% ../nbs/00_repr_str.ipynb 16
class StrProxy():
    def __init__(self, t: torch.Tensor, plain=False, verbose=False, depth=0, lvl=0, color=None):
        self.t = t
        self.plain = plain
        self.verbose = verbose
        self.depth=depth
        self.lvl=lvl
        self.color=color

    @torch.no_grad()
    def to_str(self):
        t = self.t
        if self.plain or t.is_complex():
            return torch._tensor_str._tensor_str(t, indent=0)

        color = PRINT_OPTS.color if self.color is None else self.color
        
        grey_style = "\x1b[38;2;127;127;127m" if color else ""
        red_style = "\x1b[31m" if color else ""
        end_style = "\x1b[0m" if color else ""

        dtnames = { torch.float32: "",
                    torch.float16: "f16",
                    torch.float64: "f64",
                    torch.uint8: "u8",
                    torch.int32: "i32",
                }


        tname = "tensor" if type(t) is torch.Tensor else type(t).__name__
        dev = str(t.device) if t.device.type != "cpu" else None
        dtype = dtnames[t.dtype] if t.dtype in dtnames else str(t.dtype)[6:]


        grad_fn = t.grad_fn.name() if t.grad_fn else None
        # All tensors along the compute path actually have required_grad=True.
        # Torch __repr__ just dones not show it.
        grad = "grad" if t.requires_grad else None

        shape = str(list(t.shape))

        # Later, we might be indexing 't' with a bool tensor derived from it. 
        # THis takes 4x memory and will result in a CUDA OOM if 't' is very large.
        # Move it to the cpu now - it won't matter for small tensors, and for
        # very large ones we trade a CUDA OOM for a few seconds delay.
        t = t.detach().cpu()

        zeros = grey_style+"all_zeros"+end_style if t.eq(0.).all() and t.numel() > 1 else None
        pinf = red_style+"+inf!"+end_style if t.isposinf().any() else None
        ninf = red_style+"-inf!"+end_style if t.isneginf().any() else None
        nan = red_style+"nan!"+end_style if t.isnan().any() else None


        attention = space_join([zeros,pinf,ninf,nan])

        x = ""
        summary = f"n={t.numel()}" if t.numel() > 5 else None
        if not zeros:
            if t.numel() <= 10: x = pretty_str(t)
            
            # Make sure it's float32. Also, we calculate stats on good values only.

            ft = t[ torch.isfinite(t) ].float()

            minmax = f"x∈[{pretty_str(ft.min())}, {pretty_str(ft.max())}]" if t.numel() > 2 and ft.numel() > 2 else None
            meanstd = f"μ={pretty_str(ft.mean())} σ={pretty_str(ft.std())}" if t.numel() >= 2 and ft.numel() >= 2 else None
            numel = f"n={t.numel()}" if t.numel() > 5 and max(t.shape) != t.numel() else None

            summary = space_join([numel, minmax, meanstd])




        res = tname + space_join([  shape,
                                    summary,
                                    dtype,
                                    grad,
                                    grad_fn,
                                    dev,
                                    attention,
                                    x if not self.verbose else None])

        if self.verbose:
            res += "\n" + torch._tensor_str._tensor_str(t, indent=0)

        if self.depth and t.dim() > 1:
            res += "\n" + "\n".join([
                " "*PRINT_OPTS.indent*(self.lvl+1) +
                str(StrProxy(t[i,:], depth=self.depth-1, lvl=self.lvl+1))
                for i in range(t.shape[0])])

        return res
    
    def __repr__(self):
        return self.to_str()

    def __call__(self, depth=0):
        return StrProxy(self.t, depth=depth)

# %% ../nbs/00_repr_str.ipynb 18
def lovely(t: torch.Tensor, # Tensor of interest
            verbose=False,  # Whether to show the full tensor
            plain=False,    # Just print if exactly as before
            depth=0,        # Show stats in depth
            color=None):    # Force color (True/False) or auto.
    return StrProxy(t, verbose=verbose, plain=plain, depth=depth, color=color)
