# -*- coding: utf-8 -*-
"""techsig

Automatically generated by Colaboratory.

Original file is located at
    https://colab.research.google.com/drive/15gpREQ9yiJGAkuMD2FjHCIwFOWWBV5XP
"""

import pandas as pd
import numpy as np
import yfinance as yf
from ta.trend import ADXIndicator
import plotly.express as px
import plotly.graph_objects as go
from plotly.subplots import make_subplots

def get_data(ticker, start_date, end_date):
  '''
  Import daily market data
  :param ticker: ticker name according to National Stock Exchange
  :param start_date: format 'yyyy-mm-dd'
  :param end_date: format 'yyyy-mm-dd'
  :return: pandas.DataFrame() : OHCLV data on a daily frequency
  '''
  data = yf.download(ticker,start_date,end_date)
  return data.dropna()

def moving_average(df, exponential=False, simple=False, plot=False, signal=False):
  '''
  Calculate simple and exponential moving average (ma) for given data
  :param df: pandas.DataFrame() :market data downloaded from get_data()
  :param exponential: Boolean: if True, exponential ma is displayed
  :param simple: Boolean: if True, simple ma is displayed
  :param plot: Boolean: if True, closing price with ma is plotted
  :param signal: Boolean: if True, bullish/bearish signals are returned
  :return: pandas.DataFrame() : moving average of 5 days, 10 days, 20 days, 50 days, 100 days and 200 days
  '''
  
  def signal_ma(x):
     if x == 1:
       return 'Bull'
     elif x == -1:
       return 'Bear'
     else:
       return 'Neutral'
  if simple == True:
    moving_average_5d = df['Close'].rolling(window=5).mean()
    moving_average_10d = df['Close'].rolling(window=10).mean()
    moving_average_20d = df['Close'].rolling(window=10).mean()
    moving_average_50d = df['Close'].rolling(window=50).mean()
    moving_average_100d = df['Close'].rolling(window=100).mean()
    moving_average_200d = df['Close'].rolling(window=200).mean()
    moving_average_sma = pd.DataFrame()
    moving_average_sma = pd.concat([moving_average_5d, moving_average_10d, moving_average_20d, moving_average_50d, moving_average_100d, moving_average_200d], axis=1)
    moving_average_sma.columns = ['Simple Moving Average 5d', 'Simple Moving Average 10d', 'Simple Moving Average 20d', 'Simple Moving Average 50d', 'Simple Moving Average 100d', 'Simple Moving Average 200d']
    

    if signal==True:
      short_window = 5
      moving_average_sma['signal'] = 0
      moving_average_sma['position_signal'] = 0
      moving_average_sma['position_sma'] = 'Neutral'
      moving_average_sma['signal'][short_window:] = np.where(moving_average_sma['Simple Moving Average 5d'][short_window:]
                                                             >moving_average_sma['Simple Moving Average 200d'][short_window:], 1.0, 0)
      moving_average_sma['position_signal'] = moving_average_sma['signal'].diff()
      moving_average_sma['position_sma'] = moving_average_sma['position_signal'].apply(lambda x: signal_ma(x))
      
      if plot == True:
        fig = make_subplots(rows=2, cols=1, shared_xaxes=True, vertical_spacing=0.02)
        fig.append_trace(go.Scatter(y = df['Close'], x = df.index, mode='lines', name='Close'), row=1, col=1)
        fig.append_trace(go.Scatter(y = moving_average_sma['Simple Moving Average 5d'], x = moving_average_sma.index, mode='lines', name='Simple Moving Average 5d'), row=2, col=1)
        fig.append_trace(go.Scatter(y = moving_average_sma['Simple Moving Average 200d'], x = moving_average_sma.index, mode='lines', name='Simple Moving Average 200d'), row=2, col=1)
        fig.add_trace(go.Scatter(y = moving_average_sma[moving_average_sma['position_sma']=='Bull']['Simple Moving Average 5d'], x=moving_average_sma[moving_average_sma['position_sma']=='Bull'].index, mode='markers', marker_symbol='triangle-up', marker_color='green', marker_size=15, name='Bull'), row=2, col=1)
        fig.add_trace(go.Scatter(y = moving_average_sma[moving_average_sma['position_sma']=='Bear']['Simple Moving Average 5d'], x=moving_average_sma[moving_average_sma['position_sma']=='Bear'].index, mode='markers', marker_symbol='triangle-down', marker_color='red', marker_size=15, name='Bear'), row=2, col=1)
        fig.update_layout(title='Closing Price with Simple Moving Average and Bull/Bear Signals', template='plotly_dark')
        
        fig.show()
        return moving_average_sma
      elif plot == False:
        return moving_average_sma

    elif signal==False:
     if plot == True:
        fig = make_subplots(rows=7, cols=1, shared_xaxes=True, vertical_spacing=0.02)
        fig.append_trace(go.Scatter(y = df['Close'], x = df.index, mode='lines', name='Close'), row=1, col=1)
        fig.append_trace(go.Scatter(y = moving_average_sma['Simple Moving Average 5d'], x = df.index, mode='lines', name='Simple Moving Average 5d'), row=2, col=1)
        fig.append_trace(go.Scatter(y = moving_average_sma['Simple Moving Average 10d'], x = df.index, mode='lines', name='Simple Moving Average 10d'), row=3, col=1)
        fig.append_trace(go.Scatter(y = moving_average_sma['Simple Moving Average 20d'], x = df.index, mode='lines', name='Simple Moving Average 20d'), row=4, col=1)
        fig.append_trace(go.Scatter(y = moving_average_sma['Simple Moving Average 50d'], x = df.index, mode='lines', name='Simple Moving Average 50d'), row=5, col=1)
        fig.append_trace(go.Scatter(y = moving_average_sma['Simple Moving Average 100d'], x = df.index, mode='lines', name='Simple Moving Average 100d'), row=6, col=1)
        fig.append_trace(go.Scatter(y = moving_average_sma['Simple Moving Average 200d'], x = df.index, mode='lines', name='Simple Moving Average 200d'), row=7, col=1)
        fig.update_layout(title='Closing Price with Exponential Moving Average', template='plotly_dark')
        
        fig.show()
        return moving_average_sma
     elif plot == False:
        return moving_average_sma
        
    

  if exponential == True:
    moving_average_ewm_5d = df['Close'].ewm(span=5, adjust=False).mean()
    moving_average_ewm_10d = df['Close'].ewm(span=10, adjust=False).mean()
    moving_average_ewm_20d = df['Close'].ewm(span=20, adjust=False).mean()
    moving_average_ewm_50d = df['Close'].ewm(span=50, adjust=False).mean()
    moving_average_ewm_100d = df['Close'].ewm(span=100, adjust=False).mean()
    moving_average_ewm_200d = df['Close'].ewm(span=200, adjust=False).mean()
    moving_average_ewm = pd.DataFrame()
    moving_average_ewm = pd.concat([moving_average_ewm_5d, moving_average_ewm_10d, moving_average_ewm_20d, moving_average_ewm_50d, moving_average_ewm_100d, moving_average_ewm_200d], axis=1)
    moving_average_ewm.columns = ['Exponential Moving Average 5d', 'Exponential Moving Average 10d', 'Exponential Moving Average 20d', 'Exponential Moving Average 50d', 'Exponential Moving Average 100d', 'Exponential Moving Average 200d']
    
    if signal==True:
      short_window = 5
      moving_average_ewm['signal'] = 0
      moving_average_ewm['position_signal'] = 0
      moving_average_ewm['position_ema'] = 'Neutral'
      moving_average_ewm['signal'][short_window:] = np.where(moving_average_ewm['Exponential Moving Average 5d'][short_window:]
                                                             >moving_average_ewm['Exponential Moving Average 200d'][short_window:], 1.0, 0)
      moving_average_ewm['position_signal'] = moving_average_ewm['signal'].diff()
      moving_average_ewm['position_ema'] = moving_average_ewm['position_signal'].apply(lambda x: signal_ma(x))
      
      if plot == True:
        fig = make_subplots(rows=2, cols=1, shared_xaxes=True, vertical_spacing=0.02)
        fig.append_trace(go.Scatter(y = df['Close'], x = df.index, mode='lines', name='Close'), row=1, col=1)
        fig.append_trace(go.Scatter(y = moving_average_ewm['Exponential Moving Average 5d'], x = moving_average_ewm.index, mode='lines', name='Exponential Moving Average 5d'), row=2, col=1)
        fig.append_trace(go.Scatter(y = moving_average_ewm['Exponential Moving Average 200d'], x = moving_average_ewm.index, mode='lines', name='Exponential Moving Average 200d'), row=2, col=1)
        fig.add_trace(go.Scatter(y = moving_average_ewm[moving_average_ewm['position_ema']=='Bull']['Exponential Moving Average 5d'], x=moving_average_ewm[moving_average_ewm['position_ema']=='Bull'].index, mode='markers', marker_symbol='triangle-up', marker_color='green', marker_size=15, name='Bull'), row=2, col=1)
        fig.add_trace(go.Scatter(y = moving_average_ewm[moving_average_ewm['position_ema']=='Bear']['Exponential Moving Average 5d'], x=moving_average_ewm[moving_average_ewm['position_ema']=='Bear'].index, mode='markers', marker_symbol='triangle-down', marker_color='red', marker_size=15, name='Bear'), row=2, col=1)
        fig.update_layout(title='Closing Price with Exponential Moving Average and Bull/Bear Signals', template='plotly_dark')
        
        fig.show()
        return moving_average_ewm
      elif plot == False:
        return moving_average_ewm
    
    elif signal==False:
     if plot == True:
        fig = make_subplots(rows=7, cols=1, shared_xaxes=True, vertical_spacing=0.02)
        fig.append_trace(go.Scatter(y = df['Close'], x = df.index, mode='lines', name='Close'), row=1, col=1)
        fig.append_trace(go.Scatter(y = moving_average_ewm['Exponential Moving Average 5d'], x = df.index, mode='lines', name='Exponential Moving Average 5d'), row=2, col=1)
        fig.append_trace(go.Scatter(y = moving_average_ewm['Exponential Moving Average 10d'], x = df.index, mode='lines', name='Exponential Moving Average 10d'), row=3, col=1)
        fig.append_trace(go.Scatter(y = moving_average_ewm['Exponential Moving Average 20d'], x = df.index, mode='lines', name='Exponential Moving Average 20d'), row=4, col=1)
        fig.append_trace(go.Scatter(y = moving_average_ewm['Exponential Moving Average 50d'], x = df.index, mode='lines', name='Exponential Moving Average 50d'), row=5, col=1)
        fig.append_trace(go.Scatter(y = moving_average_ewm['Exponential Moving Average 100d'], x = df.index, mode='lines', name='Exponential Moving Average 100d'), row=6, col=1)
        fig.append_trace(go.Scatter(y = moving_average_ewm['Exponential Moving Average 200d'], x = df.index, mode='lines', name='Exponential Moving Average 200d'), row=7, col=1)
        fig.update_layout(title='Closing Price with Exponential Moving Average', template='plotly_dark')
        
        fig.show()
        return moving_average_ewm
     elif plot == False:
        return moving_average_ewm

def MACD(df, a=12, b=26, c=9, signal=False, plot=False):
    '''
    Calculate moving average convergence divergence (MACD) for given data
    :param df: pandas.DataFrame() :market data downloaded from get_data()
    :param a: number of periods for moving average fast line: default = 12
    :param b: number of periods for moving average slow line: default = 26
    :param c: number of periods for macd signal line: default = 9
    :param plot: Boolean: if True, closing price with MACD is plotted
    :param signal: Boolean: if True, bullish/bearish signals are returned
    :return: pandas.DataFrame() : MA_Fast, MA_Slow, MACD, Signal and Positions are returned
    '''
    df["MA_Fast"]=df['Close'].ewm(span=a,min_periods=a).mean()
    df["MA_Slow"]=df['Close'].ewm(span=b,min_periods=b).mean()
    df["MACD"]=df["MA_Fast"]-df["MA_Slow"]
    df["Signal"]=df["MACD"].ewm(span=c,min_periods=c).mean()
    df.dropna(inplace=True)
    
    def signal_macd(x):
      if x == 1:
        return 'Bull'
      elif x == -1:
        return 'Bear'
      else:
        return 'Neutral'

    if signal == True:
      short_window = c
      df['signal_pos'] = 0
      df['position_signal'] = 0
      df['signal_pos'][short_window:] = np.where(df['Signal'][short_window:]<df['MACD'][short_window:], 1.0, 0)
      df['position_signal'] = df['signal_pos'].diff()
      df['position_macd'] = df['position_signal'].apply(lambda x: signal_macd(x))
      if plot == True:
          fig = make_subplots(rows=2, cols=1, shared_xaxes=True, vertical_spacing=0.02)
          fig.append_trace(go.Scatter(y = df['Close'], x = df.index, mode='lines', name='Close'), row=1, col=1)
          fig.append_trace(go.Scatter(y = df['MACD'], x = df.index, mode='lines', name='MACD Line'), row=2, col=1)
          fig.append_trace(go.Scatter(y = df['Signal'], x = df.index, mode='lines', name='Signal Line'), row=2, col=1)
          fig.add_trace(go.Scatter(y = df[df['position_macd']=='Bull']['Signal'], x=df[df['position_macd']=='Bull'].index, mode='markers', marker_symbol='triangle-up', marker_color='green', marker_size=15, name='Bull'), row=2, col=1)
          fig.add_trace(go.Scatter(y = df[df['position_macd']=='Bear']['Signal'], x=df[df['position_macd']=='Bear'].index, mode='markers', marker_symbol='triangle-down', marker_color='red', marker_size=15, name='Bear'), row=2, col=1)
          fig.update_layout(title='Closing Price with MACD and Bull/Bear Signals', template='plotly_dark')
         
          fig.show()
          return df

      elif plot == False:
          return df



    elif signal == False:
        if plot == True:
          fig = make_subplots(rows=2, cols=1, shared_xaxes=True, vertical_spacing=0.02)
          fig.append_trace(go.Scatter(y = df['Close'], x = df.index, mode='lines', name='Close'), row=1, col=1)
          fig.append_trace(go.Scatter(y = df['MACD'], x = df.index, mode='lines', name='MACD Line'), row=2, col=1)
          fig.append_trace(go.Scatter(y = df['Signal'], x = df.index, mode='lines', name='Signal Line'), row=2, col=1)
          fig.update_layout(title='Closing Price with MACD', template='plotly_dark')
          
          fig.show()
          return df
        elif plot == False:
          return df

def RSI (df, time_window=14, signal=False, plot=False):
    '''
    Calculate relative strength index (RSI) for given data
    :param df: pandas.DataFrame() :market data downloaded from get_data()
    :param time_window: number of periods for RSI : default = 14
    :param plot: Boolean: if True, closing price with RSI is plotted
    :param signal: Boolean: if True, bullish/bearish signals are returned
    :return: pandas.DataFrame() : RSI and Position is returned
    '''
    diff = df['Close'].diff(1).dropna()        
    up_chg = 0 * diff
    down_chg = 0 * diff    
    up_chg[diff > 0] = diff[ diff>0 ]    
    down_chg[diff < 0] = diff[ diff < 0 ]    
    up_chg_avg   = up_chg.ewm(com=time_window-1 , min_periods=time_window).mean()
    down_chg_avg = down_chg.ewm(com=time_window-1 , min_periods=time_window).mean()    
    rs = abs(up_chg_avg/down_chg_avg)
    rsi = 100 - 100/(1+rs)
    df['rsi'] = rsi
    def signal_rsi(x):
      if x<30:
        return 'bull'
      elif x>70:
        return 'bear'
      else:
        return 'Neutral'

    if signal == True:
      df['position_rsi'] = df['rsi'].apply(lambda x: signal_rsi(x)).dropna()
      df = df.dropna()
      if plot == False:
        return df
      
      elif plot == True:
         up = pd.DataFrame(index=df.index)
         up['upper'] = 70
         up['lower'] = 30
         fig = make_subplots(rows=2, cols=1, shared_xaxes=True, vertical_spacing=0.02)
         fig.append_trace(go.Scatter(y = df['Close'], x = df.index, mode='lines', name='Close'), row=1, col=1)
         fig.append_trace(go.Scatter(y = df['rsi'], x = df.index, mode='lines', name='RSI',line=dict(color='firebrick')), row=2, col=1)
         fig.add_trace(go.Scatter(y = up['upper'], x = up.index, mode='lines',line=dict(color='purple', width=2, dash='dot'), name='Over Bought'), row=2, col=1)
         fig.add_trace(go.Scatter(y = up['lower'], x = up.index, mode='lines',line=dict(color='purple', width=2, dash='dot'), name='Over Sold'), row=2, col=1)
         fig.add_trace(go.Scatter(y = df[df['position_rsi']=='bull']['rsi'], x=df[df['position_rsi']=='bull'].index, mode='markers', marker_symbol='triangle-up', marker_color='green', marker_size=15, name='Bull'), row=2, col=1)
         fig.add_trace(go.Scatter(y = df[df['position_rsi']=='bear']['rsi'], x=df[df['position_rsi']=='bear'].index, mode='markers', marker_symbol='triangle-down', marker_color='red', marker_size=15, name='Bear'), row=2, col=1)
         fig.update_layout(title='Closing Price with RSI and Bull/Bear Signals', template='plotly_dark')
         
         fig.show()
         return df.dropna()
    
    elif signal == False:
      if plot == True:
        fig = make_subplots(rows=2, cols=1, shared_xaxes=True, vertical_spacing=0.02)
        fig.append_trace(go.Scatter(y = df['Close'], x = df.index, mode='lines', name='Close'), row=1, col=1)
        fig.append_trace(go.Scatter(y = df['rsi'], x = df.index, mode='lines', name='RSI',line=dict(color='firebrick')), row=2, col=1)
        fig.update_layout(title='Closing Price with RSI', template='plotly_dark')
        
        fig.show()
      return df.dropna()

      if plot == False:
        return df.dropna()

def bollinger_bands(df, n=20, plot=True):
    '''
    Calculate Bollinger Bands for given data
    :param df: pandas.DataFrame() :market data downloaded from get_data()
    :param n: number of periods for Bollinger Bands: default = 20
    :param plot: Boolean: if True, closing price with Bollinger Bands are plotted
    :return: pandas.DataFrame(): Upper Band, Lower Band and Middle Band are returned
    '''
    df['Middle Band'] = df['Close'].rolling(window=n).mean()
    df['Upper Band'] = df['Middle Band'] + 1.96*df['Close'].rolling(window=n).std()
    df['Lower Band'] = df['Middle Band'] - 1.96*df['Close'].rolling(window=n).std()
    if plot == True:
      fig = go.Figure()
      fig.add_trace(go.Scatter(x=df.index, y= df['Middle Band'],line=dict(color='blue', width=.7), name = 'Middle Band'))
      fig.add_trace(go.Scatter(x=df.index, y= df['Upper Band'],line=dict(color='red', width=1.5), name = 'Upper Band (Sell)'))
      fig.add_trace(go.Scatter(x=df.index, y= df['Lower Band'],line=dict(color='green', width=1.5), name = 'Lower Band (Buy)'))
      fig.add_trace(go.Candlestick(x=df.index,
                      open=df['Open'],
                      high=df['High'],
                      low=df['Low'],
                      close=df['Close'], name = 'market data'))
      fig.update_layout(xaxis_rangeslider_visible=False)
      fig.update_layout(
          title='Bollinger Band Strategy', template='plotly_dark')
      
      fig.show()
          
      return df

def IchimokuCloud(df, plot=False):
  '''
  Calculate Ichimoku Clouds for given data
  :param df: pandas.DataFrame() :market data downloaded from get_data()
  :param plot: Boolean: if True, closing price with Ichimoku Clouds are plotted
  :return: pandas.DataFrame(): Conv_line, Base_line, Lead_span_A, Lead_span_B and Lagging span
  '''
  CL_period = 20 # length of Tenkan Sen or Conversion Line
  BL_period = 60 # length of Kijun Sen or Base Line
  Lead_span_B_period = 120 # length of Senkou Sen B or Leading Span B
  Lag_span_period = 30
  df['Conv_line'] = (df.High.shift(CL_period)+df.Low.shift(CL_period))/2
  df['Base_line'] = (df.High.shift(BL_period)+df.Low.shift(BL_period))/2
  df['Lead_span_A'] = (df['Conv_line'] + df['Base_line'])/2
  df['Lead_span_B'] = (df.High.shift(Lead_span_B_period)+df.Low.shift(Lead_span_B_period))/2
  df['Lagging_span'] = df.Close.shift(Lag_span_period)
  if plot == True:
    fig = go.Figure()
    fig.add_trace(go.Scatter(x=df.index, y= df['Close'],line=dict(color='blue', width=.7), name = 'Closing Price'))
    fig.add_trace(go.Scatter(x=df.index, y= df['Lead_span_A'],line=dict(color='green', width=.7), name = 'Lead span A'))
    fig.add_trace(go.Scatter(x=df.index, y= df['Lead_span_B'],line=dict(color='red', width=.7), name = 'Lead span B'))
    fig.update_layout(title = 'Closing price with Ichimoku Clouds', template='plotly_dark')
    fig.show()
    return df
  elif plot == False:
    return df

def ADX(df, trend=False, plot=False):
  '''
  Calculate average directional index for given data
  :param df: pandas.DataFrame() :market data downloaded from get_data()
  :param trend: Boolean: if True, strength of the trend is returned
  :param plot: Boolean: if True, closing price with ADX is plotted
  :return: pandas.DataFrame(): ADX, Positive Directional Index and Negative Directional Index
  '''
  def trend_strength(x):
    strength = ''
    if 0<=x and x<25:
      strength = 'Weak Trend'
    elif 25<=x and x<50:
      strength = 'Strong Trend'
    elif 50<=x and x<75:
      strength = 'Robust Trend'
    else:
      strength = 'Extremly Strong Trend'
    return strength

  df['Adj Open'] = df.Open * df['Adj Close']/df['Close']
  df['Adj High'] = df.High * df['Adj Close']/df['Close']
  df['Adj Low'] = df.Low * df['Adj Close']/df['Close']
  df.dropna(inplace=True)
  adxI = ADXIndicator(df['Adj High'],df['Adj Low'],df['Adj Close'],14,False)
  df['pos_directional_indicator'] = adxI.adx_pos()
  df['neg_directional_indicator'] = adxI.adx_neg()
  df['adx'] = adxI.adx()

  if trend == True:
    df['trend'] = df['adx'].apply(lambda x: trend_strength(x))
    if plot == True:
      fig = make_subplots(rows=2, cols=1, shared_xaxes=True, vertical_spacing=0.02)
      fig.append_trace(go.Scatter(y = df['Close'], x = df.index, mode='lines', name='Close'), row=1, col=1)
      fig.append_trace(go.Scatter(y = df['adx'], x = df.index, mode='lines', line=dict(color='lightblue', width=1),name='ADX'), row=2, col=1)
      fig.add_trace(go.Scatter(y = df['pos_directional_indicator'], x = df.index, mode='lines', line=dict(color='green', width=.7),name='Positive Directional Indicator'), row=2, col=1)
      fig.add_trace(go.Scatter(y = df['neg_directional_indicator'], x = df.index, mode='lines', line=dict(color='coral', width=.7),name='Negative Directional Indicator'), row=2, col=1)
      fig.update_layout(title = 'Close price with ADX', template='plotly_dark')
      fig.show()
      return df
    elif plot == False:
      return df 
  if trend == False:
    if plot == True:
       fig = make_subplots(rows=2, cols=1, shared_xaxes=True, vertical_spacing=0.02)
       fig.append_trace(go.Scatter(y = df['Close'], x = df.index, mode='lines', name='Close'), row=1, col=1)
       fig.append_trace(go.Scatter(y = df['adx'], x = df.index, mode='lines', line=dict(color='navy', width=1),name='ADX'), row=2, col=1)
       fig.add_trace(go.Scatter(y = df['pos_directional_indicator'], x = df.index, mode='lines', line=dict(color='green', width=.7),name='Positive Directional Indicator'), row=2, col=1)
       fig.add_trace(go.Scatter(y = df['neg_directional_indicator'], x = df.index, mode='lines', line=dict(color='coral', width=.7),name='Negative Directional Indicator'), row=2, col=1)
       fig.update_layout(title = 'Close price with ADX', template='plotly_dark')
       fig.show()
       return df
    elif plot == False:
       return df

def ATR(DF,n=14, plot=False):
    '''
    Calculate average true range (ATR) for given data
    :param DF: pandas.DataFrame() :market data downloaded from get_data()
    :param n: number of periods for ATR: default = 14
    :param plot: Boolean: if True, closing price with ATR is plotted
    :return: pandas.DataFrame(): ATR 
    '''
    df = DF.copy()
    df['H-L']=abs(df['High']-df['Low'])
    df['H-PC']=abs(df['High']-df['Close'].shift(1))
    df['L-PC']=abs(df['Low']-df['Close'].shift(1))
    df['TR']=df[['H-L','H-PC','L-PC']].max(axis=1,skipna=False)
    #df['ATR'] = df['TR'].rolling(n).mean()
    df['ATR'] = df['TR'].ewm(span=n,adjust=False,min_periods=n).mean()
    df2 = df.drop(['H-L','H-PC','L-PC'],axis=1)
    if plot == True:
      fig = make_subplots(rows=2, cols=1, shared_xaxes=True, vertical_spacing=0.02)
      fig.add_trace(go.Scatter(x=df.index, y= df2['ATR'],line=dict(color='blue', width=1.5), name = 'ATR'),row=2, col=1)
      fig.add_trace(go.Candlestick(x=df.index,
                      open=df['Open'],
                      high=df['High'],
                      low=df['Low'],
                      close=df['Close'], name = 'market data'),row=1, col=1)
      fig.update_xaxes(
    rangeslider_visible=False,
    
)
      fig.update_layout(
          title='Closing Price with Average True Range', template='plotly_dark')
      
      fig.show()
      return df2
    elif plot == False:
      return df2

def stochastic_oscillator(df, signal=False, plot=False):
    """Calculate stochastic oscillator %K and %D for given data.    
    :param df: pandas.DataFrame() :market data downloaded from get_data()
    :param plot: Boolean: if True, closing price with stochastic oscillator is plotted
    :param signal: Boolean: if True, bullish/bearish signals are returned
    :return: pandas.DataFrame(): %K and %D values
    """
    df['L14'] = df['Low'].rolling(window=14).min()
    df['H14'] = df['High'].rolling(window=14).max()
    df['%K'] = 100*((df['Close'] - df['L14']) / (df['H14'] - df['L14']) )
    df['%D'] = df['%K'].rolling(window=3).mean()
    
    up = pd.DataFrame(index=df.index)
    up['upper'] = 80
    up['lower'] = 20
    
    if signal == True:
      df['Buy Entry'] = ((df['%K'] > df['%D']) & (df['%K'].shift(1) < df['%D'].shift(1))) & (df['%D'] < 20) 
      df['Sell Entry'] = ((df['%K'] < df['%D']) & (df['%K'].shift(1) > df['%D'].shift(1))) & (df['%D'] > 80) 
      df['position_so'] = 'Neutral'
      df.loc[df['Buy Entry'], 'position_so'] = 'Bull'
      df.loc[df['Sell Entry'], 'position_so'] = 'Bear'
      
      if plot == True:
         fig = make_subplots(rows=2, cols=1, shared_xaxes=True, vertical_spacing=0.02)
         fig.append_trace(go.Scatter(y = df['Close'], x = df.index, mode='lines', name='Close'), row=1, col=1)
         fig.append_trace(go.Scatter(y = df['%K'], x = df.index, mode='lines', name='Stochastic Oscillator %K',line=dict(color='lightgreen')), row=2, col=1)
         fig.add_trace(go.Scatter(y = df['%D'], x = df.index, mode='lines', name='Stochastic Oscillator %D',line=dict(color='salmon')), row=2, col=1)
         fig.add_trace(go.Scatter(y = up['upper'], x = up.index, mode='lines',line=dict(color='white', width=2, dash='dot'), name='Over Bought'), row=2, col=1)
         fig.add_trace(go.Scatter(y = up['lower'], x = up.index, mode='lines',line=dict(color='white', width=2, dash='dot'), name='Over Sold'), row=2, col=1)
         fig.add_trace(go.Scatter(y = df[df['position_so']=='Bull']['%D'], x=df[df['position_so']=='Bull'].index, mode='markers', marker_symbol='triangle-up', marker_color='green', marker_size=15, name='Bull'), row=2, col=1)
         fig.add_trace(go.Scatter(y = df[df['position_so']=='Bear']['%D'], x=df[df['position_so']=='Bear'].index, mode='markers', marker_symbol='triangle-down', marker_color='red', marker_size=15, name='Bear'), row=2, col=1)
         fig.update_layout(title='Closing Price with Stochastic Oscillator and Bull/Bear Signals', template='plotly_dark')
         fig.show()
         return df
      elif plot==False:
         return df
    elif signal == False:
      if plot == True:
        fig = make_subplots(rows=2, cols=1, shared_xaxes=True, vertical_spacing=0.02)
        fig.append_trace(go.Scatter(y = df['Close'], x = df.index, mode='lines', name='Close'), row=1, col=1)
        fig.append_trace(go.Scatter(y = df['%K'], x = df.index, mode='lines', name='Stochastic Oscillator %K',line=dict(color='lightgreen')), row=2, col=1)
        fig.add_trace(go.Scatter(y = df['%D'], x = df.index, mode='lines', name='Stochastic Oscillator %D',line=dict(color='salmon')), row=2, col=1)
        fig.add_trace(go.Scatter(y = up['upper'], x = up.index, mode='lines',line=dict(color='white', width=2, dash='dot'), name='Over Bought'), row=2, col=1)
        fig.add_trace(go.Scatter(y = up['lower'], x = up.index, mode='lines',line=dict(color='white', width=2, dash='dot'), name='Over Sold'), row=2, col=1)
        fig.update_layout(title='Closing Price with Stochastic Oscillator', template='plotly_dark')
        fig.show()
        return df
      elif plot == False:
        return df

def OBV(DF, plot=False, signal=False):
    '''
    Calculate on balance volume (OBV) for given data
    :param DF: pandas.DataFrame() :market data downloaded from get_data()
    :param plot: Boolean: if True, closing price with OBV is plotted
    :param signal: Boolean: if True, bullish/bearish signals are returned
    :return: pandas.DataFrame(): %K and %D values
    '''
    df = DF.copy()
    df['daily_ret'] = df['Adj Close'].pct_change()
    df['direction'] = np.where(df['daily_ret']>=0,1,-1)
    df['direction'][0] = 0
    df['vol_adj'] = df['Volume'] * df['direction']
    df['obv'] = df['vol_adj'].cumsum()
    df.dropna(inplace=True)
    if signal==True:
      df['Buy Entry'] =  np.logical_and((df['Close']<df['Close'].shift(1)), (df['obv'] > df['obv'].shift(1)))  
      df['Sell Entry'] = np.logical_and((df['Close']>df['Close'].shift(1)), (df['obv'] < df['obv'].shift(1))) 
      df['position_obv'] = 'Neutral'
      df.loc[df['Buy Entry'], 'position_obv'] = 'Bull'
      df.loc[df['Sell Entry'], 'position_obv'] = 'Bear'
      if plot == True:
        fig = make_subplots(rows=2, cols=1, shared_xaxes=True, vertical_spacing=0.02)
        fig.append_trace(go.Scatter(y = df['Close'], x = df.index, mode='lines', name='Close'), row=1, col=1)
        fig.append_trace(go.Scatter(y = df['obv'], x = df.index, mode='lines', name='OBV'), row=2, col=1)
        fig.add_trace(go.Scatter(y = df[df['position_obv']=='Bull']['obv'], x=df[df['position_obv']=='Bull'].index, mode='markers', marker_symbol='triangle-up', marker_color='green', marker_size=15, name='Bull'), row=2, col=1)
        fig.add_trace(go.Scatter(y = df[df['position_obv']=='Bear']['obv'], x=df[df['position_obv']=='Bear'].index, mode='markers', marker_symbol='triangle-down', marker_color='red', marker_size=15, name='Bear'), row=2, col=1)
        fig.update_layout(title='Closing Price with OBV and Bull/Bear Signals', template='plotly_dark')
        fig.show()
        return df
      elif plot == False:
        return df


    elif signal ==False:
        if plot == True:
          fig = make_subplots(rows=2, cols=1, shared_xaxes=True, vertical_spacing=0.02)
          fig.append_trace(go.Scatter(y = df['Close'], x = df.index, mode='lines', name='Close'), row=1, col=1)
          fig.append_trace(go.Scatter(y = df['obv'], x = df.index, mode='lines', name='OBV'), row=2, col=1)
          fig.update_layout(title='Closing Price with OBV', template='plotly_dark')
          fig.show()
          return df
        elif plot == False:
          return df

def ppsr(df):
    """Calculate Pivot Points, Supports and Resistances for given data
    :param df: pandas.DataFrame() :market data downloaded from get_data()
    :return: pandas.DataFrame() : Pivot Points, Resistances and Supports
    """
    PP = pd.Series((df['High'] + df['Low'] + df['Close']) / 3)
    R1 = pd.Series(2 * PP - df['Low'])
    S1 = pd.Series(2 * PP - df['High'])
    R2 = pd.Series(PP + df['High'] - df['Low'])
    S2 = pd.Series(PP - df['High'] + df['Low'])
    R3 = pd.Series(df['High'] + 2 * (PP - df['Low']))
    S3 = pd.Series(df['Low'] - 2 * (df['High'] - PP)
    psr = {'PP': PP, 'R1': R1, 'S1': S1, 'R2': R2, 'S2': S2, 'R3': R3, 'S3': S3}
    PSR = pd.DataFrame(psr)
    df = df.join(PSR)
    return df

def semideviation(df):
    """
    Calculate semi deviation for given close price
    :param df: pandas.DataFrame(): close price of data
    :return: float: value of semi deviation
    """
    r = df.pct_change().dropna()
    if isinstance(r, pd.Series):
        is_negative = r < 0
        return r[is_negative].std(ddof=0)
    elif isinstance(r, pd.DataFrame):
        return r.aggregate(semideviation)
    else:
        raise TypeError("Expected r to be a Series or DataFrame")

def meandeviation(df):
    """
    Calculate mean deviation for given close price
    :param df: pandas.DataFrame(): close price of data
    :return: float: value of mean deviation
    """
    r = df.pct_change().dropna()
    results = r.mad()
    return results

def standard_deviation(df, n=21):
    """
    Calculate standard Deviation for given data.
    :param df: pandas.DataFrame(): close price of data
    :param n: number of periods: default = 21
    :return: pandas.DataFrame(): moving standard deviations
    """
    std = df.rolling(n, min_periods=n).std()
    return std

def TSI(df, r=25, s=13, c=9, signal=False, plot=False):
    """
    Calculate True Strength Index (TSI) for given data.
    :param df: pandas.DataFrame(): market data downloaded from get_data()
    :param r: time period for EMA_Fast: default = 25 
    :param s: time period for EMA_SLow: default = 13
    :param c: time period for Signal Line: default = 9
    :param plot: Boolean: if True, closing price with TSI is plotted
    :param signal: Boolean: if True, bullish/bearish signals are returned
    :return: pandas.DataFrame(): Price Change(pc), Price Change Smoothed(pcs), Price Change Double Smooth(pcds), Absolute Price Change(apc),
    Absolute Price Change Smoothed(apcs), Absolute Price Change Double Smooth(apcds), TSI and Signal
    """
    
    df['pc'] = df['Close'].diff(1)
    df['pcs'] = df['pc'].ewm(span=r, min_periods=r).mean()
    df['pcds'] = df['pcs'].ewm(span=s, min_periods=s).mean()
    df['apc'] = np.abs(df['pc'])
    df['apcs'] = df['apc'].ewm(span=r, min_periods=r).mean()
    df['apcds'] = df['apcs'].ewm(span=s, min_periods=s).mean()
    df['TSI'] = (df['pcds']/df['apcds'])*100
    df["Signal"]=df["TSI"].ewm(span=c,min_periods=c).mean()
    def signal_tsi(x):
      if x == 1:
        return 'Bull'
      elif x == -1:
        return 'Bear'
      else:
        return 'Neutral'
    zero = pd.DataFrame(index=df.index)
    zero['O'] = 0

    if signal == True:
      short_window = c
      df['signal_pos'] = 0
      df['position_signal'] = 0
      df['signal_pos'][short_window:] = np.where(df['Signal'][short_window:]<df['TSI'][short_window:], 1.0, 0)
      df['position_signal'] = df['signal_pos'].diff()
      df['position_tsi'] = df['position_signal'].apply(lambda x: signal_tsi(x))
      if plot == True:
        fig = make_subplots(rows=2, cols=1, shared_xaxes=True, vertical_spacing=0.02)
        fig.append_trace(go.Scatter(y = df['Close'], x = df.index, mode='lines', name='Close'), row=1, col=1)
        fig.append_trace(go.Scatter(y = df['TSI'], x = df.index, mode='lines', name='TSI Line'), row=2, col=1)
        fig.append_trace(go.Scatter(y = df['Signal'], x = df.index, mode='lines', name='Signal Line'), row=2, col=1)
        fig.add_trace(go.Scatter(y = zero['O'], x = zero.index, mode='lines', name='Centre Line'), row=2, col=1)
        fig.add_trace(go.Scatter(y = df[df['position_tsi']=='Bull']['TSI'], x=df[df['position_tsi']=='Bull'].index, mode='markers', marker_symbol='triangle-up', marker_color='green', marker_size=15, name='Bull'), row=2, col=1)
        fig.add_trace(go.Scatter(y = df[df['position_tsi']=='Bear']['TSI'], x=df[df['position_tsi']=='Bear'].index, mode='markers', marker_symbol='triangle-down', marker_color='red', marker_size=15, name='Bear'), row=2, col=1)
        fig.update_layout(title='Closing Price with TSI and Bull/Bear Signals', template='plotly_dark')
        fig.show()
        return df
      elif plot == False:
        return df
    if signal == False:
      if plot == True:
        fig = make_subplots(rows=2, cols=1, shared_xaxes=True, vertical_spacing=0.02)
        fig.append_trace(go.Scatter(y = df['Close'], x = df.index, mode='lines', name='Close'), row=1, col=1)
        fig.append_trace(go.Scatter(y = df['TSI'], x = df.index, mode='lines', name='TSI Line'), row=2, col=1)
        fig.append_trace(go.Scatter(y = df['Signal'], x = df.index, mode='lines', name='Signal Line'), row=2, col=1)
        fig.add_trace(go.Scatter(y = zero['O'], x = zero.index, mode='lines', name='Centre Line'), row=2, col=1)
        fig.update_layout(title='Closing Price with TSI', template='plotly_dark')
        fig.show()
        return df
      elif plot == False:
        return df

def MFI(df, n=14, signal = False, plot=False):
    """Calculate Money Flow Index(MFI) for given data.
    :param df: pandas.DataFrame(): market data downloaded from get_data()
    :param n: number of periods for MFI: default = 14
    :param plot: Boolean: if True, closing price with MFI is plotted
    :param signal: Boolean: if True, bullish/bearish signals are returned
    :return: pandas.DataFrame(): Typical Price, Money Flow, MFI
    """
    df['typical_price'] = (df['Close'] + df['High'] + df['Low']) / 3
    df['money_flow'] = df['typical_price'] * df['Volume']

    positive_flow = []
    negative_flow = []

    for i in range (1, len(df['typical_price'])):
      if df['typical_price'].iloc[i] > df['typical_price'].iloc[i-1]:
        positive_flow.append(df['money_flow'].iloc[i-1])
        negative_flow.append(0)
      elif df['typical_price'].iloc[i] < df['typical_price'].iloc[i-1]:
        negative_flow.append(df['money_flow'].iloc[i-1])
        positive_flow.append(0)
      else:
        negative_flow.append(0)
        positive_flow.append(0)
    
    positive_mf = []
    negative_mf = []
    for i in range(n-1, len(positive_flow)):
      positive_mf.append(sum(positive_flow[i+1-n : i+1]))
    for i in range(n-1, len(negative_flow)):
      negative_mf.append(sum(negative_flow[i+1-n : i+1]))
    mfi = 100 * (np.array(positive_mf) / (np.array(positive_mf)  + np.array(negative_mf) ))
    df['MFI'] = np.nan
    df['MFI'][n:] = mfi
    
    def signal_mfi(x):
      if x>80:
        return 'Bear'
      elif x<20:
        return 'Bull' 
      else:
        return 'Neutral' 
        
    up = pd.DataFrame(index=df.index)
    up['upper'] = 80
    up['lower'] = 20

    if signal == True:
      df['position_mfi'] = df['MFI'].apply(lambda x: signal_mfi(x))

      if plot == True:
         fig = make_subplots(rows=2, cols=1, shared_xaxes=True, vertical_spacing=0.02)
         fig.append_trace(go.Scatter(y = df['Close'], x = df.index, mode='lines', name='Close'), row=1, col=1)
         fig.append_trace(go.Scatter(y = df['MFI'], x = df.index, mode='lines', name='MFI'), row=2, col=1)
         fig.add_trace(go.Scatter(y = up['upper'], x = up.index, mode='lines',line=dict(color='white', width=2, dash='dot'), name='Over Bought'), row=2, col=1)
         fig.add_trace(go.Scatter(y = up['lower'], x = up.index, mode='lines',line=dict(color='white', width=2, dash='dot'), name='Over Sold'), row=2, col=1)
         fig.add_trace(go.Scatter(y = df[df['position_mfi']=='Bull']['MFI'], x=df[df['position_mfi']=='Bull'].index, mode='markers', marker_symbol='triangle-up', marker_color='green', marker_size=15, name='Bull'), row=2, col=1)
         fig.add_trace(go.Scatter(y = df[df['position_mfi']=='Bear']['MFI'], x=df[df['position_mfi']=='Bear'].index, mode='markers', marker_symbol='triangle-down', marker_color='red', marker_size=15, name='Bear'), row=2, col=1)
         fig.update_layout(title='Closing Price with Money Flow Index and Bull/Bear Signals', template='plotly_dark')
         fig.show()
         return df
      elif plot==False:
         return df
    
    elif signal == False:
      if plot == True:
         fig = make_subplots(rows=2, cols=1, shared_xaxes=True, vertical_spacing=0.02)
         fig.append_trace(go.Scatter(y = df['Close'], x = df.index, mode='lines', name='Close'), row=1, col=1)
         fig.append_trace(go.Scatter(y = df['MFI'], x = df.index, mode='lines', name='MFI'), row=2, col=1)
         fig.add_trace(go.Scatter(y = up['upper'], x = up.index, mode='lines',line=dict(color='white', width=2, dash='dot'), name='Over Bought'), row=2, col=1)
         fig.add_trace(go.Scatter(y = up['lower'], x = up.index, mode='lines',line=dict(color='white', width=2, dash='dot'), name='Over Sold'), row=2, col=1)
         fig.update_layout(title='Closing Price with Money Flow Index and Bull/Bear Signals', template='plotly_dark')
         fig.show()
         return df
      elif plot == False:
        return df

def summ(data):
  '''
  Calculate the summary of the latest date
  :param df: pandas.DataFrame(): market data downloaded from get_data()
  :return: pandas.DataFrame(): Three dataframes are returned viz. Moving Average, Technical Indicators and Pivot Points
  '''
  sma = moving_average(data, simple=True)
  ema = moving_average(data, exponential=True)
  ma = pd.concat([sma.tail(1), ema.tail(1)], axis=1)
  ma = ma.T
  macd = MACD(data)
  rsi = RSI(data)
  adx = ADX(data)
  atr = ATR(data)
  so = stochastic_oscillator(data)
  obv = OBV(data)
  tsi = TSI(data)
  mfi = MFI(data)
  tech = pd.concat([macd['MACD'].tail(1), rsi['rsi'].tail(1), adx['adx'].tail(1), atr['ATR'].tail(1), so['%D'].tail(1), obv['obv'].tail(1), tsi['TSI'].tail(1), mfi['MFI'].tail(1)], axis=1)
  tech.columns = ['Moving Average Convergence Divergence(12, 26)', 'Relative Strength Index(14)', 'Average Directional Index', 'Average True Range(14)', 'Stochastic Oscillator', 'On Balance Volume', 'True Strength Index(25,13)', 'Money Flow Index(14)']
  tech = tech.T
  pp = ppsr(data)
  pivot = pp[['PP', 'R1', 'R2', 'R3', 'S1', 'S2', 'S3']].tail(1)
  pivot.columns = ['Pivot Point', 'Resistance 1', 'Resistance 2', 'Resistance 3', 'Support 1', 'Support 2', 'Support 3']
  pivot = pivot.T
  
  return np.round(ma, decimals=1), np.round(tech, decimals=1), np.round(pivot, decimals=1)

def sentiment_signal(data):
  '''
  Analysing the overall sentiment based on techncial indicators
  :param df: pandas.DataFrame(): market data downloaded from get_data()
  :return: pandas.DataFrame(): bull/bear/neutral signal of the technical indicator 
  '''
  sma = moving_average(data, simple=True, signal=True)
  ema = moving_average(data, exponential=True, signal=True)
  sma_signal = sma['position_sma']
  ema_signal = ema['position_ema']
  macd = MACD(data, signal=True)
  macd_signal = macd['position_macd']
  rsi = RSI(data, signal=True)
  rsi_signal = rsi['position_rsi']
  so = stochastic_oscillator(data, signal=True)
  so_signal = so['position_so']
  obv = OBV(data, signal=True)
  obv_signal = obv['position_obv']
  tsi = TSI(data, signal=True)
  tsi_signal = tsi['position_tsi']
  mfi = MFI(data, signal=True)
  mfi_signal = mfi['position_mfi']
  tech_signals = pd.concat([macd_signal.tail(1), rsi_signal.tail(1), so_signal.tail(1), obv_signal.tail(1), tsi_signal.tail(1), mfi_signal.tail(1)], axis=1)
  tech_signals.columns = ['Moving Average Convergence Divergence(12, 26)', 'Relative Strength Index(14)', 'Stochastic Oscillator', 'On Balance Volume', 'True Strength Index(25,13)', 'Money Flow Index(14)']
  tech_signals = tech_signals.T
  tech_signals.columns = ['Signal']
  sentiment = pd.DataFrame()
  sentiment = tech_signals.value_counts()
  sentiment = sentiment.index[0]
  return tech_signals, sentiment