"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/core");
const secretsmanager = require("../lib");
let stack;
let vpc;
let secret;
let securityGroup;
let target;
module.exports = {
    'setUp'(cb) {
        stack = new cdk.Stack();
        vpc = new ec2.Vpc(stack, 'VPC');
        secret = new secretsmanager.Secret(stack, 'Secret');
        securityGroup = new ec2.SecurityGroup(stack, 'SecurityGroup', { vpc });
        target = new ec2.Connections({
            defaultPort: ec2.Port.tcp(3306),
            securityGroups: [securityGroup],
        });
        cb();
    },
    'secret rotation single user'(test) {
        // GIVEN
        const excludeCharacters = ' ;+%{}' + '@\'"`/\\#'; // DMS and BASH problem chars
        // WHEN
        new secretsmanager.SecretRotation(stack, 'SecretRotation', {
            application: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER,
            secret,
            target,
            vpc,
            excludeCharacters: excludeCharacters,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            Description: 'from SecretRotationSecurityGroupAEC520AB:3306',
            FromPort: 3306,
            GroupId: {
                'Fn::GetAtt': [
                    'SecurityGroupDD263621',
                    'GroupId',
                ],
            },
            SourceSecurityGroupId: {
                'Fn::GetAtt': [
                    'SecretRotationSecurityGroup9985012B',
                    'GroupId',
                ],
            },
            ToPort: 3306,
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::RotationSchedule', {
            SecretId: {
                Ref: 'SecretA720EF05',
            },
            RotationLambdaARN: {
                'Fn::GetAtt': [
                    'SecretRotationA9FFCFA9',
                    'Outputs.RotationLambdaARN',
                ],
            },
            RotationRules: {
                AutomaticallyAfterDays: 30,
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            GroupDescription: 'Default/SecretRotation/SecurityGroup',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Serverless::Application', {
            Location: {
                ApplicationId: 'arn:aws:serverlessrepo:us-east-1:297356227824:applications/SecretsManagerRDSMySQLRotationSingleUser',
                SemanticVersion: '1.1.60',
            },
            Parameters: {
                endpoint: {
                    'Fn::Join': [
                        '',
                        [
                            'https://secretsmanager.',
                            {
                                Ref: 'AWS::Region',
                            },
                            '.',
                            {
                                Ref: 'AWS::URLSuffix',
                            },
                        ],
                    ],
                },
                functionName: 'SecretRotation',
                excludeCharacters: excludeCharacters,
                vpcSecurityGroupIds: {
                    'Fn::GetAtt': [
                        'SecretRotationSecurityGroup9985012B',
                        'GroupId',
                    ],
                },
                vpcSubnetIds: {
                    'Fn::Join': [
                        '',
                        [
                            {
                                Ref: 'VPCPrivateSubnet1Subnet8BCA10E0',
                            },
                            ',',
                            {
                                Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A',
                            },
                        ],
                    ],
                },
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::ResourcePolicy', {
            ResourcePolicy: {
                Statement: [
                    {
                        Action: 'secretsmanager:DeleteSecret',
                        Effect: 'Deny',
                        Principal: {
                            AWS: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':iam::',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':root',
                                    ],
                                ],
                            },
                        },
                        Resource: '*',
                    },
                ],
                Version: '2012-10-17',
            },
            SecretId: {
                Ref: 'SecretA720EF05',
            },
        }));
        test.done();
    },
    'secret rotation multi user'(test) {
        // GIVEN
        const masterSecret = new secretsmanager.Secret(stack, 'MasterSecret');
        // WHEN
        new secretsmanager.SecretRotation(stack, 'SecretRotation', {
            application: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER,
            secret,
            masterSecret,
            target,
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Serverless::Application', {
            Parameters: {
                endpoint: {
                    'Fn::Join': [
                        '',
                        [
                            'https://secretsmanager.',
                            {
                                Ref: 'AWS::Region',
                            },
                            '.',
                            {
                                Ref: 'AWS::URLSuffix',
                            },
                        ],
                    ],
                },
                functionName: 'SecretRotation',
                vpcSecurityGroupIds: {
                    'Fn::GetAtt': [
                        'SecretRotationSecurityGroup9985012B',
                        'GroupId',
                    ],
                },
                vpcSubnetIds: {
                    'Fn::Join': [
                        '',
                        [
                            {
                                Ref: 'VPCPrivateSubnet1Subnet8BCA10E0',
                            },
                            ',',
                            {
                                Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A',
                            },
                        ],
                    ],
                },
                masterSecretArn: {
                    Ref: 'MasterSecretA11BF785',
                },
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::ResourcePolicy', {
            ResourcePolicy: {
                Statement: [
                    {
                        Action: 'secretsmanager:DeleteSecret',
                        Effect: 'Deny',
                        Principal: {
                            AWS: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':iam::',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':root',
                                    ],
                                ],
                            },
                        },
                        Resource: '*',
                    },
                ],
                Version: '2012-10-17',
            },
            SecretId: {
                Ref: 'MasterSecretA11BF785',
            },
        }));
        test.done();
    },
    'secret rotation allows passing an empty string for excludeCharacters'(test) {
        // WHEN
        new secretsmanager.SecretRotation(stack, 'SecretRotation', {
            application: secretsmanager.SecretRotationApplication.MARIADB_ROTATION_SINGLE_USER,
            secret,
            target,
            vpc,
            excludeCharacters: '',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Serverless::Application', {
            Parameters: {
                excludeCharacters: '',
            },
        }));
        test.done();
    },
    'throws when connections object has no default port range'(test) {
        // WHEN
        const targetWithoutDefaultPort = new ec2.Connections({
            securityGroups: [securityGroup],
        });
        // THEN
        test.throws(() => new secretsmanager.SecretRotation(stack, 'Rotation', {
            secret,
            application: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER,
            vpc,
            target: targetWithoutDefaultPort,
        }), /`target`.+default port range/);
        test.done();
    },
    'throws when master secret is missing for a multi user application'(test) {
        // THEN
        test.throws(() => new secretsmanager.SecretRotation(stack, 'Rotation', {
            secret,
            application: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER,
            vpc,
            target,
        }), /The `masterSecret` must be specified for application using the multi user scheme/);
        test.done();
    },
    'rotation function name does not exceed 64 chars'(test) {
        // WHEN
        const id = 'SecretRotation'.repeat(5);
        new secretsmanager.SecretRotation(stack, id, {
            application: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER,
            secret,
            target,
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Serverless::Application', {
            Parameters: {
                endpoint: {
                    'Fn::Join': [
                        '',
                        [
                            'https://secretsmanager.',
                            {
                                Ref: 'AWS::Region',
                            },
                            '.',
                            {
                                Ref: 'AWS::URLSuffix',
                            },
                        ],
                    ],
                },
                functionName: 'RotationSecretRotationSecretRotationSecretRotationSecretRotation',
                vpcSecurityGroupIds: {
                    'Fn::GetAtt': [
                        'SecretRotationSecretRotationSecretRotationSecretRotationSecretRotationSecurityGroupBFCB171A',
                        'GroupId',
                    ],
                },
                vpcSubnetIds: {
                    'Fn::Join': [
                        '',
                        [
                            {
                                Ref: 'VPCPrivateSubnet1Subnet8BCA10E0',
                            },
                            ',',
                            {
                                Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A',
                            },
                        ],
                    ],
                },
            },
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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