"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StackResourceRenamer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("@aws-cdk/core");
/**
 * StackResourceRenamer renames stack name and stack's subordinate resources' custom physical names, so that a CDK stack can be used to create multiple stacks in same AWS environment.
 *
 * @stability stable
 */
class StackResourceRenamer {
    /**
     * Construct a new StackResourceRenamer.
     *
     * @param renameOper RenameOperation is used to rename stack name and resources' custom physical names.
     * @param props Properties are set to customize rename operations.
     * @stability stable
     */
    constructor(renameOper, props = {}) {
        this.renameOper = renameOper;
        //mapping for resources whose physical names donot follow
        //the regular naming conventions: `${resourceType}`+'Name'
        this.irregularNames = {
            Stack: '_stackName',
            Output: '_exportName',
            ScalingPolicy: 'policyName',
            SlackChannelConfiguration: 'configurationName',
            CompositeAlarm: 'alarmName',
            SecurityGroup: 'groupName',
            DBProxy: 'dbProxyName',
        };
        this.defaultNameField = 'name';
        if (props.irregularResourceNames) {
            this.irregularNames = {
                ...this.irregularNames,
                ...props.irregularResourceNames,
            };
        }
        this.excludeResTypes = props.excludeResourceTypes;
        this.includeResTypes = props.includeResourceTypes;
    }
    /**
     * Static method to rename a stack and all its subordinate resources.
     *
     * @param stack The stack (and all its children resources) to be renamed.
     * @param renameOper RenameOperation is used to rename stack name and resources' custom physical names.
     * @param props Properties are set to customize rename operations.
     * @stability stable
     */
    static rename(stack, renameOper, props = {}) {
        cdk.Aspects.of(stack).add(new StackResourceRenamer(renameOper, props));
    }
    /**
     * Implement core.IAspect interface.
     *
     * @param node CFN resources to be renamed.
     * @stability stable
     */
    visit(node) {
        if (node instanceof cdk.Stack) {
            //rename stack
            this.renameResource(node, 'Stack');
        }
        else {
            //rename CFN resources
            let ctorName = node.constructor.name;
            if (ctorName.startsWith('Cfn')) {
                this.renameResource(node, ctorName.substring(3));
            }
        }
    }
    /**
     * Rename a CFN resource or stack.
     *
     * @param node CFN resource or stack.
     * @param resTypeName The type name of CFN resource.
     * @stability stable
     */
    renameResource(node, resTypeName) {
        //check include/exclude
        if (this.excludeResTypes && this.excludeResTypes.length > 0 &&
            this.excludeResTypes.indexOf(resTypeName) !== -1) {
            return;
        }
        if (this.includeResTypes && this.includeResTypes.length > 0 &&
            this.includeResTypes.indexOf(resTypeName) === -1) {
            return;
        }
        //find the specific "name" field for CFN resources
        let physicalName = 'name';
        if (this.irregularNames[resTypeName]) {
            physicalName = this.irregularNames[resTypeName];
        }
        else {
            //decapitalize regular resource names
            let [first, ...rest] = resTypeName;
            let decapName = first.toLowerCase() + rest.join('');
            physicalName = `${decapName}Name`;
        }
        if (physicalName.length === 0) {
            return;
        }
        //some protected fields start with underscore
        let underscoreName = '_' + physicalName;
        //rename
        let b = node;
        if (b[physicalName] && b[physicalName].length > 0 && !cdk.Token.isUnresolved(b[physicalName])) {
            b[physicalName] = this.renameOper.rename(b[physicalName], resTypeName);
        }
        else if (b[underscoreName] && b[underscoreName].length > 0 && !cdk.Token.isUnresolved(b[underscoreName])) {
            b[underscoreName] = this.renameOper.rename(b[underscoreName], resTypeName);
        }
        else if (b[this.defaultNameField] && b[this.defaultNameField].length > 0 && !cdk.Token.isUnresolved(b[this.defaultNameField])) {
            b[this.defaultNameField] = this.renameOper.rename(b[this.defaultNameField], resTypeName);
        }
    }
}
exports.StackResourceRenamer = StackResourceRenamer;
_a = JSII_RTTI_SYMBOL_1;
StackResourceRenamer[_a] = { fqn: "cdk-stack-resource-rename.StackResourceRenamer", version: "0.0.8" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvaW5kZXgudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSxxQ0FBcUM7Ozs7OztBQXNEckMsTUFBYSxvQkFBb0I7Ozs7Ozs7O0lBa0MvQixZQUFvQixVQUE0QixFQUFFLFFBQXFCLEVBQUU7UUFBckQsZUFBVSxHQUFWLFVBQVUsQ0FBa0I7UUFyQmhELHlEQUF5RDtRQUN6RCwwREFBMEQ7UUFDbEQsbUJBQWMsR0FBOEI7WUFDbEQsS0FBSyxFQUFFLFlBQVk7WUFDbkIsTUFBTSxFQUFFLGFBQWE7WUFDckIsYUFBYSxFQUFFLFlBQVk7WUFDM0IseUJBQXlCLEVBQUUsbUJBQW1CO1lBQzlDLGNBQWMsRUFBRSxXQUFXO1lBQzNCLGFBQWEsRUFBRSxXQUFXO1lBQzFCLE9BQU8sRUFBRSxhQUFhO1NBQ3ZCLENBQUM7UUFHTSxxQkFBZ0IsR0FBRyxNQUFNLENBQUM7UUFTaEMsSUFBSSxLQUFLLENBQUMsc0JBQXNCLEVBQUU7WUFDaEMsSUFBSSxDQUFDLGNBQWMsR0FBRztnQkFDcEIsR0FBRyxJQUFJLENBQUMsY0FBYztnQkFDdEIsR0FBRyxLQUFLLENBQUMsc0JBQXNCO2FBQ2hDLENBQUM7U0FDSDtRQUNELElBQUksQ0FBQyxlQUFlLEdBQUcsS0FBSyxDQUFDLG9CQUFvQixDQUFDO1FBQ2xELElBQUksQ0FBQyxlQUFlLEdBQUcsS0FBSyxDQUFDLG9CQUFvQixDQUFDO0lBQ3BELENBQUM7Ozs7Ozs7OztJQWxDRCxNQUFNLENBQUMsTUFBTSxDQUFDLEtBQXFCLEVBQUUsVUFBNEIsRUFBRSxRQUFxQixFQUFFO1FBQ3hGLEdBQUcsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLEdBQUcsQ0FBQyxJQUFJLG9CQUFvQixDQUFDLFVBQVUsRUFBRSxLQUFLLENBQUMsQ0FBQyxDQUFDO0lBQ3pFLENBQUM7Ozs7Ozs7SUFxQ0QsS0FBSyxDQUFDLElBQW9CO1FBQ3hCLElBQUksSUFBSSxZQUFZLEdBQUcsQ0FBQyxLQUFLLEVBQUU7WUFDN0IsY0FBYztZQUNkLElBQUksQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1NBQ3BDO2FBQU07WUFDTCxzQkFBc0I7WUFDdEIsSUFBSSxRQUFRLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUM7WUFDckMsSUFBSSxRQUFRLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQyxFQUFFO2dCQUM5QixJQUFJLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxRQUFRLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7YUFDbEQ7U0FDRjtJQUNILENBQUM7Ozs7Ozs7O0lBTVMsY0FBYyxDQUFDLElBQW9CLEVBQUUsV0FBbUI7UUFDaEUsdUJBQXVCO1FBQ3ZCLElBQUksSUFBSSxDQUFDLGVBQWUsSUFBSSxJQUFJLENBQUMsZUFBZSxDQUFDLE1BQU0sR0FBRyxDQUFDO1lBQ3pELElBQUksQ0FBQyxlQUFlLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFO1lBQ2xELE9BQU87U0FDUjtRQUNELElBQUksSUFBSSxDQUFDLGVBQWUsSUFBSSxJQUFJLENBQUMsZUFBZSxDQUFDLE1BQU0sR0FBRyxDQUFDO1lBQ3pELElBQUksQ0FBQyxlQUFlLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFO1lBQ2xELE9BQU87U0FDUjtRQUNELGtEQUFrRDtRQUNsRCxJQUFJLFlBQVksR0FBRyxNQUFNLENBQUM7UUFDMUIsSUFBSSxJQUFJLENBQUMsY0FBYyxDQUFDLFdBQVcsQ0FBQyxFQUFFO1lBQ3BDLFlBQVksR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLFdBQVcsQ0FBQyxDQUFDO1NBQ2pEO2FBQU07WUFDTCxxQ0FBcUM7WUFDckMsSUFBSSxDQUFDLEtBQUssRUFBRSxHQUFHLElBQUksQ0FBQyxHQUFHLFdBQVcsQ0FBQztZQUNuQyxJQUFJLFNBQVMsR0FBRyxLQUFLLENBQUMsV0FBVyxFQUFFLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUNwRCxZQUFZLEdBQUcsR0FBRyxTQUFTLE1BQU0sQ0FBQztTQUNuQztRQUNELElBQUksWUFBWSxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDN0IsT0FBTztTQUNSO1FBQ0QsNkNBQTZDO1FBQzdDLElBQUksY0FBYyxHQUFHLEdBQUcsR0FBRyxZQUFZLENBQUM7UUFDeEMsUUFBUTtRQUNSLElBQUksQ0FBQyxHQUFJLElBQVksQ0FBQztRQUN0QixJQUFJLENBQUMsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLENBQUMsWUFBWSxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxZQUFZLENBQUMsQ0FBQyxFQUFFO1lBQzdGLENBQUMsQ0FBQyxZQUFZLENBQUMsR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsWUFBWSxDQUFDLEVBQUUsV0FBVyxDQUFDLENBQUM7U0FDeEU7YUFBTSxJQUFJLENBQUMsQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLENBQUMsY0FBYyxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxjQUFjLENBQUMsQ0FBQyxFQUFFO1lBQzFHLENBQUMsQ0FBQyxjQUFjLENBQUMsR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDLEVBQUUsV0FBVyxDQUFDLENBQUM7U0FDNUU7YUFBTSxJQUFJLENBQUMsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxFQUFFO1lBQy9ILENBQUMsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLEVBQUUsV0FBVyxDQUFDLENBQUM7U0FDMUY7SUFDSCxDQUFDOztBQW5HSCxvREFvR0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBJUmVuYW1lT3BlcmF0aW9uIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVuYW1lKG9yaWdWYWw6IHN0cmluZywgdHlwZU5hbWU6IHN0cmluZyk6IHN0cmluZztcbn1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBSZW5hbWVQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBpcnJlZ3VsYXJSZXNvdXJjZU5hbWVzPzogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZXhjbHVkZVJlc291cmNlVHlwZXM/OiBzdHJpbmdbXTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGluY2x1ZGVSZXNvdXJjZVR5cGVzPzogc3RyaW5nW107XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgU3RhY2tSZXNvdXJjZVJlbmFtZXIgaW1wbGVtZW50cyBjZGsuSUFzcGVjdCB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgc3RhdGljIHJlbmFtZShzdGFjazogY2RrLklDb25zdHJ1Y3QsIHJlbmFtZU9wZXI6IElSZW5hbWVPcGVyYXRpb24sIHByb3BzOiBSZW5hbWVQcm9wcyA9IHt9KSB7XG4gICAgY2RrLkFzcGVjdHMub2Yoc3RhY2spLmFkZChuZXcgU3RhY2tSZXNvdXJjZVJlbmFtZXIocmVuYW1lT3BlciwgcHJvcHMpKTtcbiAgfVxuXG4gIC8vbWFwcGluZyBmb3IgcmVzb3VyY2VzIHdob3NlIHBoeXNpY2FsIG5hbWVzIGRvbm90IGZvbGxvd1xuICAvL3RoZSByZWd1bGFyIG5hbWluZyBjb252ZW50aW9uczogYCR7cmVzb3VyY2VUeXBlfWArJ05hbWUnXG4gIHByaXZhdGUgaXJyZWd1bGFyTmFtZXM6IHsgW2tleTogc3RyaW5nXTogc3RyaW5nIH0gPSB7XG4gICAgU3RhY2s6ICdfc3RhY2tOYW1lJyxcbiAgICBPdXRwdXQ6ICdfZXhwb3J0TmFtZScsXG4gICAgU2NhbGluZ1BvbGljeTogJ3BvbGljeU5hbWUnLFxuICAgIFNsYWNrQ2hhbm5lbENvbmZpZ3VyYXRpb246ICdjb25maWd1cmF0aW9uTmFtZScsXG4gICAgQ29tcG9zaXRlQWxhcm06ICdhbGFybU5hbWUnLFxuICAgIFNlY3VyaXR5R3JvdXA6ICdncm91cE5hbWUnLFxuICAgIERCUHJveHk6ICdkYlByb3h5TmFtZScsXG4gIH07XG4gIHByaXZhdGUgaW5jbHVkZVJlc1R5cGVzOiBzdHJpbmdbXSB8IHVuZGVmaW5lZDtcbiAgcHJpdmF0ZSBleGNsdWRlUmVzVHlwZXM6IHN0cmluZ1tdIHwgdW5kZWZpbmVkO1xuICBwcml2YXRlIGRlZmF1bHROYW1lRmllbGQgPSAnbmFtZSc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgY29uc3RydWN0b3IocHJpdmF0ZSByZW5hbWVPcGVyOiBJUmVuYW1lT3BlcmF0aW9uLCBwcm9wczogUmVuYW1lUHJvcHMgPSB7fSkge1xuICAgIGlmIChwcm9wcy5pcnJlZ3VsYXJSZXNvdXJjZU5hbWVzKSB7XG4gICAgICB0aGlzLmlycmVndWxhck5hbWVzID0ge1xuICAgICAgICAuLi50aGlzLmlycmVndWxhck5hbWVzLFxuICAgICAgICAuLi5wcm9wcy5pcnJlZ3VsYXJSZXNvdXJjZU5hbWVzLFxuICAgICAgfTtcbiAgICB9XG4gICAgdGhpcy5leGNsdWRlUmVzVHlwZXMgPSBwcm9wcy5leGNsdWRlUmVzb3VyY2VUeXBlcztcbiAgICB0aGlzLmluY2x1ZGVSZXNUeXBlcyA9IHByb3BzLmluY2x1ZGVSZXNvdXJjZVR5cGVzO1xuICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICB2aXNpdChub2RlOiBjZGsuSUNvbnN0cnVjdCk6IHZvaWQge1xuICAgIGlmIChub2RlIGluc3RhbmNlb2YgY2RrLlN0YWNrKSB7XG4gICAgICAvL3JlbmFtZSBzdGFja1xuICAgICAgdGhpcy5yZW5hbWVSZXNvdXJjZShub2RlLCAnU3RhY2snKTtcbiAgICB9IGVsc2Uge1xuICAgICAgLy9yZW5hbWUgQ0ZOIHJlc291cmNlc1xuICAgICAgbGV0IGN0b3JOYW1lID0gbm9kZS5jb25zdHJ1Y3Rvci5uYW1lO1xuICAgICAgaWYgKGN0b3JOYW1lLnN0YXJ0c1dpdGgoJ0NmbicpKSB7XG4gICAgICAgIHRoaXMucmVuYW1lUmVzb3VyY2Uobm9kZSwgY3Rvck5hbWUuc3Vic3RyaW5nKDMpKTtcbiAgICAgIH1cbiAgICB9XG4gIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHByb3RlY3RlZCByZW5hbWVSZXNvdXJjZShub2RlOiBjZGsuSUNvbnN0cnVjdCwgcmVzVHlwZU5hbWU6IHN0cmluZykge1xuICAgIC8vY2hlY2sgaW5jbHVkZS9leGNsdWRlXG4gICAgaWYgKHRoaXMuZXhjbHVkZVJlc1R5cGVzICYmIHRoaXMuZXhjbHVkZVJlc1R5cGVzLmxlbmd0aCA+IDAgJiZcbiAgICAgIHRoaXMuZXhjbHVkZVJlc1R5cGVzLmluZGV4T2YocmVzVHlwZU5hbWUpICE9PSAtMSkge1xuICAgICAgcmV0dXJuO1xuICAgIH1cbiAgICBpZiAodGhpcy5pbmNsdWRlUmVzVHlwZXMgJiYgdGhpcy5pbmNsdWRlUmVzVHlwZXMubGVuZ3RoID4gMCAmJlxuICAgICAgdGhpcy5pbmNsdWRlUmVzVHlwZXMuaW5kZXhPZihyZXNUeXBlTmFtZSkgPT09IC0xKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuICAgIC8vZmluZCB0aGUgc3BlY2lmaWMgXCJuYW1lXCIgZmllbGQgZm9yIENGTiByZXNvdXJjZXNcbiAgICBsZXQgcGh5c2ljYWxOYW1lID0gJ25hbWUnO1xuICAgIGlmICh0aGlzLmlycmVndWxhck5hbWVzW3Jlc1R5cGVOYW1lXSkge1xuICAgICAgcGh5c2ljYWxOYW1lID0gdGhpcy5pcnJlZ3VsYXJOYW1lc1tyZXNUeXBlTmFtZV07XG4gICAgfSBlbHNlIHtcbiAgICAgIC8vZGVjYXBpdGFsaXplIHJlZ3VsYXIgcmVzb3VyY2UgbmFtZXNcbiAgICAgIGxldCBbZmlyc3QsIC4uLnJlc3RdID0gcmVzVHlwZU5hbWU7XG4gICAgICBsZXQgZGVjYXBOYW1lID0gZmlyc3QudG9Mb3dlckNhc2UoKSArIHJlc3Quam9pbignJyk7XG4gICAgICBwaHlzaWNhbE5hbWUgPSBgJHtkZWNhcE5hbWV9TmFtZWA7XG4gICAgfVxuICAgIGlmIChwaHlzaWNhbE5hbWUubGVuZ3RoID09PSAwKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuICAgIC8vc29tZSBwcm90ZWN0ZWQgZmllbGRzIHN0YXJ0IHdpdGggdW5kZXJzY29yZVxuICAgIGxldCB1bmRlcnNjb3JlTmFtZSA9ICdfJyArIHBoeXNpY2FsTmFtZTtcbiAgICAvL3JlbmFtZVxuICAgIGxldCBiID0gKG5vZGUgYXMgYW55KTtcbiAgICBpZiAoYltwaHlzaWNhbE5hbWVdICYmIGJbcGh5c2ljYWxOYW1lXS5sZW5ndGggPiAwICYmICFjZGsuVG9rZW4uaXNVbnJlc29sdmVkKGJbcGh5c2ljYWxOYW1lXSkpIHtcbiAgICAgIGJbcGh5c2ljYWxOYW1lXSA9IHRoaXMucmVuYW1lT3Blci5yZW5hbWUoYltwaHlzaWNhbE5hbWVdLCByZXNUeXBlTmFtZSk7XG4gICAgfSBlbHNlIGlmIChiW3VuZGVyc2NvcmVOYW1lXSAmJiBiW3VuZGVyc2NvcmVOYW1lXS5sZW5ndGggPiAwICYmICFjZGsuVG9rZW4uaXNVbnJlc29sdmVkKGJbdW5kZXJzY29yZU5hbWVdKSkge1xuICAgICAgYlt1bmRlcnNjb3JlTmFtZV0gPSB0aGlzLnJlbmFtZU9wZXIucmVuYW1lKGJbdW5kZXJzY29yZU5hbWVdLCByZXNUeXBlTmFtZSk7XG4gICAgfSBlbHNlIGlmIChiW3RoaXMuZGVmYXVsdE5hbWVGaWVsZF0gJiYgYlt0aGlzLmRlZmF1bHROYW1lRmllbGRdLmxlbmd0aCA+IDAgJiYgIWNkay5Ub2tlbi5pc1VucmVzb2x2ZWQoYlt0aGlzLmRlZmF1bHROYW1lRmllbGRdKSkge1xuICAgICAgYlt0aGlzLmRlZmF1bHROYW1lRmllbGRdID0gdGhpcy5yZW5hbWVPcGVyLnJlbmFtZShiW3RoaXMuZGVmYXVsdE5hbWVGaWVsZF0sIHJlc1R5cGVOYW1lKTtcbiAgICB9XG4gIH1cbn1cbiJdfQ==