import os
from pathlib import Path
import pkg_resources
import shutil
import click
import dotenv 
import git
import requests

from d2s.generate_metadata import create_dataset_prompt

def get_git_path(path="."):
    """Return path of the root folder of the git repo the user is in"""
    # TODO: Does not work for some git repository, without no reason
    # Just find the .git folder and handle this shit ffs, no weird stuff to do here 
    git_repo = git.Repo(path, search_parent_directories=True)
    git_root = git_repo.working_tree_dir
    return git_root

def get_base_dir(file=''):
    """XDG base dir for d2s executables and jar in ~/.local/share/d2s"""
    return str(Path.home()) + '/.local/share/d2s/' + file

def init_d2s_java(init_file=''):
    """Check and download jar files"""
    os.makedirs(get_base_dir(), exist_ok=True)
    if init_file == 'sparql-operations' and not os.path.isfile(get_base_dir('sparql-operations.jar')):
        r = requests.get('https://github.com/MaastrichtU-IDS/d2s-sparql-operations/releases/latest/download/sparql-operations.jar')
        with open(get_base_dir('sparql-operations.jar'), 'wb') as f:
            f.write(r.content)

    if init_file == 'rmlmapper' and not os.path.isfile(get_base_dir('rmlmapper.jar')):
        # Download rmlmapper 4.9.1
        r = requests.get('https://github.com/RMLio/rmlmapper-java/releases/download/v4.9.1/rmlmapper-4.9.1.jar')
        with open(get_base_dir('rmlmapper.jar'), 'wb') as f:
            f.write(r.content)

# Init project and config
def init_folder():
    """Initialize a project in the current folder"""
    d2s_repository_url = click.prompt(click.style('[?]', bold=True) + ' Enter the Git repository URL for this projects', default="")

    # Clone git repo if URL provided
    if os.path.exists('.git'):
        click.echo(click.style('[d2s]', bold=True) + ' Git repository already initialized.')
    elif d2s_repository_url:
        click.echo(click.style('[d2s] ', bold=True) + 'Cloning the repository in the current folder...')
        git_repo = git.Repo.clone_from(d2s_repository_url, '.')
        git_repo.git.submodule('update', '--init')
        click.echo(click.style('[d2s]', bold=True) + ' Git repository cloned.')
    else:
        click.echo(click.style('[d2s] ', bold=True) + 'Initializing a bare git repository')
        git.Repo.init('.git', bare=True)

    # Create the .env file
    f = open(".env", "w").close
    if d2s_repository_url:
        dotenv.set_key('.env', 'GIT_URL', d2s_repository_url, quote_mode="noquote")
    dotenv.set_key('.env', 'VIRTUOSO_PASSWORD', 'dba', quote_mode="noquote")
    # dotenv.set_key('.env', "PATH", os.getcwd(), quote_mode="noquote")
    
    # Create required files if missing (readme, requirements...)
    for dname, dirs, files in os.walk(pkg_resources.resource_filename('d2s', 'templates/project')):
        for filename in files:
            filepath = os.path.join(dname, filename)
            if not os.path.exists(filename):
                shutil.copyfile(filepath, os.getcwd() + '/' + filename)

    # Create datasets and workflows empty folders
    folders_to_create = ['datasets', '.github/workflows']
    click.echo(click.style('[d2s]', bold=True) + ' Creating following folders in workspace: ' + ", ".join(folders_to_create))
    for file_to_create in folders_to_create:
        os.makedirs(file_to_create, exist_ok=True)

    click.echo()
    click.echo(click.style('[d2s]', bold=True) + ' Your d2s project has been created!')
    click.echo(click.style('[d2s]', bold=True) + ' Remember to commit the new files with git')

    ## Automatically commit files generated by d2s init?
    # os.system('git add .')
    # os.system('git commit -m "Initialize mapping project with d2s"')


def new_dataset():
    """Create a folder to map a new dataset"""
    # Go to the root of the git repo
    os.chdir(get_git_path())
    # Make sure datasets and .github/workflows folder have been created
    os.makedirs('datasets', exist_ok=True)
    os.makedirs('.github/workflows', exist_ok=True)

    # Ask dataset metadata to the user and generate rdflib graph
    g, dataset_metadata = create_dataset_prompt()

    dataset_id = dataset_metadata['dataset_id']
    dataset_folder_path = 'datasets/' + dataset_id

    # Copy template folder with example mappings
    shutil.copytree(pkg_resources.resource_filename('d2s', 'templates/dataset'), dataset_folder_path)
    os.rename(dataset_folder_path + '/process-dataset.ipynb', dataset_folder_path + '/process-' + dataset_id + '.ipynb')
    # for filename in os.listdir(pkg_resources.resource_filename('d2s', 'templates/dataset')):
    #     with open(pkg_resources.resource_filename('d2s', 'queries/' + filename), 'r') as f:

    # Store metadata file
    os.makedirs(dataset_folder_path + '/metadata', exist_ok=True)
    g.serialize(destination=dataset_folder_path + '/metadata/' + dataset_id + '-metadata.ttl', format='turtle')
    print("Metadata stored to " + dataset_folder_path + '/metadata/' + dataset_id + '-metadata.ttl' + ' 📝')

    # Replace metadata in all files from template for the new dataset (mainly for the dataset_id)
    for dname, dirs, files in os.walk(dataset_folder_path):
        for fname in files:
            fpath = os.path.join(dname, fname)
            with open(fpath) as f:
                file_content = f.read()
            for metadata_id, metadata_value in dataset_metadata.items():
                file_content = file_content.replace("$" + metadata_id, metadata_value)
            with open(fpath, "w") as f:
                f.write(file_content)

    # Copy example GitHub Actions workflow file, and replace dataset_id in it
    workflow_filepath = '.github/workflows/process-' + dataset_id + '.yml'
    shutil.copyfile(pkg_resources.resource_filename('d2s', 'templates/process-dataset.yml'), workflow_filepath)
    with open(workflow_filepath) as f:
        file_content = f.read()
        file_content = file_content.replace("$dataset_id", dataset_id)
    with open(workflow_filepath, "w") as f:
        f.write(file_content)

    click.echo()
    click.echo(click.style('[d2s]', bold=True) + ' Metadata, example mapping files and scripts for the ' 
        + click.style(dataset_id + ' dataset', bold=True) 
        + ' has been generated')
    click.echo(click.style('[d2s]', bold=True) + ' 📝 Start edit them in ' + click.style('datasets/' + dataset_id, bold=True))
    click.echo(click.style('[d2s]', bold=True) + ' 🐈 GitHub Actions workflow file in ' + click.style(workflow_filepath, bold=True))


def get_config():
    """Show the project configuration"""
    click.echo(click.style('[d2s]', bold=True) + ' Configuration is stored in the ' 
        + click.style('.env', bold=True) + ' file:')
    click.echo()
    with open('.env') as f:
        click.echo(f.read())


## Configure GraphDB triplestore docker deployment
# user_home_dir = str(Path.home())
# click.echo()
# shutil.copy(pkg_resources.resource_filename('d2s', 'templates/graphdb-repo-config.ttl'), 'workspace/graphdb/preload-config.ttl')
# # Copy GraphDB zip file to the right folder in d2s-core
# click.echo(click.style('[d2s]', bold=True) + ' The GraphDB triplestore needs to be downloaded for licensing reason.\n'
#     + 'Go to ' 
#     + click.style('https://ontotext.com/products/graphdb/', bold=True) 
#     + ' and provide your email to receive the URL to download the ' + click.style('GraphDB latest version standalone zip', bold=True))
# graphdb_version = click.prompt(click.style('[?]', bold=True) + ' Enter the version of the GraphDB triplestore used. Default', default='9.3.0')
# dotenv.set_key('.env', 'GRAPHDB_VERSION', graphdb_version, quote_mode="noquote")
# graphdb_heap_size = click.prompt(click.style('[?]', bold=True) + ' Enter the Java heap size allocated to GraphDB. Default', default='2G')
# dotenv.set_key('.env', 'GRAPHDB_HEAP_SIZE', graphdb_heap_size, quote_mode="noquote")
# graphdb_path = click.prompt(click.style('[?]', bold=True) + ' Enter the path to the GraphDB ' + graphdb_version + ' zip file that will be used to build the Docker image. Default', default=user_home_dir + '/graphdb-free-' + graphdb_version + '-dist.zip')
# # Get GraphDB installation file
# if os.path.exists(graphdb_path):
#     shutil.copy(graphdb_path, 'd2s-core/support/graphdb')
#     click.echo(click.style('[d2s]', bold=True) + ' GraphDB installation file copied imported successfully!')
# else:
#     click.echo(click.style('[d2s]', bold=True) + ' GraphDB installation file not found. Copy the zip file in d2s-core/support/graphdb after download.')
