from django import forms
from django.db.models.fields.related import ManyToOneRel
from django.utils.translation import gettext_lazy as _
from djangocms_text_ckeditor.fields import HTMLFormField
from entangled.forms import EntangledModelForm
from filer.fields.image import AdminImageFormField, FilerImageField
from filer.models import Image

from djangocms_frontend.fields import AttributesFormField, TagTypeFormField

from ... import settings
from ...models import FrontendUIItem
from ..link.forms import AbstractLinkForm
from .constants import (
    CAROUSEL_ASPECT_RATIO_CHOICES,
    CAROUSEL_PAUSE_CHOICES,
    CAROUSEL_RIDE_CHOICES,
    CAROUSEL_TEMPLATE_CHOICES,
)


class CarouselForm(EntangledModelForm):
    """
    Components > "Carousel" Plugin
    https://getbootstrap.com/docs/5.0/components/carousel/
    """

    class Meta:
        model = FrontendUIItem
        entangled_fields = {
            "config": [
                "template",
                "carousel_aspect_ratio",
                "carousel_controls",
                "carousel_indicators",
                "carousel_interval",
                "carousel_keyboard",
                "carousel_pause",
                "carousel_ride",
                "carousel_wrap",
                "attributes",
            ]
        }
        untangled_fields = ("tag_type",)

    template = forms.ChoiceField(
        label=_("Template"),
        choices=CAROUSEL_TEMPLATE_CHOICES,
        initial=CAROUSEL_TEMPLATE_CHOICES[0][0],
        help_text=_("This is the template that will be used for the component."),
        widget=forms.HiddenInput
        if len(CAROUSEL_TEMPLATE_CHOICES) < 2
        else forms.Select,
    )
    carousel_interval = forms.IntegerField(
        label=_("Interval"),
        initial=5000,
        help_text=_(
            "The amount of time to delay between automatically cycling "
            "an item. If false, carousel will not automatically cycle."
        ),
    )
    carousel_controls = forms.BooleanField(
        label=_("Controls"),
        initial=True,
        required=False,
        help_text=_("Adding in the previous and next controls."),
    )
    carousel_indicators = forms.BooleanField(
        label=_("Indicators"),
        initial=True,
        required=False,
        help_text=_("Adding in the indicators to the carousel."),
    )
    carousel_keyboard = forms.BooleanField(
        label=_("Keyboard"),
        initial=True,
        required=False,
        help_text=_("Whether the carousel should react to keyboard events."),
    )
    carousel_pause = forms.ChoiceField(
        label=_("Pause"),
        choices=CAROUSEL_PAUSE_CHOICES,
        initial=CAROUSEL_PAUSE_CHOICES[0][0],
        help_text=_(
            'If set to "hover", pauses the cycling of the carousel on '
            '"mouseenter" and resumes the cycling of the carousel on '
            '"mouseleave". If set to "false", hovering over the carousel '
            "won't pause it."
        ),
    )
    carousel_ride = forms.ChoiceField(
        label=_("Ride"),
        choices=CAROUSEL_RIDE_CHOICES,
        initial=CAROUSEL_RIDE_CHOICES[0][0],
        help_text=_(
            "Autoplays the carousel after the user manually cycles the "
            'first item. If "carousel", autoplays the carousel on load.'
        ),
    )
    carousel_wrap = forms.BooleanField(
        label=_("Wrap"),
        initial=True,
        required=False,
        help_text=_(
            "Whether the carousel should cycle continuously or have " "hard stops."
        ),
    )
    carousel_aspect_ratio = forms.ChoiceField(
        label=_("Aspect ratio"),
        choices=settings.EMPTY_CHOICE + CAROUSEL_ASPECT_RATIO_CHOICES,
        required=False,
        initial=settings.EMPTY_CHOICE[0][0],
        help_text=_(
            "Determines width and height of the image "
            "according to the selected ratio."
        ),
    )
    attributes = AttributesFormField(
        excluded_keys=[
            "id",
            "data-bs-interval",
            "data-bs-keyboard",
            "data-bs-pause",
            "data-bs-ride",
            "data-bs-wrap",
        ],
    )
    tag_type = TagTypeFormField()


class CarouselSlideForm(AbstractLinkForm, EntangledModelForm):
    """
    Components > "Slide" Plugin
    https://getbootstrap.com/docs/5.0/components/carousel/
    """

    class Meta:
        model = FrontendUIItem
        entangled_fields = {
            "config": [
                "carousel_image",
                "carousel_content",
                "attributes",
            ]
        }
        untangled_fields = ("tag_type",)

    link_is_optional = True

    carousel_image = AdminImageFormField(
        rel=ManyToOneRel(FilerImageField, Image, "id"),
        queryset=Image.objects.all(),
        to_field_name="id",
        label=_("Slide image"),
    )
    carousel_content = HTMLFormField(
        label=_("Content"),
        required=False,
        initial="",
        help_text=_("Content may also be added using child plugins."),
    )
    attributes = AttributesFormField()
    tag_type = TagTypeFormField()
