"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../../lib");
const cache_behavior_1 = require("../../lib/private/cache-behavior");
let app;
let stack;
beforeEach(() => {
    app = new core_1.App();
    stack = new core_1.Stack(app, 'Stack', {
        env: { account: '1234', region: 'testregion' },
    });
});
test('renders the minimum template with an origin and path specified', () => {
    const behavior = new cache_behavior_1.CacheBehavior('origin_id', {
        pathPattern: '*',
    });
    expect(behavior._renderBehavior()).toEqual({
        targetOriginId: 'origin_id',
        pathPattern: '*',
        forwardedValues: { queryString: false },
        viewerProtocolPolicy: 'allow-all',
    });
});
test('renders with all properties specified', () => {
    const fnVersion = lambda.Version.fromVersionArn(stack, 'Version', 'arn:aws:lambda:testregion:111111111111:function:myTestFun:v1');
    const behavior = new cache_behavior_1.CacheBehavior('origin_id', {
        pathPattern: '*',
        allowedMethods: lib_1.AllowedMethods.ALLOW_ALL,
        cachedMethods: lib_1.CachedMethods.CACHE_GET_HEAD_OPTIONS,
        compress: true,
        forwardQueryString: true,
        forwardQueryStringCacheKeys: ['user_id', 'auth'],
        smoothStreaming: true,
        viewerProtocolPolicy: lib_1.ViewerProtocolPolicy.HTTPS_ONLY,
        edgeLambdas: [{
                eventType: lib_1.LambdaEdgeEventType.ORIGIN_REQUEST,
                includeBody: true,
                functionVersion: fnVersion,
            }],
    });
    expect(behavior._renderBehavior()).toEqual({
        targetOriginId: 'origin_id',
        pathPattern: '*',
        allowedMethods: ['GET', 'HEAD', 'OPTIONS', 'PUT', 'PATCH', 'POST', 'DELETE'],
        cachedMethods: ['GET', 'HEAD', 'OPTIONS'],
        compress: true,
        forwardedValues: {
            queryString: true,
            queryStringCacheKeys: ['user_id', 'auth'],
        },
        smoothStreaming: true,
        viewerProtocolPolicy: 'https-only',
        lambdaFunctionAssociations: [{
                lambdaFunctionArn: 'arn:aws:lambda:testregion:111111111111:function:myTestFun:v1',
                eventType: 'origin-request',
                includeBody: true,
            }],
    });
});
test('throws if edgeLambda includeBody is set for wrong event type', () => {
    const fnVersion = lambda.Version.fromVersionArn(stack, 'Version', 'arn:aws:lambda:testregion:111111111111:function:myTestFun:v1');
    expect(() => new cache_behavior_1.CacheBehavior('origin_id', {
        pathPattern: '*',
        edgeLambdas: [{
                eventType: lib_1.LambdaEdgeEventType.ORIGIN_RESPONSE,
                includeBody: true,
                functionVersion: fnVersion,
            }],
    })).toThrow(/'includeBody' can only be true for ORIGIN_REQUEST or VIEWER_REQUEST event types./);
});
//# sourceMappingURL=data:application/json;base64,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