"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontWebDistribution = exports.ViewerCertificate = exports.CloudFrontAllowedCachedMethods = exports.CloudFrontAllowedMethods = exports.OriginSslPolicy = exports.FailoverStatusCode = void 0;
const certificatemanager = require("@aws-cdk/aws-certificatemanager");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const cloudfront_generated_1 = require("./cloudfront.generated");
const distribution_1 = require("./distribution");
/**
 * HTTP status code to failover to second origin
 */
var FailoverStatusCode;
(function (FailoverStatusCode) {
    /**
     * Forbidden (403)
     */
    FailoverStatusCode[FailoverStatusCode["FORBIDDEN"] = 403] = "FORBIDDEN";
    /**
     * Not found (404)
     */
    FailoverStatusCode[FailoverStatusCode["NOT_FOUND"] = 404] = "NOT_FOUND";
    /**
     * Internal Server Error (500)
     */
    FailoverStatusCode[FailoverStatusCode["INTERNAL_SERVER_ERROR"] = 500] = "INTERNAL_SERVER_ERROR";
    /**
     * Bad Gateway (502)
     */
    FailoverStatusCode[FailoverStatusCode["BAD_GATEWAY"] = 502] = "BAD_GATEWAY";
    /**
     * Service Unavailable (503)
     */
    FailoverStatusCode[FailoverStatusCode["SERVICE_UNAVAILABLE"] = 503] = "SERVICE_UNAVAILABLE";
    /**
     * Gateway Timeout (504)
     */
    FailoverStatusCode[FailoverStatusCode["GATEWAY_TIMEOUT"] = 504] = "GATEWAY_TIMEOUT";
})(FailoverStatusCode = exports.FailoverStatusCode || (exports.FailoverStatusCode = {}));
var OriginSslPolicy;
(function (OriginSslPolicy) {
    OriginSslPolicy["SSL_V3"] = "SSLv3";
    OriginSslPolicy["TLS_V1"] = "TLSv1";
    OriginSslPolicy["TLS_V1_1"] = "TLSv1.1";
    OriginSslPolicy["TLS_V1_2"] = "TLSv1.2";
})(OriginSslPolicy = exports.OriginSslPolicy || (exports.OriginSslPolicy = {}));
/**
 * An enum for the supported methods to a CloudFront distribution.
 */
var CloudFrontAllowedMethods;
(function (CloudFrontAllowedMethods) {
    CloudFrontAllowedMethods["GET_HEAD"] = "GH";
    CloudFrontAllowedMethods["GET_HEAD_OPTIONS"] = "GHO";
    CloudFrontAllowedMethods["ALL"] = "ALL";
})(CloudFrontAllowedMethods = exports.CloudFrontAllowedMethods || (exports.CloudFrontAllowedMethods = {}));
/**
 * Enums for the methods CloudFront can cache.
 */
var CloudFrontAllowedCachedMethods;
(function (CloudFrontAllowedCachedMethods) {
    CloudFrontAllowedCachedMethods["GET_HEAD"] = "GH";
    CloudFrontAllowedCachedMethods["GET_HEAD_OPTIONS"] = "GHO";
})(CloudFrontAllowedCachedMethods = exports.CloudFrontAllowedCachedMethods || (exports.CloudFrontAllowedCachedMethods = {}));
/**
 * Viewer certificate configuration class
 */
class ViewerCertificate {
    constructor(props, aliases = []) {
        this.props = props;
        this.aliases = aliases;
    }
    /**
     * Generate an AWS Certificate Manager (ACM) viewer certificate configuration
     *
     * @param certificate AWS Certificate Manager (ACM) certificate.
     *                    Your certificate must be located in the us-east-1 (US East (N. Virginia)) region to be accessed by CloudFront
     * @param options certificate configuration options
     */
    static fromAcmCertificate(certificate, options = {}) {
        const { sslMethod: sslSupportMethod = distribution_1.SSLMethod.SNI, securityPolicy: minimumProtocolVersion, aliases, } = options;
        return new ViewerCertificate({
            acmCertificateArn: certificate.certificateArn, sslSupportMethod, minimumProtocolVersion,
        }, aliases);
    }
    /**
     * Generate an IAM viewer certificate configuration
     *
     * @param iamCertificateId Identifier of the IAM certificate
     * @param options certificate configuration options
     */
    static fromIamCertificate(iamCertificateId, options = {}) {
        const { sslMethod: sslSupportMethod = distribution_1.SSLMethod.SNI, securityPolicy: minimumProtocolVersion, aliases, } = options;
        return new ViewerCertificate({
            iamCertificateId, sslSupportMethod, minimumProtocolVersion,
        }, aliases);
    }
    /**
     * Generate a viewer certifcate configuration using
     * the CloudFront default certificate (e.g. d111111abcdef8.cloudfront.net)
     * and a {@link SecurityPolicyProtocol.TLS_V1} security policy.
     *
     * @param aliases Alternative CNAME aliases
     *                You also must create a CNAME record with your DNS service to route queries
     */
    static fromCloudFrontDefaultCertificate(...aliases) {
        return new ViewerCertificate({ cloudFrontDefaultCertificate: true }, aliases);
    }
}
exports.ViewerCertificate = ViewerCertificate;
/**
 * Amazon CloudFront is a global content delivery network (CDN) service that securely delivers data, videos,
 * applications, and APIs to your viewers with low latency and high transfer speeds.
 * CloudFront fronts user provided content and caches it at edge locations across the world.
 *
 * Here's how you can use this construct:
 *
 * ```ts
 * import { CloudFrontWebDistribution } from '@aws-cdk/aws-cloudfront'
 *
 * const sourceBucket = new Bucket(this, 'Bucket');
 *
 * const distribution = new CloudFrontWebDistribution(this, 'MyDistribution', {
 *  originConfigs: [
 *    {
 *      s3OriginSource: {
 *      s3BucketSource: sourceBucket
 *      },
 *      behaviors : [ {isDefaultBehavior: true}]
 *    }
 *  ]
 * });
 * ```
 *
 * This will create a CloudFront distribution that uses your S3Bucket as it's origin.
 *
 * You can customize the distribution using additional properties from the CloudFrontWebDistributionProps interface.
 *
 * @resource AWS::CloudFront::Distribution
 */
class CloudFrontWebDistribution extends cdk.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        /**
         * Maps our methods to the string arrays they are
         */
        this.METHOD_LOOKUP_MAP = {
            GH: ['GET', 'HEAD'],
            GHO: ['GET', 'HEAD', 'OPTIONS'],
            ALL: ['DELETE', 'GET', 'HEAD', 'OPTIONS', 'PATCH', 'POST', 'PUT'],
        };
        /**
         * Maps for which SecurityPolicyProtocol are available to which SSLMethods
         */
        this.VALID_SSL_PROTOCOLS = {
            [distribution_1.SSLMethod.SNI]: [
                distribution_1.SecurityPolicyProtocol.TLS_V1, distribution_1.SecurityPolicyProtocol.TLS_V1_1_2016,
                distribution_1.SecurityPolicyProtocol.TLS_V1_2016, distribution_1.SecurityPolicyProtocol.TLS_V1_2_2018,
                distribution_1.SecurityPolicyProtocol.TLS_V1_2_2019,
            ],
            [distribution_1.SSLMethod.VIP]: [distribution_1.SecurityPolicyProtocol.SSL_V3, distribution_1.SecurityPolicyProtocol.TLS_V1],
        };
        let distributionConfig = {
            comment: props.comment,
            enabled: true,
            defaultRootObject: props.defaultRootObject !== undefined ? props.defaultRootObject : 'index.html',
            httpVersion: props.httpVersion || distribution_1.HttpVersion.HTTP2,
            priceClass: props.priceClass || distribution_1.PriceClass.PRICE_CLASS_100,
            ipv6Enabled: (props.enableIpV6 !== undefined) ? props.enableIpV6 : true,
            // eslint-disable-next-line max-len
            customErrorResponses: props.errorConfigurations,
            webAclId: props.webACLId,
        };
        const behaviors = [];
        const origins = [];
        const originGroups = [];
        let originIndex = 1;
        for (const originConfig of props.originConfigs) {
            let originId = `origin${originIndex}`;
            const originProperty = this.toOriginProperty(originConfig, originId);
            if (originConfig.failoverCustomOriginSource || originConfig.failoverS3OriginSource) {
                const originSecondaryId = `originSecondary${originIndex}`;
                const originSecondaryProperty = this.toOriginProperty({
                    s3OriginSource: originConfig.failoverS3OriginSource,
                    customOriginSource: originConfig.failoverCustomOriginSource,
                    originPath: originConfig.originPath,
                    originHeaders: originConfig.originHeaders,
                }, originSecondaryId);
                const originGroupsId = `OriginGroup${originIndex}`;
                const failoverCodes = (_a = originConfig.failoverCriteriaStatusCodes) !== null && _a !== void 0 ? _a : [500, 502, 503, 504];
                originGroups.push({
                    id: originGroupsId,
                    members: {
                        items: [{ originId }, { originId: originSecondaryId }],
                        quantity: 2,
                    },
                    failoverCriteria: {
                        statusCodes: {
                            items: failoverCodes,
                            quantity: failoverCodes.length,
                        },
                    },
                });
                originId = originGroupsId;
                origins.push(originSecondaryProperty);
            }
            for (const behavior of originConfig.behaviors) {
                behaviors.push({ ...behavior, targetOriginId: originId });
            }
            origins.push(originProperty);
            originIndex++;
        }
        origins.forEach(origin => {
            if (!origin.s3OriginConfig && !origin.customOriginConfig) {
                throw new Error(`Origin ${origin.domainName} is missing either S3OriginConfig or CustomOriginConfig. At least 1 must be specified.`);
            }
        });
        const originGroupsDistConfig = originGroups.length > 0
            ? {
                items: originGroups,
                quantity: originGroups.length,
            }
            : undefined;
        distributionConfig = {
            ...distributionConfig,
            origins,
            originGroups: originGroupsDistConfig,
        };
        const defaultBehaviors = behaviors.filter(behavior => behavior.isDefaultBehavior);
        if (defaultBehaviors.length !== 1) {
            throw new Error('There can only be one default behavior across all sources. [ One default behavior per distribution ].');
        }
        distributionConfig = { ...distributionConfig, defaultCacheBehavior: this.toBehavior(defaultBehaviors[0], props.viewerProtocolPolicy) };
        const otherBehaviors = [];
        for (const behavior of behaviors.filter(b => !b.isDefaultBehavior)) {
            if (!behavior.pathPattern) {
                throw new Error('pathPattern is required for all non-default behaviors');
            }
            otherBehaviors.push(this.toBehavior(behavior, props.viewerProtocolPolicy));
        }
        distributionConfig = { ...distributionConfig, cacheBehaviors: otherBehaviors.length > 0 ? otherBehaviors : undefined };
        if (props.aliasConfiguration && props.viewerCertificate) {
            throw new Error([
                'You cannot set both aliasConfiguration and viewerCertificate properties.',
                'Please only use viewerCertificate, as aliasConfiguration is deprecated.',
            ].join(' '));
        }
        let _viewerCertificate = props.viewerCertificate;
        if (props.aliasConfiguration) {
            const { acmCertRef, securityPolicy, sslMethod, names: aliases } = props.aliasConfiguration;
            _viewerCertificate = ViewerCertificate.fromAcmCertificate(certificatemanager.Certificate.fromCertificateArn(this, 'AliasConfigurationCert', acmCertRef), { securityPolicy, sslMethod, aliases });
        }
        if (_viewerCertificate) {
            const { props: viewerCertificate, aliases } = _viewerCertificate;
            Object.assign(distributionConfig, { aliases, viewerCertificate });
            const { minimumProtocolVersion, sslSupportMethod } = viewerCertificate;
            if (minimumProtocolVersion != null && sslSupportMethod != null) {
                const validProtocols = this.VALID_SSL_PROTOCOLS[sslSupportMethod];
                if (validProtocols.indexOf(minimumProtocolVersion.toString()) === -1) {
                    // eslint-disable-next-line max-len
                    throw new Error(`${minimumProtocolVersion} is not compabtible with sslMethod ${sslSupportMethod}.\n\tValid Protocols are: ${validProtocols.join(', ')}`);
                }
            }
        }
        else {
            distributionConfig = {
                ...distributionConfig,
                viewerCertificate: { cloudFrontDefaultCertificate: true },
            };
        }
        if (props.loggingConfig) {
            this.loggingBucket = props.loggingConfig.bucket || new s3.Bucket(this, 'LoggingBucket');
            distributionConfig = {
                ...distributionConfig,
                logging: {
                    bucket: this.loggingBucket.bucketDomainName,
                    includeCookies: props.loggingConfig.includeCookies || false,
                    prefix: props.loggingConfig.prefix,
                },
            };
        }
        if (props.geoRestriction) {
            distributionConfig = {
                ...distributionConfig,
                restrictions: {
                    geoRestriction: {
                        restrictionType: props.geoRestriction.restrictionType,
                        locations: props.geoRestriction.locations,
                    },
                },
            };
        }
        const distribution = new cloudfront_generated_1.CfnDistribution(this, 'CFDistribution', { distributionConfig });
        this.node.defaultChild = distribution;
        this.domainName = distribution.attrDomainName;
        this.distributionDomainName = distribution.attrDomainName;
        this.distributionId = distribution.ref;
    }
    /**
     * Creates a construct that represents an external (imported) distribution.
     */
    static fromDistributionAttributes(scope, id, attrs) {
        return new class extends cdk.Resource {
            constructor() {
                super(scope, id);
                this.domainName = attrs.domainName;
                this.distributionDomainName = attrs.domainName;
                this.distributionId = attrs.distributionId;
            }
        }();
    }
    toBehavior(input, protoPolicy) {
        let toReturn = {
            allowedMethods: this.METHOD_LOOKUP_MAP[input.allowedMethods || CloudFrontAllowedMethods.GET_HEAD],
            cachedMethods: this.METHOD_LOOKUP_MAP[input.cachedMethods || CloudFrontAllowedCachedMethods.GET_HEAD],
            compress: input.compress !== false,
            defaultTtl: input.defaultTtl && input.defaultTtl.toSeconds(),
            forwardedValues: input.forwardedValues || { queryString: false, cookies: { forward: 'none' } },
            maxTtl: input.maxTtl && input.maxTtl.toSeconds(),
            minTtl: input.minTtl && input.minTtl.toSeconds(),
            trustedSigners: input.trustedSigners,
            targetOriginId: input.targetOriginId,
            viewerProtocolPolicy: protoPolicy || distribution_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        };
        if (!input.isDefaultBehavior) {
            toReturn = Object.assign(toReturn, { pathPattern: input.pathPattern });
        }
        if (input.lambdaFunctionAssociations) {
            const includeBodyEventTypes = [distribution_1.LambdaEdgeEventType.ORIGIN_REQUEST, distribution_1.LambdaEdgeEventType.VIEWER_REQUEST];
            if (input.lambdaFunctionAssociations.some(fna => fna.includeBody && !includeBodyEventTypes.includes(fna.eventType))) {
                throw new Error('\'includeBody\' can only be true for ORIGIN_REQUEST or VIEWER_REQUEST event types.');
            }
            toReturn = Object.assign(toReturn, {
                lambdaFunctionAssociations: input.lambdaFunctionAssociations
                    .map(fna => ({
                    eventType: fna.eventType,
                    lambdaFunctionArn: fna.lambdaFunction && fna.lambdaFunction.edgeArn,
                    includeBody: fna.includeBody,
                })),
            });
            // allow edgelambda.amazonaws.com to assume the functions' execution role.
            for (const a of input.lambdaFunctionAssociations) {
                if (a.lambdaFunction.role && a.lambdaFunction.role instanceof iam.Role && a.lambdaFunction.role.assumeRolePolicy) {
                    a.lambdaFunction.role.assumeRolePolicy.addStatements(new iam.PolicyStatement({
                        actions: ['sts:AssumeRole'],
                        principals: [new iam.ServicePrincipal('edgelambda.amazonaws.com')],
                    }));
                }
            }
        }
        return toReturn;
    }
    toOriginProperty(originConfig, originId) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o;
        if (!originConfig.s3OriginSource &&
            !originConfig.customOriginSource) {
            throw new Error('There must be at least one origin source - either an s3OriginSource, a customOriginSource');
        }
        if (originConfig.customOriginSource && originConfig.s3OriginSource) {
            throw new Error('There cannot be both an s3OriginSource and a customOriginSource in the same SourceConfiguration.');
        }
        if ([
            originConfig.originHeaders,
            (_a = originConfig.s3OriginSource) === null || _a === void 0 ? void 0 : _a.originHeaders,
            (_b = originConfig.customOriginSource) === null || _b === void 0 ? void 0 : _b.originHeaders,
        ].filter(x => x).length > 1) {
            throw new Error('Only one originHeaders field allowed across origin and failover origins');
        }
        if ([
            originConfig.originPath,
            (_c = originConfig.s3OriginSource) === null || _c === void 0 ? void 0 : _c.originPath,
            (_d = originConfig.customOriginSource) === null || _d === void 0 ? void 0 : _d.originPath,
        ].filter(x => x).length > 1) {
            throw new Error('Only one originPath field allowed across origin and failover origins');
        }
        const headers = (_g = (_e = originConfig.originHeaders) !== null && _e !== void 0 ? _e : (_f = originConfig.s3OriginSource) === null || _f === void 0 ? void 0 : _f.originHeaders) !== null && _g !== void 0 ? _g : (_h = originConfig.customOriginSource) === null || _h === void 0 ? void 0 : _h.originHeaders;
        const originHeaders = [];
        if (headers) {
            Object.keys(headers).forEach((key) => {
                const oHeader = {
                    headerName: key,
                    headerValue: headers[key],
                };
                originHeaders.push(oHeader);
            });
        }
        let s3OriginConfig;
        if (originConfig.s3OriginSource) {
            // first case for backwards compatibility
            if (originConfig.s3OriginSource.originAccessIdentity) {
                // grant CloudFront OriginAccessIdentity read access to S3 bucket
                originConfig.s3OriginSource.s3BucketSource.grantRead(originConfig.s3OriginSource.originAccessIdentity);
                s3OriginConfig = {
                    originAccessIdentity: `origin-access-identity/cloudfront/${originConfig.s3OriginSource.originAccessIdentity.originAccessIdentityName}`,
                };
            }
            else {
                s3OriginConfig = {};
            }
        }
        const connectionAttempts = (_j = originConfig.connectionAttempts) !== null && _j !== void 0 ? _j : 3;
        if (connectionAttempts < 1 || 3 < connectionAttempts || !Number.isInteger(connectionAttempts)) {
            throw new Error('connectionAttempts: You can specify 1, 2, or 3 as the number of attempts.');
        }
        const connectionTimeout = (originConfig.connectionTimeout || cdk.Duration.seconds(10)).toSeconds();
        if (connectionTimeout < 1 || 10 < connectionTimeout || !Number.isInteger(connectionTimeout)) {
            throw new Error('connectionTimeout: You can specify a number of seconds between 1 and 10 (inclusive).');
        }
        const originProperty = {
            id: originId,
            domainName: originConfig.s3OriginSource
                ? originConfig.s3OriginSource.s3BucketSource.bucketRegionalDomainName
                : originConfig.customOriginSource.domainName,
            originPath: (_m = (_k = originConfig.originPath) !== null && _k !== void 0 ? _k : (_l = originConfig.customOriginSource) === null || _l === void 0 ? void 0 : _l.originPath) !== null && _m !== void 0 ? _m : (_o = originConfig.s3OriginSource) === null || _o === void 0 ? void 0 : _o.originPath,
            originCustomHeaders: originHeaders.length > 0 ? originHeaders : undefined,
            s3OriginConfig,
            customOriginConfig: originConfig.customOriginSource
                ? {
                    httpPort: originConfig.customOriginSource.httpPort || 80,
                    httpsPort: originConfig.customOriginSource.httpsPort || 443,
                    originKeepaliveTimeout: (originConfig.customOriginSource.originKeepaliveTimeout &&
                        originConfig.customOriginSource.originKeepaliveTimeout.toSeconds()) ||
                        5,
                    originReadTimeout: (originConfig.customOriginSource.originReadTimeout &&
                        originConfig.customOriginSource.originReadTimeout.toSeconds()) ||
                        30,
                    originProtocolPolicy: originConfig.customOriginSource.originProtocolPolicy ||
                        distribution_1.OriginProtocolPolicy.HTTPS_ONLY,
                    originSslProtocols: originConfig.customOriginSource
                        .allowedOriginSSLVersions || [OriginSslPolicy.TLS_V1_2],
                }
                : undefined,
            connectionAttempts,
            connectionTimeout,
        };
        return originProperty;
    }
}
exports.CloudFrontWebDistribution = CloudFrontWebDistribution;
//# sourceMappingURL=data:application/json;base64,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