"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OriginAccessIdentity = void 0;
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const cloudfront_generated_1 = require("./cloudfront.generated");
class OriginAccessIdentityBase extends cdk.Resource {
    /**
     * The ARN to include in S3 bucket policy to allow CloudFront access
     */
    arn() {
        return cdk.Stack.of(this).formatArn({
            service: 'iam',
            region: '',
            account: 'cloudfront',
            resource: 'user',
            resourceName: `CloudFront Origin Access Identity ${this.originAccessIdentityName}`,
        });
    }
}
/**
 * An origin access identity is a special CloudFront user that you can
 * associate with Amazon S3 origins, so that you can secure all or just some of
 * your Amazon S3 content.
 *
 * @resource AWS::CloudFront::CloudFrontOriginAccessIdentity
 */
class OriginAccessIdentity extends OriginAccessIdentityBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        // Comment has a max length of 128.
        const comment = ((_a = props === null || props === void 0 ? void 0 : props.comment) !== null && _a !== void 0 ? _a : 'Allows CloudFront to reach the bucket').substr(0, 128);
        this.resource = new cloudfront_generated_1.CfnCloudFrontOriginAccessIdentity(this, 'Resource', {
            cloudFrontOriginAccessIdentityConfig: { comment },
        });
        // physical id - OAI name
        this.originAccessIdentityName = this.getResourceNameAttribute(this.resource.ref);
        // Canonical user to whitelist in S3 Bucket Policy
        this.cloudFrontOriginAccessIdentityS3CanonicalUserId = this.resource.attrS3CanonicalUserId;
        // The principal for must be either the canonical user or a special ARN
        // with the CloudFront Origin Access Id (see `arn()` method). For
        // import/export the OAI is anyway required so the principal is constructed
        // with it. But for the normal case the S3 Canonical User as a nicer
        // interface and does not require constructing the ARN.
        this.grantPrincipal = new iam.CanonicalUserPrincipal(this.cloudFrontOriginAccessIdentityS3CanonicalUserId);
    }
    /**
     * Creates a OriginAccessIdentity by providing the OriginAccessIdentityName
     */
    static fromOriginAccessIdentityName(scope, id, originAccessIdentityName) {
        class Import extends OriginAccessIdentityBase {
            constructor(s, i) {
                super(s, i, { physicalName: originAccessIdentityName });
                this.originAccessIdentityName = originAccessIdentityName;
                this.grantPrincipal = new iam.ArnPrincipal(this.arn());
            }
        }
        return new Import(scope, id);
    }
}
exports.OriginAccessIdentity = OriginAccessIdentity;
//# sourceMappingURL=data:application/json;base64,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