"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaEdgeEventType = exports.CachedMethods = exports.AllowedMethods = exports.SecurityPolicyProtocol = exports.SSLMethod = exports.OriginProtocolPolicy = exports.ViewerProtocolPolicy = exports.PriceClass = exports.HttpVersion = exports.Distribution = void 0;
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const constructs_1 = require("constructs");
const cloudfront_generated_1 = require("./cloudfront.generated");
const cache_behavior_1 = require("./private/cache-behavior");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
/**
 * A CloudFront distribution with associated origin(s) and caching behavior(s).
 *
 * @experimental
 */
class Distribution extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f;
        super(scope, id);
        this.additionalBehaviors = [];
        this.boundOrigins = [];
        this.originGroups = [];
        if (props.certificate) {
            const certificateRegion = core_1.Stack.of(this).parseArn(props.certificate.certificateArn).region;
            if (!core_1.Token.isUnresolved(certificateRegion) && certificateRegion !== 'us-east-1') {
                throw new Error(`Distribution certificates must be in the us-east-1 region and the certificate you provided is in ${certificateRegion}.`);
            }
            if (((_a = props.domainNames) !== null && _a !== void 0 ? _a : []).length === 0) {
                throw new Error('Must specify at least one domain name to use a certificate with a distribution');
            }
        }
        const originId = this.addOrigin(props.defaultBehavior.origin);
        this.defaultBehavior = new cache_behavior_1.CacheBehavior(originId, { pathPattern: '*', ...props.defaultBehavior });
        if (props.additionalBehaviors) {
            Object.entries(props.additionalBehaviors).forEach(([pathPattern, behaviorOptions]) => {
                this.addBehavior(pathPattern, behaviorOptions.origin, behaviorOptions);
            });
        }
        this.certificate = props.certificate;
        this.errorResponses = (_b = props.errorResponses) !== null && _b !== void 0 ? _b : [];
        const distribution = new cloudfront_generated_1.CfnDistribution(this, 'Resource', {
            distributionConfig: {
                enabled: (_c = props.enabled) !== null && _c !== void 0 ? _c : true,
                origins: core_1.Lazy.anyValue({ produce: () => this.renderOrigins() }),
                originGroups: core_1.Lazy.anyValue({ produce: () => this.renderOriginGroups() }),
                defaultCacheBehavior: this.defaultBehavior._renderBehavior(),
                aliases: props.domainNames,
                cacheBehaviors: core_1.Lazy.anyValue({ produce: () => this.renderCacheBehaviors() }),
                comment: props.comment,
                customErrorResponses: this.renderErrorResponses(),
                defaultRootObject: props.defaultRootObject,
                httpVersion: (_d = props.httpVersion) !== null && _d !== void 0 ? _d : HttpVersion.HTTP2,
                ipv6Enabled: (_e = props.enableIpv6) !== null && _e !== void 0 ? _e : true,
                logging: this.renderLogging(props),
                priceClass: (_f = props.priceClass) !== null && _f !== void 0 ? _f : undefined,
                restrictions: this.renderRestrictions(props.geoRestriction),
                viewerCertificate: this.certificate ? this.renderViewerCertificate(this.certificate) : undefined,
                webAclId: props.webAclId,
            },
        });
        this.domainName = distribution.attrDomainName;
        this.distributionDomainName = distribution.attrDomainName;
        this.distributionId = distribution.ref;
    }
    /**
     * Creates a Distribution construct that represents an external (imported) distribution.
     */
    static fromDistributionAttributes(scope, id, attrs) {
        return new class extends core_1.Resource {
            constructor() {
                super(scope, id);
                this.domainName = attrs.domainName;
                this.distributionDomainName = attrs.domainName;
                this.distributionId = attrs.distributionId;
            }
        }();
    }
    /**
     * Adds a new behavior to this distribution for the given pathPattern.
     *
     * @param pathPattern the path pattern (e.g., 'images/*') that specifies which requests to apply the behavior to.
     * @param origin the origin to use for this behavior
     * @param behaviorOptions the options for the behavior at this path.
     */
    addBehavior(pathPattern, origin, behaviorOptions = {}) {
        if (pathPattern === '*') {
            throw new Error('Only the default behavior can have a path pattern of \'*\'');
        }
        const originId = this.addOrigin(origin);
        this.additionalBehaviors.push(new cache_behavior_1.CacheBehavior(originId, { pathPattern, ...behaviorOptions }));
    }
    addOrigin(origin, isFailoverOrigin = false) {
        var _a;
        const existingOrigin = this.boundOrigins.find(boundOrigin => boundOrigin.origin === origin);
        if (existingOrigin) {
            return (_a = existingOrigin.originGroupId) !== null && _a !== void 0 ? _a : existingOrigin.originId;
        }
        else {
            const originIndex = this.boundOrigins.length + 1;
            const scope = new core_2.Construct(this, `Origin${originIndex}`);
            const originId = constructs_1.Node.of(scope).uniqueId;
            const originBindConfig = origin.bind(scope, { originId });
            if (!originBindConfig.failoverConfig) {
                this.boundOrigins.push({ origin, originId, ...originBindConfig });
            }
            else {
                if (isFailoverOrigin) {
                    throw new Error('An Origin cannot use an Origin with its own failover configuration as its fallback origin!');
                }
                const groupIndex = this.originGroups.length + 1;
                const originGroupId = constructs_1.Node.of(new core_2.Construct(this, `OriginGroup${groupIndex}`)).uniqueId;
                this.boundOrigins.push({ origin, originId, originGroupId, ...originBindConfig });
                const failoverOriginId = this.addOrigin(originBindConfig.failoverConfig.failoverOrigin, true);
                this.addOriginGroup(originGroupId, originBindConfig.failoverConfig.statusCodes, originId, failoverOriginId);
                return originGroupId;
            }
            return originId;
        }
    }
    addOriginGroup(originGroupId, statusCodes, originId, failoverOriginId) {
        statusCodes = statusCodes !== null && statusCodes !== void 0 ? statusCodes : [500, 502, 503, 504];
        if (statusCodes.length === 0) {
            throw new Error('fallbackStatusCodes cannot be empty');
        }
        this.originGroups.push({
            failoverCriteria: {
                statusCodes: {
                    items: statusCodes,
                    quantity: statusCodes.length,
                },
            },
            id: originGroupId,
            members: {
                items: [
                    { originId },
                    { originId: failoverOriginId },
                ],
                quantity: 2,
            },
        });
    }
    renderOrigins() {
        const renderedOrigins = [];
        this.boundOrigins.forEach(boundOrigin => {
            if (boundOrigin.originProperty) {
                renderedOrigins.push(boundOrigin.originProperty);
            }
        });
        return renderedOrigins;
    }
    renderOriginGroups() {
        return this.originGroups.length === 0
            ? undefined
            : {
                items: this.originGroups,
                quantity: this.originGroups.length,
            };
    }
    renderCacheBehaviors() {
        if (this.additionalBehaviors.length === 0) {
            return undefined;
        }
        return this.additionalBehaviors.map(behavior => behavior._renderBehavior());
    }
    renderErrorResponses() {
        if (this.errorResponses.length === 0) {
            return undefined;
        }
        function validateCustomErrorResponse(errorResponse) {
            if (errorResponse.responsePagePath && !errorResponse.responseHttpStatus) {
                throw new Error('\'responseCode\' must be provided if \'responsePagePath\' is defined');
            }
            if (!errorResponse.responseHttpStatus && !errorResponse.ttl) {
                throw new Error('A custom error response without either a \'responseCode\' or \'errorCachingMinTtl\' is not valid.');
            }
        }
        this.errorResponses.forEach(e => validateCustomErrorResponse(e));
        return this.errorResponses.map(errorConfig => {
            var _a;
            return {
                errorCachingMinTtl: (_a = errorConfig.ttl) === null || _a === void 0 ? void 0 : _a.toSeconds(),
                errorCode: errorConfig.httpStatus,
                responseCode: errorConfig.responseHttpStatus,
                responsePagePath: errorConfig.responsePagePath,
            };
        });
    }
    renderLogging(props) {
        var _a;
        if (!props.enableLogging && !props.logBucket) {
            return undefined;
        }
        if (props.enableLogging === false && props.logBucket) {
            throw new Error('Explicitly disabled logging but provided a logging bucket.');
        }
        const bucket = (_a = props.logBucket) !== null && _a !== void 0 ? _a : new s3.Bucket(this, 'LoggingBucket');
        return {
            bucket: bucket.bucketDomainName,
            includeCookies: props.logIncludesCookies,
            prefix: props.logFilePrefix,
        };
    }
    renderRestrictions(geoRestriction) {
        return geoRestriction ? {
            geoRestriction: {
                restrictionType: geoRestriction.restrictionType,
                locations: geoRestriction.locations,
            },
        } : undefined;
    }
    renderViewerCertificate(certificate) {
        return {
            acmCertificateArn: certificate.certificateArn,
            sslSupportMethod: SSLMethod.SNI,
            minimumProtocolVersion: SecurityPolicyProtocol.TLS_V1_2_2019,
        };
    }
}
exports.Distribution = Distribution;
/** Maximum HTTP version to support */
var HttpVersion;
(function (HttpVersion) {
    /** HTTP 1.1 */
    HttpVersion["HTTP1_1"] = "http1.1";
    /** HTTP 2 */
    HttpVersion["HTTP2"] = "http2";
})(HttpVersion = exports.HttpVersion || (exports.HttpVersion = {}));
/**
 * The price class determines how many edge locations CloudFront will use for your distribution.
 * See https://aws.amazon.com/cloudfront/pricing/ for full list of supported regions.
 */
var PriceClass;
(function (PriceClass) {
    /** USA, Canada, Europe, & Israel */
    PriceClass["PRICE_CLASS_100"] = "PriceClass_100";
    /** PRICE_CLASS_100 + South Africa, Kenya, Middle East, Japan, Singapore, South Korea, Taiwan, Hong Kong, & Philippines */
    PriceClass["PRICE_CLASS_200"] = "PriceClass_200";
    /** All locations */
    PriceClass["PRICE_CLASS_ALL"] = "PriceClass_All";
})(PriceClass = exports.PriceClass || (exports.PriceClass = {}));
/**
 * How HTTPs should be handled with your distribution.
 */
var ViewerProtocolPolicy;
(function (ViewerProtocolPolicy) {
    /** HTTPS only */
    ViewerProtocolPolicy["HTTPS_ONLY"] = "https-only";
    /** Will redirect HTTP requests to HTTPS */
    ViewerProtocolPolicy["REDIRECT_TO_HTTPS"] = "redirect-to-https";
    /** Both HTTP and HTTPS supported */
    ViewerProtocolPolicy["ALLOW_ALL"] = "allow-all";
})(ViewerProtocolPolicy = exports.ViewerProtocolPolicy || (exports.ViewerProtocolPolicy = {}));
/**
 * Defines what protocols CloudFront will use to connect to an origin.
 */
var OriginProtocolPolicy;
(function (OriginProtocolPolicy) {
    /** Connect on HTTP only */
    OriginProtocolPolicy["HTTP_ONLY"] = "http-only";
    /** Connect with the same protocol as the viewer */
    OriginProtocolPolicy["MATCH_VIEWER"] = "match-viewer";
    /** Connect on HTTPS only */
    OriginProtocolPolicy["HTTPS_ONLY"] = "https-only";
})(OriginProtocolPolicy = exports.OriginProtocolPolicy || (exports.OriginProtocolPolicy = {}));
/**
 * The SSL method CloudFront will use for your distribution.
 *
 * Server Name Indication (SNI) - is an extension to the TLS computer networking protocol by which a client indicates
 *  which hostname it is attempting to connect to at the start of the handshaking process. This allows a server to present
 *  multiple certificates on the same IP address and TCP port number and hence allows multiple secure (HTTPS) websites
 * (or any other service over TLS) to be served by the same IP address without requiring all those sites to use the same certificate.
 *
 * CloudFront can use SNI to host multiple distributions on the same IP - which a large majority of clients will support.
 *
 * If your clients cannot support SNI however - CloudFront can use dedicated IPs for your distribution - but there is a prorated monthly charge for
 * using this feature. By default, we use SNI - but you can optionally enable dedicated IPs (VIP).
 *
 * See the CloudFront SSL for more details about pricing : https://aws.amazon.com/cloudfront/custom-ssl-domains/
 *
 */
var SSLMethod;
(function (SSLMethod) {
    SSLMethod["SNI"] = "sni-only";
    SSLMethod["VIP"] = "vip";
})(SSLMethod = exports.SSLMethod || (exports.SSLMethod = {}));
/**
 * The minimum version of the SSL protocol that you want CloudFront to use for HTTPS connections.
 * CloudFront serves your objects only to browsers or devices that support at least the SSL version that you specify.
 */
var SecurityPolicyProtocol;
(function (SecurityPolicyProtocol) {
    SecurityPolicyProtocol["SSL_V3"] = "SSLv3";
    SecurityPolicyProtocol["TLS_V1"] = "TLSv1";
    SecurityPolicyProtocol["TLS_V1_2016"] = "TLSv1_2016";
    SecurityPolicyProtocol["TLS_V1_1_2016"] = "TLSv1.1_2016";
    SecurityPolicyProtocol["TLS_V1_2_2018"] = "TLSv1.2_2018";
    SecurityPolicyProtocol["TLS_V1_2_2019"] = "TLSv1.2_2019";
})(SecurityPolicyProtocol = exports.SecurityPolicyProtocol || (exports.SecurityPolicyProtocol = {}));
/**
 * The HTTP methods that the Behavior will accept requests on.
 */
class AllowedMethods {
    constructor(methods) { this.methods = methods; }
}
exports.AllowedMethods = AllowedMethods;
/** HEAD and GET */
AllowedMethods.ALLOW_GET_HEAD = new AllowedMethods(['GET', 'HEAD']);
/** HEAD, GET, and OPTIONS */
AllowedMethods.ALLOW_GET_HEAD_OPTIONS = new AllowedMethods(['GET', 'HEAD', 'OPTIONS']);
/** All supported HTTP methods */
AllowedMethods.ALLOW_ALL = new AllowedMethods(['GET', 'HEAD', 'OPTIONS', 'PUT', 'PATCH', 'POST', 'DELETE']);
/**
 * The HTTP methods that the Behavior will cache requests on.
 */
class CachedMethods {
    constructor(methods) { this.methods = methods; }
}
exports.CachedMethods = CachedMethods;
/** HEAD and GET */
CachedMethods.CACHE_GET_HEAD = new CachedMethods(['GET', 'HEAD']);
/** HEAD, GET, and OPTIONS */
CachedMethods.CACHE_GET_HEAD_OPTIONS = new CachedMethods(['GET', 'HEAD', 'OPTIONS']);
/**
 * The type of events that a Lambda@Edge function can be invoked in response to.
 */
var LambdaEdgeEventType;
(function (LambdaEdgeEventType) {
    /**
     * The origin-request specifies the request to the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_REQUEST"] = "origin-request";
    /**
     * The origin-response specifies the response from the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_RESPONSE"] = "origin-response";
    /**
     * The viewer-request specifies the incoming request
     */
    LambdaEdgeEventType["VIEWER_REQUEST"] = "viewer-request";
    /**
     * The viewer-response specifies the outgoing reponse
     */
    LambdaEdgeEventType["VIEWER_RESPONSE"] = "viewer-response";
})(LambdaEdgeEventType = exports.LambdaEdgeEventType || (exports.LambdaEdgeEventType = {}));
//# sourceMappingURL=data:application/json;base64,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