import os
import sys
from typing import Tuple

import httpretty


def get_auth_email_token() -> Tuple[str, str]:
    try:
        email = os.environ["KTAPI_AUTH_EMAIL"]
        token = os.environ["KTAPI_AUTH_TOKEN"]
        return email, token
    except KeyError:
        print("You have to specify KTAPI_AUTH_EMAIL and KTAPI_AUTH_TOKEN first")
        sys.exit(1)


def run_stub_api_server() -> None:
    """ run stub server that will return pre-configured responses for demo purposes """

    URL = "http://www.stub-kentikapi-server.mm/query/topXchart"

    responses = [
        # first succcessful attempt
        httpretty.Response(status=200, body=query_chart_response_body),
        # second, failed attempt
        httpretty.Response(status=429, body="too many requests"),
        # active waiting - retry 5 times
        httpretty.Response(status=429, body="too many requests"),
        httpretty.Response(status=429, body="too many requests"),
        httpretty.Response(status=429, body="too many requests"),
        httpretty.Response(status=429, body="too many requests"),
        httpretty.Response(status=200, body=query_chart_response_body),
        # background queue - retry 5 times
        httpretty.Response(status=429, body="too many requests"),
        httpretty.Response(status=429, body="too many requests"),
        httpretty.Response(status=429, body="too many requests"),
        httpretty.Response(status=429, body="too many requests"),
        httpretty.Response(status=200, body=query_chart_response_body),
        # one more run in case of repeat is wanted
        httpretty.Response(status=429, body="too many requests"),
        httpretty.Response(status=429, body="too many requests"),
        httpretty.Response(status=429, body="too many requests"),
        httpretty.Response(status=429, body="too many requests"),
        httpretty.Response(status=200, body=query_chart_response_body),
    ]
    httpretty.register_uri(method=httpretty.POST, uri=URL, responses=responses)
    httpretty.enable()


# Real query chart response body from KentikAPI
query_chart_response_body = """{"dataUri":"data:image/png;base64,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"}"""
