"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubWebhookHandler = void 0;
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const access_1 = require("./access");
const utils_1 = require("./utils");
const webhook_handler_function_1 = require("./webhook-handler-function");
/**
 * Create a Lambda with a public URL to handle GitHub webhook events. After validating the event with the given secret, the orchestrator step function is called with information about the workflow job.
 *
 * @internal
 */
class GithubWebhookHandler extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.handler = new webhook_handler_function_1.WebhookHandlerFunction(this, 'webhook-handler', {
            description: 'Handle GitHub webhook and start runner orchestrator',
            environment: {
                STEP_FUNCTION_ARN: props.orchestrator.stateMachineArn,
                WEBHOOK_SECRET_ARN: props.secrets.webhook.secretArn,
                GITHUB_SECRET_ARN: props.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: props.secrets.githubPrivateKey.secretArn,
                SUPPORTED_LABELS: JSON.stringify(props.supportedLabels),
                REQUIRE_SELF_HOSTED_LABEL: props.requireSelfHostedLabel ? '1' : '0',
            },
            timeout: cdk.Duration.seconds(31),
            logGroup: (0, utils_1.singletonLogGroup)(this, utils_1.SingletonLogType.ORCHESTRATOR),
            loggingFormat: aws_cdk_lib_1.aws_lambda.LoggingFormat.JSON,
        });
        const access = props?.access ?? access_1.LambdaAccess.lambdaUrl();
        this.url = access.bind(this, 'access', this.handler);
        props.secrets.webhook.grantRead(this.handler);
        props.secrets.github.grantRead(this.handler);
        props.secrets.githubPrivateKey.grantRead(this.handler);
        props.orchestrator.grantStartExecution(this.handler);
    }
}
exports.GithubWebhookHandler = GithubWebhookHandler;
//# sourceMappingURL=data:application/json;base64,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