"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitHubRunners = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const access_1 = require("./access");
const delete_failed_runner_function_1 = require("./delete-failed-runner-function");
const idle_runner_repear_function_1 = require("./idle-runner-repear-function");
const providers_1 = require("./providers");
const secrets_1 = require("./secrets");
const setup_function_1 = require("./setup-function");
const status_function_1 = require("./status-function");
const token_retriever_function_1 = require("./token-retriever-function");
const utils_1 = require("./utils");
const webhook_1 = require("./webhook");
const webhook_redelivery_1 = require("./webhook-redelivery");
/**
 * Create all the required infrastructure to provide self-hosted GitHub runners. It creates a webhook, secrets, and a step function to orchestrate all runs. Secrets are not automatically filled. See README.md for instructions on how to setup GitHub integration.
 *
 * By default, this will create a runner provider of each available type with the defaults. This is good enough for the initial setup stage when you just want to get GitHub integration working.
 *
 * ```typescript
 * new GitHubRunners(this, 'runners');
 * ```
 *
 * Usually you'd want to configure the runner providers so the runners can run in a certain VPC or have certain permissions.
 *
 * ```typescript
 * const vpc = ec2.Vpc.fromLookup(this, 'vpc', { vpcId: 'vpc-1234567' });
 * const runnerSg = new ec2.SecurityGroup(this, 'runner security group', { vpc: vpc });
 * const dbSg = ec2.SecurityGroup.fromSecurityGroupId(this, 'database security group', 'sg-1234567');
 * const bucket = new s3.Bucket(this, 'runner bucket');
 *
 * // create a custom CodeBuild provider
 * const myProvider = new CodeBuildRunnerProvider(
 *   this, 'codebuild runner',
 *   {
 *      labels: ['my-codebuild'],
 *      vpc: vpc,
 *      securityGroups: [runnerSg],
 *   },
 * );
 * // grant some permissions to the provider
 * bucket.grantReadWrite(myProvider);
 * dbSg.connections.allowFrom(runnerSg, ec2.Port.tcp(3306), 'allow runners to connect to MySQL database');
 *
 * // create the runner infrastructure
 * new GitHubRunners(
 *   this,
 *   'runners',
 *   {
 *     providers: [myProvider],
 *   }
 * );
 * ```
 */
class GitHubRunners extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.extraLambdaEnv = {};
        this.secrets = new secrets_1.Secrets(this, 'Secrets');
        this.extraLambdaProps = {
            vpc: this.props?.vpc,
            vpcSubnets: this.props?.vpcSubnets,
            allowPublicSubnet: this.props?.allowPublicSubnet,
            securityGroups: this.lambdaSecurityGroups(),
            layers: this.props?.extraCertificates ? [new aws_cdk_lib_1.aws_lambda.LayerVersion(scope, 'Certificate Layer', {
                    description: 'Layer containing GitHub Enterprise Server certificate for cdk-github-runners',
                    code: aws_cdk_lib_1.aws_lambda.Code.fromAsset(this.props.extraCertificates),
                })] : undefined,
        };
        this.connections = new aws_cdk_lib_1.aws_ec2.Connections({ securityGroups: this.extraLambdaProps.securityGroups });
        if (this.props?.extraCertificates) {
            this.extraLambdaEnv.NODE_EXTRA_CA_CERTS = '/opt/certs.pem';
        }
        if (this.props?.providers) {
            this.providers = this.props.providers;
        }
        else {
            this.providers = [
                new providers_1.CodeBuildRunnerProvider(this, 'CodeBuild'),
                new providers_1.LambdaRunnerProvider(this, 'Lambda'),
                new providers_1.FargateRunnerProvider(this, 'Fargate'),
            ];
        }
        if (this.providers.length == 0) {
            throw new Error('At least one runner provider is required');
        }
        this.checkIntersectingLabels();
        this.orchestrator = this.stateMachine(props);
        this.webhook = new webhook_1.GithubWebhookHandler(this, 'Webhook Handler', {
            orchestrator: this.orchestrator,
            secrets: this.secrets,
            access: this.props?.webhookAccess ?? access_1.LambdaAccess.lambdaUrl(),
            supportedLabels: this.providers.map(p => {
                return {
                    provider: p.node.path,
                    labels: p.labels,
                };
            }),
            requireSelfHostedLabel: this.props?.requireSelfHostedLabel ?? true,
        });
        this.redeliverer = new webhook_redelivery_1.GithubWebhookRedelivery(this, 'Webhook Redelivery', {
            secrets: this.secrets,
        });
        this.setupUrl = this.setupFunction();
        this.statusFunction();
    }
    stateMachine(props) {
        const tokenRetrieverTask = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, 'Get Runner Token', {
            lambdaFunction: this.tokenRetriever(),
            payloadResponseOnly: true,
            resultPath: '$.runner',
        });
        let deleteFailedRunnerFunction = this.deleteFailedRunner();
        const deleteFailedRunnerTask = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, 'Delete Failed Runner', {
            lambdaFunction: deleteFailedRunnerFunction,
            payloadResponseOnly: true,
            resultPath: '$.delete',
            payload: aws_cdk_lib_1.aws_stepfunctions.TaskInput.fromObject({
                runnerName: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Name'),
                owner: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.owner'),
                repo: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.repo'),
                installationId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.numberAt('$.installationId'),
                error: aws_cdk_lib_1.aws_stepfunctions.JsonPath.objectAt('$.error'),
            }),
        });
        deleteFailedRunnerTask.addRetry({
            errors: [
                'RunnerBusy',
            ],
            interval: cdk.Duration.minutes(1),
            backoffRate: 1,
            maxAttempts: 60,
        });
        const idleReaper = this.idleReaper();
        const queueIdleReaperTask = new aws_cdk_lib_1.aws_stepfunctions_tasks.SqsSendMessage(this, 'Queue Idle Reaper', {
            queue: this.idleReaperQueue(idleReaper),
            messageBody: aws_cdk_lib_1.aws_stepfunctions.TaskInput.fromObject({
                executionArn: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Id'),
                runnerName: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Name'),
                owner: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.owner'),
                repo: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.repo'),
                installationId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.numberAt('$.installationId'),
                maxIdleSeconds: (props?.idleTimeout ?? cdk.Duration.minutes(5)).toSeconds(),
            }),
            resultPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.DISCARD,
        });
        const providerChooser = new aws_cdk_lib_1.aws_stepfunctions.Choice(this, 'Choose provider');
        for (const provider of this.providers) {
            const providerTask = provider.getStepFunctionTask({
                runnerTokenPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runner.token'),
                runnerNamePath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Name'),
                githubDomainPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runner.domain'),
                ownerPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.owner'),
                repoPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.repo'),
                registrationUrl: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runner.registrationUrl'),
            });
            providerChooser.when(aws_cdk_lib_1.aws_stepfunctions.Condition.and(aws_cdk_lib_1.aws_stepfunctions.Condition.stringEquals('$.provider', provider.node.path)), providerTask);
        }
        providerChooser.otherwise(new aws_cdk_lib_1.aws_stepfunctions.Succeed(this, 'Unknown label'));
        const runProviders = new aws_cdk_lib_1.aws_stepfunctions.Parallel(this, 'Run Providers').branch(new aws_cdk_lib_1.aws_stepfunctions.Parallel(this, 'Error Handler').branch(
        // we get a token for every retry because the token can expire faster than the job can timeout
        tokenRetrieverTask.next(providerChooser)).addCatch(
        // delete runner on failure as it won't remove itself and there is a limit on the number of registered runners
        deleteFailedRunnerTask, {
            resultPath: '$.error',
        }));
        if (props?.retryOptions?.retry ?? true) {
            const interval = props?.retryOptions?.interval ?? cdk.Duration.minutes(1);
            const maxAttempts = props?.retryOptions?.maxAttempts ?? 23;
            const backoffRate = props?.retryOptions?.backoffRate ?? 1.3;
            const totalSeconds = interval.toSeconds() * backoffRate ** maxAttempts / (backoffRate - 1);
            if (totalSeconds >= cdk.Duration.days(1).toSeconds()) {
                // https://docs.github.com/en/actions/hosting-your-own-runners/managing-self-hosted-runners/about-self-hosted-runners#usage-limits
                // "Job queue time - Each job for self-hosted runners can be queued for a maximum of 24 hours. If a self-hosted runner does not start executing the job within this limit, the job is terminated and fails to complete."
                aws_cdk_lib_1.Annotations.of(this).addWarning(`Total retry time is greater than 24 hours (${Math.floor(totalSeconds / 60 / 60)} hours). Jobs expire after 24 hours so it would be a waste of resources to retry further.`);
            }
            runProviders.addRetry({
                interval,
                maxAttempts,
                backoffRate,
                // we retry on everything
                // deleted idle runners will also fail, but the reaper will stop this step function to avoid endless retries
            });
        }
        let logOptions;
        if (this.props?.logOptions) {
            this.stateMachineLogGroup = new aws_cdk_lib_1.aws_logs.LogGroup(this, 'Logs', {
                logGroupName: props?.logOptions?.logGroupName,
                retention: props?.logOptions?.logRetention ?? aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH,
                removalPolicy: cdk.RemovalPolicy.DESTROY,
            });
            logOptions = {
                destination: this.stateMachineLogGroup,
                includeExecutionData: props?.logOptions?.includeExecutionData ?? true,
                level: props?.logOptions?.level ?? aws_cdk_lib_1.aws_stepfunctions.LogLevel.ALL,
            };
        }
        const stateMachine = new aws_cdk_lib_1.aws_stepfunctions.StateMachine(this, 'Runner Orchestrator', {
            definitionBody: aws_cdk_lib_1.aws_stepfunctions.DefinitionBody.fromChainable(queueIdleReaperTask.next(runProviders)),
            logs: logOptions,
        });
        stateMachine.grantRead(idleReaper);
        stateMachine.grantExecution(idleReaper, 'states:StopExecution');
        for (const provider of this.providers) {
            provider.grantStateMachine(stateMachine);
        }
        return stateMachine;
    }
    tokenRetriever() {
        const func = new token_retriever_function_1.TokenRetrieverFunction(this, 'token-retriever', {
            description: 'Get token from GitHub Actions used to start new self-hosted runner',
            environment: {
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                ...this.extraLambdaEnv,
            },
            timeout: cdk.Duration.seconds(30),
            logGroup: (0, utils_1.singletonLogGroup)(this, utils_1.SingletonLogType.ORCHESTRATOR),
            loggingFormat: aws_cdk_lib_1.aws_lambda.LoggingFormat.JSON,
            ...this.extraLambdaProps,
        });
        this.secrets.github.grantRead(func);
        this.secrets.githubPrivateKey.grantRead(func);
        return func;
    }
    deleteFailedRunner() {
        const func = new delete_failed_runner_function_1.DeleteFailedRunnerFunction(this, 'delete-runner', {
            description: 'Delete failed GitHub Actions runner on error',
            environment: {
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                ...this.extraLambdaEnv,
            },
            timeout: cdk.Duration.seconds(30),
            logGroup: (0, utils_1.singletonLogGroup)(this, utils_1.SingletonLogType.ORCHESTRATOR),
            loggingFormat: aws_cdk_lib_1.aws_lambda.LoggingFormat.JSON,
            ...this.extraLambdaProps,
        });
        this.secrets.github.grantRead(func);
        this.secrets.githubPrivateKey.grantRead(func);
        return func;
    }
    statusFunction() {
        const statusFunction = new status_function_1.StatusFunction(this, 'status', {
            description: 'Provide user with status about self-hosted GitHub Actions runners',
            environment: {
                WEBHOOK_SECRET_ARN: this.secrets.webhook.secretArn,
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                SETUP_SECRET_ARN: this.secrets.setup.secretArn,
                WEBHOOK_URL: this.webhook.url,
                WEBHOOK_HANDLER_ARN: this.webhook.handler.latestVersion.functionArn,
                STEP_FUNCTION_ARN: this.orchestrator.stateMachineArn,
                STEP_FUNCTION_LOG_GROUP: this.stateMachineLogGroup?.logGroupName ?? '',
                SETUP_FUNCTION_URL: this.setupUrl,
                ...this.extraLambdaEnv,
            },
            timeout: cdk.Duration.minutes(3),
            logGroup: (0, utils_1.singletonLogGroup)(this, utils_1.SingletonLogType.SETUP),
            loggingFormat: aws_cdk_lib_1.aws_lambda.LoggingFormat.JSON,
            ...this.extraLambdaProps,
        });
        const providers = this.providers.map(provider => provider.status(statusFunction));
        // expose providers as stack metadata as it's too big for Lambda environment variables
        // specifically integration testing got an error because lambda update request was >5kb
        const stack = cdk.Stack.of(this);
        const f = statusFunction.node.defaultChild;
        f.addPropertyOverride('Environment.Variables.LOGICAL_ID', f.logicalId);
        f.addPropertyOverride('Environment.Variables.STACK_NAME', stack.stackName);
        f.addMetadata('providers', providers);
        statusFunction.addToRolePolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['cloudformation:DescribeStackResource'],
            resources: [stack.stackId],
        }));
        this.secrets.webhook.grantRead(statusFunction);
        this.secrets.github.grantRead(statusFunction);
        this.secrets.githubPrivateKey.grantRead(statusFunction);
        this.secrets.setup.grantRead(statusFunction);
        this.orchestrator.grantRead(statusFunction);
        new cdk.CfnOutput(this, 'status command', {
            value: `aws --region ${stack.region} lambda invoke --function-name ${statusFunction.functionName} status.json`,
        });
        const access = this.props?.statusAccess ?? access_1.LambdaAccess.noAccess();
        const url = access.bind(this, 'status access', statusFunction);
        if (url !== '') {
            new cdk.CfnOutput(this, 'status url', {
                value: url,
            });
        }
    }
    setupFunction() {
        const setupFunction = new setup_function_1.SetupFunction(this, 'setup', {
            description: 'Setup GitHub Actions integration with self-hosted runners',
            environment: {
                SETUP_SECRET_ARN: this.secrets.setup.secretArn,
                WEBHOOK_SECRET_ARN: this.secrets.webhook.secretArn,
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                WEBHOOK_URL: this.webhook.url,
                ...this.extraLambdaEnv,
            },
            timeout: cdk.Duration.minutes(3),
            logGroup: (0, utils_1.singletonLogGroup)(this, utils_1.SingletonLogType.SETUP),
            loggingFormat: aws_cdk_lib_1.aws_lambda.LoggingFormat.JSON,
            ...this.extraLambdaProps,
        });
        // this.secrets.webhook.grantRead(setupFunction);
        this.secrets.webhook.grantWrite(setupFunction);
        this.secrets.github.grantRead(setupFunction);
        this.secrets.github.grantWrite(setupFunction);
        // this.secrets.githubPrivateKey.grantRead(setupFunction);
        this.secrets.githubPrivateKey.grantWrite(setupFunction);
        this.secrets.setup.grantRead(setupFunction);
        this.secrets.setup.grantWrite(setupFunction);
        const access = this.props?.setupAccess ?? access_1.LambdaAccess.lambdaUrl();
        return access.bind(this, 'setup access', setupFunction);
    }
    checkIntersectingLabels() {
        // this "algorithm" is very inefficient, but good enough for the tiny datasets we expect
        for (const p1 of this.providers) {
            for (const p2 of this.providers) {
                if (p1 == p2) {
                    continue;
                }
                if (p1.labels.every(l => p2.labels.includes(l))) {
                    if (p2.labels.every(l => p1.labels.includes(l))) {
                        throw new Error(`Both ${p1.node.path} and ${p2.node.path} use the same labels [${p1.labels.join(', ')}]`);
                    }
                    aws_cdk_lib_1.Annotations.of(p1).addWarning(`Labels [${p1.labels.join(', ')}] intersect with another provider (${p2.node.path} -- [${p2.labels.join(', ')}]). If a workflow specifies the labels [${p1.labels.join(', ')}], it is not guaranteed which provider will be used. It is recommended you do not use intersecting labels`);
                }
            }
        }
    }
    idleReaper() {
        return new idle_runner_repear_function_1.IdleRunnerRepearFunction(this, 'Idle Reaper', {
            description: 'Stop idle GitHub runners to avoid paying for runners when the job was already canceled',
            environment: {
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                ...this.extraLambdaEnv,
            },
            logGroup: (0, utils_1.singletonLogGroup)(this, utils_1.SingletonLogType.ORCHESTRATOR),
            loggingFormat: aws_cdk_lib_1.aws_lambda.LoggingFormat.JSON,
            timeout: cdk.Duration.minutes(5),
            ...this.extraLambdaProps,
        });
    }
    idleReaperQueue(reaper) {
        // see this comment to understand why it's a queue that's out of the step function
        // https://github.com/CloudSnorkel/cdk-github-runners/pull/314#issuecomment-1528901192
        const queue = new aws_cdk_lib_1.aws_sqs.Queue(this, 'Idle Reaper Queue', {
            deliveryDelay: cdk.Duration.minutes(10),
            visibilityTimeout: cdk.Duration.minutes(10),
        });
        reaper.addEventSource(new aws_cdk_lib_1.aws_lambda_event_sources.SqsEventSource(queue, {
            reportBatchItemFailures: true,
            maxBatchingWindow: cdk.Duration.minutes(1),
        }));
        this.secrets.github.grantRead(reaper);
        this.secrets.githubPrivateKey.grantRead(reaper);
        return queue;
    }
    lambdaSecurityGroups() {
        if (!this.props?.vpc) {
            if (this.props?.securityGroup) {
                cdk.Annotations.of(this).addWarning('securityGroup is specified, but vpc is not. securityGroup will be ignored');
            }
            if (this.props?.securityGroups) {
                cdk.Annotations.of(this).addWarning('securityGroups is specified, but vpc is not. securityGroups will be ignored');
            }
            return undefined;
        }
        if (this.props.securityGroups) {
            if (this.props.securityGroup) {
                cdk.Annotations.of(this).addWarning('Both securityGroup and securityGroups are specified. securityGroup will be ignored');
            }
            return this.props.securityGroups;
        }
        if (this.props.securityGroup) {
            return [this.props.securityGroup];
        }
        return [new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, 'Management Lambdas Security Group', { vpc: this.props.vpc })];
    }
    /**
     * Metric for the number of GitHub Actions jobs completed. It has `ProviderLabels` and `Status` dimensions. The status can be one of "Succeeded", "SucceededWithIssues", "Failed", "Canceled", "Skipped", or "Abandoned".
     *
     * **WARNING:** this method creates a metric filter for each provider. Each metric has a status dimension with six possible values. These resources may incur cost.
     */
    metricJobCompleted(props) {
        if (!this.jobsCompletedMetricFilters) {
            // we can't use logs.FilterPattern.spaceDelimited() because it has no support for ||
            // status list taken from https://github.com/actions/runner/blob/be9632302ceef50bfb36ea998cea9c94c75e5d4d/src/Sdk/DTWebApi/WebApi/TaskResult.cs
            // we need "..." for Lambda that prefixes some extra data to log lines
            const pattern = aws_cdk_lib_1.aws_logs.FilterPattern.literal('[..., marker = "CDKGHA", job = "JOB", done = "DONE", labels, status = "Succeeded" || status = "SucceededWithIssues" || status = "Failed" || status = "Canceled" || status = "Skipped" || status = "Abandoned"]');
            this.jobsCompletedMetricFilters = this.providers.map(p => p.logGroup.addMetricFilter(`${p.logGroup.node.id} filter`, {
                metricNamespace: 'GitHubRunners',
                metricName: 'JobCompleted',
                filterPattern: pattern,
                metricValue: '1',
                // can't with dimensions -- defaultValue: 0,
                dimensions: {
                    ProviderLabels: '$labels',
                    Status: '$status',
                },
            }));
            for (const metricFilter of this.jobsCompletedMetricFilters) {
                if (metricFilter.node.defaultChild instanceof aws_cdk_lib_1.aws_logs.CfnMetricFilter) {
                    metricFilter.node.defaultChild.addPropertyOverride('MetricTransformations.0.Unit', 'Count');
                }
                else {
                    aws_cdk_lib_1.Annotations.of(metricFilter).addWarning('Unable to set metric filter Unit to Count');
                }
            }
        }
        return new aws_cdk_lib_1.aws_cloudwatch.Metric({
            namespace: 'GitHubRunners',
            metricName: 'JobsCompleted',
            unit: aws_cdk_lib_1.aws_cloudwatch.Unit.COUNT,
            statistic: aws_cdk_lib_1.aws_cloudwatch.Stats.SUM,
            ...props,
        }).attachTo(this);
    }
    /**
     * Metric for successful executions.
     *
     * A successful execution doesn't always mean a runner was started. It can be successful even without any label matches.
     *
     * A successful runner doesn't mean the job it executed was successful. For that, see {@link metricJobCompleted}.
     */
    metricSucceeded(props) {
        return this.orchestrator.metricSucceeded(props);
    }
    /**
     * Metric for failed runner executions.
     *
     * A failed runner usually means the runner failed to start and so a job was never executed. It doesn't necessarily mean the job was executed and failed. For that, see {@link metricJobCompleted}.
     */
    metricFailed(props) {
        return this.orchestrator.metricFailed(props);
    }
    /**
     * Metric for the interval, in milliseconds, between the time the execution starts and the time it closes. This time may be longer than the time the runner took.
     */
    metricTime(props) {
        return this.orchestrator.metricTime(props);
    }
    /**
     * Creates a topic for notifications when a runner image build fails.
     *
     * Runner images are rebuilt every week by default. This provides the latest GitHub Runner version and software updates.
     *
     * If you want to be sure you are using the latest runner version, you can use this topic to be notified when a build fails.
     */
    failedImageBuildsTopic() {
        const topic = new aws_cdk_lib_1.aws_sns.Topic(this, 'Failed Runner Image Builds');
        const stack = cdk.Stack.of(this);
        cdk.Aspects.of(stack).add(new providers_1.CodeBuildImageBuilderFailedBuildNotifier(topic));
        cdk.Aspects.of(stack).add(new providers_1.AwsImageBuilderFailedBuildNotifier(providers_1.AwsImageBuilderFailedBuildNotifier.createFilteringTopic(this, topic)));
        return topic;
    }
    /**
     * Creates CloudWatch Logs Insights saved queries that can be used to debug issues with the runners.
     *
     * * "Webhook errors" helps diagnose configuration issues with GitHub integration
     * * "Ignored webhook" helps understand why runners aren't started
     * * "Ignored jobs based on labels" helps debug label matching issues
     * * "Webhook started runners" helps understand which runners were started
     */
    createLogsInsightsQueries() {
        new aws_cdk_lib_1.aws_logs.QueryDefinition(this, 'Webhook errors', {
            queryDefinitionName: 'GitHub Runners/Webhook errors',
            logGroups: [this.webhook.handler.logGroup],
            queryString: new aws_cdk_lib_1.aws_logs.QueryString({
                filterStatements: [
                    `strcontains(@logStream, "${this.webhook.handler.functionName}")`,
                    'level = "ERROR"',
                ],
                sort: '@timestamp desc',
                limit: 100,
            }),
        });
        new aws_cdk_lib_1.aws_logs.QueryDefinition(this, 'Orchestration errors', {
            queryDefinitionName: 'GitHub Runners/Orchestration errors',
            logGroups: [(0, utils_1.singletonLogGroup)(this, utils_1.SingletonLogType.ORCHESTRATOR)],
            queryString: new aws_cdk_lib_1.aws_logs.QueryString({
                filterStatements: [
                    'level = "ERROR"',
                ],
                sort: '@timestamp desc',
                limit: 100,
            }),
        });
        new aws_cdk_lib_1.aws_logs.QueryDefinition(this, 'Runner image build errors', {
            queryDefinitionName: 'GitHub Runners/Runner image build errors',
            logGroups: [(0, utils_1.singletonLogGroup)(this, utils_1.SingletonLogType.RUNNER_IMAGE_BUILD)],
            queryString: new aws_cdk_lib_1.aws_logs.QueryString({
                filterStatements: [
                    'strcontains(message, "error") or strcontains(message, "ERROR") or strcontains(message, "Error") or level = "ERROR"',
                ],
                sort: '@timestamp desc',
                limit: 100,
            }),
        });
        new aws_cdk_lib_1.aws_logs.QueryDefinition(this, 'Ignored webhooks', {
            queryDefinitionName: 'GitHub Runners/Ignored webhooks',
            logGroups: [this.webhook.handler.logGroup],
            queryString: new aws_cdk_lib_1.aws_logs.QueryString({
                fields: ['@timestamp', 'message.notice'],
                filterStatements: [
                    `strcontains(@logStream, "${this.webhook.handler.functionName}")`,
                    'strcontains(message.notice, "Ignoring")',
                ],
                sort: '@timestamp desc',
                limit: 100,
            }),
        });
        new aws_cdk_lib_1.aws_logs.QueryDefinition(this, 'Ignored jobs based on labels', {
            queryDefinitionName: 'GitHub Runners/Ignored jobs based on labels',
            logGroups: [this.webhook.handler.logGroup],
            queryString: new aws_cdk_lib_1.aws_logs.QueryString({
                fields: ['@timestamp', 'message.notice'],
                filterStatements: [
                    `strcontains(@logStream, "${this.webhook.handler.functionName}")`,
                    'strcontains(message.notice, "Ignoring labels")',
                ],
                sort: '@timestamp desc',
                limit: 100,
            }),
        });
        new aws_cdk_lib_1.aws_logs.QueryDefinition(this, 'Webhook started runners', {
            queryDefinitionName: 'GitHub Runners/Webhook started runners',
            logGroups: [this.webhook.handler.logGroup],
            queryString: new aws_cdk_lib_1.aws_logs.QueryString({
                fields: ['@timestamp', 'message.sfnInput.jobUrl', 'message.sfnInput.labels', 'message.sfnInput.provider'],
                filterStatements: [
                    `strcontains(@logStream, "${this.webhook.handler.functionName}")`,
                    'message.sfnInput.jobUrl like /http.*/',
                ],
                sort: '@timestamp desc',
                limit: 100,
            }),
        });
        new aws_cdk_lib_1.aws_logs.QueryDefinition(this, 'Webhook redeliveries', {
            queryDefinitionName: 'GitHub Runners/Webhook redeliveries',
            logGroups: [this.redeliverer.handler.logGroup],
            queryString: new aws_cdk_lib_1.aws_logs.QueryString({
                fields: ['@timestamp', 'message.notice', 'message.deliveryId', 'message.guid'],
                filterStatements: [
                    'isPresent(message.deliveryId)',
                ],
                sort: '@timestamp desc',
                limit: 100,
            }),
        });
    }
}
exports.GitHubRunners = GitHubRunners;
_a = JSII_RTTI_SYMBOL_1;
GitHubRunners[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.GitHubRunners", version: "0.14.13" };
//# sourceMappingURL=data:application/json;base64,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