import { aws_ec2 as ec2, aws_ecs as ecs, aws_iam as iam, aws_logs as logs, aws_stepfunctions as stepfunctions } from 'aws-cdk-lib';
import { Construct } from 'constructs';
import { BaseProvider, IRunnerProvider, IRunnerProviderStatus, Os, RunnerImage, RunnerProviderProps, RunnerRuntimeParameters } from './common';
import { IRunnerImageBuilder, RunnerImageBuilderProps } from '../image-builders';
/**
 * Properties for FargateRunnerProvider.
 */
export interface FargateRunnerProviderProps extends RunnerProviderProps {
    /**
     * Runner image builder used to build Docker images containing GitHub Runner and all requirements.
     *
     * The image builder determines the OS and architecture of the runner.
     *
     * @default FargateRunnerProvider.imageBuilder()
     */
    readonly imageBuilder?: IRunnerImageBuilder;
    /**
     * GitHub Actions label used for this provider.
     *
     * @default undefined
     * @deprecated use {@link labels} instead
     */
    readonly label?: string;
    /**
     * GitHub Actions labels used for this provider.
     *
     * These labels are used to identify which provider should spawn a new on-demand runner. Every job sends a webhook with the labels it's looking for
     * based on runs-on. We match the labels from the webhook with the labels specified here. If all the labels specified here are present in the
     * job's labels, this provider will be chosen and spawn a new runner.
     *
     * @default ['fargate']
     */
    readonly labels?: string[];
    /**
     * GitHub Actions runner group name.
     *
     * If specified, the runner will be registered with this group name. Setting a runner group can help managing access to self-hosted runners. It
     * requires a paid GitHub account.
     *
     * The group must exist or the runner will not start.
     *
     * Users will still be able to trigger this runner with the correct labels. But the runner will only be able to run jobs from repos allowed to use the group.
     *
     * @default undefined
     */
    readonly group?: string;
    /**
     * VPC to launch the runners in.
     *
     * @default default account VPC
     */
    readonly vpc?: ec2.IVpc;
    /**
     * Subnets to run the runners in.
     *
     * @default Fargate default
     */
    readonly subnetSelection?: ec2.SubnetSelection;
    /**
     * Security group to assign to the task.
     *
     * @default a new security group
     *
     * @deprecated use {@link securityGroups}
     */
    readonly securityGroup?: ec2.ISecurityGroup;
    /**
     * Security groups to assign to the task.
     *
     * @default a new security group
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * Existing Fargate cluster to use.
     *
     * @default a new cluster
     */
    readonly cluster?: ecs.Cluster;
    /**
     * Assign public IP to the runner task.
     *
     * Make sure the task will have access to GitHub. A public IP might be required unless you have NAT gateway.
     *
     * @default true
     */
    readonly assignPublicIp?: boolean;
    /**
     * The number of cpu units used by the task. For tasks using the Fargate launch type,
     * this field is required and you must use one of the following values,
     * which determines your range of valid values for the memory parameter:
     *
     * 256 (.25 vCPU) - Available memory values: 512 (0.5 GB), 1024 (1 GB), 2048 (2 GB)
     *
     * 512 (.5 vCPU) - Available memory values: 1024 (1 GB), 2048 (2 GB), 3072 (3 GB), 4096 (4 GB)
     *
     * 1024 (1 vCPU) - Available memory values: 2048 (2 GB), 3072 (3 GB), 4096 (4 GB), 5120 (5 GB), 6144 (6 GB), 7168 (7 GB), 8192 (8 GB)
     *
     * 2048 (2 vCPU) - Available memory values: Between 4096 (4 GB) and 16384 (16 GB) in increments of 1024 (1 GB)
     *
     * 4096 (4 vCPU) - Available memory values: Between 8192 (8 GB) and 30720 (30 GB) in increments of 1024 (1 GB)
     *
     * @default 1024
     */
    readonly cpu?: number;
    /**
     * The amount (in MiB) of memory used by the task. For tasks using the Fargate launch type,
     * this field is required and you must use one of the following values, which determines your range of valid values for the cpu parameter:
     *
     * 512 (0.5 GB), 1024 (1 GB), 2048 (2 GB) - Available cpu values: 256 (.25 vCPU)
     *
     * 1024 (1 GB), 2048 (2 GB), 3072 (3 GB), 4096 (4 GB) - Available cpu values: 512 (.5 vCPU)
     *
     * 2048 (2 GB), 3072 (3 GB), 4096 (4 GB), 5120 (5 GB), 6144 (6 GB), 7168 (7 GB), 8192 (8 GB) - Available cpu values: 1024 (1 vCPU)
     *
     * Between 4096 (4 GB) and 16384 (16 GB) in increments of 1024 (1 GB) - Available cpu values: 2048 (2 vCPU)
     *
     * Between 8192 (8 GB) and 30720 (30 GB) in increments of 1024 (1 GB) - Available cpu values: 4096 (4 vCPU)
     *
     * @default 2048
     */
    readonly memoryLimitMiB?: number;
    /**
     * The amount (in GiB) of ephemeral storage to be allocated to the task. The maximum supported value is 200 GiB.
     *
     * NOTE: This parameter is only supported for tasks hosted on AWS Fargate using platform version 1.4.0 or later.
     *
     * @default 20
     */
    readonly ephemeralStorageGiB?: number;
    /**
     * Use Fargate spot capacity provider to save money.
     *
     * * Runners may fail to start due to missing capacity.
     * * Runners might be stopped prematurely with spot pricing.
     *
     * @default false
     */
    readonly spot?: boolean;
}
/**
 * @internal
 */
export declare function ecsRunCommand(os: Os, dind: boolean): string[];
/**
 * GitHub Actions runner provider using Fargate to execute jobs.
 *
 * Creates a task definition with a single container that gets started for each job.
 *
 * This construct is not meant to be used by itself. It should be passed in the providers property for GitHubRunners.
 */
export declare class FargateRunnerProvider extends BaseProvider implements IRunnerProvider {
    /**
     * Path to Dockerfile for Linux x64 with all the requirement for Fargate runner. Use this Dockerfile unless you need to customize it further than allowed by hooks.
     *
     * Available build arguments that can be set in the image builder:
     * * `BASE_IMAGE` sets the `FROM` line. This should be an Ubuntu compatible image.
     * * `EXTRA_PACKAGES` can be used to install additional packages.
     *
     * @deprecated Use `imageBuilder()` instead.
     */
    static readonly LINUX_X64_DOCKERFILE_PATH: string;
    /**
     * Path to Dockerfile for Linux ARM64 with all the requirement for Fargate runner. Use this Dockerfile unless you need to customize it further than allowed by hooks.
     *
     * Available build arguments that can be set in the image builder:
     * * `BASE_IMAGE` sets the `FROM` line. This should be an Ubuntu compatible image.
     * * `EXTRA_PACKAGES` can be used to install additional packages.
     *
     * @deprecated Use `imageBuilder()` instead.
     */
    static readonly LINUX_ARM64_DOCKERFILE_PATH: string;
    /**
     * Create new image builder that builds Fargate specific runner images.
     *
     * You can customize the OS, architecture, VPC, subnet, security groups, etc. by passing in props.
     *
     * You can add components to the image builder by calling `imageBuilder.addComponent()`.
     *
     * The default OS is Ubuntu running on x64 architecture.
     *
     * Included components:
     *  * `RunnerImageComponent.requiredPackages()`
     *  * `RunnerImageComponent.runnerUser()`
     *  * `RunnerImageComponent.git()`
     *  * `RunnerImageComponent.githubCli()`
     *  * `RunnerImageComponent.awsCli()`
     *  * `RunnerImageComponent.githubRunner()`
     */
    static imageBuilder(scope: Construct, id: string, props?: RunnerImageBuilderProps): import("../image-builders").IConfigurableRunnerImageBuilder;
    /**
     * Cluster hosting the task hosting the runner.
     */
    readonly cluster: ecs.Cluster;
    /**
     * Fargate task hosting the runner.
     */
    readonly task: ecs.FargateTaskDefinition;
    /**
     * Container definition hosting the runner.
     */
    readonly container: ecs.ContainerDefinition;
    /**
     * Labels associated with this provider.
     */
    readonly labels: string[];
    /**
     * VPC used for hosting the runner task.
     */
    readonly vpc?: ec2.IVpc;
    /**
     * Subnets used for hosting the runner task.
     */
    readonly subnetSelection?: ec2.SubnetSelection;
    /**
     * Whether runner task will have a public IP.
     */
    readonly assignPublicIp: boolean;
    /**
     * Grant principal used to add permissions to the runner role.
     */
    readonly grantPrincipal: iam.IPrincipal;
    /**
     * The network connections associated with this resource.
     */
    readonly connections: ec2.Connections;
    /**
     * Use spot pricing for Fargate tasks.
     */
    readonly spot: boolean;
    /**
     * Docker image loaded with GitHub Actions Runner and its prerequisites. The image is built by an image builder and is specific to Fargate tasks.
     */
    readonly image: RunnerImage;
    /**
     * Log group where provided runners will save their logs.
     *
     * Note that this is not the job log, but the runner itself. It will not contain output from the GitHub Action but only metadata on its execution.
     */
    readonly logGroup: logs.ILogGroup;
    readonly retryableErrors: string[];
    private readonly group?;
    private readonly securityGroups;
    constructor(scope: Construct, id: string, props?: FargateRunnerProviderProps);
    /**
     * Generate step function task(s) to start a new runner.
     *
     * Called by GithubRunners and shouldn't be called manually.
     *
     * @param parameters workflow job details
     */
    getStepFunctionTask(parameters: RunnerRuntimeParameters): stepfunctions.IChainable;
    grantStateMachine(_: iam.IGrantable): void;
    status(statusFunctionRole: iam.IGrantable): IRunnerProviderStatus;
}
/**
 * @deprecated use {@link FargateRunnerProvider}
 */
export declare class FargateRunner extends FargateRunnerProvider {
}
