"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ec2Runner = exports.Ec2RunnerProvider = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_stepfunctions_1 = require("aws-cdk-lib/aws-stepfunctions");
const common_1 = require("./common");
const image_builders_1 = require("../image-builders");
const utils_1 = require("../utils");
// this script is specifically made so `poweroff` is absolutely always called
// each `{}` is a variable coming from `params` below
const linuxUserDataTemplate = `#!/bin/bash -x
TASK_TOKEN="{}"
logGroupName="{}"
runnerNamePath="{}"
githubDomainPath="{}"
ownerPath="{}"
repoPath="{}"
runnerTokenPath="{}"
labels="{}"
registrationURL="{}"
runnerGroup="{}"

heartbeat () {
  while true; do
    aws stepfunctions send-task-heartbeat --task-token "$TASK_TOKEN"
    sleep 60
  done
}
setup_logs () {
  cat <<EOF > /tmp/log.conf || exit 1
  {
    "logs": {
      "log_stream_name": "unknown",
      "logs_collected": {
        "files": {
          "collect_list": [
            {
              "file_path": "/var/log/runner.log",
              "log_group_name": "$logGroupName",
              "log_stream_name": "$runnerNamePath",
              "timezone": "UTC"
            }
          ]
        }
      }
    }
  }
EOF
  /opt/aws/amazon-cloudwatch-agent/bin/amazon-cloudwatch-agent-ctl -a fetch-config -m ec2 -s -c file:/tmp/log.conf || exit 2
}
action () {
  # Determine the value of RUNNER_FLAGS
  if [ "$(< /home/runner/RUNNER_VERSION)" = "latest" ]; then
    RUNNER_FLAGS=""
  else
    RUNNER_FLAGS="--disableupdate"
  fi

  labelsTemplate="$labels,cdkghr:started:$(date +%s)"

  # Execute the configuration command for runner registration
  sudo -Hu runner /home/runner/config.sh --unattended --url "$registrationURL" --token "$runnerTokenPath" --ephemeral --work _work --labels "$labelsTemplate" $RUNNER_FLAGS --name "$runnerNamePath" $runnerGroup || exit 1

  # Execute the run command
  sudo --preserve-env=AWS_REGION -Hu runner /home/runner/run.sh || exit 2

  # Retrieve the status
  STATUS=$(grep -Phors "finish job request for job [0-9a-f\\-]+ with result: \K.*" /home/runner/_diag/ | tail -n1)

  # Check and print the job status
  [ -n "$STATUS" ] && echo CDKGHA JOB DONE "$labels" "$STATUS"
}
heartbeat &
if setup_logs && action | tee /var/log/runner.log 2>&1; then
  aws stepfunctions send-task-success --task-token "$TASK_TOKEN" --task-output '{"ok": true}'
else
  aws stepfunctions send-task-failure --task-token "$TASK_TOKEN"
fi
sleep 10  # give cloudwatch agent its default 5 seconds buffer duration to upload logs
poweroff
`.replace(/{/g, '\\{').replace(/}/g, '\\}').replace(/\\{\\}/g, '{}');
// this script is specifically made so `poweroff` is absolutely always called
// each `{}` is a variable coming from `params` below and their order should match the linux script
const windowsUserDataTemplate = `<powershell>
$TASK_TOKEN = "{}"
$logGroupName="{}"
$runnerNamePath="{}"
$githubDomainPath="{}"
$ownerPath="{}"
$repoPath="{}"
$runnerTokenPath="{}"
$labels="{}"
$registrationURL="{}"
$runnerGroup="{}"

# EC2Launch only starts ssm agent after user data is done, so we need to start it ourselves (it is disabled by default)
Set-Service -StartupType Manual AmazonSSMAgent
Start-Service AmazonSSMAgent

Start-Job -ScriptBlock {
  while (1) {
    aws stepfunctions send-task-heartbeat --task-token "$using:TASK_TOKEN"
    sleep 60
  }
}
function setup_logs () {
  echo "{
    \`"logs\`": {
      \`"log_stream_name\`": \`"unknown\`",
      \`"logs_collected\`": {
        \`"files\`": {
         \`"collect_list\`": [
            {
              \`"file_path\`": \`"/actions/runner.log\`",
              \`"log_group_name\`": \`"$logGroupName\`",
              \`"log_stream_name\`": \`"$runnerNamePath\`",
              \`"timezone\`": \`"UTC\`"
            }
          ]
        }
      }
    }
  }" | Out-File -Encoding ASCII $Env:TEMP/log.conf
  & "C:/Program Files/Amazon/AmazonCloudWatchAgent/amazon-cloudwatch-agent-ctl.ps1" -a fetch-config -m ec2 -s -c file:$Env:TEMP/log.conf
}
function action () {
  cd /actions
  $RunnerVersion = Get-Content /actions/RUNNER_VERSION -Raw
  if ($RunnerVersion -eq "latest") { $RunnerFlags = "" } else { $RunnerFlags = "--disableupdate" }
  ./config.cmd --unattended --url "\${registrationUrl}" --token "\${runnerTokenPath}" --ephemeral --work _work --labels "\${labels},cdkghr:started:$(Get-Date -UFormat +%s)" $RunnerFlags --name "\${runnerNamePath}" \${runnerGroup} 2>&1 | Out-File -Encoding ASCII -Append /actions/runner.log

  if ($LASTEXITCODE -ne 0) { return 1 }
  ./run.cmd 2>&1 | Out-File -Encoding ASCII -Append /actions/runner.log
  if ($LASTEXITCODE -ne 0) { return 2 }

  $STATUS = Select-String -Path './_diag/*.log' -Pattern 'finish job request for job [0-9a-f\\-]+ with result: (.*)' | %{$_.Matches.Groups[1].Value} | Select-Object -Last 1

  if ($STATUS) {
      echo "CDKGHA JOB DONE \${labels} $STATUS" | Out-File -Encoding ASCII -Append /actions/runner.log
  }

  return 0
}
setup_logs
$r = action
if ($r -eq 0) {
  aws stepfunctions send-task-success --task-token "$TASK_TOKEN" --task-output '{ }'
} else {
  aws stepfunctions send-task-failure --task-token "$TASK_TOKEN"
}
Start-Sleep -Seconds 10  # give cloudwatch agent its default 5 seconds buffer duration to upload logs
Stop-Computer -ComputerName localhost -Force
</powershell>
`.replace(/{/g, '\\{').replace(/}/g, '\\}').replace(/\\{\\}/g, '{}');
/**
 * GitHub Actions runner provider using EC2 to execute jobs.
 *
 * This construct is not meant to be used by itself. It should be passed in the providers property for GitHubRunners.
 */
class Ec2RunnerProvider extends common_1.BaseProvider {
    /**
     * Create new image builder that builds EC2 specific runner images.
     *
     * You can customize the OS, architecture, VPC, subnet, security groups, etc. by passing in props.
     *
     * You can add components to the image builder by calling `imageBuilder.addComponent()`.
     *
     * The default OS is Ubuntu running on x64 architecture.
     *
     * Included components:
     *  * `RunnerImageComponent.requiredPackages()`
     *  * `RunnerImageComponent.cloudWatchAgent()`
     *  * `RunnerImageComponent.runnerUser()`
     *  * `RunnerImageComponent.git()`
     *  * `RunnerImageComponent.githubCli()`
     *  * `RunnerImageComponent.awsCli()`
     *  * `RunnerImageComponent.docker()`
     *  * `RunnerImageComponent.githubRunner()`
     */
    static imageBuilder(scope, id, props) {
        return image_builders_1.RunnerImageBuilder.new(scope, id, {
            os: common_1.Os.LINUX_UBUNTU,
            architecture: common_1.Architecture.X86_64,
            builderType: image_builders_1.RunnerImageBuilderType.AWS_IMAGE_BUILDER,
            components: [
                image_builders_1.RunnerImageComponent.requiredPackages(),
                image_builders_1.RunnerImageComponent.cloudWatchAgent(),
                image_builders_1.RunnerImageComponent.runnerUser(),
                image_builders_1.RunnerImageComponent.git(),
                image_builders_1.RunnerImageComponent.githubCli(),
                image_builders_1.RunnerImageComponent.awsCli(),
                image_builders_1.RunnerImageComponent.docker(),
                image_builders_1.RunnerImageComponent.githubRunner(props?.runnerVersion ?? common_1.RunnerVersion.latest()),
            ],
            ...props,
        });
    }
    constructor(scope, id, props) {
        super(scope, id, props);
        this.retryableErrors = [
            'Ec2.Ec2Exception',
            'States.Timeout',
        ];
        this.labels = props?.labels ?? ['ec2'];
        this.group = props?.group;
        this.vpc = props?.vpc ?? aws_cdk_lib_1.aws_ec2.Vpc.fromLookup(this, 'Default VPC', { isDefault: true });
        this.securityGroups = props?.securityGroup ? [props.securityGroup] : (props?.securityGroups ?? [new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, 'SG', { vpc: this.vpc })]);
        this.subnets = props?.subnet ? [props.subnet] : this.vpc.selectSubnets(props?.subnetSelection).subnets;
        this.instanceType = props?.instanceType ?? aws_cdk_lib_1.aws_ec2.InstanceType.of(aws_cdk_lib_1.aws_ec2.InstanceClass.M6I, aws_cdk_lib_1.aws_ec2.InstanceSize.LARGE);
        this.storageSize = props?.storageSize ?? cdk.Size.gibibytes(30); // 30 is the minimum for Windows
        this.storageOptions = props?.storageOptions;
        this.spot = props?.spot ?? false;
        this.spotMaxPrice = props?.spotMaxPrice;
        this.amiBuilder = props?.imageBuilder ?? props?.amiBuilder ?? Ec2RunnerProvider.imageBuilder(this, 'Ami Builder', {
            vpc: props?.vpc,
            subnetSelection: props?.subnetSelection,
            securityGroups: this.securityGroups,
        });
        this.ami = this.amiBuilder.bindAmi();
        if (this.amiBuilder instanceof image_builders_1.AwsImageBuilderRunnerImageBuilder) {
            if (this.amiBuilder.storageSize && this.storageSize.toBytes() < this.amiBuilder.storageSize.toBytes()) {
                throw new Error(`Runner storage size (${this.storageSize.toGibibytes()} GiB) must be at least the same as the image builder storage size (${this.amiBuilder.storageSize.toGibibytes()} GiB)`);
            }
        }
        if (!this.ami.architecture.instanceTypeMatch(this.instanceType)) {
            throw new Error(`AMI architecture (${this.ami.architecture.name}) doesn't match runner instance type (${this.instanceType} / ${this.instanceType.architecture})`);
        }
        this.grantPrincipal = this.role = new aws_cdk_lib_1.aws_iam.Role(this, 'Role', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        this.grantPrincipal.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['states:SendTaskFailure', 'states:SendTaskSuccess', 'states:SendTaskHeartbeat'],
            resources: ['*'], // no support for stateMachine.stateMachineArn :(
            conditions: {
                StringEquals: {
                    'aws:ResourceTag/aws:cloudformation:stack-id': cdk.Stack.of(this).stackId,
                },
            },
        }));
        this.grantPrincipal.addToPrincipalPolicy(utils_1.MINIMAL_EC2_SSM_SESSION_MANAGER_POLICY_STATEMENT);
        this.logGroup = new aws_cdk_lib_1.aws_logs.LogGroup(this, 'Logs', {
            retention: props?.logRetention ?? aws_logs_1.RetentionDays.ONE_MONTH,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        this.logGroup.grantWrite(this);
    }
    /**
     * Generate step function task(s) to start a new runner.
     *
     * Called by GithubRunners and shouldn't be called manually.
     *
     * @param parameters workflow job details
     */
    getStepFunctionTask(parameters) {
        // we need to build user data in two steps because passing the template as the first parameter to stepfunctions.JsonPath.format fails on syntax
        const params = [
            aws_cdk_lib_1.aws_stepfunctions.JsonPath.taskToken,
            this.logGroup.logGroupName,
            parameters.runnerNamePath,
            parameters.githubDomainPath,
            parameters.ownerPath,
            parameters.repoPath,
            parameters.runnerTokenPath,
            this.labels.join(','),
            parameters.registrationUrl,
            this.group ? `--runnergroup ${this.group}` : '',
        ];
        const passUserData = new aws_cdk_lib_1.aws_stepfunctions.Pass(this, `${this.labels.join(', ')} data`, {
            parameters: {
                userdataTemplate: this.ami.os.is(common_1.Os.WINDOWS) ? windowsUserDataTemplate : linuxUserDataTemplate,
            },
            resultPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.ec2'),
        });
        // we use ec2:RunInstances because we must
        // we can't use fleets because they don't let us override user data, security groups or even disk size
        // we can't use requestSpotInstances because it doesn't support launch templates, and it's deprecated
        // ec2:RunInstances also seemed like the only one to immediately return an error when spot capacity is not available
        // we build a complicated chain of states here because ec2:RunInstances can only try one subnet at a time
        // if someone can figure out a good way to use Map for this, please open a PR
        // build a state for each subnet we want to try
        const instanceProfile = new aws_cdk_lib_1.aws_iam.CfnInstanceProfile(this, 'Instance Profile', {
            roles: [this.role.roleName],
        });
        const rootDeviceResource = (0, common_1.amiRootDevice)(this, this.ami.launchTemplate.launchTemplateId);
        rootDeviceResource.node.addDependency(this.amiBuilder);
        const subnetRunners = this.subnets.map((subnet, index) => {
            return new aws_cdk_lib_1.aws_stepfunctions_tasks.CallAwsService(this, `${this.labels.join(', ')} subnet${index + 1}`, {
                comment: subnet.subnetId,
                integrationPattern: aws_stepfunctions_1.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
                service: 'ec2',
                action: 'runInstances',
                heartbeatTimeout: aws_cdk_lib_1.aws_stepfunctions.Timeout.duration(aws_cdk_lib_1.Duration.minutes(10)),
                parameters: {
                    LaunchTemplate: {
                        LaunchTemplateId: this.ami.launchTemplate.launchTemplateId,
                    },
                    MinCount: 1,
                    MaxCount: 1,
                    InstanceType: this.instanceType.toString(),
                    UserData: aws_cdk_lib_1.aws_stepfunctions.JsonPath.base64Encode(aws_cdk_lib_1.aws_stepfunctions.JsonPath.format(aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.ec2.userdataTemplate'), ...params)),
                    InstanceInitiatedShutdownBehavior: aws_cdk_lib_1.aws_ec2.InstanceInitiatedShutdownBehavior.TERMINATE,
                    IamInstanceProfile: {
                        Arn: instanceProfile.attrArn,
                    },
                    MetadataOptions: {
                        HttpTokens: 'required',
                    },
                    SecurityGroupIds: this.securityGroups.map(sg => sg.securityGroupId),
                    SubnetId: subnet.subnetId,
                    BlockDeviceMappings: [{
                            DeviceName: rootDeviceResource.ref,
                            Ebs: {
                                DeleteOnTermination: true,
                                VolumeSize: this.storageSize.toGibibytes(),
                                VolumeType: this.storageOptions?.volumeType,
                                Iops: this.storageOptions?.iops,
                                Throughput: this.storageOptions?.throughput,
                            },
                        }],
                    InstanceMarketOptions: this.spot ? {
                        MarketType: 'spot',
                        SpotOptions: {
                            MaxPrice: this.spotMaxPrice,
                            SpotInstanceType: 'one-time',
                        },
                    } : undefined,
                    TagSpecifications: [
                        {
                            ResourceType: 'instance',
                            Tags: [{
                                    Key: 'GitHubRunners:Provider',
                                    Value: this.node.path,
                                }],
                        },
                        {
                            ResourceType: 'volume',
                            Tags: [{
                                    Key: 'GitHubRunners:Provider',
                                    Value: this.node.path,
                                }],
                        },
                    ],
                },
                iamResources: ['*'],
            });
        });
        // start with the first subnet
        passUserData.next(subnetRunners[0]);
        // chain up the rest of the subnets
        for (let i = 1; i < subnetRunners.length; i++) {
            subnetRunners[i - 1].addCatch(subnetRunners[i], {
                errors: ['Ec2.Ec2Exception', 'States.Timeout'],
                resultPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.lastSubnetError'),
            });
        }
        return passUserData;
    }
    grantStateMachine(stateMachineRole) {
        stateMachineRole.grantPrincipal.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [this.role.roleArn],
            conditions: {
                StringEquals: {
                    'iam:PassedToService': 'ec2.amazonaws.com',
                },
            },
        }));
        stateMachineRole.grantPrincipal.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['ec2:createTags'],
            resources: [aws_cdk_lib_1.Stack.of(this).formatArn({
                    service: 'ec2',
                    resource: '*',
                })],
        }));
        stateMachineRole.grantPrincipal.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['iam:CreateServiceLinkedRole'],
            resources: ['*'],
            conditions: {
                StringEquals: {
                    'iam:AWSServiceName': 'spot.amazonaws.com',
                },
            },
        }));
    }
    status(statusFunctionRole) {
        statusFunctionRole.grantPrincipal.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['ec2:DescribeLaunchTemplateVersions'],
            resources: ['*'],
        }));
        return {
            type: this.constructor.name,
            labels: this.labels,
            securityGroups: this.securityGroups.map(sg => sg.securityGroupId),
            roleArn: this.role.roleArn,
            logGroup: this.logGroup.logGroupName,
            ami: {
                launchTemplate: this.ami.launchTemplate.launchTemplateId || 'unknown',
                amiBuilderLogGroup: this.ami.logGroup?.logGroupName,
            },
        };
    }
    /**
     * The network connections associated with this resource.
     */
    get connections() {
        return new aws_cdk_lib_1.aws_ec2.Connections({ securityGroups: this.securityGroups });
    }
}
exports.Ec2RunnerProvider = Ec2RunnerProvider;
_a = JSII_RTTI_SYMBOL_1;
Ec2RunnerProvider[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.Ec2RunnerProvider", version: "0.14.13" };
/**
 * @deprecated use {@link Ec2RunnerProvider}
 */
class Ec2Runner extends Ec2RunnerProvider {
}
exports.Ec2Runner = Ec2Runner;
_b = JSII_RTTI_SYMBOL_1;
Ec2Runner[_b] = { fqn: "@cloudsnorkel/cdk-github-runners.Ec2Runner", version: "0.14.13" };
//# sourceMappingURL=data:application/json;base64,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