"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseProvider = exports.Os = exports.Architecture = exports.RunnerVersion = void 0;
exports.amiRootDevice = amiRootDevice;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const ami_root_device_function_1 = require("./ami-root-device-function");
const utils_1 = require("../utils");
/**
 * Defines desired GitHub Actions runner version.
 */
class RunnerVersion {
    /**
     * Use the latest version available at the time the runner provider image is built.
     */
    static latest() {
        return new RunnerVersion('latest');
    }
    /**
     * Use a specific version.
     *
     * @see https://github.com/actions/runner/releases
     *
     * @param version GitHub Runner version
     */
    static specific(version) {
        return new RunnerVersion(version);
    }
    constructor(version) {
        this.version = version;
    }
    /**
     * Check if two versions are the same.
     *
     * @param other version to compare
     */
    is(other) {
        return this.version == other.version;
    }
}
exports.RunnerVersion = RunnerVersion;
_a = JSII_RTTI_SYMBOL_1;
RunnerVersion[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.RunnerVersion", version: "0.14.13" };
/**
 * CPU architecture enum for an image.
 */
class Architecture {
    static of(architecture) {
        return new Architecture(architecture);
    }
    constructor(name) {
        this.name = name;
    }
    /**
    * Checks if the given architecture is the same as this one.
    *
    * @param arch architecture to compare
    */
    is(arch) {
        return arch.name == this.name;
    }
    /**
     * Checks if this architecture is in a given list.
     *
     * @param arches architectures to check
     */
    isIn(arches) {
        for (const arch of arches) {
            if (this.is(arch)) {
                return true;
            }
        }
        return false;
    }
    /**
     * Checks if a given EC2 instance type matches this architecture.
     *
     * @param instanceType instance type to check
     */
    instanceTypeMatch(instanceType) {
        if (instanceType.architecture == aws_cdk_lib_1.aws_ec2.InstanceArchitecture.X86_64) {
            return this.is(Architecture.X86_64);
        }
        if (instanceType.architecture == aws_cdk_lib_1.aws_ec2.InstanceArchitecture.ARM_64) {
            return this.is(Architecture.ARM64);
        }
        throw new Error('Unknown instance type architecture');
    }
}
exports.Architecture = Architecture;
_b = JSII_RTTI_SYMBOL_1;
Architecture[_b] = { fqn: "@cloudsnorkel/cdk-github-runners.Architecture", version: "0.14.13" };
/**
 * ARM64
 */
Architecture.ARM64 = Architecture.of('ARM64');
/**
 * X86_64
 */
Architecture.X86_64 = Architecture.of('X86_64');
/**
 * OS enum for an image.
 */
class Os {
    static of(os) {
        return new Os(os);
    }
    constructor(name) {
        this.name = name;
    }
    /**
    * Checks if the given OS is the same as this one.
    *
    * @param os OS to compare
    */
    is(os) {
        return os.name == this.name;
    }
    /**
     * Checks if this OS is in a given list.
     *
     * @param oses list of OS to check
     */
    isIn(oses) {
        for (const os of oses) {
            if (this.is(os)) {
                return true;
            }
        }
        return false;
    }
}
exports.Os = Os;
_c = JSII_RTTI_SYMBOL_1;
Os[_c] = { fqn: "@cloudsnorkel/cdk-github-runners.Os", version: "0.14.13" };
/**
* Linux
*
* @deprecated use {@link LINUX_UBUNTU}, {@link LINUX_UBUNTU_2404}, {@link LINUX_AMAZON_2} or {@link LINUX_AMAZON_2023}
*/
Os.LINUX = Os.of('Linux');
/**
 * Ubuntu Linux
 */
Os.LINUX_UBUNTU = Os.of('Ubuntu Linux');
/**
* Ubuntu Linux 22.04
*/
Os.LINUX_UBUNTU_2204 = Os.of('Ubuntu Linux 22.04');
/**
 * Ubuntu Linux 24.04
 */
Os.LINUX_UBUNTU_2404 = Os.of('Ubuntu Linux 24.04');
/**
 * Amazon Linux 2
 */
Os.LINUX_AMAZON_2 = Os.of('Amazon Linux 2');
/**
 * Amazon Linux 2023
 */
Os.LINUX_AMAZON_2023 = Os.of('Amazon Linux 2023');
/**
 * @internal
 */
Os._ALL_LINUX_VERSIONS = [Os.LINUX, Os.LINUX_UBUNTU, Os.LINUX_UBUNTU_2204, Os.LINUX_UBUNTU_2404, Os.LINUX_AMAZON_2, Os.LINUX_AMAZON_2023];
/**
   * @internal
   */
Os._ALL_LINUX_AMAZON_VERSIONS = [Os.LINUX_AMAZON_2, Os.LINUX_AMAZON_2023];
/**
   * @internal
   */
Os._ALL_LINUX_UBUNTU_VERSIONS = [Os.LINUX_UBUNTU, Os.LINUX_UBUNTU_2204, Os.LINUX_UBUNTU_2404];
/**
* Windows
*/
Os.WINDOWS = Os.of('Windows');
/**
 * Base class for all providers with common methods used by all providers.
 *
 * @internal
 */
class BaseProvider extends constructs_1.Construct {
    constructor(scope, id, _props) {
        super(scope, id);
        cdk.Tags.of(this).add('GitHubRunners:Provider', this.node.path);
    }
    labelsFromProperties(defaultLabel, propsLabel, propsLabels) {
        if (propsLabels && propsLabel) {
            throw new Error('Must supply either `label` or `labels` in runner properties, but not both. Try removing the `label` property.');
        }
        if (propsLabels) {
            return propsLabels;
        }
        if (propsLabel) {
            return [propsLabel];
        }
        return [defaultLabel];
    }
}
exports.BaseProvider = BaseProvider;
/**
 * Use custom resource to determine the root device name of a given AMI, Launch Template, or SSM parameter pointing to AMI.
 *
 * TODO move somewhere more common as it's used by both providers and AMI builder now
 *
 * @internal
 */
function amiRootDevice(scope, ami) {
    const crHandler = (0, utils_1.singletonLambda)(ami_root_device_function_1.AmiRootDeviceFunction, scope, 'AMI Root Device Reader', {
        description: 'Custom resource handler that discovers the boot drive device name for a given AMI',
        timeout: cdk.Duration.minutes(1),
        logGroup: (0, utils_1.singletonLogGroup)(scope, utils_1.SingletonLogType.RUNNER_IMAGE_BUILD),
        loggingFormat: aws_cdk_lib_1.aws_lambda.LoggingFormat.JSON,
        initialPolicy: [
            new aws_cdk_lib_1.aws_iam.PolicyStatement({
                actions: [
                    'ssm:GetParameter',
                    'ec2:DescribeImages',
                    'ec2:DescribeLaunchTemplateVersions',
                    'imagebuilder:GetImage',
                ],
                resources: ['*'],
            }),
        ],
    });
    return new aws_cdk_lib_1.CustomResource(scope, 'AMI Root Device', {
        serviceToken: crHandler.functionArn,
        resourceType: 'Custom::AmiRootDevice',
        properties: {
            Ami: ami ?? '',
        },
    });
}
//# sourceMappingURL=data:application/json;base64,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