"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RunnerImageComponent = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_image_builder_1 = require("./aws-image-builder");
const providers_1 = require("../providers");
/**
 * Components are used to build runner images. They can run commands in the image, copy files into the image, and run some Docker commands.
 */
class RunnerImageComponent {
    /**
     * Define a custom component that can run commands in the image, copy files into the image, and run some Docker commands.
     *
     * The order of operations is (1) assets (2) commands (3) docker commands.
     *
     * Use this to customize the image for the runner.
     *
     * **WARNING:** Docker commands are not guaranteed to be included before the next component
     */
    static custom(props) {
        return new class extends RunnerImageComponent {
            get name() {
                if (props.name && !props.name.match(/[a-zA-Z0-9\-]/)) {
                    throw new Error(`Invalid component name: ${props.name}. Name must only contain alphanumeric characters and dashes.`);
                }
                return `Custom-${props.name ?? 'Undefined'}`;
            }
            getCommands(_os, _architecture) {
                return props.commands ?? [];
            }
            getAssets(_os, _architecture) {
                return props.assets ?? [];
            }
            getDockerCommands(_os, _architecture) {
                return props.dockerCommands ?? [];
            }
        }();
    }
    /**
     * A component to install the required packages for the runner.
     */
    static requiredPackages() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'RequiredPackages';
            }
            getCommands(os, _architecture) {
                if (os.isIn(providers_1.Os._ALL_LINUX_UBUNTU_VERSIONS)) {
                    return [
                        'apt-get update',
                        'DEBIAN_FRONTEND=noninteractive apt-get upgrade -y',
                        'DEBIAN_FRONTEND=noninteractive apt-get install -y curl sudo jq bash zip unzip iptables software-properties-common ca-certificates',
                    ];
                }
                else if (os.is(providers_1.Os.LINUX_AMAZON_2)) {
                    return [
                        'yum update -y',
                        'yum install -y jq tar gzip bzip2 which binutils zip unzip sudo shadow-utils',
                    ];
                }
                else if (os.is(providers_1.Os.LINUX_AMAZON_2023)) {
                    return [
                        'dnf upgrade -y',
                        'dnf install -y jq tar gzip bzip2 which binutils zip unzip sudo shadow-utils findutils',
                    ];
                }
                else if (os.is(providers_1.Os.WINDOWS)) {
                    return [];
                }
                throw new Error(`Unsupported OS for required packages: ${os.name}`);
            }
        };
    }
    /**
     * A component to install CloudWatch Agent for the runner so we can send logs.
     */
    static cloudWatchAgent() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'CloudWatchAgent';
            }
            getCommands(os, architecture) {
                if (os.isIn(providers_1.Os._ALL_LINUX_UBUNTU_VERSIONS)) {
                    let archUrl;
                    if (architecture.is(providers_1.Architecture.X86_64)) {
                        archUrl = 'amd64';
                    }
                    else if (architecture.is(providers_1.Architecture.ARM64)) {
                        archUrl = 'arm64';
                    }
                    else {
                        throw new Error(`Unsupported architecture for required packages: ${architecture.name}`);
                    }
                    return [
                        `curl -sfLo /tmp/amazon-cloudwatch-agent.deb https://s3.amazonaws.com/amazoncloudwatch-agent/ubuntu/${archUrl}/latest/amazon-cloudwatch-agent.deb`,
                        'dpkg -i -E /tmp/amazon-cloudwatch-agent.deb',
                        'rm /tmp/amazon-cloudwatch-agent.deb',
                    ];
                }
                else if (os.is(providers_1.Os.LINUX_AMAZON_2)) {
                    return [
                        'yum install -y amazon-cloudwatch-agent',
                    ];
                }
                else if (os.is(providers_1.Os.LINUX_AMAZON_2023)) {
                    return [
                        'dnf install -y amazon-cloudwatch-agent',
                    ];
                }
                else if (os.is(providers_1.Os.WINDOWS)) {
                    return [
                        '$p = Start-Process msiexec.exe -PassThru -Wait -ArgumentList \'/i https://s3.amazonaws.com/amazoncloudwatch-agent/windows/amd64/latest/amazon-cloudwatch-agent.msi /qn\'',
                        'if ($p.ExitCode -ne 0) { throw "Exit code is $p.ExitCode" }',
                    ];
                }
                throw new Error(`Unsupported OS for required packages: ${os.name}`);
            }
        };
    }
    /**
     * A component to prepare the required runner user.
     */
    static runnerUser() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'RunnerUser';
            }
            getCommands(os, _architecture) {
                if (os.isIn(providers_1.Os._ALL_LINUX_UBUNTU_VERSIONS)) {
                    return [
                        'addgroup runner',
                        'adduser --system --disabled-password --home /home/runner --ingroup runner runner',
                        'echo "%runner   ALL=(ALL:ALL) NOPASSWD: ALL" > /etc/sudoers.d/runner',
                    ];
                }
                else if (os.isIn(providers_1.Os._ALL_LINUX_AMAZON_VERSIONS)) {
                    return [
                        '/usr/sbin/groupadd runner',
                        '/usr/sbin/useradd --system --shell /usr/sbin/nologin --home-dir /home/runner --gid runner runner',
                        'mkdir -p /home/runner',
                        'chown runner /home/runner',
                        'echo "%runner   ALL=(ALL:ALL) NOPASSWD: ALL" > /etc/sudoers.d/runner',
                    ];
                }
                else if (os.is(providers_1.Os.WINDOWS)) {
                    return [];
                }
                throw new Error(`Unsupported OS for runner user: ${os.name}`);
            }
        };
    }
    /**
     * A component to install the AWS CLI.
     */
    static awsCli() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'AwsCli';
            }
            getCommands(os, architecture) {
                if (os.isIn(providers_1.Os._ALL_LINUX_UBUNTU_VERSIONS) || os.isIn(providers_1.Os._ALL_LINUX_AMAZON_VERSIONS)) {
                    let archUrl;
                    if (architecture.is(providers_1.Architecture.X86_64)) {
                        archUrl = 'x86_64';
                    }
                    else if (architecture.is(providers_1.Architecture.ARM64)) {
                        archUrl = 'aarch64';
                    }
                    else {
                        throw new Error(`Unsupported architecture for awscli: ${architecture.name}`);
                    }
                    return [
                        `curl -fsSL "https://awscli.amazonaws.com/awscli-exe-linux-${archUrl}.zip" -o awscliv2.zip`,
                        'unzip -q awscliv2.zip',
                        './aws/install',
                        'rm -rf awscliv2.zip aws',
                    ];
                }
                else if (os.is(providers_1.Os.WINDOWS)) {
                    return [
                        '$p = Start-Process msiexec.exe -PassThru -Wait -ArgumentList \'/i https://awscli.amazonaws.com/AWSCLIV2.msi /qn\'',
                        'if ($p.ExitCode -ne 0) { throw "Exit code is $p.ExitCode" }',
                    ];
                }
                throw new Error(`Unknown os/architecture combo for awscli: ${os.name}/${architecture.name}`);
            }
        }();
    }
    /**
     * A component to install the GitHub CLI.
     */
    static githubCli() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'GithubCli';
            }
            getCommands(os, architecture) {
                if (os.isIn(providers_1.Os._ALL_LINUX_UBUNTU_VERSIONS)) {
                    return [
                        'curl -fsSL https://cli.github.com/packages/githubcli-archive-keyring.gpg | sudo dd of=/usr/share/keyrings/githubcli-archive-keyring.gpg',
                        'echo "deb [arch=$(dpkg --print-architecture) signed-by=/usr/share/keyrings/githubcli-archive-keyring.gpg] ' +
                            '  https://cli.github.com/packages stable main" | sudo tee /etc/apt/sources.list.d/github-cli.list > /dev/null',
                        'apt-get update',
                        'DEBIAN_FRONTEND=noninteractive apt-get install -y gh',
                    ];
                }
                else if (os.is(providers_1.Os.LINUX_AMAZON_2)) {
                    return [
                        'curl -fsSSL https://cli.github.com/packages/rpm/gh-cli.repo -o /etc/yum.repos.d/gh-cli.repo',
                        'yum install -y gh',
                    ];
                }
                else if (os.is(providers_1.Os.LINUX_AMAZON_2023)) {
                    return [
                        'curl -fsSSL https://cli.github.com/packages/rpm/gh-cli.repo -o /etc/yum.repos.d/gh-cli.repo',
                        'dnf install -y gh',
                    ];
                }
                else if (os.is(providers_1.Os.WINDOWS)) {
                    return [
                        'cmd /c curl -w "%{redirect_url}" -fsS https://github.com/cli/cli/releases/latest > $Env:TEMP\\latest-gh',
                        '$LatestUrl = Get-Content $Env:TEMP\\latest-gh',
                        '$GH_VERSION = ($LatestUrl -Split \'/\')[-1].substring(1)',
                        'Invoke-WebRequest -UseBasicParsing -Uri "https://github.com/cli/cli/releases/download/v${GH_VERSION}/gh_${GH_VERSION}_windows_amd64.msi" -OutFile gh.msi',
                        '$p = Start-Process msiexec.exe -PassThru -Wait -ArgumentList \'/i gh.msi /qn\'',
                        'if ($p.ExitCode -ne 0) { throw "Exit code is $p.ExitCode" }',
                        'del gh.msi',
                    ];
                }
                throw new Error(`Unknown os/architecture combo for github cli: ${os.name}/${architecture.name}`);
            }
        }();
    }
    /**
     * A component to install the GitHub CLI.
     */
    static git() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'Git';
            }
            getCommands(os, architecture) {
                if (os.isIn(providers_1.Os._ALL_LINUX_UBUNTU_VERSIONS)) {
                    return [
                        'add-apt-repository ppa:git-core/ppa',
                        'apt-get update',
                        'DEBIAN_FRONTEND=noninteractive apt-get install -y git',
                    ];
                }
                else if (os.is(providers_1.Os.LINUX_AMAZON_2)) {
                    return [
                        'yum install -y git',
                    ];
                }
                else if (os.is(providers_1.Os.LINUX_AMAZON_2023)) {
                    return [
                        'dnf install -y git',
                    ];
                }
                else if (os.is(providers_1.Os.WINDOWS)) {
                    return [
                        'cmd /c curl -w "%{redirect_url}" -fsS https://github.com/git-for-windows/git/releases/latest > $Env:TEMP\\latest-git',
                        '$LatestUrl = Get-Content $Env:TEMP\\latest-git',
                        '$GIT_VERSION = ($LatestUrl -Split \'/\')[-1].substring(1)',
                        '$GIT_VERSION_SHORT = ($GIT_VERSION -Split \'.windows.\')[0]',
                        '$GIT_REVISION = ($GIT_VERSION -Split \'.windows.\')[1]',
                        'If ($GIT_REVISION -gt 1) {$GIT_VERSION_SHORT = "$GIT_VERSION_SHORT.$GIT_REVISION"}',
                        'Invoke-WebRequest -UseBasicParsing -Uri https://github.com/git-for-windows/git/releases/download/v${GIT_VERSION}/Git-${GIT_VERSION_SHORT}-64-bit.exe -OutFile git-setup.exe',
                        '$p = Start-Process git-setup.exe -PassThru -Wait -ArgumentList \'/VERYSILENT\'',
                        'if ($p.ExitCode -ne 0) { throw "Exit code is $p.ExitCode" }',
                        'del git-setup.exe',
                    ];
                }
                throw new Error(`Unknown os/architecture combo for git: ${os.name}/${architecture.name}`);
            }
        }();
    }
    /**
     * A component to install the GitHub Actions Runner. This is the actual executable that connects to GitHub to ask for jobs and then execute them.
     *
     * @param runnerVersion The version of the runner to install. Usually you would set this to latest.
     */
    static githubRunner(runnerVersion) {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'GithubRunner';
            }
            getCommands(os, architecture) {
                if (os.isIn(providers_1.Os._ALL_LINUX_UBUNTU_VERSIONS) || os.isIn(providers_1.Os._ALL_LINUX_AMAZON_VERSIONS)) {
                    let versionCommand;
                    if (runnerVersion.is(providers_1.RunnerVersion.latest())) {
                        versionCommand = 'RUNNER_VERSION=`curl -w "%{redirect_url}" -fsS https://github.com/actions/runner/releases/latest | grep -oE "[^/v]+$"`';
                    }
                    else {
                        versionCommand = `RUNNER_VERSION='${runnerVersion.version}'`;
                    }
                    let archUrl;
                    if (architecture.is(providers_1.Architecture.X86_64)) {
                        archUrl = 'x64';
                    }
                    else if (architecture.is(providers_1.Architecture.ARM64)) {
                        archUrl = 'arm64';
                    }
                    else {
                        throw new Error(`Unsupported architecture for GitHub Runner: ${architecture.name}`);
                    }
                    let commands = [
                        versionCommand,
                        `curl -fsSLO "https://github.com/actions/runner/releases/download/v\${RUNNER_VERSION}/actions-runner-linux-${archUrl}-\${RUNNER_VERSION}.tar.gz"`,
                        `tar -C /home/runner -xzf "actions-runner-linux-${archUrl}-\${RUNNER_VERSION}.tar.gz"`,
                        `rm actions-runner-linux-${archUrl}-\${RUNNER_VERSION}.tar.gz`,
                        `echo -n ${runnerVersion.version} > /home/runner/RUNNER_VERSION`,
                    ];
                    if (os.isIn(providers_1.Os._ALL_LINUX_UBUNTU_VERSIONS)) {
                        commands.push('/home/runner/bin/installdependencies.sh');
                    }
                    else if (os.is(providers_1.Os.LINUX_AMAZON_2)) {
                        commands.push('yum install -y openssl-libs krb5-libs zlib libicu60');
                    }
                    else if (os.is(providers_1.Os.LINUX_AMAZON_2023)) {
                        commands.push('dnf install -y openssl-libs krb5-libs zlib libicu-67.1');
                    }
                    commands.push('mkdir -p /opt/hostedtoolcache', 'chown runner /opt/hostedtoolcache');
                    return commands;
                }
                else if (os.is(providers_1.Os.WINDOWS)) {
                    let runnerCommands;
                    if (runnerVersion.is(providers_1.RunnerVersion.latest())) {
                        runnerCommands = [
                            'cmd /c curl -w "%{redirect_url}" -fsS https://github.com/actions/runner/releases/latest > $Env:TEMP\\latest-gha',
                            '$LatestUrl = Get-Content $Env:TEMP\\latest-gha',
                            '$RUNNER_VERSION = ($LatestUrl -Split \'/\')[-1].substring(1)',
                        ];
                    }
                    else {
                        runnerCommands = [`$RUNNER_VERSION = '${runnerVersion.version}'`];
                    }
                    runnerCommands = runnerCommands.concat([
                        // create directories
                        'mkdir C:\\hostedtoolcache\\windows',
                        'mkdir C:\\tools',
                        // download zstd and extract to C:\tools
                        'cmd /c curl -w "%{redirect_url}" -fsS https://github.com/facebook/zstd/releases/latest > $Env:TEMP\\latest-zstd',
                        '$LatestUrl = Get-Content $Env:TEMP\\latest-zstd',
                        '$ZSTD_VERSION = ($LatestUrl -Split \'/\')[-1].substring(1)',
                        'Invoke-WebRequest -UseBasicParsing -Uri "https://github.com/facebook/zstd/releases/download/v$ZSTD_VERSION/zstd-v$ZSTD_VERSION-win64.zip" -OutFile zstd.zip',
                        'Expand-Archive zstd.zip -DestinationPath C:\\tools',
                        'Move-Item -Path C:\\tools\\zstd-v$ZSTD_VERSION-win64\\zstd.exe C:\\tools',
                        'Remove-Item -LiteralPath "C:\\tools\\zstd-v$ZSTD_VERSION-win64" -Force -Recurse',
                        'del zstd.zip',
                        // add C:\tools to PATH
                        '$persistedPaths = [Environment]::GetEnvironmentVariable(\'Path\', [EnvironmentVariableTarget]::Machine)',
                        '[Environment]::SetEnvironmentVariable("PATH", $persistedPaths + ";C:\\tools", [EnvironmentVariableTarget]::Machine)',
                    ]);
                    return runnerCommands.concat([
                        'Invoke-WebRequest -UseBasicParsing -Uri "https://github.com/actions/runner/releases/download/v${RUNNER_VERSION}/actions-runner-win-x64-${RUNNER_VERSION}.zip" -OutFile actions.zip',
                        'Expand-Archive actions.zip -DestinationPath C:\\actions',
                        'del actions.zip',
                        `echo ${runnerVersion.version} | Out-File -Encoding ASCII -NoNewline C:\\actions\\RUNNER_VERSION`,
                    ]);
                }
                throw new Error(`Unknown os/architecture combo for github runner: ${os.name}/${architecture.name}`);
            }
            getDockerCommands(_os, _architecture) {
                return [
                    `ENV RUNNER_VERSION=${runnerVersion.version}`,
                ];
            }
        }();
    }
    /**
     * A component to install Docker.
     *
     * On Windows this sets up dockerd for Windows containers without Docker Desktop. If you need Linux containers on Windows, you'll need to install Docker Desktop which doesn't seem to play well with servers (PRs welcome).
     */
    static docker() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'Docker';
            }
            getCommands(os, architecture) {
                if (os.isIn(providers_1.Os._ALL_LINUX_UBUNTU_VERSIONS)) {
                    return [
                        'curl -fsSL https://download.docker.com/linux/ubuntu/gpg | sudo gpg --dearmor -o /usr/share/keyrings/docker.gpg',
                        'echo ' +
                            '  "deb [arch=$(dpkg --print-architecture) signed-by=/usr/share/keyrings/docker.gpg] https://download.docker.com/linux/ubuntu ' +
                            '  $(lsb_release -cs) stable" | sudo tee /etc/apt/sources.list.d/docker.list > /dev/null',
                        'apt-get update',
                        'DEBIAN_FRONTEND=noninteractive apt-get install -y docker-ce docker-ce-cli containerd.io docker-compose-plugin',
                        'usermod -aG docker runner',
                        'ln -s /usr/libexec/docker/cli-plugins/docker-compose /usr/bin/docker-compose',
                    ];
                }
                else if (os.is(providers_1.Os.LINUX_AMAZON_2)) {
                    return [
                        'amazon-linux-extras install docker',
                        'usermod -a -G docker runner',
                        'curl -sfLo /usr/bin/docker-compose https://github.com/docker/compose/releases/latest/download/docker-compose-$(uname -s | tr \'[:upper:]\' \'[:lower:]\')-$(uname -m)',
                        'chmod +x /usr/bin/docker-compose',
                        'ln -s /usr/bin/docker-compose /usr/libexec/docker/cli-plugins/docker-compose',
                    ];
                }
                else if (os.is(providers_1.Os.LINUX_AMAZON_2023)) {
                    return [
                        'dnf install -y docker',
                        'usermod -a -G docker runner',
                        'curl -sfLo /usr/bin/docker-compose https://github.com/docker/compose/releases/latest/download/docker-compose-$(uname -s | tr \'[:upper:]\' \'[:lower:]\')-$(uname -m)',
                        'chmod +x /usr/bin/docker-compose',
                        'ln -s /usr/bin/docker-compose /usr/libexec/docker/cli-plugins/docker-compose',
                    ];
                }
                else if (os.is(providers_1.Os.WINDOWS)) {
                    return [
                        // figure out latest docker version
                        'cmd /c curl -w "%{redirect_url}" -fsS https://github.com/moby/moby/releases/latest > $Env:TEMP\\latest-docker',
                        '$LatestUrl = Get-Content $Env:TEMP\\latest-docker',
                        '$DOCKER_VERSION = ($LatestUrl -Split \'/\')[-1].substring(1)',
                        // download static binaries
                        'Invoke-WebRequest -UseBasicParsing -Uri "https://download.docker.com/win/static/stable/x86_64/docker-${DOCKER_VERSION}.zip" -OutFile docker.zip',
                        // extract to C:\Program Files\Docker
                        'Expand-Archive docker.zip -DestinationPath "$Env:ProgramFiles"',
                        'del docker.zip',
                        // add to path
                        '$persistedPaths = [Environment]::GetEnvironmentVariable(\'Path\', [EnvironmentVariableTarget]::Machine)',
                        '[Environment]::SetEnvironmentVariable("PATH", $persistedPaths + ";$Env:ProgramFiles\\Docker", [EnvironmentVariableTarget]::Machine)',
                        '$env:PATH = $env:PATH + ";$Env:ProgramFiles\\Docker"',
                        // register docker service
                        'dockerd --register-service',
                        'if ($LASTEXITCODE -ne 0) { throw "Exit code is $LASTEXITCODE" }',
                        // enable containers feature
                        'Enable-WindowsOptionalFeature -Online -FeatureName containers -All -NoRestart',
                        // install docker-compose
                        'cmd /c curl -w "%{redirect_url}" -fsS https://github.com/docker/compose/releases/latest > $Env:TEMP\\latest-docker-compose',
                        '$LatestUrl = Get-Content $Env:TEMP\\latest-docker-compose',
                        '$LatestDockerCompose = ($LatestUrl -Split \'/\')[-1]',
                        'Invoke-WebRequest -UseBasicParsing -Uri  "https://github.com/docker/compose/releases/download/${LatestDockerCompose}/docker-compose-Windows-x86_64.exe" -OutFile $Env:ProgramFiles\\Docker\\docker-compose.exe',
                        'New-Item -ItemType directory -Path "$Env:ProgramFiles\\Docker\\cli-plugins"',
                        'Copy-Item -Path "$Env:ProgramFiles\\Docker\\docker-compose.exe" -Destination "$Env:ProgramFiles\\Docker\\cli-plugins\\docker-compose.exe"',
                    ];
                }
                throw new Error(`Unknown os/architecture combo for docker: ${os.name}/${architecture.name}`);
            }
            shouldReboot(os, _architecture) {
                return os.is(providers_1.Os.WINDOWS);
            }
        }();
    }
    /**
     * A component to install Docker-in-Docker.
     *
     * @deprecated use `docker()`
     */
    static dockerInDocker() {
        return RunnerImageComponent.docker();
    }
    /**
     * A component to add a trusted certificate authority. This can be used to support GitHub Enterprise Server with self-signed certificate.
     *
     * @param source path to certificate file in PEM format
     * @param name unique certificate name to be used on runner file system
     */
    static extraCertificates(source, name) {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = `Extra-Certificates-${name}`;
            }
            getCommands(os, architecture) {
                if (!name.match(/^[a-zA-Z0-9_-]+$/)) {
                    throw new Error(`Invalid certificate name: ${name}. Name must only contain alphanumeric characters, dashes and underscores.`);
                }
                if (os.isIn(providers_1.Os._ALL_LINUX_UBUNTU_VERSIONS)) {
                    return [
                        'update-ca-certificates',
                    ];
                }
                else if (os.isIn(providers_1.Os._ALL_LINUX_AMAZON_VERSIONS)) {
                    return [
                        'update-ca-trust',
                    ];
                }
                else if (os.is(providers_1.Os.WINDOWS)) {
                    return [
                        `Import-Certificate -FilePath C:\\${name}.crt -CertStoreLocation Cert:\\LocalMachine\\Root`,
                        `Remove-Item C:\\${name}.crt`,
                    ];
                }
                throw new Error(`Unknown os/architecture combo for extra certificates: ${os.name}/${architecture.name}`);
            }
            getAssets(os, _architecture) {
                if (os.isIn(providers_1.Os._ALL_LINUX_UBUNTU_VERSIONS)) {
                    return [
                        { source, target: `/usr/local/share/ca-certificates/${name}.crt` },
                    ];
                }
                else if (os.isIn(providers_1.Os._ALL_LINUX_AMAZON_VERSIONS)) {
                    return [
                        { source, target: `/etc/pki/ca-trust/source/anchors/${name}.crt` },
                    ];
                }
                else if (os.is(providers_1.Os.WINDOWS)) {
                    return [
                        { source, target: `C:\\${name}.crt` },
                    ];
                }
                throw new Error(`Unsupported OS for extra certificates: ${os.name}`);
            }
        }();
    }
    /**
     * A component to set up the required Lambda entrypoint for Lambda runners.
     */
    static lambdaEntrypoint() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'Lambda-Entrypoint';
            }
            getCommands(os, _architecture) {
                if (!os.isIn(providers_1.Os._ALL_LINUX_VERSIONS)) {
                    throw new Error(`Unsupported OS for Lambda entrypoint: ${os.name}`);
                }
                return [];
            }
            getAssets(_os, _architecture) {
                return [
                    {
                        source: path.join(__dirname, '..', '..', 'assets', 'providers', 'lambda-bootstrap.sh'),
                        target: '/bootstrap.sh',
                    },
                    {
                        source: path.join(__dirname, '..', '..', 'assets', 'providers', 'lambda-runner.sh'),
                        target: '/runner.sh',
                    },
                ];
            }
            getDockerCommands(_os, _architecture) {
                return [
                    'LABEL DISABLE_SOCI=1', // hacky way to disable soci v2 indexing on lambda as lambda will fail to start with an index
                    'ENTRYPOINT ["bash", "/bootstrap.sh"]',
                ];
            }
        };
    }
    /**
     * A component to add environment variables for jobs the runner executes.
     *
     * These variables only affect the jobs ran by the runner. They are not global. They do not affect other components.
     *
     * It is not recommended to use this component to pass secrets. Instead, use GitHub Secrets or AWS Secrets Manager.
     *
     * Must be used after the {@link githubRunner} component.
     */
    static environmentVariables(vars) {
        Object.entries(vars).forEach(e => {
            if (e[0].includes('\n') || e[1].includes('\n')) {
                throw new Error(`Environment variable cannot contain newlines: ${e}`);
            }
        });
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'EnvironmentVariables';
            }
            getCommands(os, _architecture) {
                if (os.isIn(providers_1.Os._ALL_LINUX_VERSIONS)) {
                    return Object.entries(vars).map(e => `echo '${e[0]}=${e[1].replace(/'/g, "'\"'\"'")}' >> /home/runner/.env`);
                }
                else if (os.is(providers_1.Os.WINDOWS)) {
                    return Object.entries(vars).map(e => `Add-Content -Path C:\\actions\\.env -Value '${e[0]}=${e[1].replace(/'/g, "''")}'`);
                }
                else {
                    throw new Error(`Unsupported OS for environment variables component: ${os.name}`);
                }
            }
        };
    }
    /**
     * Returns assets to copy into the built image. Can be used to copy files into the image.
     */
    getAssets(_os, _architecture) {
        return [];
    }
    /**
     * Returns Docker commands to run to in built image. Can be used to add commands like `VOLUME`, `ENTRYPOINT`, `CMD`, etc.
     *
     * Docker commands are added after assets and normal commands.
     */
    getDockerCommands(_os, _architecture) {
        return [];
    }
    /**
     * Returns true if the image builder should be rebooted after this component is installed.
     */
    shouldReboot(_os, _architecture) {
        return false;
    }
    /**
     * Convert component to an AWS Image Builder component.
     *
     * @internal
     */
    _asAwsImageBuilderComponent(scope, id, os, architecture) {
        let platform;
        if (os.isIn(providers_1.Os._ALL_LINUX_UBUNTU_VERSIONS) || os.isIn(providers_1.Os._ALL_LINUX_AMAZON_VERSIONS)) {
            platform = 'Linux';
        }
        else if (os.is(providers_1.Os.WINDOWS)) {
            platform = 'Windows';
        }
        else {
            throw new Error(`Unknown os/architecture combo for image builder component: ${os.name}/${architecture.name}`);
        }
        return new aws_image_builder_1.ImageBuilderComponent(scope, id, {
            platform: platform,
            commands: this.getCommands(os, architecture),
            assets: this.getAssets(os, architecture).map((asset, index) => {
                return {
                    asset: new aws_cdk_lib_1.aws_s3_assets.Asset(scope, `${id} asset ${index}`, { path: asset.source }),
                    path: asset.target,
                };
            }),
            displayName: id,
            description: id,
            reboot: this.shouldReboot(os, architecture),
        });
    }
}
exports.RunnerImageComponent = RunnerImageComponent;
_a = JSII_RTTI_SYMBOL_1;
RunnerImageComponent[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.RunnerImageComponent", version: "0.14.13" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29tcG9uZW50cy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uL3NyYy9pbWFnZS1idWlsZGVycy9jb21wb25lbnRzLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsNkJBQTZCO0FBQzdCLDZDQUF5RDtBQUV6RCwyREFBNEQ7QUFFNUQsNENBQStEO0FBOEIvRDs7R0FFRztBQUNILE1BQXNCLG9CQUFvQjtJQUN4Qzs7Ozs7Ozs7T0FRRztJQUNILE1BQU0sQ0FBQyxNQUFNLENBQUMsS0FBc0M7UUFDbEQsT0FBTyxJQUFJLEtBQU0sU0FBUSxvQkFBb0I7WUFDM0MsSUFBSSxJQUFJO2dCQUNOLElBQUksS0FBSyxDQUFDLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLGVBQWUsQ0FBQyxFQUFFLENBQUM7b0JBQ3JELE1BQU0sSUFBSSxLQUFLLENBQUMsMkJBQTJCLEtBQUssQ0FBQyxJQUFJLDhEQUE4RCxDQUFDLENBQUM7Z0JBQ3ZILENBQUM7Z0JBQ0QsT0FBTyxVQUFVLEtBQUssQ0FBQyxJQUFJLElBQUksV0FBVyxFQUFFLENBQUM7WUFDL0MsQ0FBQztZQUVELFdBQVcsQ0FBQyxHQUFPLEVBQUUsYUFBMkI7Z0JBQzlDLE9BQU8sS0FBSyxDQUFDLFFBQVEsSUFBSSxFQUFFLENBQUM7WUFDOUIsQ0FBQztZQUNELFNBQVMsQ0FBQyxHQUFPLEVBQUUsYUFBMkI7Z0JBQzVDLE9BQU8sS0FBSyxDQUFDLE1BQU0sSUFBSSxFQUFFLENBQUM7WUFDNUIsQ0FBQztZQUVELGlCQUFpQixDQUFDLEdBQU8sRUFBRSxhQUEyQjtnQkFDcEQsT0FBTyxLQUFLLENBQUMsY0FBYyxJQUFJLEVBQUUsQ0FBQztZQUNwQyxDQUFDO1NBQ0YsRUFBRSxDQUFDO0lBQ04sQ0FBQztJQUVEOztPQUVHO0lBQ0gsTUFBTSxDQUFDLGdCQUFnQjtRQUNyQixPQUFPLElBQUksS0FBTSxTQUFRLG9CQUFvQjtZQUFsQzs7Z0JBQ1QsU0FBSSxHQUFHLGtCQUFrQixDQUFDO1lBeUI1QixDQUFDO1lBdkJDLFdBQVcsQ0FBQyxFQUFNLEVBQUUsYUFBMkI7Z0JBQzdDLElBQUksRUFBRSxDQUFDLElBQUksQ0FBQyxjQUFFLENBQUMsMEJBQTBCLENBQUMsRUFBRSxDQUFDO29CQUMzQyxPQUFPO3dCQUNMLGdCQUFnQjt3QkFDaEIsbURBQW1EO3dCQUNuRCxtSUFBbUk7cUJBQ3BJLENBQUM7Z0JBQ0osQ0FBQztxQkFBTSxJQUFJLEVBQUUsQ0FBQyxFQUFFLENBQUMsY0FBRSxDQUFDLGNBQWMsQ0FBQyxFQUFFLENBQUM7b0JBQ3BDLE9BQU87d0JBQ0wsZUFBZTt3QkFDZiw2RUFBNkU7cUJBQzlFLENBQUM7Z0JBQ0osQ0FBQztxQkFBTSxJQUFJLEVBQUUsQ0FBQyxFQUFFLENBQUMsY0FBRSxDQUFDLGlCQUFpQixDQUFDLEVBQUUsQ0FBQztvQkFDdkMsT0FBTzt3QkFDTCxnQkFBZ0I7d0JBQ2hCLHVGQUF1RjtxQkFDeEYsQ0FBQztnQkFDSixDQUFDO3FCQUFNLElBQUksRUFBRSxDQUFDLEVBQUUsQ0FBQyxjQUFFLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztvQkFDN0IsT0FBTyxFQUFFLENBQUM7Z0JBQ1osQ0FBQztnQkFFRCxNQUFNLElBQUksS0FBSyxDQUFDLHlDQUF5QyxFQUFFLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQztZQUN0RSxDQUFDO1NBQ0YsQ0FBQztJQUNKLENBQUM7SUFFRDs7T0FFRztJQUNILE1BQU0sQ0FBQyxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxLQUFNLFNBQVEsb0JBQW9CO1lBQWxDOztnQkFDVCxTQUFJLEdBQUcsaUJBQWlCLENBQUM7WUFtQzNCLENBQUM7WUFqQ0MsV0FBVyxDQUFDLEVBQU0sRUFBRSxZQUEwQjtnQkFDNUMsSUFBSSxFQUFFLENBQUMsSUFBSSxDQUFDLGNBQUUsQ0FBQywwQkFBMEIsQ0FBQyxFQUFFLENBQUM7b0JBQzNDLElBQUksT0FBTyxDQUFDO29CQUNaLElBQUksWUFBWSxDQUFDLEVBQUUsQ0FBQyx3QkFBWSxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUM7d0JBQ3pDLE9BQU8sR0FBRyxPQUFPLENBQUM7b0JBQ3BCLENBQUM7eUJBQU0sSUFBSSxZQUFZLENBQUMsRUFBRSxDQUFDLHdCQUFZLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQzt3QkFDL0MsT0FBTyxHQUFHLE9BQU8sQ0FBQztvQkFDcEIsQ0FBQzt5QkFBTSxDQUFDO3dCQUNOLE1BQU0sSUFBSSxLQUFLLENBQUMsbURBQW1ELFlBQVksQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDO29CQUMxRixDQUFDO29CQUVELE9BQU87d0JBQ0wsc0dBQXNHLE9BQU8scUNBQXFDO3dCQUNsSiw2Q0FBNkM7d0JBQzdDLHFDQUFxQztxQkFDdEMsQ0FBQztnQkFDSixDQUFDO3FCQUFNLElBQUksRUFBRSxDQUFDLEVBQUUsQ0FBQyxjQUFFLENBQUMsY0FBYyxDQUFDLEVBQUUsQ0FBQztvQkFDcEMsT0FBTzt3QkFDTCx3Q0FBd0M7cUJBQ3pDLENBQUM7Z0JBQ0osQ0FBQztxQkFBTSxJQUFJLEVBQUUsQ0FBQyxFQUFFLENBQUMsY0FBRSxDQUFDLGlCQUFpQixDQUFDLEVBQUUsQ0FBQztvQkFDdkMsT0FBTzt3QkFDTCx3Q0FBd0M7cUJBQ3pDLENBQUM7Z0JBQ0osQ0FBQztxQkFBTSxJQUFJLEVBQUUsQ0FBQyxFQUFFLENBQUMsY0FBRSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7b0JBQzdCLE9BQU87d0JBQ0wsMEtBQTBLO3dCQUMxSyw2REFBNkQ7cUJBQzlELENBQUM7Z0JBQ0osQ0FBQztnQkFFRCxNQUFNLElBQUksS0FBSyxDQUFDLHlDQUF5QyxFQUFFLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQztZQUN0RSxDQUFDO1NBQ0YsQ0FBQztJQUNKLENBQUM7SUFFRDs7T0FFRztJQUNILE1BQU0sQ0FBQyxVQUFVO1FBQ2YsT0FBTyxJQUFJLEtBQU0sU0FBUSxvQkFBb0I7WUFBbEM7O2dCQUNULFNBQUksR0FBRyxZQUFZLENBQUM7WUF1QnRCLENBQUM7WUFyQkMsV0FBVyxDQUFDLEVBQU0sRUFBRSxhQUEyQjtnQkFDN0MsSUFBSSxFQUFFLENBQUMsSUFBSSxDQUFDLGNBQUUsQ0FBQywwQkFBMEIsQ0FBQyxFQUFFLENBQUM7b0JBQzNDLE9BQU87d0JBQ0wsaUJBQWlCO3dCQUNqQixrRkFBa0Y7d0JBQ2xGLHNFQUFzRTtxQkFDdkUsQ0FBQztnQkFDSixDQUFDO3FCQUFNLElBQUksRUFBRSxDQUFDLElBQUksQ0FBQyxjQUFFLENBQUMsMEJBQTBCLENBQUMsRUFBRSxDQUFDO29CQUNsRCxPQUFPO3dCQUNMLDJCQUEyQjt3QkFDM0Isa0dBQWtHO3dCQUNsRyx1QkFBdUI7d0JBQ3ZCLDJCQUEyQjt3QkFDM0Isc0VBQXNFO3FCQUN2RSxDQUFDO2dCQUNKLENBQUM7cUJBQU0sSUFBSSxFQUFFLENBQUMsRUFBRSxDQUFDLGNBQUUsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO29CQUM3QixPQUFPLEVBQUUsQ0FBQztnQkFDWixDQUFDO2dCQUVELE1BQU0sSUFBSSxLQUFLLENBQUMsbUNBQW1DLEVBQUUsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDO1lBQ2hFLENBQUM7U0FDRixDQUFDO0lBQ0osQ0FBQztJQUVEOztPQUVHO0lBQ0gsTUFBTSxDQUFDLE1BQU07UUFDWCxPQUFPLElBQUksS0FBTSxTQUFRLG9CQUFvQjtZQUFsQzs7Z0JBQ1QsU0FBSSxHQUFHLFFBQVEsQ0FBQztZQTRCbEIsQ0FBQztZQTFCQyxXQUFXLENBQUMsRUFBTSxFQUFFLFlBQTBCO2dCQUM1QyxJQUFJLEVBQUUsQ0FBQyxJQUFJLENBQUMsY0FBRSxDQUFDLDBCQUEwQixDQUFDLElBQUksRUFBRSxDQUFDLElBQUksQ0FBQyxjQUFFLENBQUMsMEJBQTBCLENBQUMsRUFBRSxDQUFDO29CQUNyRixJQUFJLE9BQWUsQ0FBQztvQkFDcEIsSUFBSSxZQUFZLENBQUMsRUFBRSxDQUFDLHdCQUFZLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQzt3QkFDekMsT0FBTyxHQUFHLFFBQVEsQ0FBQztvQkFDckIsQ0FBQzt5QkFBTSxJQUFJLFlBQVksQ0FBQyxFQUFFLENBQUMsd0JBQVksQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDO3dCQUMvQyxPQUFPLEdBQUcsU0FBUyxDQUFDO29CQUN0QixDQUFDO3lCQUFNLENBQUM7d0JBQ04sTUFBTSxJQUFJLEtBQUssQ0FBQyx3Q0FBd0MsWUFBWSxDQUFDLElBQUksRUFBRSxDQUFDLENBQUM7b0JBQy9FLENBQUM7b0JBRUQsT0FBTzt3QkFDTCw2REFBNkQsT0FBTyx1QkFBdUI7d0JBQzNGLHVCQUF1Qjt3QkFDdkIsZUFBZTt3QkFDZix5QkFBeUI7cUJBQzFCLENBQUM7Z0JBQ0osQ0FBQztxQkFBTSxJQUFJLEVBQUUsQ0FBQyxFQUFFLENBQUMsY0FBRSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7b0JBQzdCLE9BQU87d0JBQ0wsbUhBQW1IO3dCQUNuSCw2REFBNkQ7cUJBQzlELENBQUM7Z0JBQ0osQ0FBQztnQkFFRCxNQUFNLElBQUksS0FBSyxDQUFDLDZDQUE2QyxFQUFFLENBQUMsSUFBSSxJQUFJLFlBQVksQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDO1lBQy9GLENBQUM7U0FDRixFQUFFLENBQUM7SUFDTixDQUFDO0lBRUQ7O09BRUc7SUFDSCxNQUFNLENBQUMsU0FBUztRQUNkLE9BQU8sSUFBSSxLQUFNLFNBQVEsb0JBQW9CO1lBQWxDOztnQkFDVCxTQUFJLEdBQUcsV0FBVyxDQUFDO1lBbUNyQixDQUFDO1lBakNDLFdBQVcsQ0FBQyxFQUFNLEVBQUUsWUFBMEI7Z0JBQzVDLElBQUksRUFBRSxDQUFDLElBQUksQ0FBQyxjQUFFLENBQUMsMEJBQTBCLENBQUMsRUFBRSxDQUFDO29CQUMzQyxPQUFPO3dCQUNMLHlJQUF5STt3QkFDekksNEdBQTRHOzRCQUM1RywrR0FBK0c7d0JBQy9HLGdCQUFnQjt3QkFDaEIsc0RBQXNEO3FCQUN2RCxDQUFDO2dCQUNKLENBQUM7cUJBQU0sSUFBSSxFQUFFLENBQUMsRUFBRSxDQUFDLGNBQUUsQ0FBQyxjQUFjLENBQUMsRUFBRSxDQUFDO29CQUNwQyxPQUFPO3dCQUNMLDZGQUE2Rjt3QkFDN0YsbUJBQW1CO3FCQUNwQixDQUFDO2dCQUNKLENBQUM7cUJBQU0sSUFBSSxFQUFFLENBQUMsRUFBRSxDQUFDLGNBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxFQUFFLENBQUM7b0JBQ3ZDLE9BQU87d0JBQ0wsNkZBQTZGO3dCQUM3RixtQkFBbUI7cUJBQ3BCLENBQUM7Z0JBQ0osQ0FBQztxQkFBTSxJQUFJLEVBQUUsQ0FBQyxFQUFFLENBQUMsY0FBRSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7b0JBQzdCLE9BQU87d0JBQ0wseUdBQXlHO3dCQUN6RywrQ0FBK0M7d0JBQy9DLDBEQUEwRDt3QkFDMUQsMEpBQTBKO3dCQUMxSixnRkFBZ0Y7d0JBQ2hGLDZEQUE2RDt3QkFDN0QsWUFBWTtxQkFDYixDQUFDO2dCQUNKLENBQUM7Z0JBRUQsTUFBTSxJQUFJLEtBQUssQ0FBQyxpREFBaUQsRUFBRSxDQUFDLElBQUksSUFBSSxZQUFZLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQztZQUNuRyxDQUFDO1NBQ0YsRUFBRSxDQUFDO0lBQ04sQ0FBQztJQUVEOztPQUVHO0lBQ0gsTUFBTSxDQUFDLEdBQUc7UUFDUixPQUFPLElBQUksS0FBTSxTQUFRLG9CQUFvQjtZQUFsQzs7Z0JBQ1QsU0FBSSxHQUFHLEtBQUssQ0FBQztZQWtDZixDQUFDO1lBaENDLFdBQVcsQ0FBQyxFQUFNLEVBQUUsWUFBMEI7Z0JBQzVDLElBQUksRUFBRSxDQUFDLElBQUksQ0FBQyxjQUFFLENBQUMsMEJBQTBCLENBQUMsRUFBRSxDQUFDO29CQUMzQyxPQUFPO3dCQUNMLHFDQUFxQzt3QkFDckMsZ0JBQWdCO3dCQUNoQix1REFBdUQ7cUJBQ3hELENBQUM7Z0JBQ0osQ0FBQztxQkFBTSxJQUFJLEVBQUUsQ0FBQyxFQUFFLENBQUMsY0FBRSxDQUFDLGNBQWMsQ0FBQyxFQUFFLENBQUM7b0JBQ3BDLE9BQU87d0JBQ0wsb0JBQW9CO3FCQUNyQixDQUFDO2dCQUNKLENBQUM7cUJBQU0sSUFBSSxFQUFFLENBQUMsRUFBRSxDQUFDLGNBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxFQUFFLENBQUM7b0JBQ3ZDLE9BQU87d0JBQ0wsb0JBQW9CO3FCQUNyQixDQUFDO2dCQUNKLENBQUM7cUJBQU0sSUFBSSxFQUFFLENBQUMsRUFBRSxDQUFDLGNBQUUsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO29CQUM3QixPQUFPO3dCQUNMLHNIQUFzSDt3QkFDdEgsZ0RBQWdEO3dCQUNoRCwyREFBMkQ7d0JBQzNELDZEQUE2RDt3QkFDN0Qsd0RBQXdEO3dCQUN4RCxvRkFBb0Y7d0JBQ3BGLDZLQUE2Szt3QkFDN0ssZ0ZBQWdGO3dCQUNoRiw2REFBNkQ7d0JBQzdELG1CQUFtQjtxQkFDcEIsQ0FBQztnQkFDSixDQUFDO2dCQUVELE1BQU0sSUFBSSxLQUFLLENBQUMsMENBQTBDLEVBQUUsQ0FBQyxJQUFJLElBQUksWUFBWSxDQUFDLElBQUksRUFBRSxDQUFDLENBQUM7WUFDNUYsQ0FBQztTQUNGLEVBQUUsQ0FBQztJQUNOLENBQUM7SUFFRDs7OztPQUlHO0lBQ0gsTUFBTSxDQUFDLFlBQVksQ0FBQyxhQUE0QjtRQUM5QyxPQUFPLElBQUksS0FBTSxTQUFRLG9CQUFvQjtZQUFsQzs7Z0JBQ1QsU0FBSSxHQUFHLGNBQWMsQ0FBQztZQXFGeEIsQ0FBQztZQW5GQyxXQUFXLENBQUMsRUFBTSxFQUFFLFlBQTBCO2dCQUM1QyxJQUFJLEVBQUUsQ0FBQyxJQUFJLENBQUMsY0FBRSxDQUFDLDBCQUEwQixDQUFDLElBQUksRUFBRSxDQUFDLElBQUksQ0FBQyxjQUFFLENBQUMsMEJBQTBCLENBQUMsRUFBRSxDQUFDO29CQUNyRixJQUFJLGNBQXNCLENBQUM7b0JBQzNCLElBQUksYUFBYSxDQUFDLEVBQUUsQ0FBQyx5QkFBYSxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsQ0FBQzt3QkFDN0MsY0FBYyxHQUFHLHdIQUF3SCxDQUFDO29CQUM1SSxDQUFDO3lCQUFNLENBQUM7d0JBQ04sY0FBYyxHQUFHLG1CQUFtQixhQUFhLENBQUMsT0FBTyxHQUFHLENBQUM7b0JBQy9ELENBQUM7b0JBRUQsSUFBSSxPQUFPLENBQUM7b0JBQ1osSUFBSSxZQUFZLENBQUMsRUFBRSxDQUFDLHdCQUFZLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQzt3QkFDekMsT0FBTyxHQUFHLEtBQUssQ0FBQztvQkFDbEIsQ0FBQzt5QkFBTSxJQUFJLFlBQVksQ0FBQyxFQUFFLENBQUMsd0JBQVksQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDO3dCQUMvQyxPQUFPLEdBQUcsT0FBTyxDQUFDO29CQUNwQixDQUFDO3lCQUFNLENBQUM7d0JBQ04sTUFBTSxJQUFJLEtBQUssQ0FBQywrQ0FBK0MsWUFBWSxDQUFDLElBQUksRUFBRSxDQUFDLENBQUM7b0JBQ3RGLENBQUM7b0JBRUQsSUFBSSxRQUFRLEdBQUc7d0JBQ2IsY0FBYzt3QkFDZCw2R0FBNkcsT0FBTyw2QkFBNkI7d0JBQ2pKLGtEQUFrRCxPQUFPLDZCQUE2Qjt3QkFDdEYsMkJBQTJCLE9BQU8sNEJBQTRCO3dCQUM5RCxXQUFXLGFBQWEsQ0FBQyxPQUFPLGdDQUFnQztxQkFDakUsQ0FBQztvQkFFRixJQUFJLEVBQUUsQ0FBQyxJQUFJLENBQUMsY0FBRSxDQUFDLDBCQUEwQixDQUFDLEVBQUUsQ0FBQzt3QkFDM0MsUUFBUSxDQUFDLElBQUksQ0FBQyx5Q0FBeUMsQ0FBQyxDQUFDO29CQUMzRCxDQUFDO3lCQUFNLElBQUksRUFBRSxDQUFDLEVBQUUsQ0FBQyxjQUFFLENBQUMsY0FBYyxDQUFDLEVBQUUsQ0FBQzt3QkFDcEMsUUFBUSxDQUFDLElBQUksQ0FBQyxxREFBcUQsQ0FBQyxDQUFDO29CQUN2RSxDQUFDO3lCQUFNLElBQUksRUFBRSxDQUFDLEVBQUUsQ0FBQyxjQUFFLENBQUMsaUJBQWlCLENBQUMsRUFBRSxDQUFDO3dCQUN2QyxRQUFRLENBQUMsSUFBSSxDQUFDLHdEQUF3RCxDQUFDLENBQUM7b0JBQzFFLENBQUM7b0JBRUQsUUFBUSxDQUFDLElBQUksQ0FBQywrQkFBK0IsRUFBRSxtQ0FBbUMsQ0FBQyxDQUFDO29CQUVwRixPQUFPLFFBQVEsQ0FBQztnQkFDbEIsQ0FBQztxQkFBTSxJQUFJLEVBQUUsQ0FBQyxFQUFFLENBQUMsY0FBRSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7b0JBQzdCLElBQUksY0FBd0IsQ0FBQztvQkFDN0IsSUFBSSxhQUFhLENBQUMsRUFBRSxDQUFDLHlCQUFhLENBQUMsTUFBTSxFQUFFLENBQUMsRUFBRSxDQUFDO3dCQUM3QyxjQUFjLEdBQUc7NEJBQ2YsaUhBQWlIOzRCQUNqSCxnREFBZ0Q7NEJBQ2hELDhEQUE4RDt5QkFDL0QsQ0FBQztvQkFDSixDQUFDO3lCQUFNLENBQUM7d0JBQ04sY0FBYyxHQUFHLENBQUMsc0JBQXNCLGFBQWEsQ0FBQyxPQUFPLEdBQUcsQ0FBQyxDQUFDO29CQUNwRSxDQUFDO29CQUVELGNBQWMsR0FBRyxjQUFjLENBQUMsTUFBTSxDQUFDO3dCQUNyQyxxQkFBcUI7d0JBQ3JCLG9DQUFvQzt3QkFDcEMsaUJBQWlCO3dCQUNqQix3Q0FBd0M7d0JBQ3hDLGlIQUFpSDt3QkFDakgsaURBQWlEO3dCQUNqRCw0REFBNEQ7d0JBQzVELDZKQUE2Sjt3QkFDN0osb0RBQW9EO3dCQUNwRCwwRUFBMEU7d0JBQzFFLGlGQUFpRjt3QkFDakYsY0FBYzt3QkFDZCx1QkFBdUI7d0JBQ3ZCLHlHQUF5Rzt3QkFDekcscUhBQXFIO3FCQUN0SCxDQUFDLENBQUM7b0JBRUgsT0FBTyxjQUFjLENBQUMsTUFBTSxDQUFDO3dCQUMzQixvTEFBb0w7d0JBQ3BMLHlEQUF5RDt3QkFDekQsaUJBQWlCO3dCQUNqQixRQUFRLGFBQWEsQ0FBQyxPQUFPLG9FQUFvRTtxQkFDbEcsQ0FBQyxDQUFDO2dCQUNMLENBQUM7Z0JBRUQsTUFBTSxJQUFJLEtBQUssQ0FBQyxvREFBb0QsRUFBRSxDQUFDLElBQUksSUFBSSxZQUFZLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQztZQUN0RyxDQUFDO1lBRUQsaUJBQWlCLENBQUMsR0FBTyxFQUFFLGFBQTJCO2dCQUNwRCxPQUFPO29CQUNMLHNCQUFzQixhQUFhLENBQUMsT0FBTyxFQUFFO2lCQUM5QyxDQUFDO1lBQ0osQ0FBQztTQUNGLEVBQUUsQ0FBQztJQUNOLENBQUM7SUFFRDs7OztPQUlHO0lBQ0gsTUFBTSxDQUFDLE1BQU07UUFDWCxPQUFPLElBQUksS0FBTSxTQUFRLG9CQUFvQjtZQUFsQzs7Z0JBQ1QsU0FBSSxHQUFHLFFBQVEsQ0FBQztZQWtFbEIsQ0FBQztZQWhFQyxXQUFXLENBQUMsRUFBTSxFQUFFLFlBQTBCO2dCQUM1QyxJQUFJLEVBQUUsQ0FBQyxJQUFJLENBQUMsY0FBRSxDQUFDLDBCQUEwQixDQUFDLEVBQUUsQ0FBQztvQkFDM0MsT0FBTzt3QkFDTCxnSEFBZ0g7d0JBQ2hILE9BQU87NEJBQ1AsK0hBQStIOzRCQUMvSCx5RkFBeUY7d0JBQ3pGLGdCQUFnQjt3QkFDaEIsK0dBQStHO3dCQUMvRywyQkFBMkI7d0JBQzNCLDhFQUE4RTtxQkFDL0UsQ0FBQztnQkFDSixDQUFDO3FCQUFNLElBQUksRUFBRSxDQUFDLEVBQUUsQ0FBQyxjQUFFLENBQUMsY0FBYyxDQUFDLEVBQUUsQ0FBQztvQkFDcEMsT0FBTzt3QkFDTCxvQ0FBb0M7d0JBQ3BDLDZCQUE2Qjt3QkFDN0IsdUtBQXVLO3dCQUN2SyxrQ0FBa0M7d0JBQ2xDLDhFQUE4RTtxQkFDL0UsQ0FBQztnQkFDSixDQUFDO3FCQUFNLElBQUksRUFBRSxDQUFDLEVBQUUsQ0FBQyxjQUFFLENBQUMsaUJBQWlCLENBQUMsRUFBRSxDQUFDO29CQUN2QyxPQUFPO3dCQUNMLHVCQUF1Qjt3QkFDdkIsNkJBQTZCO3dCQUM3Qix1S0FBdUs7d0JBQ3ZLLGtDQUFrQzt3QkFDbEMsOEVBQThFO3FCQUMvRSxDQUFDO2dCQUNKLENBQUM7cUJBQU0sSUFBSSxFQUFFLENBQUMsRUFBRSxDQUFDLGNBQUUsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO29CQUM3QixPQUFPO3dCQUNMLG1DQUFtQzt3QkFDbkMsK0dBQStHO3dCQUMvRyxtREFBbUQ7d0JBQ25ELDhEQUE4RDt3QkFDOUQsMkJBQTJCO3dCQUMzQixpSkFBaUo7d0JBQ2pKLHFDQUFxQzt3QkFDckMsZ0VBQWdFO3dCQUNoRSxnQkFBZ0I7d0JBQ2hCLGNBQWM7d0JBQ2QseUdBQXlHO3dCQUN6RyxxSUFBcUk7d0JBQ3JJLHNEQUFzRDt3QkFDdEQsMEJBQTBCO3dCQUMxQiw0QkFBNEI7d0JBQzVCLGlFQUFpRTt3QkFDakUsNEJBQTRCO3dCQUM1QiwrRUFBK0U7d0JBQy9FLHlCQUF5Qjt3QkFDekIsNEhBQTRIO3dCQUM1SCwyREFBMkQ7d0JBQzNELHNEQUFzRDt3QkFDdEQsZ05BQWdOO3dCQUNoTiw2RUFBNkU7d0JBQzdFLDJJQUEySTtxQkFDNUksQ0FBQztnQkFDSixDQUFDO2dCQUVELE1BQU0sSUFBSSxLQUFLLENBQUMsNkNBQTZDLEVBQUUsQ0FBQyxJQUFJLElBQUksWUFBWSxDQUFDLElBQUksRUFBRSxDQUFDLENBQUM7WUFDL0YsQ0FBQztZQUVELFlBQVksQ0FBQyxFQUFNLEVBQUUsYUFBMkI7Z0JBQzlDLE9BQU8sRUFBRSxDQUFDLEVBQUUsQ0FBQyxjQUFFLENBQUMsT0FBTyxDQUFDLENBQUM7WUFDM0IsQ0FBQztTQUNGLEVBQUUsQ0FBQztJQUNOLENBQUM7SUFFRDs7OztPQUlHO0lBQ0gsTUFBTSxDQUFDLGNBQWM7UUFDbkIsT0FBTyxvQkFBb0IsQ0FBQyxNQUFNLEVBQUUsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSCxNQUFNLENBQUMsaUJBQWlCLENBQUMsTUFBYyxFQUFFLElBQVk7UUFDbkQsT0FBTyxJQUFJLEtBQU0sU0FBUSxvQkFBb0I7WUFBbEM7O2dCQUNULFNBQUksR0FBRyxzQkFBc0IsSUFBSSxFQUFFLENBQUM7WUEwQ3RDLENBQUM7WUF4Q0MsV0FBVyxDQUFDLEVBQU0sRUFBRSxZQUEwQjtnQkFDNUMsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsa0JBQWtCLENBQUMsRUFBRSxDQUFDO29CQUNwQyxNQUFNLElBQUksS0FBSyxDQUFDLDZCQUE2QixJQUFJLDJFQUEyRSxDQUFDLENBQUM7Z0JBQ2hJLENBQUM7Z0JBRUQsSUFBSSxFQUFFLENBQUMsSUFBSSxDQUFDLGNBQUUsQ0FBQywwQkFBMEIsQ0FBQyxFQUFFLENBQUM7b0JBQzNDLE9BQU87d0JBQ0wsd0JBQXdCO3FCQUN6QixDQUFDO2dCQUNKLENBQUM7cUJBQU0sSUFBSSxFQUFFLENBQUMsSUFBSSxDQUFDLGNBQUUsQ0FBQywwQkFBMEIsQ0FBQyxFQUFFLENBQUM7b0JBQ2xELE9BQU87d0JBQ0wsaUJBQWlCO3FCQUNsQixDQUFDO2dCQUNKLENBQUM7cUJBQU0sSUFBSSxFQUFFLENBQUMsRUFBRSxDQUFDLGNBQUUsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO29CQUM3QixPQUFPO3dCQUNMLG9DQUFvQyxJQUFJLG1EQUFtRDt3QkFDM0YsbUJBQW1CLElBQUksTUFBTTtxQkFDOUIsQ0FBQztnQkFDSixDQUFDO2dCQUVELE1BQU0sSUFBSSxLQUFLLENBQUMseURBQXlELEVBQUUsQ0FBQyxJQUFJLElBQUksWUFBWSxDQUFDLElBQUksRUFBRSxDQUFDLENBQUM7WUFDM0csQ0FBQztZQUVELFNBQVMsQ0FBQyxFQUFNLEVBQUUsYUFBMkI7Z0JBQzNDLElBQUksRUFBRSxDQUFDLElBQUksQ0FBQyxjQUFFLENBQUMsMEJBQTBCLENBQUMsRUFBRSxDQUFDO29CQUMzQyxPQUFPO3dCQUNMLEVBQUUsTUFBTSxFQUFFLE1BQU0sRUFBRSxvQ0FBb0MsSUFBSSxNQUFNLEVBQUU7cUJBQ25FLENBQUM7Z0JBQ0osQ0FBQztxQkFBTSxJQUFJLEVBQUUsQ0FBQyxJQUFJLENBQUMsY0FBRSxDQUFDLDBCQUEwQixDQUFDLEVBQUUsQ0FBQztvQkFDbEQsT0FBTzt3QkFDTCxFQUFFLE1BQU0sRUFBRSxNQUFNLEVBQUUsb0NBQW9DLElBQUksTUFBTSxFQUFFO3FCQUNuRSxDQUFDO2dCQUNKLENBQUM7cUJBQU0sSUFBSSxFQUFFLENBQUMsRUFBRSxDQUFDLGNBQUUsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO29CQUM3QixPQUFPO3dCQUNMLEVBQUUsTUFBTSxFQUFFLE1BQU0sRUFBRSxPQUFPLElBQUksTUFBTSxFQUFFO3FCQUN0QyxDQUFDO2dCQUNKLENBQUM7Z0JBRUQsTUFBTSxJQUFJLEtBQUssQ0FBQywwQ0FBMEMsRUFBRSxDQUFDLElBQUksRUFBRSxDQUFDLENBQUM7WUFDdkUsQ0FBQztTQUNGLEVBQUUsQ0FBQztJQUNOLENBQUM7SUFFRDs7T0FFRztJQUNILE1BQU0sQ0FBQyxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLEtBQU0sU0FBUSxvQkFBb0I7WUFBbEM7O2dCQUNULFNBQUksR0FBRyxtQkFBbUIsQ0FBQztZQTZCN0IsQ0FBQztZQTNCQyxXQUFXLENBQUMsRUFBTSxFQUFFLGFBQTJCO2dCQUM3QyxJQUFJLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxjQUFFLENBQUMsbUJBQW1CLENBQUMsRUFBRSxDQUFDO29CQUNyQyxNQUFNLElBQUksS0FBSyxDQUFDLHlDQUF5QyxFQUFFLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQztnQkFDdEUsQ0FBQztnQkFFRCxPQUFPLEVBQUUsQ0FBQztZQUNaLENBQUM7WUFFRCxTQUFTLENBQUMsR0FBTyxFQUFFLGFBQTJCO2dCQUM1QyxPQUFPO29CQUNMO3dCQUNFLE1BQU0sRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxJQUFJLEVBQUUsSUFBSSxFQUFFLFFBQVEsRUFBRSxXQUFXLEVBQUUscUJBQXFCLENBQUM7d0JBQ3RGLE1BQU0sRUFBRSxlQUFlO3FCQUN4QjtvQkFDRDt3QkFDRSxNQUFNLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsV0FBVyxFQUFFLGtCQUFrQixDQUFDO3dCQUNuRixNQUFNLEVBQUUsWUFBWTtxQkFDckI7aUJBQ0YsQ0FBQztZQUNKLENBQUM7WUFFRCxpQkFBaUIsQ0FBQyxHQUFPLEVBQUUsYUFBMkI7Z0JBQ3BELE9BQU87b0JBQ0wsc0JBQXNCLEVBQUUsNkZBQTZGO29CQUNySCxzQ0FBc0M7aUJBQ3ZDLENBQUM7WUFDSixDQUFDO1NBQ0YsQ0FBQztJQUNKLENBQUM7SUFFRDs7Ozs7Ozs7T0FRRztJQUNILE1BQU0sQ0FBQyxvQkFBb0IsQ0FBQyxJQUE0QjtRQUN0RCxNQUFNLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRTtZQUMvQixJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDO2dCQUMvQyxNQUFNLElBQUksS0FBSyxDQUFDLGlEQUFpRCxDQUFDLEVBQUUsQ0FBQyxDQUFDO1lBQ3hFLENBQUM7UUFDSCxDQUFDLENBQUMsQ0FBQztRQUVILE9BQU8sSUFBSSxLQUFNLFNBQVEsb0JBQW9CO1lBQWxDOztnQkFDVCxTQUFJLEdBQUcsc0JBQXNCLENBQUM7WUFXaEMsQ0FBQztZQVRDLFdBQVcsQ0FBQyxFQUFNLEVBQUUsYUFBMkI7Z0JBQzdDLElBQUksRUFBRSxDQUFDLElBQUksQ0FBQyxjQUFFLENBQUMsbUJBQW1CLENBQUMsRUFBRSxDQUFDO29CQUNwQyxPQUFPLE1BQU0sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsU0FBUyxDQUFDLHdCQUF3QixDQUFDLENBQUM7Z0JBQy9HLENBQUM7cUJBQU0sSUFBSSxFQUFFLENBQUMsRUFBRSxDQUFDLGNBQUUsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO29CQUM3QixPQUFPLE1BQU0sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsK0NBQStDLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7Z0JBQzNILENBQUM7cUJBQU0sQ0FBQztvQkFDTixNQUFNLElBQUksS0FBSyxDQUFDLHVEQUF1RCxFQUFFLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQztnQkFDcEYsQ0FBQztZQUNILENBQUM7U0FDRixDQUFDO0lBQ0osQ0FBQztJQWNEOztPQUVHO0lBQ0gsU0FBUyxDQUFDLEdBQU8sRUFBRSxhQUEyQjtRQUM1QyxPQUFPLEVBQUUsQ0FBQztJQUNaLENBQUM7SUFFRDs7OztPQUlHO0lBQ0gsaUJBQWlCLENBQUMsR0FBTyxFQUFFLGFBQTJCO1FBQ3BELE9BQU8sRUFBRSxDQUFDO0lBQ1osQ0FBQztJQUVEOztPQUVHO0lBQ0gsWUFBWSxDQUFDLEdBQU8sRUFBRSxhQUEyQjtRQUMvQyxPQUFPLEtBQUssQ0FBQztJQUNmLENBQUM7SUFFRDs7OztPQUlHO0lBQ0gsMkJBQTJCLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsRUFBTSxFQUFFLFlBQTBCO1FBQzFGLElBQUksUUFBNkIsQ0FBQztRQUNsQyxJQUFJLEVBQUUsQ0FBQyxJQUFJLENBQUMsY0FBRSxDQUFDLDBCQUEwQixDQUFDLElBQUksRUFBRSxDQUFDLElBQUksQ0FBQyxjQUFFLENBQUMsMEJBQTBCLENBQUMsRUFBRSxDQUFDO1lBQ3JGLFFBQVEsR0FBRyxPQUFPLENBQUM7UUFDckIsQ0FBQzthQUFNLElBQUksRUFBRSxDQUFDLEVBQUUsQ0FBQyxjQUFFLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztZQUM3QixRQUFRLEdBQUcsU0FBUyxDQUFDO1FBQ3ZCLENBQUM7YUFBTSxDQUFDO1lBQ04sTUFBTSxJQUFJLEtBQUssQ0FBQyw4REFBOEQsRUFBRSxDQUFDLElBQUksSUFBSSxZQUFZLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQztRQUNoSCxDQUFDO1FBRUQsT0FBTyxJQUFJLHlDQUFxQixDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDMUMsUUFBUSxFQUFFLFFBQVE7WUFDbEIsUUFBUSxFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsRUFBRSxFQUFFLFlBQVksQ0FBQztZQUM1QyxNQUFNLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFLEVBQUUsWUFBWSxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRSxFQUFFO2dCQUM1RCxPQUFPO29CQUNMLEtBQUssRUFBRSxJQUFJLDJCQUFTLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxHQUFHLEVBQUUsVUFBVSxLQUFLLEVBQUUsRUFBRSxFQUFFLElBQUksRUFBRSxLQUFLLENBQUMsTUFBTSxFQUFFLENBQUM7b0JBQ2pGLElBQUksRUFBRSxLQUFLLENBQUMsTUFBTTtpQkFDbkIsQ0FBQztZQUNKLENBQUMsQ0FBQztZQUNGLFdBQVcsRUFBRSxFQUFFO1lBQ2YsV0FBVyxFQUFFLEVBQUU7WUFDZixNQUFNLEVBQUUsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLEVBQUUsWUFBWSxDQUFDO1NBQzVDLENBQUMsQ0FBQztJQUNMLENBQUM7O0FBaG5CSCxvREFpbkJDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCB7IGF3c19zM19hc3NldHMgYXMgczNfYXNzZXRzIH0gZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBJbWFnZUJ1aWxkZXJDb21wb25lbnQgfSBmcm9tICcuL2F3cy1pbWFnZS1idWlsZGVyJztcbmltcG9ydCB7IFJ1bm5lckltYWdlQXNzZXQgfSBmcm9tICcuL2NvbW1vbic7XG5pbXBvcnQgeyBBcmNoaXRlY3R1cmUsIE9zLCBSdW5uZXJWZXJzaW9uIH0gZnJvbSAnLi4vcHJvdmlkZXJzJztcblxuZXhwb3J0IGludGVyZmFjZSBSdW5uZXJJbWFnZUNvbXBvbmVudEN1c3RvbVByb3BzIHtcbiAgLyoqXG4gICAqIENvbXBvbmVudCBuYW1lIHVzZWQgZm9yICgxKSBpbWFnZSBidWlsZCBsb2dnaW5nIGFuZCAoMikgaWRlbnRpZmllciBmb3Ige0BsaW5rIElDb25maWd1cmFibGVSdW5uZXJJbWFnZUJ1aWxkZXIucmVtb3ZlQ29tcG9uZW50fS5cbiAgICpcbiAgICogTmFtZSBtdXN0IG9ubHkgY29udGFpbiBhbHBoYW51bWVyaWMgY2hhcmFjdGVycyBhbmQgZGFzaGVzLlxuICAgKi9cbiAgcmVhZG9ubHkgbmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogQ29tbWFuZHMgdG8gcnVuIGluIHRoZSBidWlsdCBpbWFnZS5cbiAgICovXG4gIHJlYWRvbmx5IGNvbW1hbmRzPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIEFzc2V0cyB0byBjb3B5IGludG8gdGhlIGJ1aWx0IGltYWdlLlxuICAgKi9cbiAgcmVhZG9ubHkgYXNzZXRzPzogUnVubmVySW1hZ2VBc3NldFtdO1xuXG4gIC8qKlxuICAgKiBEb2NrZXIgY29tbWFuZHMgdG8gcnVuIGluIHRoZSBidWlsdCBpbWFnZS5cbiAgICpcbiAgICogRm9yIGV4YW1wbGU6IGBbJ0VOViBmb289YmFyJywgJ1JVTiBlY2hvICRmb28nXWBcbiAgICpcbiAgICogVGhlc2UgY29tbWFuZHMgYXJlIGlnbm9yZWQgd2hlbiBidWlsZGluZyBBTUlzLlxuICAgKi9cbiAgcmVhZG9ubHkgZG9ja2VyQ29tbWFuZHM/OiBzdHJpbmdbXTtcbn1cblxuLyoqXG4gKiBDb21wb25lbnRzIGFyZSB1c2VkIHRvIGJ1aWxkIHJ1bm5lciBpbWFnZXMuIFRoZXkgY2FuIHJ1biBjb21tYW5kcyBpbiB0aGUgaW1hZ2UsIGNvcHkgZmlsZXMgaW50byB0aGUgaW1hZ2UsIGFuZCBydW4gc29tZSBEb2NrZXIgY29tbWFuZHMuXG4gKi9cbmV4cG9ydCBhYnN0cmFjdCBjbGFzcyBSdW5uZXJJbWFnZUNvbXBvbmVudCB7XG4gIC8qKlxuICAgKiBEZWZpbmUgYSBjdXN0b20gY29tcG9uZW50IHRoYXQgY2FuIHJ1biBjb21tYW5kcyBpbiB0aGUgaW1hZ2UsIGNvcHkgZmlsZXMgaW50byB0aGUgaW1hZ2UsIGFuZCBydW4gc29tZSBEb2NrZXIgY29tbWFuZHMuXG4gICAqXG4gICAqIFRoZSBvcmRlciBvZiBvcGVyYXRpb25zIGlzICgxKSBhc3NldHMgKDIpIGNvbW1hbmRzICgzKSBkb2NrZXIgY29tbWFuZHMuXG4gICAqXG4gICAqIFVzZSB0aGlzIHRvIGN1c3RvbWl6ZSB0aGUgaW1hZ2UgZm9yIHRoZSBydW5uZXIuXG4gICAqXG4gICAqICoqV0FSTklORzoqKiBEb2NrZXIgY29tbWFuZHMgYXJlIG5vdCBndWFyYW50ZWVkIHRvIGJlIGluY2x1ZGVkIGJlZm9yZSB0aGUgbmV4dCBjb21wb25lbnRcbiAgICovXG4gIHN0YXRpYyBjdXN0b20ocHJvcHM6IFJ1bm5lckltYWdlQ29tcG9uZW50Q3VzdG9tUHJvcHMpOiBSdW5uZXJJbWFnZUNvbXBvbmVudCB7XG4gICAgcmV0dXJuIG5ldyBjbGFzcyBleHRlbmRzIFJ1bm5lckltYWdlQ29tcG9uZW50IHtcbiAgICAgIGdldCBuYW1lKCkge1xuICAgICAgICBpZiAocHJvcHMubmFtZSAmJiAhcHJvcHMubmFtZS5tYXRjaCgvW2EtekEtWjAtOVxcLV0vKSkge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgSW52YWxpZCBjb21wb25lbnQgbmFtZTogJHtwcm9wcy5uYW1lfS4gTmFtZSBtdXN0IG9ubHkgY29udGFpbiBhbHBoYW51bWVyaWMgY2hhcmFjdGVycyBhbmQgZGFzaGVzLmApO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBgQ3VzdG9tLSR7cHJvcHMubmFtZSA/PyAnVW5kZWZpbmVkJ31gO1xuICAgICAgfVxuXG4gICAgICBnZXRDb21tYW5kcyhfb3M6IE9zLCBfYXJjaGl0ZWN0dXJlOiBBcmNoaXRlY3R1cmUpIHtcbiAgICAgICAgcmV0dXJuIHByb3BzLmNvbW1hbmRzID8/IFtdO1xuICAgICAgfVxuICAgICAgZ2V0QXNzZXRzKF9vczogT3MsIF9hcmNoaXRlY3R1cmU6IEFyY2hpdGVjdHVyZSkge1xuICAgICAgICByZXR1cm4gcHJvcHMuYXNzZXRzID8/IFtdO1xuICAgICAgfVxuXG4gICAgICBnZXREb2NrZXJDb21tYW5kcyhfb3M6IE9zLCBfYXJjaGl0ZWN0dXJlOiBBcmNoaXRlY3R1cmUpIHtcbiAgICAgICAgcmV0dXJuIHByb3BzLmRvY2tlckNvbW1hbmRzID8/IFtdO1xuICAgICAgfVxuICAgIH0oKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBIGNvbXBvbmVudCB0byBpbnN0YWxsIHRoZSByZXF1aXJlZCBwYWNrYWdlcyBmb3IgdGhlIHJ1bm5lci5cbiAgICovXG4gIHN0YXRpYyByZXF1aXJlZFBhY2thZ2VzKCk6IFJ1bm5lckltYWdlQ29tcG9uZW50IHtcbiAgICByZXR1cm4gbmV3IGNsYXNzIGV4dGVuZHMgUnVubmVySW1hZ2VDb21wb25lbnQge1xuICAgICAgbmFtZSA9ICdSZXF1aXJlZFBhY2thZ2VzJztcblxuICAgICAgZ2V0Q29tbWFuZHMob3M6IE9zLCBfYXJjaGl0ZWN0dXJlOiBBcmNoaXRlY3R1cmUpOiBzdHJpbmdbXSB7XG4gICAgICAgIGlmIChvcy5pc0luKE9zLl9BTExfTElOVVhfVUJVTlRVX1ZFUlNJT05TKSkge1xuICAgICAgICAgIHJldHVybiBbXG4gICAgICAgICAgICAnYXB0LWdldCB1cGRhdGUnLFxuICAgICAgICAgICAgJ0RFQklBTl9GUk9OVEVORD1ub25pbnRlcmFjdGl2ZSBhcHQtZ2V0IHVwZ3JhZGUgLXknLFxuICAgICAgICAgICAgJ0RFQklBTl9GUk9OVEVORD1ub25pbnRlcmFjdGl2ZSBhcHQtZ2V0IGluc3RhbGwgLXkgY3VybCBzdWRvIGpxIGJhc2ggemlwIHVuemlwIGlwdGFibGVzIHNvZnR3YXJlLXByb3BlcnRpZXMtY29tbW9uIGNhLWNlcnRpZmljYXRlcycsXG4gICAgICAgICAgXTtcbiAgICAgICAgfSBlbHNlIGlmIChvcy5pcyhPcy5MSU5VWF9BTUFaT05fMikpIHtcbiAgICAgICAgICByZXR1cm4gW1xuICAgICAgICAgICAgJ3l1bSB1cGRhdGUgLXknLFxuICAgICAgICAgICAgJ3l1bSBpbnN0YWxsIC15IGpxIHRhciBnemlwIGJ6aXAyIHdoaWNoIGJpbnV0aWxzIHppcCB1bnppcCBzdWRvIHNoYWRvdy11dGlscycsXG4gICAgICAgICAgXTtcbiAgICAgICAgfSBlbHNlIGlmIChvcy5pcyhPcy5MSU5VWF9BTUFaT05fMjAyMykpIHtcbiAgICAgICAgICByZXR1cm4gW1xuICAgICAgICAgICAgJ2RuZiB1cGdyYWRlIC15JyxcbiAgICAgICAgICAgICdkbmYgaW5zdGFsbCAteSBqcSB0YXIgZ3ppcCBiemlwMiB3aGljaCBiaW51dGlscyB6aXAgdW56aXAgc3VkbyBzaGFkb3ctdXRpbHMgZmluZHV0aWxzJyxcbiAgICAgICAgICBdO1xuICAgICAgICB9IGVsc2UgaWYgKG9zLmlzKE9zLldJTkRPV1MpKSB7XG4gICAgICAgICAgcmV0dXJuIFtdO1xuICAgICAgICB9XG5cbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBVbnN1cHBvcnRlZCBPUyBmb3IgcmVxdWlyZWQgcGFja2FnZXM6ICR7b3MubmFtZX1gKTtcbiAgICAgIH1cbiAgICB9O1xuICB9XG5cbiAgLyoqXG4gICAqIEEgY29tcG9uZW50IHRvIGluc3RhbGwgQ2xvdWRXYXRjaCBBZ2VudCBmb3IgdGhlIHJ1bm5lciBzbyB3ZSBjYW4gc2VuZCBsb2dzLlxuICAgKi9cbiAgc3RhdGljIGNsb3VkV2F0Y2hBZ2VudCgpOiBSdW5uZXJJbWFnZUNvbXBvbmVudCB7XG4gICAgcmV0dXJuIG5ldyBjbGFzcyBleHRlbmRzIFJ1bm5lckltYWdlQ29tcG9uZW50IHtcbiAgICAgIG5hbWUgPSAnQ2xvdWRXYXRjaEFnZW50JztcblxuICAgICAgZ2V0Q29tbWFuZHMob3M6IE9zLCBhcmNoaXRlY3R1cmU6IEFyY2hpdGVjdHVyZSk6IHN0cmluZ1tdIHtcbiAgICAgICAgaWYgKG9zLmlzSW4oT3MuX0FMTF9MSU5VWF9VQlVOVFVfVkVSU0lPTlMpKSB7XG4gICAgICAgICAgbGV0IGFyY2hVcmw7XG4gICAgICAgICAgaWYgKGFyY2hpdGVjdHVyZS5pcyhBcmNoaXRlY3R1cmUuWDg2XzY0KSkge1xuICAgICAgICAgICAgYXJjaFVybCA9ICdhbWQ2NCc7XG4gICAgICAgICAgfSBlbHNlIGlmIChhcmNoaXRlY3R1cmUuaXMoQXJjaGl0ZWN0dXJlLkFSTTY0KSkge1xuICAgICAgICAgICAgYXJjaFVybCA9ICdhcm02NCc7XG4gICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgVW5zdXBwb3J0ZWQgYXJjaGl0ZWN0dXJlIGZvciByZXF1aXJlZCBwYWNrYWdlczogJHthcmNoaXRlY3R1cmUubmFtZX1gKTtcbiAgICAgICAgICB9XG5cbiAgICAgICAgICByZXR1cm4gW1xuICAgICAgICAgICAgYGN1cmwgLXNmTG8gL3RtcC9hbWF6b24tY2xvdWR3YXRjaC1hZ2VudC5kZWIgaHR0cHM6Ly9zMy5hbWF6b25hd3MuY29tL2FtYXpvbmNsb3Vkd2F0Y2gtYWdlbnQvdWJ1bnR1LyR7YXJjaFVybH0vbGF0ZXN0L2FtYXpvbi1jbG91ZHdhdGNoLWFnZW50LmRlYmAsXG4gICAgICAgICAgICAnZHBrZyAtaSAtRSAvdG1wL2FtYXpvbi1jbG91ZHdhdGNoLWFnZW50LmRlYicsXG4gICAgICAgICAgICAncm0gL3RtcC9hbWF6b24tY2xvdWR3YXRjaC1hZ2VudC5kZWInLFxuICAgICAgICAgIF07XG4gICAgICAgIH0gZWxzZSBpZiAob3MuaXMoT3MuTElOVVhfQU1BWk9OXzIpKSB7XG4gICAgICAgICAgcmV0dXJuIFtcbiAgICAgICAgICAgICd5dW0gaW5zdGFsbCAteSBhbWF6b24tY2xvdWR3YXRjaC1hZ2VudCcsXG4gICAgICAgICAgXTtcbiAgICAgICAgfSBlbHNlIGlmIChvcy5pcyhPcy5MSU5VWF9BTUFaT05fMjAyMykpIHtcbiAgICAgICAgICByZXR1cm4gW1xuICAgICAgICAgICAgJ2RuZiBpbnN0YWxsIC15IGFtYXpvbi1jbG91ZHdhdGNoLWFnZW50JyxcbiAgICAgICAgICBdO1xuICAgICAgICB9IGVsc2UgaWYgKG9zLmlzKE9zLldJTkRPV1MpKSB7XG4gICAgICAgICAgcmV0dXJuIFtcbiAgICAgICAgICAgICckcCA9IFN0YXJ0LVByb2Nlc3MgbXNpZXhlYy5leGUgLVBhc3NUaHJ1IC1XYWl0IC1Bcmd1bWVudExpc3QgXFwnL2kgaHR0cHM6Ly9zMy5hbWF6b25hd3MuY29tL2FtYXpvbmNsb3Vkd2F0Y2gtYWdlbnQvd2luZG93cy9hbWQ2NC9sYXRlc3QvYW1hem9uLWNsb3Vkd2F0Y2gtYWdlbnQubXNpIC9xblxcJycsXG4gICAgICAgICAgICAnaWYgKCRwLkV4aXRDb2RlIC1uZSAwKSB7IHRocm93IFwiRXhpdCBjb2RlIGlzICRwLkV4aXRDb2RlXCIgfScsXG4gICAgICAgICAgXTtcbiAgICAgICAgfVxuXG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgVW5zdXBwb3J0ZWQgT1MgZm9yIHJlcXVpcmVkIHBhY2thZ2VzOiAke29zLm5hbWV9YCk7XG4gICAgICB9XG4gICAgfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBIGNvbXBvbmVudCB0byBwcmVwYXJlIHRoZSByZXF1aXJlZCBydW5uZXIgdXNlci5cbiAgICovXG4gIHN0YXRpYyBydW5uZXJVc2VyKCk6IFJ1bm5lckltYWdlQ29tcG9uZW50IHtcbiAgICByZXR1cm4gbmV3IGNsYXNzIGV4dGVuZHMgUnVubmVySW1hZ2VDb21wb25lbnQge1xuICAgICAgbmFtZSA9ICdSdW5uZXJVc2VyJztcblxuICAgICAgZ2V0Q29tbWFuZHMob3M6IE9zLCBfYXJjaGl0ZWN0dXJlOiBBcmNoaXRlY3R1cmUpOiBzdHJpbmdbXSB7XG4gICAgICAgIGlmIChvcy5pc0luKE9zLl9BTExfTElOVVhfVUJVTlRVX1ZFUlNJT05TKSkge1xuICAgICAgICAgIHJldHVybiBbXG4gICAgICAgICAgICAnYWRkZ3JvdXAgcnVubmVyJyxcbiAgICAgICAgICAgICdhZGR1c2VyIC0tc3lzdGVtIC0tZGlzYWJsZWQtcGFzc3dvcmQgLS1ob21lIC9ob21lL3J1bm5lciAtLWluZ3JvdXAgcnVubmVyIHJ1bm5lcicsXG4gICAgICAgICAgICAnZWNobyBcIiVydW5uZXIgICBBTEw9KEFMTDpBTEwpIE5PUEFTU1dEOiBBTExcIiA+IC9ldGMvc3Vkb2Vycy5kL3J1bm5lcicsXG4gICAgICAgICAgXTtcbiAgICAgICAgfSBlbHNlIGlmIChvcy5pc0luKE9zLl9BTExfTElOVVhfQU1BWk9OX1ZFUlNJT05TKSkge1xuICAgICAgICAgIHJldHVybiBbXG4gICAgICAgICAgICAnL3Vzci9zYmluL2dyb3VwYWRkIHJ1bm5lcicsXG4gICAgICAgICAgICAnL3Vzci9zYmluL3VzZXJhZGQgLS1zeXN0ZW0gLS1zaGVsbCAvdXNyL3NiaW4vbm9sb2dpbiAtLWhvbWUtZGlyIC9ob21lL3J1bm5lciAtLWdpZCBydW5uZXIgcnVubmVyJyxcbiAgICAgICAgICAgICdta2RpciAtcCAvaG9tZS9ydW5uZXInLFxuICAgICAgICAgICAgJ2Nob3duIHJ1bm5lciAvaG9tZS9ydW5uZXInLFxuICAgICAgICAgICAgJ2VjaG8gXCIlcnVubmVyICAgQUxMPShBTEw6QUxMKSBOT1BBU1NXRDogQUxMXCIgPiAvZXRjL3N1ZG9lcnMuZC9ydW5uZXInLFxuICAgICAgICAgIF07XG4gICAgICAgIH0gZWxzZSBpZiAob3MuaXMoT3MuV0lORE9XUykpIHtcbiAgICAgICAgICByZXR1cm4gW107XG4gICAgICAgIH1cblxuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFVuc3VwcG9ydGVkIE9TIGZvciBydW5uZXIgdXNlcjogJHtvcy5uYW1lfWApO1xuICAgICAgfVxuICAgIH07XG4gIH1cblxuICAvKipcbiAgICogQSBjb21wb25lbnQgdG8gaW5zdGFsbCB0aGUgQVdTIENMSS5cbiAgICovXG4gIHN0YXRpYyBhd3NDbGkoKTogUnVubmVySW1hZ2VDb21wb25lbnQge1xuICAgIHJldHVybiBuZXcgY2xhc3MgZXh0ZW5kcyBSdW5uZXJJbWFnZUNvbXBvbmVudCB7XG4gICAgICBuYW1lID0gJ0F3c0NsaSc7XG5cbiAgICAgIGdldENvbW1hbmRzKG9zOiBPcywgYXJjaGl0ZWN0dXJlOiBBcmNoaXRlY3R1cmUpIHtcbiAgICAgICAgaWYgKG9zLmlzSW4oT3MuX0FMTF9MSU5VWF9VQlVOVFVfVkVSU0lPTlMpIHx8IG9zLmlzSW4oT3MuX0FMTF9MSU5VWF9BTUFaT05fVkVSU0lPTlMpKSB7XG4gICAgICAgICAgbGV0IGFyY2hVcmw6IHN0cmluZztcbiAgICAgICAgICBpZiAoYXJjaGl0ZWN0dXJlLmlzKEFyY2hpdGVjdHVyZS5YODZfNjQpKSB7XG4gICAgICAgICAgICBhcmNoVXJsID0gJ3g4Nl82NCc7XG4gICAgICAgICAgfSBlbHNlIGlmIChhcmNoaXRlY3R1cmUuaXMoQXJjaGl0ZWN0dXJlLkFSTTY0KSkge1xuICAgICAgICAgICAgYXJjaFVybCA9ICdhYXJjaDY0JztcbiAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBVbnN1cHBvcnRlZCBhcmNoaXRlY3R1cmUgZm9yIGF3c2NsaTogJHthcmNoaXRlY3R1cmUubmFtZX1gKTtcbiAgICAgICAgICB9XG5cbiAgICAgICAgICByZXR1cm4gW1xuICAgICAgICAgICAgYGN1cmwgLWZzU0wgXCJodHRwczovL2F3c2NsaS5hbWF6b25hd3MuY29tL2F3c2NsaS1leGUtbGludXgtJHthcmNoVXJsfS56aXBcIiAtbyBhd3NjbGl2Mi56aXBgLFxuICAgICAgICAgICAgJ3VuemlwIC1xIGF3c2NsaXYyLnppcCcsXG4gICAgICAgICAgICAnLi9hd3MvaW5zdGFsbCcsXG4gICAgICAgICAgICAncm0gLXJmIGF3c2NsaXYyLnppcCBhd3MnLFxuICAgICAgICAgIF07XG4gICAgICAgIH0gZWxzZSBpZiAob3MuaXMoT3MuV0lORE9XUykpIHtcbiAgICAgICAgICByZXR1cm4gW1xuICAgICAgICAgICAgJyRwID0gU3RhcnQtUHJvY2VzcyBtc2lleGVjLmV4ZSAtUGFzc1RocnUgLVdhaXQgLUFyZ3VtZW50TGlzdCBcXCcvaSBodHRwczovL2F3c2NsaS5hbWF6b25hd3MuY29tL0FXU0NMSVYyLm1zaSAvcW5cXCcnLFxuICAgICAgICAgICAgJ2lmICgkcC5FeGl0Q29kZSAtbmUgMCkgeyB0aHJvdyBcIkV4aXQgY29kZSBpcyAkcC5FeGl0Q29kZVwiIH0nLFxuICAgICAgICAgIF07XG4gICAgICAgIH1cblxuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFVua25vd24gb3MvYXJjaGl0ZWN0dXJlIGNvbWJvIGZvciBhd3NjbGk6ICR7b3MubmFtZX0vJHthcmNoaXRlY3R1cmUubmFtZX1gKTtcbiAgICAgIH1cbiAgICB9KCk7XG4gIH1cblxuICAvKipcbiAgICogQSBjb21wb25lbnQgdG8gaW5zdGFsbCB0aGUgR2l0SHViIENMSS5cbiAgICovXG4gIHN0YXRpYyBnaXRodWJDbGkoKTogUnVubmVySW1hZ2VDb21wb25lbnQge1xuICAgIHJldHVybiBuZXcgY2xhc3MgZXh0ZW5kcyBSdW5uZXJJbWFnZUNvbXBvbmVudCB7XG4gICAgICBuYW1lID0gJ0dpdGh1YkNsaSc7XG5cbiAgICAgIGdldENvbW1hbmRzKG9zOiBPcywgYXJjaGl0ZWN0dXJlOiBBcmNoaXRlY3R1cmUpIHtcbiAgICAgICAgaWYgKG9zLmlzSW4oT3MuX0FMTF9MSU5VWF9VQlVOVFVfVkVSU0lPTlMpKSB7XG4gICAgICAgICAgcmV0dXJuIFtcbiAgICAgICAgICAgICdjdXJsIC1mc1NMIGh0dHBzOi8vY2xpLmdpdGh1Yi5jb20vcGFja2FnZXMvZ2l0aHViY2xpLWFyY2hpdmUta2V5cmluZy5ncGcgfCBzdWRvIGRkIG9mPS91c3Ivc2hhcmUva2V5cmluZ3MvZ2l0aHViY2xpLWFyY2hpdmUta2V5cmluZy5ncGcnLFxuICAgICAgICAgICAgJ2VjaG8gXCJkZWIgW2FyY2g9JChkcGtnIC0tcHJpbnQtYXJjaGl0ZWN0dXJlKSBzaWduZWQtYnk9L3Vzci9zaGFyZS9rZXlyaW5ncy9naXRodWJjbGktYXJjaGl2ZS1rZXlyaW5nLmdwZ10gJyArXG4gICAgICAgICAgICAnICBodHRwczovL2NsaS5naXRodWIuY29tL3BhY2thZ2VzIHN0YWJsZSBtYWluXCIgfCBzdWRvIHRlZSAvZXRjL2FwdC9zb3VyY2VzLmxpc3QuZC9naXRodWItY2xpLmxpc3QgPiAvZGV2L251bGwnLFxuICAgICAgICAgICAgJ2FwdC1nZXQgdXBkYXRlJyxcbiAgICAgICAgICAgICdERUJJQU5fRlJPTlRFTkQ9bm9uaW50ZXJhY3RpdmUgYXB0LWdldCBpbnN0YWxsIC15IGdoJyxcbiAgICAgICAgICBdO1xuICAgICAgICB9IGVsc2UgaWYgKG9zLmlzKE9zLkxJTlVYX0FNQVpPTl8yKSkge1xuICAgICAgICAgIHJldHVybiBbXG4gICAgICAgICAgICAnY3VybCAtZnNTU0wgaHR0cHM6Ly9jbGkuZ2l0aHViLmNvbS9wYWNrYWdlcy9ycG0vZ2gtY2xpLnJlcG8gLW8gL2V0Yy95dW0ucmVwb3MuZC9naC1jbGkucmVwbycsXG4gICAgICAgICAgICAneXVtIGluc3RhbGwgLXkgZ2gnLFxuICAgICAgICAgIF07XG4gICAgICAgIH0gZWxzZSBpZiAob3MuaXMoT3MuTElOVVhfQU1BWk9OXzIwMjMpKSB7XG4gICAgICAgICAgcmV0dXJuIFtcbiAgICAgICAgICAgICdjdXJsIC1mc1NTTCBodHRwczovL2NsaS5naXRodWIuY29tL3BhY2thZ2VzL3JwbS9naC1jbGkucmVwbyAtbyAvZXRjL3l1bS5yZXBvcy5kL2doLWNsaS5yZXBvJyxcbiAgICAgICAgICAgICdkbmYgaW5zdGFsbCAteSBnaCcsXG4gICAgICAgICAgXTtcbiAgICAgICAgfSBlbHNlIGlmIChvcy5pcyhPcy5XSU5ET1dTKSkge1xuICAgICAgICAgIHJldHVybiBbXG4gICAgICAgICAgICAnY21kIC9jIGN1cmwgLXcgXCIle3JlZGlyZWN0X3VybH1cIiAtZnNTIGh0dHBzOi8vZ2l0aHViLmNvbS9jbGkvY2xpL3JlbGVhc2VzL2xhdGVzdCA+ICRFbnY6VEVNUFxcXFxsYXRlc3QtZ2gnLFxuICAgICAgICAgICAgJyRMYXRlc3RVcmwgPSBHZXQtQ29udGVudCAkRW52OlRFTVBcXFxcbGF0ZXN0LWdoJyxcbiAgICAgICAgICAgICckR0hfVkVSU0lPTiA9ICgkTGF0ZXN0VXJsIC1TcGxpdCBcXCcvXFwnKVstMV0uc3Vic3RyaW5nKDEpJyxcbiAgICAgICAgICAgICdJbnZva2UtV2ViUmVxdWVzdCAtVXNlQmFzaWNQYXJzaW5nIC1VcmkgXCJodHRwczovL2dpdGh1Yi5jb20vY2xpL2NsaS9yZWxlYXNlcy9kb3dubG9hZC92JHtHSF9WRVJTSU9OfS9naF8ke0dIX1ZFUlNJT059X3dpbmRvd3NfYW1kNjQubXNpXCIgLU91dEZpbGUgZ2gubXNpJyxcbiAgICAgICAgICAgICckcCA9IFN0YXJ0LVByb2Nlc3MgbXNpZXhlYy5leGUgLVBhc3NUaHJ1IC1XYWl0IC1Bcmd1bWVudExpc3QgXFwnL2kgZ2gubXNpIC9xblxcJycsXG4gICAgICAgICAgICAnaWYgKCRwLkV4aXRDb2RlIC1uZSAwKSB7IHRocm93IFwiRXhpdCBjb2RlIGlzICRwLkV4aXRDb2RlXCIgfScsXG4gICAgICAgICAgICAnZGVsIGdoLm1zaScsXG4gICAgICAgICAgXTtcbiAgICAgICAgfVxuXG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgVW5rbm93biBvcy9hcmNoaXRlY3R1cmUgY29tYm8gZm9yIGdpdGh1YiBjbGk6ICR7b3MubmFtZX0vJHthcmNoaXRlY3R1cmUubmFtZX1gKTtcbiAgICAgIH1cbiAgICB9KCk7XG4gIH1cblxuICAvKipcbiAgICogQSBjb21wb25lbnQgdG8gaW5zdGFsbCB0aGUgR2l0SHViIENMSS5cbiAgICovXG4gIHN0YXRpYyBnaXQoKTogUnVubmVySW1hZ2VDb21wb25lbnQge1xuICAgIHJldHVybiBuZXcgY2xhc3MgZXh0ZW5kcyBSdW5uZXJJbWFnZUNvbXBvbmVudCB7XG4gICAgICBuYW1lID0gJ0dpdCc7XG5cbiAgICAgIGdldENvbW1hbmRzKG9zOiBPcywgYXJjaGl0ZWN0dXJlOiBBcmNoaXRlY3R1cmUpIHtcbiAgICAgICAgaWYgKG9zLmlzSW4oT3MuX0FMTF9MSU5VWF9VQlVOVFVfVkVSU0lPTlMpKSB7XG4gICAgICAgICAgcmV0dXJuIFtcbiAgICAgICAgICAgICdhZGQtYXB0LXJlcG9zaXRvcnkgcHBhOmdpdC1jb3JlL3BwYScsXG4gICAgICAgICAgICAnYXB0LWdldCB1cGRhdGUnLFxuICAgICAgICAgICAgJ0RFQklBTl9GUk9OVEVORD1ub25pbnRlcmFjdGl2ZSBhcHQtZ2V0IGluc3RhbGwgLXkgZ2l0JyxcbiAgICAgICAgICBdO1xuICAgICAgICB9IGVsc2UgaWYgKG9zLmlzKE9zLkxJTlVYX0FNQVpPTl8yKSkge1xuICAgICAgICAgIHJldHVybiBbXG4gICAgICAgICAgICAneXVtIGluc3RhbGwgLXkgZ2l0JyxcbiAgICAgICAgICBdO1xuICAgICAgICB9IGVsc2UgaWYgKG9zLmlzKE9zLkxJTlVYX0FNQVpPTl8yMDIzKSkge1xuICAgICAgICAgIHJldHVybiBbXG4gICAgICAgICAgICAnZG5mIGluc3RhbGwgLXkgZ2l0JyxcbiAgICAgICAgICBdO1xuICAgICAgICB9IGVsc2UgaWYgKG9zLmlzKE9zLldJTkRPV1MpKSB7XG4gICAgICAgICAgcmV0dXJuIFtcbiAgICAgICAgICAgICdjbWQgL2MgY3VybCAtdyBcIiV7cmVkaXJlY3RfdXJsfVwiIC1mc1MgaHR0cHM6Ly9naXRodWIuY29tL2dpdC1mb3Itd2luZG93cy9naXQvcmVsZWFzZXMvbGF0ZXN0ID4gJEVudjpURU1QXFxcXGxhdGVzdC1naXQnLFxuICAgICAgICAgICAgJyRMYXRlc3RVcmwgPSBHZXQtQ29udGVudCAkRW52OlRFTVBcXFxcbGF0ZXN0LWdpdCcsXG4gICAgICAgICAgICAnJEdJVF9WRVJTSU9OID0gKCRMYXRlc3RVcmwgLVNwbGl0IFxcJy9cXCcpWy0xXS5zdWJzdHJpbmcoMSknLFxuICAgICAgICAgICAgJyRHSVRfVkVSU0lPTl9TSE9SVCA9ICgkR0lUX1ZFUlNJT04gLVNwbGl0IFxcJy53aW5kb3dzLlxcJylbMF0nLFxuICAgICAgICAgICAgJyRHSVRfUkVWSVNJT04gPSAoJEdJVF9WRVJTSU9OIC1TcGxpdCBcXCcud2luZG93cy5cXCcpWzFdJyxcbiAgICAgICAgICAgICdJZiAoJEdJVF9SRVZJU0lPTiAtZ3QgMSkgeyRHSVRfVkVSU0lPTl9TSE9SVCA9IFwiJEdJVF9WRVJTSU9OX1NIT1JULiRHSVRfUkVWSVNJT05cIn0nLFxuICAgICAgICAgICAgJ0ludm9rZS1XZWJSZXF1ZXN0IC1Vc2VCYXNpY1BhcnNpbmcgLVVyaSBodHRwczovL2dpdGh1Yi5jb20vZ2l0LWZvci13aW5kb3dzL2dpdC9yZWxlYXNlcy9kb3dubG9hZC92JHtHSVRfVkVSU0lPTn0vR2l0LSR7R0lUX1ZFUlNJT05fU0hPUlR9LTY0LWJpdC5leGUgLU91dEZpbGUgZ2l0LXNldHVwLmV4ZScsXG4gICAgICAgICAgICAnJHAgPSBTdGFydC1Qcm9jZXNzIGdpdC1zZXR1cC5leGUgLVBhc3NUaHJ1IC1XYWl0IC1Bcmd1bWVudExpc3QgXFwnL1ZFUllTSUxFTlRcXCcnLFxuICAgICAgICAgICAgJ2lmICgkcC5FeGl0Q29kZSAtbmUgMCkgeyB0aHJvdyBcIkV4aXQgY29kZSBpcyAkcC5FeGl0Q29kZVwiIH0nLFxuICAgICAgICAgICAgJ2RlbCBnaXQtc2V0dXAuZXhlJyxcbiAgICAgICAgICBdO1xuICAgICAgICB9XG5cbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBVbmtub3duIG9zL2FyY2hpdGVjdHVyZSBjb21ibyBmb3IgZ2l0OiAke29zLm5hbWV9LyR7YXJjaGl0ZWN0dXJlLm5hbWV9YCk7XG4gICAgICB9XG4gICAgfSgpO1xuICB9XG5cbiAgLyoqXG4gICAqIEEgY29tcG9uZW50IHRvIGluc3RhbGwgdGhlIEdpdEh1YiBBY3Rpb25zIFJ1bm5lci4gVGhpcyBpcyB0aGUgYWN0dWFsIGV4ZWN1dGFibGUgdGhhdCBjb25uZWN0cyB0byBHaXRIdWIgdG8gYXNrIGZvciBqb2JzIGFuZCB0aGVuIGV4ZWN1dGUgdGhlbS5cbiAgICpcbiAgICogQHBhcmFtIHJ1bm5lclZlcnNpb24gVGhlIHZlcnNpb24gb2YgdGhlIHJ1bm5lciB0byBpbnN0YWxsLiBVc3VhbGx5IHlvdSB3b3VsZCBzZXQgdGhpcyB0byBsYXRlc3QuXG4gICAqL1xuICBzdGF0aWMgZ2l0aHViUnVubmVyKHJ1bm5lclZlcnNpb246IFJ1bm5lclZlcnNpb24pOiBSdW5uZXJJbWFnZUNvbXBvbmVudCB7XG4gICAgcmV0dXJuIG5ldyBjbGFzcyBleHRlbmRzIFJ1bm5lckltYWdlQ29tcG9uZW50IHtcbiAgICAgIG5hbWUgPSAnR2l0aHViUnVubmVyJztcblxuICAgICAgZ2V0Q29tbWFuZHMob3M6IE9zLCBhcmNoaXRlY3R1cmU6IEFyY2hpdGVjdHVyZSkge1xuICAgICAgICBpZiAob3MuaXNJbihPcy5fQUxMX0xJTlVYX1VCVU5UVV9WRVJTSU9OUykgfHwgb3MuaXNJbihPcy5fQUxMX0xJTlVYX0FNQVpPTl9WRVJTSU9OUykpIHtcbiAgICAgICAgICBsZXQgdmVyc2lvbkNvbW1hbmQ6IHN0cmluZztcbiAgICAgICAgICBpZiAocnVubmVyVmVyc2lvbi5pcyhSdW5uZXJWZXJzaW9uLmxhdGVzdCgpKSkge1xuICAgICAgICAgICAgdmVyc2lvbkNvbW1hbmQgPSAnUlVOTkVSX1ZFUlNJT049YGN1cmwgLXcgXCIle3JlZGlyZWN0X3VybH1cIiAtZnNTIGh0dHBzOi8vZ2l0aHViLmNvbS9hY3Rpb25zL3J1bm5lci9yZWxlYXNlcy9sYXRlc3QgfCBncmVwIC1vRSBcIlteL3ZdKyRcImAnO1xuICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICB2ZXJzaW9uQ29tbWFuZCA9IGBSVU5ORVJfVkVSU0lPTj0nJHtydW5uZXJWZXJzaW9uLnZlcnNpb259J2A7XG4gICAgICAgICAgfVxuXG4gICAgICAgICAgbGV0IGFyY2hVcmw7XG4gICAgICAgICAgaWYgKGFyY2hpdGVjdHVyZS5pcyhBcmNoaXRlY3R1cmUuWDg2XzY0KSkge1xuICAgICAgICAgICAgYXJjaFVybCA9ICd4NjQnO1xuICAgICAgICAgIH0gZWxzZSBpZiAoYXJjaGl0ZWN0dXJlLmlzKEFyY2hpdGVjdHVyZS5BUk02NCkpIHtcbiAgICAgICAgICAgIGFyY2hVcmwgPSAnYXJtNjQnO1xuICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFVuc3VwcG9ydGVkIGFyY2hpdGVjdHVyZSBmb3IgR2l0SHViIFJ1bm5lcjogJHthcmNoaXRlY3R1cmUubmFtZX1gKTtcbiAgICAgICAgICB9XG5cbiAgICAgICAgICBsZXQgY29tbWFuZHMgPSBbXG4gICAgICAgICAgICB2ZXJzaW9uQ29tbWFuZCxcbiAgICAgICAgICAgIGBjdXJsIC1mc1NMTyBcImh0dHBzOi8vZ2l0aHViLmNvbS9hY3Rpb25zL3J1bm5lci9yZWxlYXNlcy9kb3dubG9hZC92XFwke1JVTk5FUl9WRVJTSU9OfS9hY3Rpb25zLXJ1bm5lci1saW51eC0ke2FyY2hVcmx9LVxcJHtSVU5ORVJfVkVSU0lPTn0udGFyLmd6XCJgLFxuICAgICAgICAgICAgYHRhciAtQyAvaG9tZS9ydW5uZXIgLXh6ZiBcImFjdGlvbnMtcnVubmVyLWxpbnV4LSR7YXJjaFVybH0tXFwke1JVTk5FUl9WRVJTSU9OfS50YXIuZ3pcImAsXG4gICAgICAgICAgICBgcm0gYWN0aW9ucy1ydW5uZXItbGludXgtJHthcmNoVXJsfS1cXCR7UlVOTkVSX1ZFUlNJT059LnRhci5nemAsXG4gICAgICAgICAgICBgZWNobyAtbiAke3J1bm5lclZlcnNpb24udmVyc2lvbn0gPiAvaG9tZS9ydW5uZXIvUlVOTkVSX1ZFUlNJT05gLFxuICAgICAgICAgIF07XG5cbiAgICAgICAgICBpZiAob3MuaXNJbihPcy5fQUxMX0xJTlVYX1VCVU5UVV9WRVJTSU9OUykpIHtcbiAgICAgICAgICAgIGNvbW1hbmRzLnB1c2goJy9ob21lL3J1bm5lci9iaW4vaW5zdGFsbGRlcGVuZGVuY2llcy5zaCcpO1xuICAgICAgICAgIH0gZWxzZSBpZiAob3MuaXMoT3MuTElOVVhfQU1BWk9OXzIpKSB7XG4gICAgICAgICAgICBjb21tYW5kcy5wdXNoKCd5dW0gaW5zdGFsbCAteSBvcGVuc3NsLWxpYnMga3JiNS1saWJzIHpsaWIgbGliaWN1NjAnKTtcbiAgICAgICAgICB9IGVsc2UgaWYgKG9zLmlzKE9zLkxJTlVYX0FNQVpPTl8yMDIzKSkge1xuICAgICAgICAgICAgY29tbWFuZHMucHVzaCgnZG5mIGluc3RhbGwgLXkgb3BlbnNzbC1saWJzIGtyYjUtbGlicyB6bGliIGxpYmljdS02Ny4xJyk7XG4gICAgICAgICAgfVxuXG4gICAgICAgICAgY29tbWFuZHMucHVzaCgnbWtkaXIgLXAgL29wdC9ob3N0ZWR0b29sY2FjaGUnLCAnY2hvd24gcnVubmVyIC9vcHQvaG9zdGVkdG9vbGNhY2hlJyk7XG5cbiAgICAgICAgICByZXR1cm4gY29tbWFuZHM7XG4gICAgICAgIH0gZWxzZSBpZiAob3MuaXMoT3MuV0lORE9XUykpIHtcbiAgICAgICAgICBsZXQgcnVubmVyQ29tbWFuZHM6IHN0cmluZ1tdO1xuICAgICAgICAgIGlmIChydW5uZXJWZXJzaW9uLmlzKFJ1bm5lclZlcnNpb24ubGF0ZXN0KCkpKSB7XG4gICAgICAgICAgICBydW5uZXJDb21tYW5kcyA9IFtcbiAgICAgICAgICAgICAgJ2NtZCAvYyBjdXJsIC13IFwiJXtyZWRpcmVjdF91cmx9XCIgLWZzUyBodHRwczovL2dpdGh1Yi5jb20vYWN0aW9ucy9ydW5uZXIvcmVsZWFzZXMvbGF0ZXN0ID4gJEVudjpURU1QXFxcXGxhdGVzdC1naGEnLFxuICAgICAgICAgICAgICAnJExhdGVzdFVybCA9IEdldC1Db250ZW50ICRFbnY6VEVNUFxcXFxsYXRlc3QtZ2hhJyxcbiAgICAgICAgICAgICAgJyRSVU5ORVJfVkVSU0lPTiA9ICgkTGF0ZXN0VXJsIC1TcGxpdCBcXCcvXFwnKVstMV0uc3Vic3RyaW5nKDEpJyxcbiAgICAgICAgICAgIF07XG4gICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgIHJ1bm5lckNvbW1hbmRzID0gW2AkUlVOTkVSX1ZFUlNJT04gPSAnJHtydW5uZXJWZXJzaW9uLnZlcnNpb259J2BdO1xuICAgICAgICAgIH1cblxuICAgICAgICAgIHJ1bm5lckNvbW1hbmRzID0gcnVubmVyQ29tbWFuZHMuY29uY2F0KFtcbiAgICAgICAgICAgIC8vIGNyZWF0ZSBkaXJlY3Rvcmllc1xuICAgICAgICAgICAgJ21rZGlyIEM6XFxcXGhvc3RlZHRvb2xjYWNoZVxcXFx3aW5kb3dzJyxcbiAgICAgICAgICAgICdta2RpciBDOlxcXFx0b29scycsXG4gICAgICAgICAgICAvLyBkb3dubG9hZCB6c3RkIGFuZCBleHRyYWN0IHRvIEM6XFx0b29sc1xuICAgICAgICAgICAgJ2NtZCAvYyBjdXJsIC13IFwiJXtyZWRpcmVjdF91cmx9XCIgLWZzUyBodHRwczovL2dpdGh1Yi5jb20vZmFjZWJvb2svenN0ZC9yZWxlYXNlcy9sYXRlc3QgPiAkRW52OlRFTVBcXFxcbGF0ZXN0LXpzdGQnLFxuICAgICAgICAgICAgJyRMYXRlc3RVcmwgPSBHZXQtQ29udGVudCAkRW52OlRFTVBcXFxcbGF0ZXN0LXpzdGQnLFxuICAgICAgICAgICAgJyRaU1REX1ZFUlNJT04gPSAoJExhdGVzdFVybCAtU3BsaXQgXFwnL1xcJylbLTFdLnN1YnN0cmluZygxKScsXG4gICAgICAgICAgICAnSW52b2tlLVdlYlJlcXVlc3QgLVVzZUJhc2ljUGFyc2luZyAtVXJpIFwiaHR0cHM6Ly9naXRodWIuY29tL2ZhY2Vib29rL3pzdGQvcmVsZWFzZXMvZG93bmxvYWQvdiRaU1REX1ZFUlNJT04venN0ZC12JFpTVERfVkVSU0lPTi13aW42NC56aXBcIiAtT3V0RmlsZSB6c3RkLnppcCcsXG4gICAgICAgICAgICAnRXhwYW5kLUFyY2hpdmUgenN0ZC56aXAgLURlc3RpbmF0aW9uUGF0aCBDOlxcXFx0b29scycsXG4gICAgICAgICAgICAnTW92ZS1JdGVtIC1QYXRoIEM6XFxcXHRvb2xzXFxcXHpzdGQtdiRaU1REX1ZFUlNJT04td2luNjRcXFxcenN0ZC5leGUgQzpcXFxcdG9vbHMnLFxuICAgICAgICAgICAgJ1JlbW92ZS1JdGVtIC1MaXRlcmFsUGF0aCBcIkM6XFxcXHRvb2xzXFxcXHpzdGQtdiRaU1REX1ZFUlNJT04td2luNjRcIiAtRm9yY2UgLVJlY3Vyc2UnLFxuICAgICAgICAgICAgJ2RlbCB6c3RkLnppcCcsXG4gICAgICAgICAgICAvLyBhZGQgQzpcXHRvb2xzIHRvIFBBVEhcbiAgICAgICAgICAgICckcGVyc2lzdGVkUGF0aHMgPSBbRW52aXJvbm1lbnRdOjpHZXRFbnZpcm9ubWVudFZhcmlhYmxlKFxcJ1BhdGhcXCcsIFtFbnZpcm9ubWVudFZhcmlhYmxlVGFyZ2V0XTo6TWFjaGluZSknLFxuICAgICAgICAgICAgJ1tFbnZpcm9ubWVudF06OlNldEVudmlyb25tZW50VmFyaWFibGUoXCJQQVRIXCIsICRwZXJzaXN0ZWRQYXRocyArIFwiO0M6XFxcXHRvb2xzXCIsIFtFbnZpcm9ubWVudFZhcmlhYmxlVGFyZ2V0XTo6TWFjaGluZSknLFxuICAgICAgICAgIF0pO1xuXG4gICAgICAgICAgcmV0dXJuIHJ1bm5lckNvbW1hbmRzLmNvbmNhdChbXG4gICAgICAgICAgICAnSW52b2tlLVdlYlJlcXVlc3QgLVVzZUJhc2ljUGFyc2luZyAtVXJpIFwiaHR0cHM6Ly9naXRodWIuY29tL2FjdGlvbnMvcnVubmVyL3JlbGVhc2VzL2Rvd25sb2FkL3Yke1JVTk5FUl9WRVJTSU9OfS9hY3Rpb25zLXJ1bm5lci13aW4teDY0LSR7UlVOTkVSX1ZFUlNJT059LnppcFwiIC1PdXRGaWxlIGFjdGlvbnMuemlwJyxcbiAgICAgICAgICAgICdFeHBhbmQtQXJjaGl2ZSBhY3Rpb25zLnppcCAtRGVzdGluYXRpb25QYXRoIEM6XFxcXGFjdGlvbnMnLFxuICAgICAgICAgICAgJ2RlbCBhY3Rpb25zLnppcCcsXG4gICAgICAgICAgICBgZWNobyAke3J1bm5lclZlcnNpb24udmVyc2lvbn0gfCBPdXQtRmlsZSAtRW5jb2RpbmcgQVNDSUkgLU5vTmV3bGluZSBDOlxcXFxhY3Rpb25zXFxcXFJVTk5FUl9WRVJTSU9OYCxcbiAgICAgICAgICBdKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgVW5rbm93biBvcy9hcmNoaXRlY3R1cmUgY29tYm8gZm9yIGdpdGh1YiBydW5uZXI6ICR7b3MubmFtZX0vJHthcmNoaXRlY3R1cmUubmFtZX1gKTtcbiAgICAgIH1cblxuICAgICAgZ2V0RG9ja2VyQ29tbWFuZHMoX29zOiBPcywgX2FyY2hpdGVjdHVyZTogQXJjaGl0ZWN0dXJlKTogc3RyaW5nW10ge1xuICAgICAgICByZXR1cm4gW1xuICAgICAgICAgIGBFTlYgUlVOTkVSX1ZFUlNJT049JHtydW5uZXJWZXJzaW9uLnZlcnNpb259YCxcbiAgICAgICAgXTtcbiAgICAgIH1cbiAgICB9KCk7XG4gIH1cblxuICAvKipcbiAgICogQSBjb21wb25lbnQgdG8gaW5zdGFsbCBEb2NrZXIuXG4gICAqXG4gICAqIE9uIFdpbmRvd3MgdGhpcyBzZXRzIHVwIGRvY2tlcmQgZm9yIFdpbmRvd3MgY29udGFpbmVycyB3aXRob3V0IERvY2tlciBEZXNrdG9wLiBJZiB5b3UgbmVlZCBMaW51eCBjb250YWluZXJzIG9uIFdpbmRvd3MsIHlvdSdsbCBuZWVkIHRvIGluc3RhbGwgRG9ja2VyIERlc2t0b3Agd2hpY2ggZG9lc24ndCBzZWVtIHRvIHBsYXkgd2VsbCB3aXRoIHNlcnZlcnMgKFBScyB3ZWxjb21lKS5cbiAgICovXG4gIHN0YXRpYyBkb2NrZXIoKTogUnVubmVySW1hZ2VDb21wb25lbnQge1xuICAgIHJldHVybiBuZXcgY2xhc3MgZXh0ZW5kcyBSdW5uZXJJbWFnZUNvbXBvbmVudCB7XG4gICAgICBuYW1lID0gJ0RvY2tlcic7XG5cbiAgICAgIGdldENvbW1hbmRzKG9zOiBPcywgYXJjaGl0ZWN0dXJlOiBBcmNoaXRlY3R1cmUpIHtcbiAgICAgICAgaWYgKG9zLmlzSW4oT3MuX0FMTF9MSU5VWF9VQlVOVFVfVkVSU0lPTlMpKSB7XG4gICAgICAgICAgcmV0dXJuIFtcbiAgICAgICAgICAgICdjdXJsIC1mc1NMIGh0dHBzOi8vZG93bmxvYWQuZG9ja2VyLmNvbS9saW51eC91YnVudHUvZ3BnIHwgc3VkbyBncGcgLS1kZWFybW9yIC1vIC91c3Ivc2hhcmUva2V5cmluZ3MvZG9ja2VyLmdwZycsXG4gICAgICAgICAgICAnZWNobyAnICtcbiAgICAgICAgICAgICcgIFwiZGViIFthcmNoPSQoZHBrZyAtLXByaW50LWFyY2hpdGVjdHVyZSkgc2lnbmVkLWJ5PS91c3Ivc2hhcmUva2V5cmluZ3MvZG9ja2VyLmdwZ10gaHR0cHM6Ly9kb3dubG9hZC5kb2NrZXIuY29tL2xpbnV4L3VidW50dSAnICtcbiAgICAgICAgICAgICcgICQobHNiX3JlbGVhc2UgLWNzKSBzdGFibGVcIiB8IHN1ZG8gdGVlIC9ldGMvYXB0L3NvdXJjZXMubGlzdC5kL2RvY2tlci5saXN0ID4gL2Rldi9udWxsJyxcbiAgICAgICAgICAgICdhcHQtZ2V0IHVwZGF0ZScsXG4gICAgICAgICAgICAnREVCSUFOX0ZST05URU5EPW5vbmludGVyYWN0aXZlIGFwdC1nZXQgaW5zdGFsbCAteSBkb2NrZXItY2UgZG9ja2VyLWNlLWNsaSBjb250YWluZXJkLmlvIGRvY2tlci1jb21wb3NlLXBsdWdpbicsXG4gICAgICAgICAgICAndXNlcm1vZCAtYUcgZG9ja2VyIHJ1bm5lcicsXG4gICAgICAgICAgICAnbG4gLXMgL3Vzci9saWJleGVjL2RvY2tlci9jbGktcGx1Z2lucy9kb2NrZXItY29tcG9zZSAvdXNyL2Jpbi9kb2NrZXItY29tcG9zZScsXG4gICAgICAgICAgXTtcbiAgICAgICAgfSBlbHNlIGlmIChvcy5pcyhPcy5MSU5VWF9BTUFaT05fMikpIHtcbiAgICAgICAgICByZXR1cm4gW1xuICAgICAgICAgICAgJ2FtYXpvbi1saW51eC1leHRyYXMgaW5zdGFsbCBkb2NrZXInLFxuICAgICAgICAgICAgJ3VzZXJtb2QgLWEgLUcgZG9ja2VyIHJ1bm5lcicsXG4gICAgICAgICAgICAnY3VybCAtc2ZMbyAvdXNyL2Jpbi9kb2NrZXItY29tcG9zZSBodHRwczovL2dpdGh1Yi5jb20vZG9ja2VyL2NvbXBvc2UvcmVsZWFzZXMvbGF0ZXN0L2Rvd25sb2FkL2RvY2tlci1jb21wb3NlLSQodW5hbWUgLXMgfCB0ciBcXCdbOnVwcGVyOl1cXCcgXFwnWzpsb3dlcjpdXFwnKS0kKHVuYW1lIC1tKScsXG4gICAgICAgICAgICAnY2htb2QgK3ggL3Vzci9iaW4vZG9ja2VyLWNvbXBvc2UnLFxuICAgICAgICAgICAgJ2xuIC1zIC91c3IvYmluL2RvY2tlci1jb21wb3NlIC91c3IvbGliZXhlYy9kb2NrZXIvY2xpLXBsdWdpbnMvZG9ja2VyLWNvbXBvc2UnLFxuICAgICAgICAgIF07XG4gICAgICAgIH0gZWxzZSBpZiAob3MuaXMoT3MuTElOVVhfQU1BWk9OXzIwMjMpKSB7XG4gICAgICAgICAgcmV0dXJuIFtcbiAgICAgICAgICAgICdkbmYgaW5zdGFsbCAteSBkb2NrZXInLFxuICAgICAgICAgICAgJ3VzZXJtb2QgLWEgLUcgZG9ja2VyIHJ1bm5lcicsXG4gICAgICAgICAgICAnY3VybCAtc2ZMbyAvdXNyL2Jpbi9kb2NrZXItY29tcG9zZSBodHRwczovL2dpdGh1Yi5jb20vZG9ja2VyL2NvbXBvc2UvcmVsZWFzZXMvbGF0ZXN0L2Rvd25sb2FkL2RvY2tlci1jb21wb3NlLSQodW5hbWUgLXMgfCB0ciBcXCdbOnVwcGVyOl1cXCcgXFwnWzpsb3dlcjpdXFwnKS0kKHVuYW1lIC1tKScsXG4gICAgICAgICAgICAnY2htb2QgK3ggL3Vzci9iaW4vZG9ja2VyLWNvbXBvc2UnLFxuICAgICAgICAgICAgJ2xuIC1zIC91c3IvYmluL2RvY2tlci1jb21wb3NlIC91c3IvbGliZXhlYy9kb2NrZXIvY2xpLXBsdWdpbnMvZG9ja2VyLWNvbXBvc2UnLFxuICAgICAgICAgIF07XG4gICAgICAgIH0gZWxzZSBpZiAob3MuaXMoT3MuV0lORE9XUykpIHtcbiAgICAgICAgICByZXR1cm4gW1xuICAgICAgICAgICAgLy8gZmlndXJlIG91dCBsYXRlc3QgZG9ja2VyIHZlcnNpb25cbiAgICAgICAgICAgICdjbWQgL2MgY3VybCAtdyBcIiV7cmVkaXJlY3RfdXJsfVwiIC1mc1MgaHR0cHM6Ly9naXRodWIuY29tL21vYnkvbW9ieS9yZWxlYXNlcy9sYXRlc3QgPiAkRW52OlRFTVBcXFxcbGF0ZXN0LWRvY2tlcicsXG4gICAgICAgICAgICAnJExhdGVzdFVybCA9IEdldC1Db250ZW50ICRFbnY6VEVNUFxcXFxsYXRlc3QtZG9ja2VyJyxcbiAgICAgICAgICAgICckRE9DS0VSX1ZFUlNJT04gPSAoJExhdGVzdFVybCAtU3BsaXQgXFwnL1xcJylbLTFdLnN1YnN0cmluZygxKScsXG4gICAgICAgICAgICAvLyBkb3dubG9hZCBzdGF0aWMgYmluYXJpZXNcbiAgICAgICAgICAgICdJbnZva2UtV2ViUmVxdWVzdCAtVXNlQmFzaWNQYXJzaW5nIC1VcmkgXCJodHRwczovL2Rvd25sb2FkLmRvY2tlci5jb20vd2luL3N0YXRpYy9zdGFibGUveDg2XzY0L2RvY2tlci0ke0RPQ0tFUl9WRVJTSU9OfS56aXBcIiAtT3V0RmlsZSBkb2NrZXIuemlwJyxcbiAgICAgICAgICAgIC8vIGV4dHJhY3QgdG8gQzpcXFByb2dyYW0gRmlsZXNcXERvY2tlclxuICAgICAgICAgICAgJ0V4cGFuZC1BcmNoaXZlIGRvY2tlci56aXAgLURlc3RpbmF0aW9uUGF0aCBcIiRFbnY6UHJvZ3JhbUZpbGVzXCInLFxuICAgICAgICAgICAgJ2RlbCBkb2NrZXIuemlwJyxcbiAgICAgICAgICAgIC8vIGFkZCB0byBwYXRoXG4gICAgICAgICAgICAnJHBlcnNpc3RlZFBhdGhzID0gW0Vudmlyb25tZW50XTo6R2V0RW52aXJvbm1lbnRWYXJpYWJsZShcXCdQYXRoXFwnLCBbRW52aXJvbm1lbnRWYXJpYWJsZVRhcmdldF06Ok1hY2hpbmUpJyxcbiAgICAgICAgICAgICdbRW52aXJvbm1lbnRdOjpTZXRFbnZpcm9ubWVudFZhcmlhYmxlKFwiUEFUSFwiLCAkcGVyc2lzdGVkUGF0aHMgKyBcIjskRW52OlByb2dyYW1GaWxlc1xcXFxEb2NrZXJcIiwgW0Vudmlyb25tZW50VmFyaWFibGVUYXJnZXRdOjpNYWNoaW5lKScsXG4gICAgICAgICAgICAnJGVudjpQQVRIID0gJGVudjpQQVRIICsgXCI7JEVudjpQcm9ncmFtRmlsZXNcXFxcRG9ja2VyXCInLFxuICAgICAgICAgICAgLy8gcmVnaXN0ZXIgZG9ja2VyIHNlcnZpY2VcbiAgICAgICAgICAgICdkb2NrZXJkIC0tcmVnaXN0ZXItc2VydmljZScsXG4gICAgICAgICAgICAnaWYgKCRMQVNURVhJVENPREUgLW5lIDApIHsgdGhyb3cgXCJFeGl0IGNvZGUgaXMgJExBU1RFWElUQ09ERVwiIH0nLFxuICAgICAgICAgICAgLy8gZW5hYmxlIGNvbnRhaW5lcnMgZmVhdHVyZVxuICAgICAgICAgICAgJ0VuYWJsZS1XaW5kb3dzT3B0aW9uYWxGZWF0dXJlIC1PbmxpbmUgLUZlYXR1cmVOYW1lIGNvbnRhaW5lcnMgLUFsbCAtTm9SZXN0YXJ0JyxcbiAgICAgICAgICAgIC8vIGluc3RhbGwgZG9ja2VyLWNvbXBvc2VcbiAgICAgICAgICAgICdjbWQgL2MgY3VybCAtdyBcIiV7cmVkaXJlY3RfdXJsfVwiIC1mc1MgaHR0cHM6Ly9naXRodWIuY29tL2RvY2tlci9jb21wb3NlL3JlbGVhc2VzL2xhdGVzdCA+ICRFbnY6VEVNUFxcXFxsYXRlc3QtZG9ja2VyLWNvbXBvc2UnLFxuICAgICAgICAgICAgJyRMYXRlc3RVcmwgPSBHZXQtQ29udGVudCAkRW52OlRFTVBcXFxcbGF0ZXN0LWRvY2tlci1jb21wb3NlJyxcbiAgICAgICAgICAgICckTGF0ZXN0RG9ja2VyQ29tcG9zZSA9ICgkTGF0ZXN0VXJsIC1TcGxpdCBcXCcvXFwnKVstMV0nLFxuICAgICAgICAgICAgJ0ludm9rZS1XZWJSZXF1ZXN0IC1Vc2VCYXNpY1BhcnNpbmcgLVVyaSAgXCJodHRwczovL2dpdGh1Yi5jb20vZG9ja2VyL2NvbXBvc2UvcmVsZWFzZXMvZG93bmxvYWQvJHtMYXRlc3REb2NrZXJDb21wb3NlfS9kb2NrZXItY29tcG9zZS1XaW5kb3dzLXg4Nl82NC5leGVcIiAtT3V0RmlsZSAkRW52OlByb2dyYW1GaWxlc1xcXFxEb2NrZXJcXFxcZG9ja2VyLWNvbXBvc2UuZXhlJyxcbiAgICAgICAgICAgICdOZXctSXRlbSAtSXRlbVR5cGUgZGlyZWN0b3J5IC1QYXRoIFwiJEVudjpQcm9ncmFtRmlsZXNcXFxcRG9ja2VyXFxcXGNsaS1wbHVnaW5zXCInLFxuICAgICAgICAgICAgJ0NvcHktSXRlbSAtUGF0aCBcIiRFbnY6UHJvZ3JhbUZpbGVzXFxcXERvY2tlclxcXFxkb2NrZXItY29tcG9zZS5leGVcIiAtRGVzdGluYXRpb24gXCIkRW52OlByb2dyYW1GaWxlc1xcXFxEb2NrZXJcXFxcY2xpLXBsdWdpbnNcXFxcZG9ja2VyLWNvbXBvc2UuZXhlXCInLFxuICAgICAgICAgIF07XG4gICAgICAgIH1cblxuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFVua25vd24gb3MvYXJjaGl0ZWN0dXJlIGNvbWJvIGZvciBkb2NrZXI6ICR7b3MubmFtZX0vJHthcmNoaXRlY3R1cmUubmFtZX1gKTtcbiAgICAgIH1cblxuICAgICAgc2hvdWxkUmVib290KG9zOiBPcywgX2FyY2hpdGVjdHVyZTogQXJjaGl0ZWN0dXJlKTogYm9vbGVhbiB7XG4gICAgICAgIHJldHVybiBvcy5pcyhPcy5XSU5ET1dTKTtcbiAgICAgIH1cbiAgICB9KCk7XG4gIH1cblxuICAvKipcbiAgICogQSBjb21wb25lbnQgdG8gaW5zdGFsbCBEb2NrZXItaW4tRG9ja2VyLlxuICAgKlxuICAgKiBAZGVwcmVjYXRlZCB1c2UgYGRvY2tlcigpYFxuICAgKi9cbiAgc3RhdGljIGRvY2tlckluRG9ja2VyKCk6IFJ1bm5lckltYWdlQ29tcG9uZW50IHtcbiAgICByZXR1cm4gUnVubmVySW1hZ2VDb21wb25lbnQuZG9ja2VyKCk7XG4gIH1cblxuICAvKipcbiAgICogQSBjb21wb25lbnQgdG8gYWRkIGEgdHJ1c3RlZCBjZXJ0aWZpY2F0ZSBhdXRob3JpdHkuIFRoaXMgY2FuIGJlIHVzZWQgdG8gc3VwcG9ydCBHaXRIdWIgRW50ZXJwcmlzZSBTZXJ2ZXIgd2l0aCBzZWxmLXNpZ25lZCBjZXJ0aWZpY2F0ZS5cbiAgICpcbiAgICogQHBhcmFtIHNvdXJjZSBwYXRoIHRvIGNlcnRpZmljYXRlIGZpbGUgaW4gUEVNIGZvcm1hdFxuICAgKiBAcGFyYW0gbmFtZSB1bmlxdWUgY2VydGlmaWNhdGUgbmFtZSB0byBiZSB1c2VkIG9uIHJ1bm5lciBmaWxlIHN5c3RlbVxuICAgKi9cbiAgc3RhdGljIGV4dHJhQ2VydGlmaWNhdGVzKHNvdXJjZTogc3RyaW5nLCBuYW1lOiBzdHJpbmcpOiBSdW5uZXJJbWFnZUNvbXBvbmVudCB7XG4gICAgcmV0dXJuIG5ldyBjbGFzcyBleHRlbmRzIFJ1bm5lckltYWdlQ29tcG9uZW50IHtcbiAgICAgIG5hbWUgPSBgRXh0cmEtQ2VydGlmaWNhdGVzLSR7bmFtZX1gO1xuXG4gICAgICBnZXRDb21tYW5kcyhvczogT3MsIGFyY2hpdGVjdHVyZTogQXJjaGl0ZWN0dXJlKSB7XG4gICAgICAgIGlmICghbmFtZS5tYXRjaCgvXlthLXpBLVowLTlfLV0rJC8pKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBJbnZhbGlkIGNlcnRpZmljYXRlIG5hbWU6ICR7bmFtZX0uIE5hbWUgbXVzdCBvbmx5IGNvbnRhaW4gYWxwaGFudW1lcmljIGNoYXJhY3RlcnMsIGRhc2hlcyBhbmQgdW5kZXJzY29yZXMuYCk7XG4gICAgICAgIH1cblxuICAgICAgICBpZiAob3MuaXNJbihPcy5fQUxMX0xJTlVYX1VCVU5UVV9WRVJTSU9OUykpIHtcbiAgICAgICAgICByZXR1cm4gW1xuICAgICAgICAgICAgJ3VwZGF0ZS1jYS1jZXJ0aWZpY2F0ZXMnLFxuICAgICAgICAgIF07XG4gICAgICAgIH0gZWxzZSBpZiAob3MuaXNJbihPcy5fQUxMX0xJTlVYX0FNQVpPTl9WRVJTSU9OUykpIHtcbiAgICAgICAgICByZXR1cm4gW1xuICAgICAgICAgICAgJ3VwZGF0ZS1jYS10cnVzdCcsXG4gICAgICAgICAgXTtcbiAgICAgICAgfSBlbHNlIGlmIChvcy5pcyhPcy5XSU5ET1dTKSkge1xuICAgICAgICAgIHJldHVybiBbXG4gICAgICAgICAgICBgSW1wb3J0LUNlcnRpZmljYXRlIC1GaWxlUGF0aCBDOlxcXFwke25hbWV9LmNydCAtQ2VydFN0b3JlTG9jYXRpb24gQ2VydDpcXFxcTG9jYWxNYWNoaW5lXFxcXFJvb3RgLFxuICAgICAgICAgICAgYFJlbW92ZS1JdGVtIEM6XFxcXCR7bmFtZX0uY3J0YCxcbiAgICAgICAgICBdO1xuICAgICAgICB9XG5cbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBVbmtub3duIG9zL2FyY2hpdGVjdHVyZSBjb21ibyBmb3IgZXh0cmEgY2VydGlmaWNhdGVzOiAke29zLm5hbWV9LyR7YXJjaGl0ZWN0dXJlLm5hbWV9YCk7XG4gICAgICB9XG5cbiAgICAgIGdldEFzc2V0cyhvczogT3MsIF9hcmNoaXRlY3R1cmU6IEFyY2hpdGVjdHVyZSk6IFJ1bm5lckltYWdlQXNzZXRbXSB7XG4gICAgICAgIGlmIChvcy5pc0luKE9zLl9BTExfTElOVVhfVUJVTlRVX1ZFUlNJT05TKSkge1xuICAgICAgICAgIHJldHVybiBbXG4gICAgICAgICAgICB7IHNvdXJjZSwgdGFyZ2V0OiBgL3Vzci9sb2NhbC9zaGFyZS9jYS1jZXJ0aWZpY2F0ZXMvJHtuYW1lfS5jcnRgIH0sXG4gICAgICAgICAgXTtcbiAgICAgICAgfSBlbHNlIGlmIChvcy5pc0luKE9zLl9BTExfTElOVVhfQU1BWk9OX1ZFUlNJT05TKSkge1xuICAgICAgICAgIHJldHVybiBbXG4gICAgICAgICAgICB7IHNvdXJjZSwgdGFyZ2V0OiBgL2V0Yy9wa2kvY2EtdHJ1c3Qvc291cmNlL2FuY2hvcnMvJHtuYW1lfS5jcnRgIH0sXG4gICAgICAgICAgXTtcbiAgICAgICAgfSBlbHNlIGlmIChvcy5pcyhPcy5XSU5ET1dTKSkge1xuICAgICAgICAgIHJldHVybiBbXG4gICAgICAgICAgICB7IHNvdXJjZSwgdGFyZ2V0OiBgQzpcXFxcJHtuYW1lfS5jcnRgIH0sXG4gICAgICAgICAgXTtcbiAgICAgICAgfVxuXG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgVW5zdXBwb3J0ZWQgT1MgZm9yIGV4dHJhIGNlcnRpZmljYXRlczogJHtvcy5uYW1lfWApO1xuICAgICAgfVxuICAgIH0oKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBIGNvbXBvbmVudCB0byBzZXQgdXAgdGhlIHJlcXVpcmVkIExhbWJkYSBlbnRyeXBvaW50IGZvciBMYW1iZGEgcnVubmVycy5cbiAgICovXG4gIHN0YXRpYyBsYW1iZGFFbnRyeXBvaW50KCk6IFJ1bm5lckltYWdlQ29tcG9uZW50IHtcbiAgICByZXR1cm4gbmV3IGNsYXNzIGV4dGVuZHMgUnVubmVySW1hZ2VDb21wb25lbnQge1xuICAgICAgbmFtZSA9ICdMYW1iZGEtRW50cnlwb2ludCc7XG5cbiAgICAgIGdldENvbW1hbmRzKG9zOiBPcywgX2FyY2hpdGVjdHVyZTogQXJjaGl0ZWN0dXJlKSB7XG4gICAgICAgIGlmICghb3MuaXNJbihPcy5fQUxMX0xJTlVYX1ZFUlNJT05TKSkge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgVW5zdXBwb3J0ZWQgT1MgZm9yIExhbWJkYSBlbnRyeXBvaW50OiAke29zLm5hbWV9YCk7XG4gICAgICAgIH1cblxuICAgICAgICByZXR1cm4gW107XG4gICAgICB9XG5cbiAgICAgIGdldEFzc2V0cyhfb3M6IE9zLCBfYXJjaGl0ZWN0dXJlOiBBcmNoaXRlY3R1cmUpOiBSdW5uZXJJbWFnZUFzc2V0W10ge1xuICAgICAgICByZXR1cm4gW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIHNvdXJjZTogcGF0aC5qb2luKF9fZGlybmFtZSwgJy4uJywgJy4uJywgJ2Fzc2V0cycsICdwcm92aWRlcnMnLCAnbGFtYmRhLWJvb3RzdHJhcC5zaCcpLFxuICAgICAgICAgICAgdGFyZ2V0OiAnL2Jvb3RzdHJhcC5zaCcsXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBzb3VyY2U6IHBhdGguam9pbihfX2Rpcm5hbWUsICcuLicsICcuLicsICdhc3NldHMnLCAncHJvdmlkZXJzJywgJ2xhbWJkYS1ydW5uZXIuc2gnKSxcbiAgICAgICAgICAgIHRhcmdldDogJy9ydW5uZXIuc2gnLFxuICAgICAgICAgIH0sXG4gICAgICAgIF07XG4gICAgICB9XG5cbiAgICAgIGdldERvY2tlckNvbW1hbmRzKF9vczogT3MsIF9hcmNoaXRlY3R1cmU6IEFyY2hpdGVjdHVyZSk6IHN0cmluZ1tdIHtcbiAgICAgICAgcmV0dXJuIFtcbiAgICAgICAgICAnTEFCRUwgRElTQUJMRV9TT0NJPTEnLCAvLyBoYWNreSB3YXkgdG8gZGlzYWJsZSBzb2NpIHYyIGluZGV4aW5nIG9uIGxhbWJkYSBhcyBsYW1iZGEgd2lsbCBmYWlsIHRvIHN0YXJ0IHdpdGggYW4gaW5kZXhcbiAgICAgICAgICAnRU5UUllQT0lOVCBbXCJiYXNoXCIsIFwiL2Jvb3RzdHJhcC5zaFwiXScsXG4gICAgICAgIF07XG4gICAgICB9XG4gICAgfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBIGNvbXBvbmVudCB0byBhZGQgZW52aXJvbm1lbnQgdmFyaWFibGVzIGZvciBqb2JzIHRoZSBydW5uZXIgZXhlY3V0ZXMuXG4gICAqXG4gICAqIFRoZXNlIHZhcmlhYmxlcyBvbmx5IGFmZmVjdCB0aGUgam9icyByYW4gYnkgdGhlIHJ1bm5lci4gVGhleSBhcmUgbm90IGdsb2JhbC4gVGhleSBkbyBub3QgYWZmZWN0IG90aGVyIGNvbXBvbmVudHMuXG4gICAqXG4gICAqIEl0IGlzIG5vdCByZWNvbW1lbmRlZCB0byB1c2UgdGhpcyBjb21wb25lbnQgdG8gcGFzcyBzZWNyZXRzLiBJbnN0ZWFkLCB1c2UgR2l0SHViIFNlY3JldHMgb3IgQVdTIFNlY3JldHMgTWFuYWdlci5cbiAgICpcbiAgICogTXVzdCBiZSB1c2VkIGFmdGVyIHRoZSB7QGxpbmsgZ2l0aHViUnVubmVyfSBjb21wb25lbnQuXG4gICAqL1xuICBzdGF0aWMgZW52aXJvbm1lbnRWYXJpYWJsZXModmFyczogUmVjb3JkPHN0cmluZywgc3RyaW5nPik6IFJ1bm5lckltYWdlQ29tcG9uZW50IHtcbiAgICBPYmplY3QuZW50cmllcyh2YXJzKS5mb3JFYWNoKGUgPT4ge1xuICAgICAgaWYgKGVbMF0uaW5jbHVkZXMoJ1xcbicpIHx8IGVbMV0uaW5jbHVkZXMoJ1xcbicpKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgRW52aXJvbm1lbnQgdmFyaWFibGUgY2Fubm90IGNvbnRhaW4gbmV3bGluZXM6ICR7ZX1gKTtcbiAgICAgIH1cbiAgICB9KTtcblxuICAgIHJldHVybiBuZXcgY2xhc3MgZXh0ZW5kcyBSdW5uZXJJbWFnZUNvbXBvbmVudCB7XG4gICAgICBuYW1lID0gJ0Vudmlyb25tZW50VmFyaWFibGVzJztcblxuICAgICAgZ2V0Q29tbWFuZHMob3M6IE9zLCBfYXJjaGl0ZWN0dXJlOiBBcmNoaXRlY3R1cmUpIHtcbiAgICAgICAgaWYgKG9zLmlzSW4oT3MuX0FMTF9MSU5VWF9WRVJTSU9OUykpIHtcbiAgICAgICAgICByZXR1cm4gT2JqZWN0LmVudHJpZXModmFycykubWFwKGUgPT4gYGVjaG8gJyR7ZVswXX09JHtlWzFdLnJlcGxhY2UoLycvZywgXCInXFxcIidcXFwiJ1wiKX0nID4+IC9ob21lL3J1bm5lci8uZW52YCk7XG4gICAgICAgIH0gZWxzZSBpZiAob3MuaXMoT3MuV0lORE9XUykpIHtcbiAgICAgICAgICByZXR1cm4gT2JqZWN0LmVudHJpZXModmFycykubWFwKGUgPT4gYEFkZC1Db250ZW50IC1QYXRoIEM6XFxcXGFjdGlvbnNcXFxcLmVudiAtVmFsdWUgJyR7ZVswXX09JHtlWzFdLnJlcGxhY2UoLycvZywgXCInJ1wiKX0nYCk7XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBVbnN1cHBvcnRlZCBPUyBmb3IgZW52aXJvbm1lbnQgdmFyaWFibGVzIGNvbXBvbmVudDogJHtvcy5uYW1lfWApO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDb21wb25lbnQgbmFtZS5cbiAgICpcbiAgICogVXNlZCB0byBpZGVudGlmeSBjb21wb25lbnQgaW4gaW1hZ2UgYnVpbGQgbG9ncywgYW5kIGZvciB7QGxpbmsgSUNvbmZpZ3VyYWJsZVJ1bm5lckltYWdlQnVpbGRlci5yZW1vdmVDb21wb25lbnR9XG4gICAqL1xuICBhYnN0cmFjdCByZWFkb25seSBuYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFJldHVybnMgY29tbWFuZHMgdG8gcnVuIHRvIGluIGJ1aWx0IGltYWdlLiBDYW4gYmUgdXNlZCB0byBpbnN0YWxsIHBhY2thZ2VzLCBzZXR1cCBidWlsZCBwcmVyZXF1aXNpdGVzLCBldGMuXG4gICAqL1xuICBhYnN0cmFjdCBnZXRDb21tYW5kcyhfb3M6IE9zLCBfYXJjaGl0ZWN0dXJlOiBBcmNoaXRlY3R1cmUpOiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogUmV0dXJucyBhc3NldHMgdG8gY29weSBpbnRvIHRoZSBidWlsdCBpbWFnZS4gQ2FuIGJlIHVzZWQgdG8gY29weSBmaWxlcyBpbnRvIHRoZSBpbWFnZS5cbiAgICovXG4gIGdldEFzc2V0cyhfb3M6IE9zLCBfYXJjaGl0ZWN0dXJlOiBBcmNoaXRlY3R1cmUpOiBSdW5uZXJJbWFnZUFzc2V0W10ge1xuICAgIHJldHVybiBbXTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIERvY2tlciBjb21tYW5kcyB0byBydW4gdG8gaW4gYnVpbHQgaW1hZ2UuIENhbiBiZSB1c2VkIHRvIGFkZCBjb21tYW5kcyBsaWtlIGBWT0xVTUVgLCBgRU5UUllQT0lOVGAsIGBDTURgLCBldGMuXG4gICAqXG4gICAqIERvY2tlciBjb21tYW5kcyBhcmUgYWRkZWQgYWZ0ZXIgYXNzZXRzIGFuZCBub3JtYWwgY29tbWFuZHMuXG4gICAqL1xuICBnZXREb2NrZXJDb21tYW5kcyhfb3M6IE9zLCBfYXJjaGl0ZWN0dXJlOiBBcmNoaXRlY3R1cmUpOiBzdHJpbmdbXSB7XG4gICAgcmV0dXJuIFtdO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdHJ1ZSBpZiB0aGUgaW1hZ2UgYnVpbGRlciBzaG91bGQgYmUgcmVib290ZWQgYWZ0ZXIgdGhpcyBjb21wb25lbnQgaXMgaW5zdGFsbGVkLlxuICAgKi9cbiAgc2hvdWxkUmVib290KF9vczogT3MsIF9hcmNoaXRlY3R1cmU6IEFyY2hpdGVjdHVyZSk6IGJvb2xlYW4ge1xuICAgIHJldHVybiBmYWxzZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDb252ZXJ0IGNvbXBvbmVudCB0byBhbiBBV1MgSW1hZ2UgQnVpbGRlciBjb21wb25lbnQuXG4gICAqXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgX2FzQXdzSW1hZ2VCdWlsZGVyQ29tcG9uZW50KHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIG9zOiBPcywgYXJjaGl0ZWN0dXJlOiBBcmNoaXRlY3R1cmUpIHtcbiAgICBsZXQgcGxhdGZvcm06ICdMaW51eCcgfCAnV2luZG93cyc7XG4gICAgaWYgKG9zLmlzSW4oT3MuX0FMTF9MSU5VWF9VQlVOVFVfVkVSU0lPTlMpIHx8IG9zLmlzSW4oT3MuX0FMTF9MSU5VWF9BTUFaT05fVkVSU0lPTlMpKSB7XG4gICAgICBwbGF0Zm9ybSA9ICdMaW51eCc7XG4gICAgfSBlbHNlIGlmIChvcy5pcyhPcy5XSU5ET1dTKSkge1xuICAgICAgcGxhdGZvcm0gPSAnV2luZG93cyc7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgVW5rbm93biBvcy9hcmNoaXRlY3R1cmUgY29tYm8gZm9yIGltYWdlIGJ1aWxkZXIgY29tcG9uZW50OiAke29zLm5hbWV9LyR7YXJjaGl0ZWN0dXJlLm5hbWV9YCk7XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBJbWFnZUJ1aWxkZXJDb21wb25lbnQoc2NvcGUsIGlkLCB7XG4gICAgICBwbGF0Zm9ybTogcGxhdGZvcm0sXG4gICAgICBjb21tYW5kczogdGhpcy5nZXRDb21tYW5kcyhvcywgYXJjaGl0ZWN0dXJlKSxcbiAgICAgIGFzc2V0czogdGhpcy5nZXRBc3NldHMob3MsIGFyY2hpdGVjdHVyZSkubWFwKChhc3NldCwgaW5kZXgpID0+IHtcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICBhc3NldDogbmV3IHMzX2Fzc2V0cy5Bc3NldChzY29wZSwgYCR7aWR9IGFzc2V0ICR7aW5kZXh9YCwgeyBwYXRoOiBhc3NldC5zb3VyY2UgfSksXG4gICAgICAgICAgcGF0aDogYXNzZXQudGFyZ2V0LFxuICAgICAgICB9O1xuICAgICAgfSksXG4gICAgICBkaXNwbGF5TmFtZTogaWQsXG4gICAgICBkZXNjcmlwdGlvbjogaWQsXG4gICAgICByZWJvb3Q6IHRoaXMuc2hvdWxkUmVib290KG9zLCBhcmNoaXRlY3R1cmUpLFxuICAgIH0pO1xuICB9XG59XG5cbiJdfQ==