"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsImageBuilderFailedBuildNotifier = exports.AwsImageBuilderRunnerImageBuilder = exports.ImageBuilderComponent = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ecr_1 = require("aws-cdk-lib/aws-ecr");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const ami_1 = require("./ami");
const common_1 = require("./common");
const container_1 = require("./container");
const delete_resources_function_1 = require("./delete-resources-function");
const filter_failed_builds_function_1 = require("./filter-failed-builds-function");
const workflow_1 = require("./workflow");
const providers_1 = require("../../providers");
const utils_1 = require("../../utils");
const build_image_function_1 = require("../build-image-function");
const common_2 = require("../common");
/**
 * Components are a set of commands to run and optional files to add to an image. Components are the building blocks of images built by Image Builder.
 *
 * Example:
 *
 * ```
 * new ImageBuilderComponent(this, 'AWS CLI', {
 *   platform: 'Windows',
 *   displayName: 'AWS CLI',
 *   description: 'Install latest version of AWS CLI',
 *   commands: [
 *     '$p = Start-Process msiexec.exe -PassThru -Wait -ArgumentList \'/i https://awscli.amazonaws.com/AWSCLIV2.msi /qn\'',
 *     'if ($p.ExitCode -ne 0) { throw "Exit code is $p.ExitCode" }',
 *   ],
 * }
 * ```
 *
 * @deprecated Use `RunnerImageComponent` instead as this be internal soon.
 */
class ImageBuilderComponent extends common_1.ImageBuilderObjectBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.assets = [];
        this.platform = props.platform;
        let steps = [];
        if (props.assets) {
            let inputs = [];
            let extractCommands = [];
            for (const asset of props.assets) {
                this.assets.push(asset.asset);
                if (asset.asset.isFile) {
                    inputs.push({
                        source: asset.asset.s3ObjectUrl,
                        destination: asset.path,
                    });
                }
                else if (asset.asset.isZipArchive) {
                    inputs.push({
                        source: asset.asset.s3ObjectUrl,
                        destination: `${asset.path}.zip`,
                    });
                    if (props.platform === 'Windows') {
                        extractCommands.push(`Expand-Archive "${asset.path}.zip" -DestinationPath "${asset.path}"`);
                        extractCommands.push(`del "${asset.path}.zip"`);
                    }
                    else {
                        extractCommands.push(`unzip "${asset.path}.zip" -d "${asset.path}"`);
                        extractCommands.push(`rm "${asset.path}.zip"`);
                    }
                }
                else {
                    throw new Error(`Unknown asset type: ${asset.asset}`);
                }
            }
            steps.push({
                name: 'Download',
                action: 'S3Download',
                inputs,
            });
            if (extractCommands.length > 0) {
                steps.push({
                    name: 'Extract',
                    action: props.platform === 'Linux' ? 'ExecuteBash' : 'ExecutePowerShell',
                    inputs: {
                        commands: this.prefixCommandsWithErrorHandling(props.platform, extractCommands),
                    },
                });
            }
        }
        if (props.commands.length > 0) {
            steps.push({
                name: 'Run',
                action: props.platform === 'Linux' ? 'ExecuteBash' : 'ExecutePowerShell',
                inputs: {
                    commands: this.prefixCommandsWithErrorHandling(props.platform, props.commands),
                },
            });
        }
        if (props.reboot ?? false) {
            steps.push({
                name: 'Reboot',
                action: 'Reboot',
                inputs: {},
            });
        }
        const data = {
            name: props.displayName,
            schemaVersion: '1.0',
            phases: [
                {
                    name: 'build',
                    steps,
                },
            ],
        };
        const name = (0, common_2.uniqueImageBuilderName)(this);
        const component = new aws_cdk_lib_1.aws_imagebuilder.CfnComponent(this, 'Component', {
            name: name,
            description: props.description,
            platform: props.platform,
            version: this.generateVersion('Component', name, {
                platform: props.platform,
                data,
                description: props.description,
            }),
            data: JSON.stringify(data),
        });
        this.arn = component.attrArn;
    }
    /**
     * Grants read permissions to the principal on the assets buckets.
     *
     * @param grantee
     */
    grantAssetsRead(grantee) {
        for (const asset of this.assets) {
            asset.grantRead(grantee);
        }
    }
    prefixCommandsWithErrorHandling(platform, commands) {
        if (platform == 'Windows') {
            return [
                '$ErrorActionPreference = \'Stop\'',
                '$ProgressPreference = \'SilentlyContinue\'',
                'Set-PSDebug -Trace 1',
            ].concat(commands);
        }
        else {
            return [
                'set -ex',
            ].concat(commands);
        }
    }
}
exports.ImageBuilderComponent = ImageBuilderComponent;
_a = JSII_RTTI_SYMBOL_1;
ImageBuilderComponent[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.ImageBuilderComponent", version: "0.14.13" };
/**
 * @internal
 */
class AwsImageBuilderRunnerImageBuilder extends common_2.RunnerImageBuilderBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.boundComponents = [];
        if (props?.codeBuildOptions) {
            aws_cdk_lib_1.Annotations.of(this).addWarning('codeBuildOptions are ignored when using AWS Image Builder to build runner images.');
        }
        this.os = props?.os ?? providers_1.Os.LINUX_UBUNTU;
        this.architecture = props?.architecture ?? providers_1.Architecture.X86_64;
        this.rebuildInterval = props?.rebuildInterval ?? aws_cdk_lib_1.Duration.days(7);
        this.logRetention = props?.logRetention ?? aws_logs_1.RetentionDays.ONE_MONTH;
        this.logRemovalPolicy = props?.logRemovalPolicy ?? aws_cdk_lib_1.RemovalPolicy.DESTROY;
        this.vpc = props?.vpc ?? aws_cdk_lib_1.aws_ec2.Vpc.fromLookup(this, 'VPC', { isDefault: true });
        this.securityGroups = props?.securityGroups ?? [new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, 'SG', { vpc: this.vpc })];
        this.subnetSelection = props?.subnetSelection;
        this.baseImage = props?.baseDockerImage ?? (0, container_1.defaultBaseDockerImage)(this.os);
        this.baseAmi = props?.baseAmi ?? (0, ami_1.defaultBaseAmi)(this, this.os, this.architecture);
        this.instanceType = props?.awsImageBuilderOptions?.instanceType ?? aws_cdk_lib_1.aws_ec2.InstanceType.of(aws_cdk_lib_1.aws_ec2.InstanceClass.M6I, aws_cdk_lib_1.aws_ec2.InstanceSize.LARGE);
        this.fastLaunchOptions = props?.awsImageBuilderOptions?.fastLaunchOptions;
        this.storageSize = props?.awsImageBuilderOptions?.storageSize;
        this.waitOnDeploy = props?.waitOnDeploy ?? true;
        this.dockerSetupCommands = props?.dockerSetupCommands ?? [];
        // tags for finding resources
        this.tags = {
            'GitHubRunners:Stack': cdk.Stack.of(this).stackName,
            'GitHubRunners:Builder': this.node.path,
        };
        // confirm instance type
        if (!this.architecture.instanceTypeMatch(this.instanceType)) {
            throw new Error(`Builder architecture (${this.architecture.name}) doesn't match selected instance type (${this.instanceType} / ${this.instanceType.architecture})`);
        }
        // warn against isolated networks
        if (props?.subnetSelection?.subnetType == aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_ISOLATED) {
            aws_cdk_lib_1.Annotations.of(this).addWarning('Private isolated subnets cannot pull from public ECR and VPC endpoint is not supported yet. ' +
                'See https://github.com/aws/containers-roadmap/issues/1160');
        }
        // role to be used by AWS Image Builder
        this.role = new aws_cdk_lib_1.aws_iam.Role(this, 'Role', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('ec2.amazonaws.com'),
        });
    }
    platform() {
        if (this.os.is(providers_1.Os.WINDOWS)) {
            return 'Windows';
        }
        if (this.os.isIn(providers_1.Os._ALL_LINUX_VERSIONS)) {
            return 'Linux';
        }
        throw new Error(`OS ${this.os.name} is not supported by AWS Image Builder`);
    }
    /**
     * Called by IRunnerProvider to finalize settings and create the image builder.
     */
    bindDockerImage() {
        if (this.boundDockerImage) {
            return this.boundDockerImage;
        }
        // create repository that only keeps one tag
        const repository = new aws_cdk_lib_1.aws_ecr.Repository(this, 'Repository', {
            imageScanOnPush: true,
            imageTagMutability: aws_ecr_1.TagMutability.MUTABLE,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            emptyOnDelete: true,
        });
        const dist = new aws_cdk_lib_1.aws_imagebuilder.CfnDistributionConfiguration(this, 'Docker Distribution', {
            name: (0, common_2.uniqueImageBuilderName)(this),
            // description: this.description,
            distributions: [
                {
                    region: aws_cdk_lib_1.Stack.of(this).region,
                    containerDistributionConfiguration: {
                        ContainerTags: ['latest'],
                        TargetRepository: {
                            Service: 'ECR',
                            RepositoryName: repository.repositoryName,
                        },
                    },
                },
            ],
            tags: this.tags,
        });
        let dockerfileTemplate = `FROM {{{ imagebuilder:parentImage }}}
{{{ imagebuilder:environments }}}
{{{ imagebuilder:components }}}`;
        for (const c of this.components) {
            const commands = c.getDockerCommands(this.os, this.architecture);
            if (commands.length > 0) {
                dockerfileTemplate += '\n' + commands.join('\n') + '\n';
            }
        }
        const recipe = new container_1.ContainerRecipe(this, 'Container Recipe', {
            platform: this.platform(),
            components: this.bindComponents(),
            targetRepository: repository,
            dockerfileTemplate: dockerfileTemplate,
            parentImage: this.baseImage,
            tags: this.tags,
        });
        const log = this.createLog('Docker Log', recipe.name);
        const infra = this.createInfrastructure([
            aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
            aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilderECRContainerBuilds'),
        ]);
        if (this.waitOnDeploy) {
            this.createImage(infra, dist, log, undefined, recipe.arn);
        }
        this.dockerImageCleaner(recipe, repository);
        this.createPipeline(infra, dist, log, undefined, recipe.arn);
        this.boundDockerImage = {
            imageRepository: repository,
            imageTag: 'latest',
            os: this.os,
            architecture: this.architecture,
            logGroup: log,
            runnerVersion: providers_1.RunnerVersion.specific('unknown'),
            // no dependable as CloudFormation will fail to get image ARN once the image is deleted (we delete old images daily)
        };
        return this.boundDockerImage;
    }
    dockerImageCleaner(recipe, repository) {
        // this is here to provide safe upgrade from old cdk-github-runners versions
        // this lambda was used by a custom resource to delete all images builds on cleanup
        // if we remove the custom resource and the lambda, the old images will be deleted on update
        // keeping the lambda but removing the permissions will make sure that deletion will fail
        const oldDeleter = (0, utils_1.singletonLambda)(build_image_function_1.BuildImageFunction, this, 'build-image', {
            description: 'Custom resource handler that triggers CodeBuild to build runner images',
            timeout: cdk.Duration.minutes(3),
            logGroup: (0, utils_1.singletonLogGroup)(this, utils_1.SingletonLogType.RUNNER_IMAGE_BUILD),
            loggingFormat: aws_cdk_lib_1.aws_lambda.LoggingFormat.JSON,
        });
        oldDeleter.addToRolePolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            effect: aws_cdk_lib_1.aws_iam.Effect.DENY,
            actions: ['imagebuilder:DeleteImage'],
            resources: ['*'],
        }));
        // delete old version on update and on stack deletion
        this.imageCleaner('Container', recipe.name.toLowerCase(), recipe.version);
        // delete old docker images + IB resources daily
        new aws_cdk_lib_1.aws_imagebuilder.CfnLifecyclePolicy(this, 'Lifecycle Policy Docker', {
            name: (0, common_2.uniqueImageBuilderName)(this),
            description: `Delete old GitHub Runner Docker images for ${this.node.path}`,
            executionRole: new aws_cdk_lib_1.aws_iam.Role(this, 'Lifecycle Policy Docker Role', {
                assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('imagebuilder.amazonaws.com'),
                inlinePolicies: {
                    ib: new aws_cdk_lib_1.aws_iam.PolicyDocument({
                        statements: [
                            new aws_cdk_lib_1.aws_iam.PolicyStatement({
                                actions: ['tag:GetResources', 'imagebuilder:DeleteImage'],
                                resources: ['*'], // Image Builder doesn't support scoping this :(
                            }),
                        ],
                    }),
                    ecr: new aws_cdk_lib_1.aws_iam.PolicyDocument({
                        statements: [
                            new aws_cdk_lib_1.aws_iam.PolicyStatement({
                                actions: ['ecr:BatchGetImage', 'ecr:BatchDeleteImage'],
                                resources: [repository.repositoryArn],
                            }),
                        ],
                    }),
                },
            }).roleArn,
            policyDetails: [{
                    action: {
                        type: 'DELETE',
                        includeResources: {
                            containers: true,
                        },
                    },
                    filter: {
                        type: 'COUNT',
                        value: 2,
                    },
                }],
            resourceType: 'CONTAINER_IMAGE',
            resourceSelection: {
                recipes: [
                    {
                        name: recipe.name,
                        semanticVersion: recipe.version,
                    },
                ],
            },
        });
    }
    createLog(id, recipeName) {
        return new aws_cdk_lib_1.aws_logs.LogGroup(this, id, {
            logGroupName: `/aws/imagebuilder/${recipeName}`,
            retention: this.logRetention,
            removalPolicy: this.logRemovalPolicy,
        });
    }
    createInfrastructure(managedPolicies) {
        if (this.infrastructure) {
            return this.infrastructure;
        }
        for (const managedPolicy of managedPolicies) {
            this.role.addManagedPolicy(managedPolicy);
        }
        for (const component of this.boundComponents) {
            component.grantAssetsRead(this.role);
        }
        this.infrastructure = new aws_cdk_lib_1.aws_imagebuilder.CfnInfrastructureConfiguration(this, 'Infrastructure', {
            name: (0, common_2.uniqueImageBuilderName)(this),
            // description: this.description,
            subnetId: this.vpc?.selectSubnets(this.subnetSelection).subnetIds[0],
            securityGroupIds: this.securityGroups?.map(sg => sg.securityGroupId),
            instanceTypes: [this.instanceType.toString()],
            instanceMetadataOptions: {
                httpTokens: 'required',
                // Container builds require a minimum of two hops.
                httpPutResponseHopLimit: 2,
            },
            instanceProfileName: new aws_cdk_lib_1.aws_iam.CfnInstanceProfile(this, 'Instance Profile', {
                roles: [
                    this.role.roleName,
                ],
            }).ref,
        });
        return this.infrastructure;
    }
    createImage(infra, dist, log, imageRecipeArn, containerRecipeArn) {
        const image = new aws_cdk_lib_1.aws_imagebuilder.CfnImage(this, this.amiOrContainerId('Image', imageRecipeArn, containerRecipeArn), {
            infrastructureConfigurationArn: infra.attrArn,
            distributionConfigurationArn: dist.attrArn,
            imageRecipeArn,
            containerRecipeArn,
            imageTestsConfiguration: {
                imageTestsEnabled: false,
            },
            tags: this.tags,
        });
        image.node.addDependency(infra);
        image.node.addDependency(log);
        // do not delete the image as it will be deleted by imageCleaner().
        // if we delete it here, imageCleaner() won't be able to find the image.
        // if imageCleaner() can't find the image, it won't be able to delete the linked AMI/Docker image.
        // use RETAIN_ON_UPDATE_OR_DELETE, so everything is cleaned only on rollback.
        image.applyRemovalPolicy(aws_cdk_lib_1.RemovalPolicy.RETAIN_ON_UPDATE_OR_DELETE);
        return image;
    }
    amiOrContainerId(baseId, imageRecipeArn, containerRecipeArn) {
        if (imageRecipeArn) {
            return `AMI ${baseId}`;
        }
        if (containerRecipeArn) {
            return `Docker ${baseId}`;
        }
        throw new Error('Either imageRecipeArn or containerRecipeArn must be defined');
    }
    createPipeline(infra, dist, log, imageRecipeArn, containerRecipeArn) {
        // set schedule
        let scheduleOptions;
        if (this.rebuildInterval.toDays() > 0) {
            scheduleOptions = {
                scheduleExpression: aws_cdk_lib_1.aws_events.Schedule.rate(this.rebuildInterval).expressionString,
                pipelineExecutionStartCondition: 'EXPRESSION_MATCH_ONLY',
            };
        }
        // generate workflows, if needed
        let workflows;
        let executionRole;
        if (this.dockerSetupCommands.length > 0) {
            workflows = [{
                    workflowArn: (0, workflow_1.generateBuildWorkflowWithDockerSetupCommands)(this, 'Build', this.dockerSetupCommands).arn,
                }];
            executionRole = aws_cdk_lib_1.aws_iam.Role.fromRoleArn(this, 'Image Builder Role', cdk.Stack.of(this).formatArn({
                service: 'iam',
                region: '',
                resource: 'role',
                resourceName: 'aws-service-role/imagebuilder.amazonaws.com/AWSServiceRoleForImageBuilder',
            }));
        }
        // generate pipeline
        const pipeline = new aws_cdk_lib_1.aws_imagebuilder.CfnImagePipeline(this, this.amiOrContainerId('Pipeline', imageRecipeArn, containerRecipeArn), {
            name: (0, common_2.uniqueImageBuilderName)(this),
            // description: this.description,
            infrastructureConfigurationArn: infra.attrArn,
            distributionConfigurationArn: dist.attrArn,
            imageRecipeArn,
            containerRecipeArn,
            schedule: scheduleOptions,
            imageTestsConfiguration: {
                imageTestsEnabled: false,
            },
            workflows: workflows,
            executionRole: executionRole?.roleArn,
            tags: this.tags,
        });
        pipeline.node.addDependency(infra);
        pipeline.node.addDependency(log);
        return pipeline;
    }
    /**
     * The network connections associated with this resource.
     */
    get connections() {
        return new aws_cdk_lib_1.aws_ec2.Connections({ securityGroups: this.securityGroups });
    }
    get grantPrincipal() {
        return this.role;
    }
    bindAmi() {
        if (this.boundAmi) {
            return this.boundAmi;
        }
        const launchTemplate = new aws_cdk_lib_1.aws_ec2.LaunchTemplate(this, 'Launch template', {
            requireImdsv2: true,
        });
        const launchTemplateConfigs = [{
                launchTemplateId: launchTemplate.launchTemplateId,
                setDefaultVersion: true,
            }];
        const fastLaunchConfigs = [];
        if (this.fastLaunchOptions?.enabled ?? false) {
            if (!this.os.is(providers_1.Os.WINDOWS)) {
                throw new Error('Fast launch is only supported for Windows');
            }
            // create a separate launch template for fast launch so:
            //  - settings don't affect the runners
            //  - enabling fast launch on an existing builder works (without a new launch template, EC2 Image Builder will use the first version of the launch template, which doesn't have instance or VPC config)
            //  - setting vpc + subnet on the main launch template will cause RunInstances to fail
            //  - EC2 Image Builder seems to get confused with which launch template version to base any new version on, so a new template is always best
            const fastLaunchTemplate = new aws_cdk_lib_1.aws_ec2.CfnLaunchTemplate(this, 'Fast Launch Template', {
                launchTemplateData: {
                    metadataOptions: {
                        httpTokens: 'required',
                    },
                    instanceType: this.instanceType.toString(),
                    networkInterfaces: [{
                            subnetId: this.vpc?.selectSubnets(this.subnetSelection).subnetIds[0],
                            deviceIndex: 0,
                            groups: this.securityGroups.map(sg => sg.securityGroupId),
                        }],
                    tagSpecifications: [
                        {
                            resourceType: 'instance',
                            tags: [{
                                    key: 'Name',
                                    value: `${this.node.path}/Fast Launch Instance`,
                                }],
                        },
                        {
                            resourceType: 'volume',
                            tags: [{
                                    key: 'Name',
                                    value: `${this.node.path}/Fast Launch Instance`,
                                }],
                        },
                    ],
                },
                tagSpecifications: [{
                        resourceType: 'launch-template',
                        tags: [{
                                key: 'Name',
                                value: `${this.node.path}/Fast Launch Template`,
                            }],
                    }],
            });
            launchTemplateConfigs.push({
                launchTemplateId: fastLaunchTemplate.attrLaunchTemplateId,
                setDefaultVersion: true,
            });
            fastLaunchConfigs.push({
                enabled: true,
                launchTemplate: {
                    launchTemplateId: fastLaunchTemplate.attrLaunchTemplateId,
                },
                maxParallelLaunches: this.fastLaunchOptions?.maxParallelLaunches ?? 6,
                snapshotConfiguration: {
                    targetResourceCount: this.fastLaunchOptions?.targetResourceCount ?? 1,
                },
            });
        }
        const stackName = cdk.Stack.of(this).stackName;
        const builderName = this.node.path;
        const dist = new aws_cdk_lib_1.aws_imagebuilder.CfnDistributionConfiguration(this, 'AMI Distribution', {
            name: (0, common_2.uniqueImageBuilderName)(this),
            // description: this.description,
            distributions: [
                {
                    region: aws_cdk_lib_1.Stack.of(this).region,
                    amiDistributionConfiguration: {
                        Name: `${cdk.Names.uniqueResourceName(this, {
                            maxLength: 100,
                            separator: '-',
                            allowedSpecialCharacters: '_-',
                        })}-{{ imagebuilder:buildDate }}`,
                        AmiTags: {
                            'Name': this.node.id,
                            'GitHubRunners:Stack': stackName,
                            'GitHubRunners:Builder': builderName,
                        },
                    },
                    launchTemplateConfigurations: launchTemplateConfigs,
                    fastLaunchConfigurations: fastLaunchConfigs.length > 0 ? fastLaunchConfigs : undefined,
                },
            ],
            tags: this.tags,
        });
        const recipe = new ami_1.AmiRecipe(this, 'Ami Recipe', {
            platform: this.platform(),
            components: this.bindComponents(),
            architecture: this.architecture,
            baseAmi: this.baseAmi,
            storageSize: this.storageSize,
            tags: this.tags,
        });
        const log = this.createLog('Ami Log', recipe.name);
        const infra = this.createInfrastructure([
            aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
            aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilder'),
        ]);
        if (this.waitOnDeploy) {
            this.createImage(infra, dist, log, recipe.arn, undefined);
        }
        this.createPipeline(infra, dist, log, recipe.arn, undefined);
        this.boundAmi = {
            launchTemplate: launchTemplate,
            architecture: this.architecture,
            os: this.os,
            logGroup: log,
            runnerVersion: providers_1.RunnerVersion.specific('unknown'),
        };
        this.amiCleaner(recipe, stackName, builderName);
        return this.boundAmi;
    }
    amiCleaner(recipe, stackName, builderName) {
        // this is here to provide safe upgrade from old cdk-github-runners versions
        // this lambda was used by a custom resource to delete all amis when the builder was removed
        // if we remove the custom resource, role and lambda, all amis will be deleted on update
        // keeping the just role but removing the permissions along with the custom resource will make sure that deletion will fail
        const stack = cdk.Stack.of(this);
        if (stack.node.tryFindChild('delete-ami-dcc036c8-876b-451e-a2c1-552f9e06e9e1') == undefined) {
            const role = new aws_cdk_lib_1.aws_iam.Role(stack, 'delete-ami-dcc036c8-876b-451e-a2c1-552f9e06e9e1', {
                description: 'Empty role to prevent deletion of AMIs on cdk-github-runners upgrade',
                assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('lambda.amazonaws.com'),
                inlinePolicies: {
                    deny: new aws_cdk_lib_1.aws_iam.PolicyDocument({
                        statements: [
                            new aws_cdk_lib_1.aws_iam.PolicyStatement({
                                actions: ['ec2:DeregisterImage', 'ec2:DeleteSnapshot'],
                                resources: ['*'],
                                effect: aws_cdk_lib_1.aws_iam.Effect.DENY,
                            }),
                        ],
                    }),
                },
            });
            const l1role = role.node.defaultChild;
            l1role.overrideLogicalId('deleteamidcc036c8876b451ea2c1552f9e06e9e1ServiceRole1CC58A6F');
        }
        // delete old version on update and on stack deletion
        this.imageCleaner('Image', recipe.name.toLowerCase(), recipe.version);
        // delete old AMIs + IB resources daily
        new aws_cdk_lib_1.aws_imagebuilder.CfnLifecyclePolicy(this, 'Lifecycle Policy AMI', {
            name: (0, common_2.uniqueImageBuilderName)(this),
            description: `Delete old GitHub Runner AMIs for ${this.node.path}`,
            executionRole: new aws_cdk_lib_1.aws_iam.Role(this, 'Lifecycle Policy AMI Role', {
                assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('imagebuilder.amazonaws.com'),
                inlinePolicies: {
                    ib: new aws_cdk_lib_1.aws_iam.PolicyDocument({
                        statements: [
                            new aws_cdk_lib_1.aws_iam.PolicyStatement({
                                actions: ['tag:GetResources', 'imagebuilder:DeleteImage'],
                                resources: ['*'], // Image Builder doesn't support scoping this :(
                            }),
                        ],
                    }),
                    ami: new aws_cdk_lib_1.aws_iam.PolicyDocument({
                        statements: [
                            new aws_cdk_lib_1.aws_iam.PolicyStatement({
                                actions: ['ec2:DescribeImages', 'ec2:DescribeImageAttribute'],
                                resources: ['*'],
                            }),
                            new aws_cdk_lib_1.aws_iam.PolicyStatement({
                                actions: ['ec2:DeregisterImage', 'ec2:DeleteSnapshot'],
                                resources: ['*'],
                                conditions: {
                                    StringEquals: {
                                        'aws:ResourceTag/GitHubRunners:Stack': stackName,
                                        'aws:ResourceTag/GitHubRunners:Builder': builderName,
                                    },
                                },
                            }),
                        ],
                    }),
                },
            }).roleArn,
            policyDetails: [{
                    action: {
                        type: 'DELETE',
                        includeResources: {
                            amis: true,
                            snapshots: true,
                        },
                    },
                    filter: {
                        type: 'COUNT',
                        value: 2,
                    },
                }],
            resourceType: 'AMI_IMAGE',
            resourceSelection: {
                recipes: [
                    {
                        name: recipe.name,
                        semanticVersion: recipe.version, // docs say it's optional, but it's not
                    },
                ],
            },
        });
    }
    bindComponents() {
        if (this.boundComponents.length == 0) {
            this.boundComponents.push(...this.components.map((c, i) => c._asAwsImageBuilderComponent(this, `Component ${i} ${c.name}`, this.os, this.architecture)));
        }
        return this.boundComponents;
    }
    imageCleaner(type, recipeName, version) {
        const cleanerFunction = (0, utils_1.singletonLambda)(delete_resources_function_1.DeleteResourcesFunction, this, 'aws-image-builder-delete-resources', {
            description: 'Custom resource handler that deletes resources of old versions of EC2 Image Builder images',
            initialPolicy: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: [
                        'imagebuilder:ListImageBuildVersions',
                        'imagebuilder:DeleteImage',
                    ],
                    resources: ['*'],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['ec2:DescribeImages'],
                    resources: ['*'],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['ec2:DeregisterImage', 'ec2:DeleteSnapshot'],
                    resources: ['*'],
                    conditions: {
                        StringEquals: {
                            'aws:ResourceTag/GitHubRunners:Stack': cdk.Stack.of(this).stackName,
                        },
                    },
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['ecr:BatchDeleteImage'],
                    resources: ['*'],
                }),
            ],
            logGroup: (0, utils_1.singletonLogGroup)(this, utils_1.SingletonLogType.RUNNER_IMAGE_BUILD),
            loggingFormat: aws_cdk_lib_1.aws_lambda.LoggingFormat.JSON,
            timeout: cdk.Duration.minutes(10),
        });
        new aws_cdk_lib_1.CustomResource(this, `${type} Cleaner`, {
            serviceToken: cleanerFunction.functionArn,
            resourceType: 'Custom::ImageBuilder-Delete-Resources',
            properties: {
                ImageVersionArn: cdk.Stack.of(this).formatArn({
                    service: 'imagebuilder',
                    resource: 'image',
                    resourceName: `${recipeName}/${version}`,
                    arnFormat: cdk.ArnFormat.SLASH_RESOURCE_NAME,
                }),
            },
        });
    }
}
exports.AwsImageBuilderRunnerImageBuilder = AwsImageBuilderRunnerImageBuilder;
/**
 * @internal
 */
class AwsImageBuilderFailedBuildNotifier {
    static createFilteringTopic(scope, targetTopic) {
        const topic = new aws_cdk_lib_1.aws_sns.Topic(scope, 'Image Builder Builds');
        const filter = new filter_failed_builds_function_1.FilterFailedBuildsFunction(scope, 'Image Builder Builds Filter', {
            logGroup: (0, utils_1.singletonLogGroup)(scope, utils_1.SingletonLogType.RUNNER_IMAGE_BUILD),
            loggingFormat: aws_cdk_lib_1.aws_lambda.LoggingFormat.JSON,
            environment: {
                TARGET_TOPIC_ARN: targetTopic.topicArn,
            },
        });
        topic.addSubscription(new aws_cdk_lib_1.aws_sns_subscriptions.LambdaSubscription(filter));
        targetTopic.grantPublish(filter);
        return topic;
    }
    constructor(topic) {
        this.topic = topic;
    }
    visit(node) {
        if (node instanceof AwsImageBuilderRunnerImageBuilder) {
            const builder = node;
            const infraNode = builder.node.tryFindChild('Infrastructure');
            if (infraNode) {
                const infra = infraNode;
                infra.snsTopicArn = this.topic.topicArn;
            }
            else {
                cdk.Annotations.of(builder).addWarning('Unused builder cannot get notifications of failed builds');
            }
        }
    }
}
exports.AwsImageBuilderFailedBuildNotifier = AwsImageBuilderFailedBuildNotifier;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYnVpbGRlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9pbWFnZS1idWlsZGVycy9hd3MtaW1hZ2UtYnVpbGRlci9idWlsZGVyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsbUNBQW1DO0FBQ25DLDZDQWdCcUI7QUFDckIsaURBQW9EO0FBQ3BELG1EQUFxRDtBQUVyRCwrQkFBa0Q7QUFDbEQscUNBQWtEO0FBQ2xELDJDQUFzRTtBQUN0RSwyRUFBc0U7QUFFdEUsbUZBQTZFO0FBQzdFLHlDQUEwRTtBQUMxRSwrQ0FBMEY7QUFDMUYsdUNBQW1GO0FBQ25GLGtFQUE2RDtBQUM3RCxzQ0FBb0c7QUFvSHBHOzs7Ozs7Ozs7Ozs7Ozs7Ozs7R0FrQkc7QUFDSCxNQUFhLHFCQUFzQixTQUFRLCtCQUFzQjtJQWEvRCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXNDO1FBQzlFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFIRixXQUFNLEdBQXNCLEVBQUUsQ0FBQztRQUs5QyxJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUM7UUFFL0IsSUFBSSxLQUFLLEdBQVUsRUFBRSxDQUFDO1FBRXRCLElBQUksS0FBSyxDQUFDLE1BQU0sRUFBRSxDQUFDO1lBQ2pCLElBQUksTUFBTSxHQUFVLEVBQUUsQ0FBQztZQUN2QixJQUFJLGVBQWUsR0FBYSxFQUFFLENBQUM7WUFDbkMsS0FBSyxNQUFNLEtBQUssSUFBSSxLQUFLLENBQUMsTUFBTSxFQUFFLENBQUM7Z0JBQ2pDLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztnQkFFOUIsSUFBSSxLQUFLLENBQUMsS0FBSyxDQUFDLE1BQU0sRUFBRSxDQUFDO29CQUN2QixNQUFNLENBQUMsSUFBSSxDQUFDO3dCQUNWLE1BQU0sRUFBRSxLQUFLLENBQUMsS0FBSyxDQUFDLFdBQVc7d0JBQy9CLFdBQVcsRUFBRSxLQUFLLENBQUMsSUFBSTtxQkFDeEIsQ0FBQyxDQUFDO2dCQUNMLENBQUM7cUJBQU0sSUFBSSxLQUFLLENBQUMsS0FBSyxDQUFDLFlBQVksRUFBRSxDQUFDO29CQUNwQyxNQUFNLENBQUMsSUFBSSxDQUFDO3dCQUNWLE1BQU0sRUFBRSxLQUFLLENBQUMsS0FBSyxDQUFDLFdBQVc7d0JBQy9CLFdBQVcsRUFBRSxHQUFHLEtBQUssQ0FBQyxJQUFJLE1BQU07cUJBQ2pDLENBQUMsQ0FBQztvQkFDSCxJQUFJLEtBQUssQ0FBQyxRQUFRLEtBQUssU0FBUyxFQUFFLENBQUM7d0JBQ2pDLGVBQWUsQ0FBQyxJQUFJLENBQUMsbUJBQW1CLEtBQUssQ0FBQyxJQUFJLDJCQUEyQixLQUFLLENBQUMsSUFBSSxHQUFHLENBQUMsQ0FBQzt3QkFDNUYsZUFBZSxDQUFDLElBQUksQ0FBQyxRQUFRLEtBQUssQ0FBQyxJQUFJLE9BQU8sQ0FBQyxDQUFDO29CQUNsRCxDQUFDO3lCQUFNLENBQUM7d0JBQ04sZUFBZSxDQUFDLElBQUksQ0FBQyxVQUFVLEtBQUssQ0FBQyxJQUFJLGFBQWEsS0FBSyxDQUFDLElBQUksR0FBRyxDQUFDLENBQUM7d0JBQ3JFLGVBQWUsQ0FBQyxJQUFJLENBQUMsT0FBTyxLQUFLLENBQUMsSUFBSSxPQUFPLENBQUMsQ0FBQztvQkFDakQsQ0FBQztnQkFDSCxDQUFDO3FCQUFNLENBQUM7b0JBQ04sTUFBTSxJQUFJLEtBQUssQ0FBQyx1QkFBdUIsS0FBSyxDQUFDLEtBQUssRUFBRSxDQUFDLENBQUM7Z0JBQ3hELENBQUM7WUFDSCxDQUFDO1lBRUQsS0FBSyxDQUFDLElBQUksQ0FBQztnQkFDVCxJQUFJLEVBQUUsVUFBVTtnQkFDaEIsTUFBTSxFQUFFLFlBQVk7Z0JBQ3BCLE1BQU07YUFDUCxDQUFDLENBQUM7WUFFSCxJQUFJLGVBQWUsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFLENBQUM7Z0JBQy9CLEtBQUssQ0FBQyxJQUFJLENBQUM7b0JBQ1QsSUFBSSxFQUFFLFNBQVM7b0JBQ2YsTUFBTSxFQUFFLEtBQUssQ0FBQyxRQUFRLEtBQUssT0FBTyxDQUFDLENBQUMsQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLG1CQUFtQjtvQkFDeEUsTUFBTSxFQUFFO3dCQUNOLFFBQVEsRUFBRSxJQUFJLENBQUMsK0JBQStCLENBQUMsS0FBSyxDQUFDLFFBQVEsRUFBRSxlQUFlLENBQUM7cUJBQ2hGO2lCQUNGLENBQUMsQ0FBQztZQUNMLENBQUM7UUFDSCxDQUFDO1FBRUQsSUFBSSxLQUFLLENBQUMsUUFBUSxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUUsQ0FBQztZQUM5QixLQUFLLENBQUMsSUFBSSxDQUFDO2dCQUNULElBQUksRUFBRSxLQUFLO2dCQUNYLE1BQU0sRUFBRSxLQUFLLENBQUMsUUFBUSxLQUFLLE9BQU8sQ0FBQyxDQUFDLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxtQkFBbUI7Z0JBQ3hFLE1BQU0sRUFBRTtvQkFDTixRQUFRLEVBQUUsSUFBSSxDQUFDLCtCQUErQixDQUFDLEtBQUssQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLFFBQVEsQ0FBQztpQkFDL0U7YUFDRixDQUFDLENBQUM7UUFDTCxDQUFDO1FBRUQsSUFBSSxLQUFLLENBQUMsTUFBTSxJQUFJLEtBQUssRUFBRSxDQUFDO1lBQzFCLEtBQUssQ0FBQyxJQUFJLENBQUM7Z0JBQ1QsSUFBSSxFQUFFLFFBQVE7Z0JBQ2QsTUFBTSxFQUFFLFFBQVE7Z0JBQ2hCLE1BQU0sRUFBRSxFQUFFO2FBQ1gsQ0FBQyxDQUFDO1FBQ0wsQ0FBQztRQUVELE1BQU0sSUFBSSxHQUFHO1lBQ1gsSUFBSSxFQUFFLEtBQUssQ0FBQyxXQUFXO1lBQ3ZCLGFBQWEsRUFBRSxLQUFLO1lBQ3BCLE1BQU0sRUFBRTtnQkFDTjtvQkFDRSxJQUFJLEVBQUUsT0FBTztvQkFDYixLQUFLO2lCQUNOO2FBQ0Y7U0FDRixDQUFDO1FBRUYsTUFBTSxJQUFJLEdBQUcsSUFBQSwrQkFBc0IsRUFBQyxJQUFJLENBQUMsQ0FBQztRQUMxQyxNQUFNLFNBQVMsR0FBRyxJQUFJLDhCQUFZLENBQUMsWUFBWSxDQUFDLElBQUksRUFBRSxXQUFXLEVBQUU7WUFDakUsSUFBSSxFQUFFLElBQUk7WUFDVixXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7WUFDOUIsUUFBUSxFQUFFLEtBQUssQ0FBQyxRQUFRO1lBQ3hCLE9BQU8sRUFBRSxJQUFJLENBQUMsZUFBZSxDQUFDLFdBQVcsRUFBRSxJQUFJLEVBQUU7Z0JBQy9DLFFBQVEsRUFBRSxLQUFLLENBQUMsUUFBUTtnQkFDeEIsSUFBSTtnQkFDSixXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7YUFDL0IsQ0FBQztZQUNGLElBQUksRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQztTQUMzQixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsR0FBRyxHQUFHLFNBQVMsQ0FBQyxPQUFPLENBQUM7SUFDL0IsQ0FBQztJQUVEOzs7O09BSUc7SUFDSCxlQUFlLENBQUMsT0FBdUI7UUFDckMsS0FBSyxNQUFNLEtBQUssSUFBSSxJQUFJLENBQUMsTUFBTSxFQUFFLENBQUM7WUFDaEMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUMzQixDQUFDO0lBQ0gsQ0FBQztJQUVELCtCQUErQixDQUFDLFFBQTZCLEVBQUUsUUFBa0I7UUFDL0UsSUFBSSxRQUFRLElBQUksU0FBUyxFQUFFLENBQUM7WUFDMUIsT0FBTztnQkFDTCxtQ0FBbUM7Z0JBQ25DLDRDQUE0QztnQkFDNUMsc0JBQXNCO2FBQ3ZCLENBQUMsTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQ3JCLENBQUM7YUFBTSxDQUFDO1lBQ04sT0FBTztnQkFDTCxTQUFTO2FBQ1YsQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDckIsQ0FBQztJQUNILENBQUM7O0FBcklILHNEQXNJQzs7O0FBRUQ7O0dBRUc7QUFDSCxNQUFhLGlDQUFrQyxTQUFRLCtCQUFzQjtJQXVCM0UsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUErQjtRQUN2RSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztRQVhULG9CQUFlLEdBQTRCLEVBQUUsQ0FBQztRQWE3RCxJQUFJLEtBQUssRUFBRSxnQkFBZ0IsRUFBRSxDQUFDO1lBQzVCLHlCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFVBQVUsQ0FBQyxtRkFBbUYsQ0FBQyxDQUFDO1FBQ3ZILENBQUM7UUFFRCxJQUFJLENBQUMsRUFBRSxHQUFHLEtBQUssRUFBRSxFQUFFLElBQUksY0FBRSxDQUFDLFlBQVksQ0FBQztRQUN2QyxJQUFJLENBQUMsWUFBWSxHQUFHLEtBQUssRUFBRSxZQUFZLElBQUksd0JBQVksQ0FBQyxNQUFNLENBQUM7UUFDL0QsSUFBSSxDQUFDLGVBQWUsR0FBRyxLQUFLLEVBQUUsZUFBZSxJQUFJLHNCQUFRLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ2xFLElBQUksQ0FBQyxZQUFZLEdBQUcsS0FBSyxFQUFFLFlBQVksSUFBSSx3QkFBYSxDQUFDLFNBQVMsQ0FBQztRQUNuRSxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsS0FBSyxFQUFFLGdCQUFnQixJQUFJLDJCQUFhLENBQUMsT0FBTyxDQUFDO1FBQ3pFLElBQUksQ0FBQyxHQUFHLEdBQUcsS0FBSyxFQUFFLEdBQUcsSUFBSSxxQkFBRyxDQUFDLEdBQUcsQ0FBQyxVQUFVLENBQUMsSUFBSSxFQUFFLEtBQUssRUFBRSxFQUFFLFNBQVMsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBQzlFLElBQUksQ0FBQyxjQUFjLEdBQUcsS0FBSyxFQUFFLGNBQWMsSUFBSSxDQUFDLElBQUkscUJBQUcsQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLElBQUksRUFBRSxFQUFFLEdBQUcsRUFBRSxJQUFJLENBQUMsR0FBRyxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBQ3RHLElBQUksQ0FBQyxlQUFlLEdBQUcsS0FBSyxFQUFFLGVBQWUsQ0FBQztRQUM5QyxJQUFJLENBQUMsU0FBUyxHQUFHLEtBQUssRUFBRSxlQUFlLElBQUksSUFBQSxrQ0FBc0IsRUFBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDM0UsSUFBSSxDQUFDLE9BQU8sR0FBRyxLQUFLLEVBQUUsT0FBTyxJQUFJLElBQUEsb0JBQWMsRUFBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLEVBQUUsRUFBRSxJQUFJLENBQUMsWUFBWSxDQUFDLENBQUM7UUFDbEYsSUFBSSxDQUFDLFlBQVksR0FBRyxLQUFLLEVBQUUsc0JBQXNCLEVBQUUsWUFBWSxJQUFJLHFCQUFHLENBQUMsWUFBWSxDQUFDLEVBQUUsQ0FBQyxxQkFBRyxDQUFDLGFBQWEsQ0FBQyxHQUFHLEVBQUUscUJBQUcsQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDdEksSUFBSSxDQUFDLGlCQUFpQixHQUFHLEtBQUssRUFBRSxzQkFBc0IsRUFBRSxpQkFBaUIsQ0FBQztRQUMxRSxJQUFJLENBQUMsV0FBVyxHQUFHLEtBQUssRUFBRSxzQkFBc0IsRUFBRSxXQUFXLENBQUM7UUFDOUQsSUFBSSxDQUFDLFlBQVksR0FBRyxLQUFLLEVBQUUsWUFBWSxJQUFJLElBQUksQ0FBQztRQUNoRCxJQUFJLENBQUMsbUJBQW1CLEdBQUcsS0FBSyxFQUFFLG1CQUFtQixJQUFJLEVBQUUsQ0FBQztRQUU1RCw2QkFBNkI7UUFDN0IsSUFBSSxDQUFDLElBQUksR0FBRztZQUNWLHFCQUFxQixFQUFFLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFNBQVM7WUFDbkQsdUJBQXVCLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJO1NBQ3hDLENBQUM7UUFFRix3QkFBd0I7UUFDeEIsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsaUJBQWlCLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUM7WUFDNUQsTUFBTSxJQUFJLEtBQUssQ0FBQyx5QkFBeUIsSUFBSSxDQUFDLFlBQVksQ0FBQyxJQUFJLDJDQUEyQyxJQUFJLENBQUMsWUFBWSxNQUFNLElBQUksQ0FBQyxZQUFZLENBQUMsWUFBWSxHQUFHLENBQUMsQ0FBQztRQUN0SyxDQUFDO1FBRUQsaUNBQWlDO1FBQ2pDLElBQUksS0FBSyxFQUFFLGVBQWUsRUFBRSxVQUFVLElBQUkscUJBQUcsQ0FBQyxVQUFVLENBQUMsZ0JBQWdCLEVBQUUsQ0FBQztZQUMxRSx5QkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxVQUFVLENBQUMsOEZBQThGO2dCQUM1SCwyREFBMkQsQ0FBQyxDQUFDO1FBQ2pFLENBQUM7UUFFRCx1Q0FBdUM7UUFDdkMsSUFBSSxDQUFDLElBQUksR0FBRyxJQUFJLHFCQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxNQUFNLEVBQUU7WUFDckMsU0FBUyxFQUFFLElBQUkscUJBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxtQkFBbUIsQ0FBQztTQUN6RCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU8sUUFBUTtRQUNkLElBQUksSUFBSSxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsY0FBRSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7WUFDM0IsT0FBTyxTQUFTLENBQUM7UUFDbkIsQ0FBQztRQUNELElBQUksSUFBSSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsY0FBRSxDQUFDLG1CQUFtQixDQUFDLEVBQUUsQ0FBQztZQUN6QyxPQUFPLE9BQU8sQ0FBQztRQUNqQixDQUFDO1FBQ0QsTUFBTSxJQUFJLEtBQUssQ0FBQyxNQUFNLElBQUksQ0FBQyxFQUFFLENBQUMsSUFBSSx3Q0FBd0MsQ0FBQyxDQUFDO0lBQzlFLENBQUM7SUFFRDs7T0FFRztJQUNILGVBQWU7UUFDYixJQUFJLElBQUksQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDO1lBQzFCLE9BQU8sSUFBSSxDQUFDLGdCQUFnQixDQUFDO1FBQy9CLENBQUM7UUFFRCw0Q0FBNEM7UUFDNUMsTUFBTSxVQUFVLEdBQUcsSUFBSSxxQkFBRyxDQUFDLFVBQVUsQ0FBQyxJQUFJLEVBQUUsWUFBWSxFQUFFO1lBQ3hELGVBQWUsRUFBRSxJQUFJO1lBQ3JCLGtCQUFrQixFQUFFLHVCQUFhLENBQUMsT0FBTztZQUN6QyxhQUFhLEVBQUUsMkJBQWEsQ0FBQyxPQUFPO1lBQ3BDLGFBQWEsRUFBRSxJQUFJO1NBQ3BCLENBQUMsQ0FBQztRQUVILE1BQU0sSUFBSSxHQUFHLElBQUksOEJBQVksQ0FBQyw0QkFBNEIsQ0FBQyxJQUFJLEVBQUUscUJBQXFCLEVBQUU7WUFDdEYsSUFBSSxFQUFFLElBQUEsK0JBQXNCLEVBQUMsSUFBSSxDQUFDO1lBQ2xDLGlDQUFpQztZQUNqQyxhQUFhLEVBQUU7Z0JBQ2I7b0JBQ0UsTUFBTSxFQUFFLG1CQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU07b0JBQzdCLGtDQUFrQyxFQUFFO3dCQUNsQyxhQUFhLEVBQUUsQ0FBQyxRQUFRLENBQUM7d0JBQ3pCLGdCQUFnQixFQUFFOzRCQUNoQixPQUFPLEVBQUUsS0FBSzs0QkFDZCxjQUFjLEVBQUUsVUFBVSxDQUFDLGNBQWM7eUJBQzFDO3FCQUNGO2lCQUNGO2FBQ0Y7WUFDRCxJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUk7U0FDaEIsQ0FBQyxDQUFDO1FBRUgsSUFBSSxrQkFBa0IsR0FBRzs7Z0NBRUcsQ0FBQztRQUU3QixLQUFLLE1BQU0sQ0FBQyxJQUFJLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQztZQUNoQyxNQUFNLFFBQVEsR0FBRyxDQUFDLENBQUMsaUJBQWlCLENBQUMsSUFBSSxDQUFDLEVBQUUsRUFBRSxJQUFJLENBQUMsWUFBWSxDQUFDLENBQUM7WUFDakUsSUFBSSxRQUFRLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRSxDQUFDO2dCQUN4QixrQkFBa0IsSUFBSSxJQUFJLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxJQUFJLENBQUM7WUFDMUQsQ0FBQztRQUNILENBQUM7UUFFRCxNQUFNLE1BQU0sR0FBRyxJQUFJLDJCQUFlLENBQUMsSUFBSSxFQUFFLGtCQUFrQixFQUFFO1lBQzNELFFBQVEsRUFBRSxJQUFJLENBQUMsUUFBUSxFQUFFO1lBQ3pCLFVBQVUsRUFBRSxJQUFJLENBQUMsY0FBYyxFQUFFO1lBQ2pDLGdCQUFnQixFQUFFLFVBQVU7WUFDNUIsa0JBQWtCLEVBQUUsa0JBQWtCO1lBQ3RDLFdBQVcsRUFBRSxJQUFJLENBQUMsU0FBUztZQUMzQixJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUk7U0FDaEIsQ0FBQyxDQUFDO1FBRUgsTUFBTSxHQUFHLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxZQUFZLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3RELE1BQU0sS0FBSyxHQUFHLElBQUksQ0FBQyxvQkFBb0IsQ0FBQztZQUN0QyxxQkFBRyxDQUFDLGFBQWEsQ0FBQyx3QkFBd0IsQ0FBQyw4QkFBOEIsQ0FBQztZQUMxRSxxQkFBRyxDQUFDLGFBQWEsQ0FBQyx3QkFBd0IsQ0FBQyxxREFBcUQsQ0FBQztTQUNsRyxDQUFDLENBQUM7UUFFSCxJQUFJLElBQUksQ0FBQyxZQUFZLEVBQUUsQ0FBQztZQUN0QixJQUFJLENBQUMsV0FBVyxDQUFDLEtBQUssRUFBRSxJQUFJLEVBQUUsR0FBRyxFQUFFLFNBQVMsRUFBRSxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDNUQsQ0FBQztRQUNELElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxNQUFNLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFFNUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFLEdBQUcsRUFBRSxTQUFTLEVBQUUsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBRTdELElBQUksQ0FBQyxnQkFBZ0IsR0FBRztZQUN0QixlQUFlLEVBQUUsVUFBVTtZQUMzQixRQUFRLEVBQUUsUUFBUTtZQUNsQixFQUFFLEVBQUUsSUFBSSxDQUFDLEVBQUU7WUFDWCxZQUFZLEVBQUUsSUFBSSxDQUFDLFlBQVk7WUFDL0IsUUFBUSxFQUFFLEdBQUc7WUFDYixhQUFhLEVBQUUseUJBQWEsQ0FBQyxRQUFRLENBQUMsU0FBUyxDQUFDO1lBQ2hELG9IQUFvSDtTQUNySCxDQUFDO1FBRUYsT0FBTyxJQUFJLENBQUMsZ0JBQWdCLENBQUM7SUFDL0IsQ0FBQztJQUVPLGtCQUFrQixDQUFDLE1BQXVCLEVBQUUsVUFBMkI7UUFDN0UsNEVBQTRFO1FBQzVFLG1GQUFtRjtRQUNuRiw0RkFBNEY7UUFDNUYseUZBQXlGO1FBQ3pGLE1BQU0sVUFBVSxHQUFHLElBQUEsdUJBQWUsRUFBQyx5Q0FBa0IsRUFBRSxJQUFJLEVBQUUsYUFBYSxFQUFFO1lBQzFFLFdBQVcsRUFBRSx3RUFBd0U7WUFDckYsT0FBTyxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztZQUNoQyxRQUFRLEVBQUUsSUFBQSx5QkFBaUIsRUFBQyxJQUFJLEVBQUUsd0JBQWdCLENBQUMsa0JBQWtCLENBQUM7WUFDdEUsYUFBYSxFQUFFLHdCQUFNLENBQUMsYUFBYSxDQUFDLElBQUk7U0FDekMsQ0FBQyxDQUFDO1FBQ0gsVUFBVSxDQUFDLGVBQWUsQ0FBQyxJQUFJLHFCQUFHLENBQUMsZUFBZSxDQUFDO1lBQ2pELE1BQU0sRUFBRSxxQkFBRyxDQUFDLE1BQU0sQ0FBQyxJQUFJO1lBQ3ZCLE9BQU8sRUFBRSxDQUFDLDBCQUEwQixDQUFDO1lBQ3JDLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztTQUNqQixDQUFDLENBQUMsQ0FBQztRQUVKLHFEQUFxRDtRQUNyRCxJQUFJLENBQUMsWUFBWSxDQUFDLFdBQVcsRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUUxRSxnREFBZ0Q7UUFDaEQsSUFBSSw4QkFBWSxDQUFDLGtCQUFrQixDQUFDLElBQUksRUFBRSx5QkFBeUIsRUFBRTtZQUNuRSxJQUFJLEVBQUUsSUFBQSwrQkFBc0IsRUFBQyxJQUFJLENBQUM7WUFDbEMsV0FBVyxFQUFFLDhDQUE4QyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRTtZQUMzRSxhQUFhLEVBQUUsSUFBSSxxQkFBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsOEJBQThCLEVBQUU7Z0JBQ2hFLFNBQVMsRUFBRSxJQUFJLHFCQUFHLENBQUMsZ0JBQWdCLENBQUMsNEJBQTRCLENBQUM7Z0JBQ2pFLGNBQWMsRUFBRTtvQkFDZCxFQUFFLEVBQUUsSUFBSSxxQkFBRyxDQUFDLGNBQWMsQ0FBQzt3QkFDekIsVUFBVSxFQUFFOzRCQUNWLElBQUkscUJBQUcsQ0FBQyxlQUFlLENBQUM7Z0NBQ3RCLE9BQU8sRUFBRSxDQUFDLGtCQUFrQixFQUFFLDBCQUEwQixDQUFDO2dDQUN6RCxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUMsRUFBRSxnREFBZ0Q7NkJBQ25FLENBQUM7eUJBQ0g7cUJBQ0YsQ0FBQztvQkFDRixHQUFHLEVBQUUsSUFBSSxxQkFBRyxDQUFDLGNBQWMsQ0FBQzt3QkFDMUIsVUFBVSxFQUFFOzRCQUNWLElBQUkscUJBQUcsQ0FBQyxlQUFlLENBQUM7Z0NBQ3RCLE9BQU8sRUFBRSxDQUFDLG1CQUFtQixFQUFFLHNCQUFzQixDQUFDO2dDQUN0RCxTQUFTLEVBQUUsQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDOzZCQUN0QyxDQUFDO3lCQUNIO3FCQUNGLENBQUM7aUJBQ0g7YUFDRixDQUFDLENBQUMsT0FBTztZQUNWLGFBQWEsRUFBRSxDQUFDO29CQUNkLE1BQU0sRUFBRTt3QkFDTixJQUFJLEVBQUUsUUFBUTt3QkFDZCxnQkFBZ0IsRUFBRTs0QkFDaEIsVUFBVSxFQUFFLElBQUk7eUJBQ2pCO3FCQUNGO29CQUNELE1BQU0sRUFBRTt3QkFDTixJQUFJLEVBQUUsT0FBTzt3QkFDYixLQUFLLEVBQUUsQ0FBQztxQkFDVDtpQkFDRixDQUFDO1lBQ0YsWUFBWSxFQUFFLGlCQUFpQjtZQUMvQixpQkFBaUIsRUFBRTtnQkFDakIsT0FBTyxFQUFFO29CQUNQO3dCQUNFLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSTt3QkFDakIsZUFBZSxFQUFFLE1BQU0sQ0FBQyxPQUFPO3FCQUNoQztpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVTLFNBQVMsQ0FBQyxFQUFVLEVBQUUsVUFBa0I7UUFDaEQsT0FBTyxJQUFJLHNCQUFJLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxFQUFFLEVBQUU7WUFDakMsWUFBWSxFQUFFLHFCQUFxQixVQUFVLEVBQUU7WUFDL0MsU0FBUyxFQUFFLElBQUksQ0FBQyxZQUFZO1lBQzVCLGFBQWEsRUFBRSxJQUFJLENBQUMsZ0JBQWdCO1NBQ3JDLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFUyxvQkFBb0IsQ0FBQyxlQUFxQztRQUNsRSxJQUFJLElBQUksQ0FBQyxjQUFjLEVBQUUsQ0FBQztZQUN4QixPQUFPLElBQUksQ0FBQyxjQUFjLENBQUM7UUFDN0IsQ0FBQztRQUVELEtBQUssTUFBTSxhQUFhLElBQUksZUFBZSxFQUFFLENBQUM7WUFDNUMsSUFBSSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUM1QyxDQUFDO1FBRUQsS0FBSyxNQUFNLFNBQVMsSUFBSSxJQUFJLENBQUMsZUFBZSxFQUFFLENBQUM7WUFDN0MsU0FBUyxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDdkMsQ0FBQztRQUVELElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSw4QkFBWSxDQUFDLDhCQUE4QixDQUFDLElBQUksRUFBRSxnQkFBZ0IsRUFBRTtZQUM1RixJQUFJLEVBQUUsSUFBQSwrQkFBc0IsRUFBQyxJQUFJLENBQUM7WUFDbEMsaUNBQWlDO1lBQ2pDLFFBQVEsRUFBRSxJQUFJLENBQUMsR0FBRyxFQUFFLGFBQWEsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQztZQUNwRSxnQkFBZ0IsRUFBRSxJQUFJLENBQUMsY0FBYyxFQUFFLEdBQUcsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUM7WUFDcEUsYUFBYSxFQUFFLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxRQUFRLEVBQUUsQ0FBQztZQUM3Qyx1QkFBdUIsRUFBRTtnQkFDdkIsVUFBVSxFQUFFLFVBQVU7Z0JBQ3RCLGtEQUFrRDtnQkFDbEQsdUJBQXVCLEVBQUUsQ0FBQzthQUMzQjtZQUNELG1CQUFtQixFQUFFLElBQUkscUJBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUU7Z0JBQ3hFLEtBQUssRUFBRTtvQkFDTCxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVE7aUJBQ25CO2FBQ0YsQ0FBQyxDQUFDLEdBQUc7U0FDUCxDQUFDLENBQUM7UUFFSCxPQUFPLElBQUksQ0FBQyxjQUFjLENBQUM7SUFDN0IsQ0FBQztJQUVTLFdBQVcsQ0FBQyxLQUFrRCxFQUFFLElBQStDLEVBQUUsR0FBa0IsRUFDM0ksY0FBdUIsRUFBRSxrQkFBMkI7UUFDcEQsTUFBTSxLQUFLLEdBQUcsSUFBSSw4QkFBWSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLGdCQUFnQixDQUFDLE9BQU8sRUFBRSxjQUFjLEVBQUUsa0JBQWtCLENBQUMsRUFBRTtZQUNoSCw4QkFBOEIsRUFBRSxLQUFLLENBQUMsT0FBTztZQUM3Qyw0QkFBNEIsRUFBRSxJQUFJLENBQUMsT0FBTztZQUMxQyxjQUFjO1lBQ2Qsa0JBQWtCO1lBQ2xCLHVCQUF1QixFQUFFO2dCQUN2QixpQkFBaUIsRUFBRSxLQUFLO2FBQ3pCO1lBQ0QsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJO1NBQ2hCLENBQUMsQ0FBQztRQUNILEtBQUssQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ2hDLEtBQUssQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBRTlCLG1FQUFtRTtRQUNuRSx3RUFBd0U7UUFDeEUsa0dBQWtHO1FBQ2xHLDZFQUE2RTtRQUM3RSxLQUFLLENBQUMsa0JBQWtCLENBQUMsMkJBQWEsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO1FBRW5FLE9BQU8sS0FBSyxDQUFDO0lBQ2YsQ0FBQztJQUVPLGdCQUFnQixDQUFDLE1BQWMsRUFBRSxjQUF1QixFQUFFLGtCQUEyQjtRQUMzRixJQUFJLGNBQWMsRUFBRSxDQUFDO1lBQ25CLE9BQU8sT0FBTyxNQUFNLEVBQUUsQ0FBQztRQUN6QixDQUFDO1FBQ0QsSUFBSSxrQkFBa0IsRUFBRSxDQUFDO1lBQ3ZCLE9BQU8sVUFBVSxNQUFNLEVBQUUsQ0FBQztRQUM1QixDQUFDO1FBQ0QsTUFBTSxJQUFJLEtBQUssQ0FBQyw2REFBNkQsQ0FBQyxDQUFDO0lBQ2pGLENBQUM7SUFFUyxjQUFjLENBQUMsS0FBa0QsRUFBRSxJQUErQyxFQUFFLEdBQWtCLEVBQzlJLGNBQXVCLEVBQUUsa0JBQTJCO1FBQ3BELGVBQWU7UUFDZixJQUFJLGVBQTJFLENBQUM7UUFDaEYsSUFBSSxJQUFJLENBQUMsZUFBZSxDQUFDLE1BQU0sRUFBRSxHQUFHLENBQUMsRUFBRSxDQUFDO1lBQ3RDLGVBQWUsR0FBRztnQkFDaEIsa0JBQWtCLEVBQUUsd0JBQU0sQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsQ0FBQyxnQkFBZ0I7Z0JBQy9FLCtCQUErQixFQUFFLHVCQUF1QjthQUN6RCxDQUFDO1FBQ0osQ0FBQztRQUVELGdDQUFnQztRQUNoQyxJQUFJLFNBQW9GLENBQUM7UUFDekYsSUFBSSxhQUFvQyxDQUFDO1FBQ3pDLElBQUksSUFBSSxDQUFDLG1CQUFtQixDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUUsQ0FBQztZQUN4QyxTQUFTLEdBQUcsQ0FBQztvQkFDWCxXQUFXLEVBQUUsSUFBQSx1REFBNEMsRUFBQyxJQUFJLEVBQUUsT0FBTyxFQUFFLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDLEdBQUc7aUJBQ3ZHLENBQUMsQ0FBQztZQUNILGFBQWEsR0FBRyxxQkFBRyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxFQUFFLG9CQUFvQixFQUFFLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFNBQVMsQ0FBQztnQkFDNUYsT0FBTyxFQUFFLEtBQUs7Z0JBQ2QsTUFBTSxFQUFFLEVBQUU7Z0JBQ1YsUUFBUSxFQUFFLE1BQU07Z0JBQ2hCLFlBQVksRUFBRSwyRUFBMkU7YUFDMUYsQ0FBQyxDQUFDLENBQUM7UUFDTixDQUFDO1FBRUQsb0JBQW9CO1FBQ3BCLE1BQU0sUUFBUSxHQUFHLElBQUksOEJBQVksQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLGdCQUFnQixDQUFDLFVBQVUsRUFBRSxjQUFjLEVBQUUsa0JBQWtCLENBQUMsRUFBRTtZQUM5SCxJQUFJLEVBQUUsSUFBQSwrQkFBc0IsRUFBQyxJQUFJLENBQUM7WUFDbEMsaUNBQWlDO1lBQ2pDLDhCQUE4QixFQUFFLEtBQUssQ0FBQyxPQUFPO1lBQzdDLDRCQUE0QixFQUFFLElBQUksQ0FBQyxPQUFPO1lBQzFDLGNBQWM7WUFDZCxrQkFBa0I7WUFDbEIsUUFBUSxFQUFFLGVBQWU7WUFDekIsdUJBQXVCLEVBQUU7Z0JBQ3ZCLGlCQUFpQixFQUFFLEtBQUs7YUFDekI7WUFDRCxTQUFTLEVBQUUsU0FBUztZQUNwQixhQUFhLEVBQUUsYUFBYSxFQUFFLE9BQU87WUFDckMsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJO1NBQ2hCLENBQUMsQ0FBQztRQUNILFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ25DLFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBRWpDLE9BQU8sUUFBUSxDQUFDO0lBQ2xCLENBQUM7SUFFRDs7T0FFRztJQUNILElBQVcsV0FBVztRQUNwQixPQUFPLElBQUkscUJBQUcsQ0FBQyxXQUFXLENBQUMsRUFBRSxjQUFjLEVBQUUsSUFBSSxDQUFDLGNBQWMsRUFBRSxDQUFDLENBQUM7SUFDdEUsQ0FBQztJQUVELElBQVcsY0FBYztRQUN2QixPQUFPLElBQUksQ0FBQyxJQUFJLENBQUM7SUFDbkIsQ0FBQztJQUVELE9BQU87UUFDTCxJQUFJLElBQUksQ0FBQyxRQUFRLEVBQUUsQ0FBQztZQUNsQixPQUFPLElBQUksQ0FBQyxRQUFRLENBQUM7UUFDdkIsQ0FBQztRQUVELE1BQU0sY0FBYyxHQUFHLElBQUkscUJBQUcsQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLGlCQUFpQixFQUFFO1lBQ3JFLGFBQWEsRUFBRSxJQUFJO1NBQ3BCLENBQUMsQ0FBQztRQUVILE1BQU0scUJBQXFCLEdBQW9GLENBQUM7Z0JBQzlHLGdCQUFnQixFQUFFLGNBQWMsQ0FBQyxnQkFBZ0I7Z0JBQ2pELGlCQUFpQixFQUFFLElBQUk7YUFDeEIsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxpQkFBaUIsR0FBZ0YsRUFBRSxDQUFDO1FBRTFHLElBQUksSUFBSSxDQUFDLGlCQUFpQixFQUFFLE9BQU8sSUFBSSxLQUFLLEVBQUUsQ0FBQztZQUM3QyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsY0FBRSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7Z0JBQzVCLE1BQU0sSUFBSSxLQUFLLENBQUMsMkNBQTJDLENBQUMsQ0FBQztZQUMvRCxDQUFDO1lBRUQsd0RBQXdEO1lBQ3hELHVDQUF1QztZQUN2Qyx1TUFBdU07WUFDdk0sc0ZBQXNGO1lBQ3RGLDZJQUE2STtZQUM3SSxNQUFNLGtCQUFrQixHQUFHLElBQUkscUJBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLEVBQUUsc0JBQXNCLEVBQUU7Z0JBQ2pGLGtCQUFrQixFQUFFO29CQUNsQixlQUFlLEVBQUU7d0JBQ2YsVUFBVSxFQUFFLFVBQVU7cUJBQ3ZCO29CQUNELFlBQVksRUFBRSxJQUFJLENBQUMsWUFBWSxDQUFDLFFBQVEsRUFBRTtvQkFDMUMsaUJBQWlCLEVBQUUsQ0FBQzs0QkFDbEIsUUFBUSxFQUFFLElBQUksQ0FBQyxHQUFHLEVBQUUsYUFBYSxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDOzRCQUNwRSxXQUFXLEVBQUUsQ0FBQzs0QkFDZCxNQUFNLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDO3lCQUMxRCxDQUFDO29CQUNGLGlCQUFpQixFQUFFO3dCQUNqQjs0QkFDRSxZQUFZLEVBQUUsVUFBVTs0QkFDeEIsSUFBSSxFQUFFLENBQUM7b0NBQ0wsR0FBRyxFQUFFLE1BQU07b0NBQ1gsS0FBSyxFQUFFLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLHVCQUF1QjtpQ0FDaEQsQ0FBQzt5QkFDSDt3QkFDRDs0QkFDRSxZQUFZLEVBQUUsUUFBUTs0QkFDdEIsSUFBSSxFQUFFLENBQUM7b0NBQ0wsR0FBRyxFQUFFLE1BQU07b0NBQ1gsS0FBSyxFQUFFLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLHVCQUF1QjtpQ0FDaEQsQ0FBQzt5QkFDSDtxQkFDRjtpQkFDRjtnQkFDRCxpQkFBaUIsRUFBRSxDQUFDO3dCQUNsQixZQUFZLEVBQUUsaUJBQWlCO3dCQUMvQixJQUFJLEVBQUUsQ0FBQztnQ0FDTCxHQUFHLEVBQUUsTUFBTTtnQ0FDWCxLQUFLLEVBQUUsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksdUJBQXVCOzZCQUNoRCxDQUFDO3FCQUNILENBQUM7YUFDSCxDQUFDLENBQUM7WUFFSCxxQkFBcUIsQ0FBQyxJQUFJLENBQUM7Z0JBQ3pCLGdCQUFnQixFQUFFLGtCQUFrQixDQUFDLG9CQUFvQjtnQkFDekQsaUJBQWlCLEVBQUUsSUFBSTthQUN4QixDQUFDLENBQUM7WUFDSCxpQkFBaUIsQ0FBQyxJQUFJLENBQUM7Z0JBQ3JCLE9BQU8sRUFBRSxJQUFJO2dCQUNiLGNBQWMsRUFBRTtvQkFDZCxnQkFBZ0IsRUFBRSxrQkFBa0IsQ0FBQyxvQkFBb0I7aUJBQzFEO2dCQUNELG1CQUFtQixFQUFFLElBQUksQ0FBQyxpQkFBaUIsRUFBRSxtQkFBbUIsSUFBSSxDQUFDO2dCQUNyRSxxQkFBcUIsRUFBRTtvQkFDckIsbUJBQW1CLEVBQUUsSUFBSSxDQUFDLGlCQUFpQixFQUFFLG1CQUFtQixJQUFJLENBQUM7aUJBQ3RFO2FBQ0YsQ0FBQyxDQUFDO1FBQ0wsQ0FBQztRQUVELE1BQU0sU0FBUyxHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFNBQVMsQ0FBQztRQUMvQyxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQztRQUVuQyxNQUFNLElBQUksR0FBRyxJQUFJLDhCQUFZLENBQUMsNEJBQTRCLENBQUMsSUFBSSxFQUFFLGtCQUFrQixFQUFFO1lBQ25GLElBQUksRUFBRSxJQUFBLCtCQUFzQixFQUFDLElBQUksQ0FBQztZQUNsQyxpQ0FBaUM7WUFDakMsYUFBYSxFQUFFO2dCQUNiO29CQUNFLE1BQU0sRUFBRSxtQkFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxNQUFNO29CQUM3Qiw0QkFBNEIsRUFBRTt3QkFDNUIsSUFBSSxFQUFFLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLEVBQUU7NEJBQzFDLFNBQVMsRUFBRSxHQUFHOzRCQUNkLFNBQVMsRUFBRSxHQUFHOzRCQUNkLHdCQUF3QixFQUFFLElBQUk7eUJBQy9CLENBQUMsK0JBQStCO3dCQUNqQyxPQUFPLEVBQUU7NEJBQ1AsTUFBTSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRTs0QkFDcEIscUJBQXFCLEVBQUUsU0FBUzs0QkFDaEMsdUJBQXVCLEVBQUUsV0FBVzt5QkFDckM7cUJBQ0Y7b0JBQ0QsNEJBQTRCLEVBQUUscUJBQXFCO29CQUNuRCx3QkFBd0IsRUFBRSxpQkFBaUIsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLENBQUMsU0FBUztpQkFDdkY7YUFDRjtZQUNELElBQUksRUFBRSxJQUFJLENBQUMsSUFBSTtTQUNoQixDQUFDLENBQUM7UUFFSCxNQUFNLE1BQU0sR0FBRyxJQUFJLGVBQVMsQ0FBQyxJQUFJLEVBQUUsWUFBWSxFQUFFO1lBQy9DLFFBQVEsRUFBRSxJQUFJLENBQUMsUUFBUSxFQUFFO1lBQ3pCLFVBQVUsRUFBRSxJQUFJLENBQUMsY0FBYyxFQUFFO1lBQ2pDLFlBQVksRUFBRSxJQUFJLENBQUMsWUFBWTtZQUMvQixPQUFPLEVBQUUsSUFBSSxDQUFDLE9BQU87WUFDckIsV0FBVyxFQUFFLElBQUksQ0FBQyxXQUFXO1lBQzdCLElBQUksRUFBRSxJQUFJLENBQUMsSUFBSTtTQUNoQixDQUFDLENBQUM7UUFFSCxNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLFNBQVMsRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDbkQsTUFBTSxLQUFLLEdBQUcsSUFBSSxDQUFDLG9CQUFvQixDQUFDO1lBQ3RDLHFCQUFHLENBQUMsYUFBYSxDQUFDLHdCQUF3QixDQUFDLDhCQUE4QixDQUFDO1lBQzFFLHFCQUFHLENBQUMsYUFBYSxDQUFDLHdCQUF3QixDQUFDLG1DQUFtQyxDQUFDO1NBQ2hGLENBQUMsQ0FBQztRQUNILElBQUksSUFBSSxDQUFDLFlBQVksRUFBRSxDQUFDO1lBQ3RCLElBQUksQ0FBQyxXQUFXLENBQUMsS0FBSyxFQUFFLElBQUksRUFBRSxHQUFHLEVBQUUsTUFBTSxDQUFDLEdBQUcsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUM1RCxDQUFDO1FBQ0QsSUFBSSxDQUFDLGNBQWMsQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFLEdBQUcsRUFBRSxNQUFNLENBQUMsR0FBRyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBRTdELElBQUksQ0FBQyxRQUFRLEdBQUc7WUFDZCxjQUFjLEVBQUUsY0FBYztZQUM5QixZQUFZLEVBQUUsSUFBSSxDQUFDLFlBQVk7WUFDL0IsRUFBRSxFQUFFLElBQUksQ0FBQyxFQUFFO1lBQ1gsUUFBUSxFQUFFLEdBQUc7WUFDYixhQUFhLEVBQUUseUJBQWEsQ0FBQyxRQUFRLENBQUMsU0FBUyxDQUFDO1NBQ2pELENBQUM7UUFFRixJQUFJLENBQUMsVUFBVSxDQUFDLE1BQU0sRUFBRSxTQUFTLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFFaEQsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDO0lBQ3ZCLENBQUM7SUFFTyxVQUFVLENBQUMsTUFBaUIsRUFBRSxTQUFpQixFQUFFLFdBQW1CO1FBQzFFLDRFQUE0RTtRQUM1RSw0RkFBNEY7UUFDNUYsd0ZBQXdGO1FBQ3hGLDJIQUEySDtRQUMzSCxNQUFNLEtBQUssR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNqQyxJQUFJLEtBQUssQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLGlEQUFpRCxDQUFDLElBQUksU0FBUyxFQUFFLENBQUM7WUFDNUYsTUFBTSxJQUFJLEdBQUcsSUFBSSxxQkFBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsaURBQWlELEVBQUU7Z0JBQ2xGLFdBQVcsRUFBRSxzRUFBc0U7Z0JBQ25GLFNBQVMsRUFBRSxJQUFJLHFCQUFHLENBQUMsZ0JBQWdCLENBQUMsc0JBQXNCLENBQUM7Z0JBQzNELGNBQWMsRUFBRTtvQkFDZCxJQUFJLEVBQUUsSUFBSSxxQkFBRyxDQUFDLGNBQWMsQ0FBQzt3QkFDM0IsVUFBVSxFQUFFOzRCQUNWLElBQUkscUJBQUcsQ0FBQyxlQUFlLENBQUM7Z0NBQ3RCLE9BQU8sRUFBRSxDQUFDLHFCQUFxQixFQUFFLG9CQUFvQixDQUFDO2dDQUN0RCxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7Z0NBQ2hCLE1BQU0sRUFBRSxxQkFBRyxDQUFDLE1BQU0sQ0FBQyxJQUFJOzZCQUN4QixDQUFDO3lCQUNIO3FCQUNGLENBQUM7aUJBQ0g7YUFDRixDQUFDLENBQUM7WUFDSCxNQUFNLE1BQU0sR0FBZ0IsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUEyQixDQUFDO1lBQ2xFLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQyw4REFBOEQsQ0FBQyxDQUFDO1FBQzNGLENBQUM7UUFFRCxxREFBcUQ7UUFDckQsSUFBSSxDQUFDLFlBQVksQ0FBQyxPQUFPLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLENBQUM7UUFFdEUsdUNBQXVDO1FBQ3ZDLElBQUksOEJBQVksQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLEVBQUUsc0JBQXNCLEVBQUU7WUFDaEUsSUFBSSxFQUFFLElBQUEsK0JBQXNCLEVBQUMsSUFBSSxDQUFDO1lBQ2xDLFdBQVcsRUFBRSxxQ0FBcUMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUU7WUFDbEUsYUFBYSxFQUFFLElBQUkscUJBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLDJCQUEyQixFQUFFO2dCQUM3RCxTQUFTLEVBQUUsSUFBSSxxQkFBRyxDQUFDLGdCQUFnQixDQUFDLDRCQUE0QixDQUFDO2dCQUNqRSxjQUFjLEVBQUU7b0JBQ2QsRUFBRSxFQUFFLElBQUkscUJBQUcsQ0FBQyxjQUFjLENBQUM7d0JBQ3pCLFVBQVUsRUFBRTs0QkFDVixJQUFJLHFCQUFHLENBQUMsZUFBZSxDQUFDO2dDQUN0QixPQUFPLEVBQUUsQ0FBQyxrQkFBa0IsRUFBRSwwQkFBMEIsQ0FBQztnQ0FDekQsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDLEVBQUUsZ0RBQWdEOzZCQUNuRSxDQUFDO3lCQUNIO3FCQUNGLENBQUM7b0JBQ0YsR0FBRyxFQUFFLElBQUkscUJBQUcsQ0FBQyxjQUFjLENBQUM7d0JBQzFCLFVBQVUsRUFBRTs0QkFDVixJQUFJLHFCQUFHLENBQUMsZUFBZSxDQUFDO2dDQUN0QixPQUFPLEVBQUUsQ0FBQyxvQkFBb0IsRUFBRSw0QkFBNEIsQ0FBQztnQ0FDN0QsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDOzZCQUNqQixDQUFDOzRCQUNGLElBQUkscUJBQUcsQ0FBQyxlQUFlLENBQUM7Z0NBQ3RCLE9BQU8sRUFBRSxDQUFDLHFCQUFxQixFQUFFLG9CQUFvQixDQUFDO2dDQUN0RCxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7Z0NBQ2hCLFVBQVUsRUFBRTtvQ0FDVixZQUFZLEVBQUU7d0NBQ1oscUNBQXFDLEVBQUUsU0FBUzt3Q0FDaEQsdUNBQXVDLEVBQUUsV0FBVztxQ0FDckQ7aUNBQ0Y7NkJBQ0YsQ0FBQzt5QkFDSDtxQkFDRixDQUFDO2lCQUNIO2FBQ0YsQ0FBQyxDQUFDLE9BQU87WUFDVixhQUFhLEVBQUUsQ0FBQztvQkFDZCxNQUFNLEVBQUU7d0JBQ04sSUFBSSxFQUFFLFFBQVE7d0JBQ2QsZ0JBQWdCLEVBQUU7NEJBQ2hCLElBQUksRUFBRSxJQUFJOzRCQUNWLFNBQVMsRUFBRSxJQUFJO3lCQUNoQjtxQkFDRjtvQkFDRCxNQUFNLEVBQUU7d0JBQ04sSUFBSSxFQUFFLE9BQU87d0JBQ2IsS0FBSyxFQUFFLENBQUM7cUJBQ1Q7aUJBQ0YsQ0FBQztZQUNGLFlBQVksRUFBRSxXQUFXO1lBQ3pCLGlCQUFpQixFQUFFO2dCQUNqQixPQUFPLEVBQUU7b0JBQ1A7d0JBQ0UsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJO3dCQUNqQixlQUFlLEVBQUUsTUFBTSxDQUFDLE9BQU8sRUFBRSx1Q0FBdUM7cUJBQ3pFO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU8sY0FBYztRQUNwQixJQUFJLElBQUksQ0FBQyxlQUFlLENBQUMsTUFBTSxJQUFJLENBQUMsRUFBRSxDQUFDO1lBQ3JDLElBQUksQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsMkJBQTJCLENBQUMsSUFBSSxFQUFFLGFBQWEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRSxJQUFJLENBQUMsRUFBRSxFQUFFLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDM0osQ0FBQztRQUVELE9BQU8sSUFBSSxDQUFDLGVBQWUsQ0FBQztJQUM5QixDQUFDO0lBRU8sWUFBWSxDQUFDLElBQTJCLEVBQUUsVUFBa0IsRUFBRSxPQUFlO1FBQ25GLE1BQU0sZUFBZSxHQUFHLElBQUEsdUJBQWUsRUFBQyxtREFBdUIsRUFBRSxJQUFJLEVBQUUsb0NBQW9DLEVBQUU7WUFDM0csV0FBVyxFQUFFLDRGQUE0RjtZQUN6RyxhQUFhLEVBQUU7Z0JBQ2IsSUFBSSxxQkFBRyxDQUFDLGVBQWUsQ0FBQztvQkFDdEIsT0FBTyxFQUFFO3dCQUNQLHFDQUFxQzt3QkFDckMsMEJBQTBCO3FCQUMzQjtvQkFDRCxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7aUJBQ2pCLENBQUM7Z0JBQ0YsSUFBSSxxQkFBRyxDQUFDLGVBQWUsQ0FBQztvQkFDdEIsT0FBTyxFQUFFLENBQUMsb0JBQW9CLENBQUM7b0JBQy9CLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztpQkFDakIsQ0FBQztnQkFDRixJQUFJLHFCQUFHLENBQUMsZUFBZSxDQUFDO29CQUN0QixPQUFPLEVBQUUsQ0FBQyxxQkFBcUIsRUFBRSxvQkFBb0IsQ0FBQztvQkFDdEQsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO29CQUNoQixVQUFVLEVBQUU7d0JBQ1YsWUFBWSxFQUFFOzRCQUNaLHFDQUFxQyxFQUFFLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFNBQVM7eUJBQ3BFO3FCQUNGO2lCQUNGLENBQUM7Z0JBQ0YsSUFBSSxxQkFBRyxDQUFDLGVBQWUsQ0FBQztvQkFDdEIsT0FBTyxFQUFFLENBQUMsc0JBQXNCLENBQUM7b0JBQ2pDLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztpQkFDakIsQ0FBQzthQUNIO1lBQ0QsUUFBUSxFQUFFLElBQUEseUJBQWlCLEVBQUMsSUFBSSxFQUFFLHdCQUFnQixDQUFDLGtCQUFrQixDQUFDO1lBQ3RFLGFBQWEsRUFBRSx3QkFBTSxDQUFDLGFBQWEsQ0FBQyxJQUFJO1lBQ3hDLE9BQU8sRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7U0FDbEMsQ0FBQyxDQUFDO1FBRUgsSUFBSSw0QkFBYyxDQUFDLElBQUksRUFBRSxHQUFHLElBQUksVUFBVSxFQUFFO1lBQzFDLFlBQVksRUFBRSxlQUFlLENBQUMsV0FBVztZQUN6QyxZQUFZLEVBQUUsdUNBQXVDO1lBQ3JELFVBQVUsRUFBd0I7Z0JBQ2hDLGVBQWUsRUFBRSxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxTQUFTLENBQUM7b0JBQzVDLE9BQU8sRUFBRSxjQUFjO29CQUN2QixRQUFRLEVBQUUsT0FBTztvQkFDakIsWUFBWSxFQUFFLEdBQUcsVUFBVSxJQUFJLE9BQU8sRUFBRTtvQkFDeEMsU0FBUyxFQUFFLEdBQUcsQ0FBQyxTQUFTLENBQUMsbUJBQW1CO2lCQUM3QyxDQUFDO2FBQ0g7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDO0NBQ0Y7QUFyb0JELDhFQXFvQkM7QUFFRDs7R0FFRztBQUNILE1BQWEsa0NBQWtDO0lBQ3RDLE1BQU0sQ0FBQyxvQkFBb0IsQ0FBQyxLQUFnQixFQUFFLFdBQXNCO1FBQ3pFLE1BQU0sS0FBSyxHQUFHLElBQUkscUJBQUcsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLHNCQUFzQixDQUFDLENBQUM7UUFDM0QsTUFBTSxNQUFNLEdBQUcsSUFBSSwwREFBMEIsQ0FBQyxLQUFLLEVBQUUsNkJBQTZCLEVBQUU7WUFDbEYsUUFBUSxFQUFFLElBQUEseUJBQWlCLEVBQUMsS0FBSyxFQUFFLHdCQUFnQixDQUFDLGtCQUFrQixDQUFDO1lBQ3ZFLGFBQWEsRUFBRSx3QkFBTSxDQUFDLGFBQWEsQ0FBQyxJQUFJO1lBQ3hDLFdBQVcsRUFBRTtnQkFDWCxnQkFBZ0IsRUFBRSxXQUFXLENBQUMsUUFBUTthQUN2QztTQUNGLENBQUMsQ0FBQztRQUVILEtBQUssQ0FBQyxlQUFlLENBQUMsSUFBSSxtQ0FBSSxDQUFDLGtCQUFrQixDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUM7UUFDM0QsV0FBVyxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUVqQyxPQUFPLEtBQUssQ0FBQztJQUNmLENBQUM7SUFFRCxZQUFvQixLQUFpQjtRQUFqQixVQUFLLEdBQUwsS0FBSyxDQUFZO0lBQ3JDLENBQUM7SUFFTSxLQUFLLENBQUMsSUFBZ0I7UUFDM0IsSUFBSSxJQUFJLFlBQVksaUNBQWlDLEVBQUUsQ0FBQztZQUN0RCxNQUFNLE9BQU8sR0FBRyxJQUF5QyxDQUFDO1lBQzFELE1BQU0sU0FBUyxHQUFHLE9BQU8sQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLGdCQUFnQixDQUFDLENBQUM7WUFDOUQsSUFBSSxTQUFTLEVBQUUsQ0FBQztnQkFDZCxNQUFNLEtBQUssR0FBRyxTQUF3RCxDQUFDO2dCQUN2RSxLQUFLLENBQUMsV0FBVyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDO1lBQzFDLENBQUM7aUJBQU0sQ0FBQztnQkFDTixHQUFHLENBQUMsV0FBVyxDQUFDLEVBQUUsQ0FBQyxPQUFPLENBQUMsQ0FBQyxVQUFVLENBQUMsMERBQTBELENBQUMsQ0FBQztZQUNyRyxDQUFDO1FBQ0gsQ0FBQztJQUNILENBQUM7Q0FDRjtBQWhDRCxnRkFnQ0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjZGsgZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHtcbiAgQW5ub3RhdGlvbnMsXG4gIGF3c19lYzIgYXMgZWMyLFxuICBhd3NfZWNyIGFzIGVjcixcbiAgYXdzX2V2ZW50cyBhcyBldmVudHMsXG4gIGF3c19pYW0gYXMgaWFtLFxuICBhd3NfaW1hZ2VidWlsZGVyIGFzIGltYWdlYnVpbGRlcixcbiAgYXdzX2xhbWJkYSBhcyBsYW1iZGEsXG4gIGF3c19sb2dzIGFzIGxvZ3MsXG4gIGF3c19zM19hc3NldHMgYXMgczNfYXNzZXRzLFxuICBhd3Nfc25zIGFzIHNucyxcbiAgYXdzX3Nuc19zdWJzY3JpcHRpb25zIGFzIHN1YnMsXG4gIEN1c3RvbVJlc291cmNlLFxuICBEdXJhdGlvbixcbiAgUmVtb3ZhbFBvbGljeSxcbiAgU3RhY2ssXG59IGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCB7IFRhZ011dGFiaWxpdHkgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtZWNyJztcbmltcG9ydCB7IFJldGVudGlvbkRheXMgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtbG9ncyc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QsIElDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IEFtaVJlY2lwZSwgZGVmYXVsdEJhc2VBbWkgfSBmcm9tICcuL2FtaSc7XG5pbXBvcnQgeyBJbWFnZUJ1aWxkZXJPYmplY3RCYXNlIH0gZnJvbSAnLi9jb21tb24nO1xuaW1wb3J0IHsgQ29udGFpbmVyUmVjaXBlLCBkZWZhdWx0QmFzZURvY2tlckltYWdlIH0gZnJvbSAnLi9jb250YWluZXInO1xuaW1wb3J0IHsgRGVsZXRlUmVzb3VyY2VzRnVuY3Rpb24gfSBmcm9tICcuL2RlbGV0ZS1yZXNvdXJjZXMtZnVuY3Rpb24nO1xuaW1wb3J0IHsgRGVsZXRlUmVzb3VyY2VzUHJvcHMgfSBmcm9tICcuL2RlbGV0ZS1yZXNvdXJjZXMubGFtYmRhJztcbmltcG9ydCB7IEZpbHRlckZhaWxlZEJ1aWxkc0Z1bmN0aW9uIH0gZnJvbSAnLi9maWx0ZXItZmFpbGVkLWJ1aWxkcy1mdW5jdGlvbic7XG5pbXBvcnQgeyBnZW5lcmF0ZUJ1aWxkV29ya2Zsb3dXaXRoRG9ja2VyU2V0dXBDb21tYW5kcyB9IGZyb20gJy4vd29ya2Zsb3cnO1xuaW1wb3J0IHsgQXJjaGl0ZWN0dXJlLCBPcywgUnVubmVyQW1pLCBSdW5uZXJJbWFnZSwgUnVubmVyVmVyc2lvbiB9IGZyb20gJy4uLy4uL3Byb3ZpZGVycyc7XG5pbXBvcnQgeyBzaW5nbGV0b25Mb2dHcm91cCwgc2luZ2xldG9uTGFtYmRhLCBTaW5nbGV0b25Mb2dUeXBlIH0gZnJvbSAnLi4vLi4vdXRpbHMnO1xuaW1wb3J0IHsgQnVpbGRJbWFnZUZ1bmN0aW9uIH0gZnJvbSAnLi4vYnVpbGQtaW1hZ2UtZnVuY3Rpb24nO1xuaW1wb3J0IHsgUnVubmVySW1hZ2VCdWlsZGVyQmFzZSwgUnVubmVySW1hZ2VCdWlsZGVyUHJvcHMsIHVuaXF1ZUltYWdlQnVpbGRlck5hbWUgfSBmcm9tICcuLi9jb21tb24nO1xuXG5leHBvcnQgaW50ZXJmYWNlIEF3c0ltYWdlQnVpbGRlclJ1bm5lckltYWdlQnVpbGRlclByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBpbnN0YW5jZSB0eXBlIHVzZWQgdG8gYnVpbGQgdGhlIGltYWdlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBtNmkubGFyZ2VcbiAgICovXG4gIHJlYWRvbmx5IGluc3RhbmNlVHlwZT86IGVjMi5JbnN0YW5jZVR5cGU7XG5cbiAgLyoqXG4gICAqIFNpemUgb2Ygdm9sdW1lIGF2YWlsYWJsZSBmb3IgYnVpbGRlciBpbnN0YW5jZXMuIFRoaXMgbW9kaWZpZXMgdGhlIGJvb3Qgdm9sdW1lIHNpemUgYW5kIGRvZXNuJ3QgYWRkIGFueSBhZGRpdGlvbmFsIHZvbHVtZXMuXG4gICAqXG4gICAqIFVzZSB0aGlzIGlmIHlvdSdyZSBidWlsZGluZyBpbWFnZXMgd2l0aCBiaWcgY29tcG9uZW50cyBhbmQgbmVlZCBtb3JlIHNwYWNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBkZWZhdWx0IHNpemUgZm9yIEFNSSAodXN1YWxseSAzMEdCIGZvciBMaW51eCBhbmQgNTBHQiBmb3IgV2luZG93cylcbiAgICovXG4gIHJlYWRvbmx5IHN0b3JhZ2VTaXplPzogY2RrLlNpemU7XG5cbiAgLyoqXG4gICAqIE9wdGlvbnMgZm9yIGZhc3QgbGF1bmNoLlxuICAgKlxuICAgKiBUaGlzIGlzIG9ubHkgc3VwcG9ydGVkIGZvciBXaW5kb3dzIEFNSXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IGRpc2FibGVkXG4gICAqL1xuICByZWFkb25seSBmYXN0TGF1bmNoT3B0aW9ucz86IEZhc3RMYXVuY2hPcHRpb25zO1xufVxuXG4vKipcbiAqIE9wdGlvbnMgZm9yIGZhc3QgbGF1bmNoLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEZhc3RMYXVuY2hPcHRpb25zIHtcbiAgLyoqXG4gICAqIEVuYWJsZSBmYXN0IGxhdW5jaCBmb3IgQU1JcyBnZW5lcmF0ZWQgYnkgdGhpcyBidWlsZGVyLiBJdCBjcmVhdGVzIGEgc25hcHNob3Qgb2YgdGhlIHJvb3Qgdm9sdW1lIGFuZCB1c2VzIGl0IHRvIGxhdW5jaCBuZXcgaW5zdGFuY2VzIGZhc3Rlci5cbiAgICpcbiAgICogVGhpcyBpcyBvbmx5IHN1cHBvcnRlZCBmb3IgV2luZG93cyBBTUlzLlxuICAgKlxuICAgKiBAbm90ZSB0aGlzIGZlYXR1cmUgY29tZXMgd2l0aCBhZGRpdGlvbmFsIHJlc291cmNlIGNvc3RzLiBTZWUgdGhlIGRvY3VtZW50YXRpb24gZm9yIG1vcmUgZGV0YWlscy4gaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0VDMi9sYXRlc3QvV2luZG93c0d1aWRlL3dpbi1mYXN0LWxhdW5jaC1tYW5hZ2UtY29zdHMuaHRtbFxuICAgKiBAbm90ZSBlbmFibGluZyBmYXN0IGxhdW5jaCBvbiBhbiBleGlzdGluZyBidWlsZGVyIHdpbGwgbm90IGVuYWJsZSBpdCBmb3IgZXhpc3RpbmcgQU1Jcy4gSXQgd2lsbCBvbmx5IGFmZmVjdCBuZXcgQU1Jcy4gSWYgeW91IHdhbnQgaW1tZWRpYXRlIGVmZmVjdCwgdHJpZ2dlciBhIG5ldyBpbWFnZSBidWlsZC4gQWx0ZXJuYXRpdmVseSwgeW91IGNhbiBjcmVhdGUgYSBuZXcgYnVpbGRlciB3aXRoIGZhc3QgbGF1bmNoIGVuYWJsZWQgYW5kIHVzZSBpdCBmb3IgbmV3IEFNSXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBlbmFibGVkPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIG1heGltdW0gbnVtYmVyIG9mIHBhcmFsbGVsIGluc3RhbmNlcyB0aGF0IGFyZSBsYXVuY2hlZCBmb3IgY3JlYXRpbmcgcmVzb3VyY2VzLlxuICAgKlxuICAgKiBNdXN0IGJlIGF0IGxlYXN0IDYuXG4gICAqXG4gICAqIEBkZWZhdWx0IDZcbiAgICovXG4gIHJlYWRvbmx5IG1heFBhcmFsbGVsTGF1bmNoZXM/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2YgcHJlLXByb3Zpc2lvbmVkIHNuYXBzaG90cyB0byBrZWVwIG9uIGhhbmQgZm9yIGEgZmFzdC1sYXVuY2ggZW5hYmxlZCBXaW5kb3dzIEFNSS5cbiAgICpcbiAgICogQGRlZmF1bHQgMVxuICAgKi9cbiAgcmVhZG9ubHkgdGFyZ2V0UmVzb3VyY2VDb3VudD86IG51bWJlcjtcbn1cblxuLyoqXG4gKiBBbiBhc3NldCBpbmNsdWRpbmcgZmlsZSBvciBkaXJlY3RvcnkgdG8gcGxhY2UgaW5zaWRlIHRoZSBidWlsdCBpbWFnZS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJbWFnZUJ1aWxkZXJBc3NldCB7XG4gIC8qKlxuICAgKiBQYXRoIHRvIHBsYWNlIGFzc2V0IGluIHRoZSBpbWFnZS5cbiAgICovXG4gIHJlYWRvbmx5IHBhdGg6IHN0cmluZztcblxuICAvKipcbiAgICogQXNzZXQgdG8gcGxhY2UgaW4gdGhlIGltYWdlLlxuICAgKi9cbiAgcmVhZG9ubHkgYXNzZXQ6IHMzX2Fzc2V0cy5Bc3NldDtcbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBJbWFnZUJ1aWxkZXJDb21wb25lbnQgY29uc3RydWN0LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEltYWdlQnVpbGRlckNvbXBvbmVudFByb3BlcnRpZXMge1xuICAvKipcbiAgICogQ29tcG9uZW50IHBsYXRmb3JtLiBNdXN0IG1hdGNoIHRoZSBidWlsZGVyIHBsYXRmb3JtLlxuICAgKi9cbiAgcmVhZG9ubHkgcGxhdGZvcm06ICdMaW51eCcgfCAnV2luZG93cyc7XG5cbiAgLyoqXG4gICAqIENvbXBvbmVudCBkaXNwbGF5IG5hbWUuXG4gICAqL1xuICByZWFkb25seSBkaXNwbGF5TmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBDb21wb25lbnQgZGVzY3JpcHRpb24uXG4gICAqL1xuICByZWFkb25seSBkZXNjcmlwdGlvbjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBTaGVsbCBjb21tYW5kcyB0byBydW4gd2hlbiBhZGRpbmcgdGhpcyBjb21wb25lbnQgdG8gdGhlIGltYWdlLlxuICAgKlxuICAgKiBPbiBMaW51eCwgdGhlc2UgYXJlIGJhc2ggY29tbWFuZHMuIE9uIFdpbmRvd3MsIHRoZXJlIGFyZSBQb3dlclNoZWxsIGNvbW1hbmRzLlxuICAgKi9cbiAgcmVhZG9ubHkgY29tbWFuZHM6IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBPcHRpb25hbCBhc3NldHMgdG8gYWRkIHRvIHRoZSBidWlsdCBpbWFnZS5cbiAgICovXG4gIHJlYWRvbmx5IGFzc2V0cz86IEltYWdlQnVpbGRlckFzc2V0W107XG5cbiAgLyoqXG4gICAqIFJlcXVpcmUgYSByZWJvb3QgYWZ0ZXIgaW5zdGFsbGluZyB0aGlzIGNvbXBvbmVudC5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHJlYm9vdD86IGJvb2xlYW47XG59XG5cbi8qKlxuICogQ29tcG9uZW50cyBhcmUgYSBzZXQgb2YgY29tbWFuZHMgdG8gcnVuIGFuZCBvcHRpb25hbCBmaWxlcyB0byBhZGQgdG8gYW4gaW1hZ2UuIENvbXBvbmVudHMgYXJlIHRoZSBidWlsZGluZyBibG9ja3Mgb2YgaW1hZ2VzIGJ1aWx0IGJ5IEltYWdlIEJ1aWxkZXIuXG4gKlxuICogRXhhbXBsZTpcbiAqXG4gKiBgYGBcbiAqIG5ldyBJbWFnZUJ1aWxkZXJDb21wb25lbnQodGhpcywgJ0FXUyBDTEknLCB7XG4gKiAgIHBsYXRmb3JtOiAnV2luZG93cycsXG4gKiAgIGRpc3BsYXlOYW1lOiAnQVdTIENMSScsXG4gKiAgIGRlc2NyaXB0aW9uOiAnSW5zdGFsbCBsYXRlc3QgdmVyc2lvbiBvZiBBV1MgQ0xJJyxcbiAqICAgY29tbWFuZHM6IFtcbiAqICAgICAnJHAgPSBTdGFydC1Qcm9jZXNzIG1zaWV4ZWMuZXhlIC1QYXNzVGhydSAtV2FpdCAtQXJndW1lbnRMaXN0IFxcJy9pIGh0dHBzOi8vYXdzY2xpLmFtYXpvbmF3cy5jb20vQVdTQ0xJVjIubXNpIC9xblxcJycsXG4gKiAgICAgJ2lmICgkcC5FeGl0Q29kZSAtbmUgMCkgeyB0aHJvdyBcIkV4aXQgY29kZSBpcyAkcC5FeGl0Q29kZVwiIH0nLFxuICogICBdLFxuICogfVxuICogYGBgXG4gKlxuICogQGRlcHJlY2F0ZWQgVXNlIGBSdW5uZXJJbWFnZUNvbXBvbmVudGAgaW5zdGVhZCBhcyB0aGlzIGJlIGludGVybmFsIHNvb24uXG4gKi9cbmV4cG9ydCBjbGFzcyBJbWFnZUJ1aWxkZXJDb21wb25lbnQgZXh0ZW5kcyBJbWFnZUJ1aWxkZXJPYmplY3RCYXNlIHtcbiAgLyoqXG4gICAqIENvbXBvbmVudCBBUk4uXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgYXJuOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFN1cHBvcnRlZCBwbGF0Zm9ybSBmb3IgdGhlIGNvbXBvbmVudC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBwbGF0Zm9ybTogJ1dpbmRvd3MnIHwgJ0xpbnV4JztcblxuICBwcml2YXRlIHJlYWRvbmx5IGFzc2V0czogczNfYXNzZXRzLkFzc2V0W10gPSBbXTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogSW1hZ2VCdWlsZGVyQ29tcG9uZW50UHJvcGVydGllcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICB0aGlzLnBsYXRmb3JtID0gcHJvcHMucGxhdGZvcm07XG5cbiAgICBsZXQgc3RlcHM6IGFueVtdID0gW107XG5cbiAgICBpZiAocHJvcHMuYXNzZXRzKSB7XG4gICAgICBsZXQgaW5wdXRzOiBhbnlbXSA9IFtdO1xuICAgICAgbGV0IGV4dHJhY3RDb21tYW5kczogc3RyaW5nW10gPSBbXTtcbiAgICAgIGZvciAoY29uc3QgYXNzZXQgb2YgcHJvcHMuYXNzZXRzKSB7XG4gICAgICAgIHRoaXMuYXNzZXRzLnB1c2goYXNzZXQuYXNzZXQpO1xuXG4gICAgICAgIGlmIChhc3NldC5hc3NldC5pc0ZpbGUpIHtcbiAgICAgICAgICBpbnB1dHMucHVzaCh7XG4gICAgICAgICAgICBzb3VyY2U6IGFzc2V0LmFzc2V0LnMzT2JqZWN0VXJsLFxuICAgICAgICAgICAgZGVzdGluYXRpb246IGFzc2V0LnBhdGgsXG4gICAgICAgICAgfSk7XG4gICAgICAgIH0gZWxzZSBpZiAoYXNzZXQuYXNzZXQuaXNaaXBBcmNoaXZlKSB7XG4gICAgICAgICAgaW5wdXRzLnB1c2goe1xuICAgICAgICAgICAgc291cmNlOiBhc3NldC5hc3NldC5zM09iamVjdFVybCxcbiAgICAgICAgICAgIGRlc3RpbmF0aW9uOiBgJHthc3NldC5wYXRofS56aXBgLFxuICAgICAgICAgIH0pO1xuICAgICAgICAgIGlmIChwcm9wcy5wbGF0Zm9ybSA9PT0gJ1dpbmRvd3MnKSB7XG4gICAgICAgICAgICBleHRyYWN0Q29tbWFuZHMucHVzaChgRXhwYW5kLUFyY2hpdmUgXCIke2Fzc2V0LnBhdGh9LnppcFwiIC1EZXN0aW5hdGlvblBhdGggXCIke2Fzc2V0LnBhdGh9XCJgKTtcbiAgICAgICAgICAgIGV4dHJhY3RDb21tYW5kcy5wdXNoKGBkZWwgXCIke2Fzc2V0LnBhdGh9LnppcFwiYCk7XG4gICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgIGV4dHJhY3RDb21tYW5kcy5wdXNoKGB1bnppcCBcIiR7YXNzZXQucGF0aH0uemlwXCIgLWQgXCIke2Fzc2V0LnBhdGh9XCJgKTtcbiAgICAgICAgICAgIGV4dHJhY3RDb21tYW5kcy5wdXNoKGBybSBcIiR7YXNzZXQucGF0aH0uemlwXCJgKTtcbiAgICAgICAgICB9XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBVbmtub3duIGFzc2V0IHR5cGU6ICR7YXNzZXQuYXNzZXR9YCk7XG4gICAgICAgIH1cbiAgICAgIH1cblxuICAgICAgc3RlcHMucHVzaCh7XG4gICAgICAgIG5hbWU6ICdEb3dubG9hZCcsXG4gICAgICAgIGFjdGlvbjogJ1MzRG93bmxvYWQnLFxuICAgICAgICBpbnB1dHMsXG4gICAgICB9KTtcblxuICAgICAgaWYgKGV4dHJhY3RDb21tYW5kcy5sZW5ndGggPiAwKSB7XG4gICAgICAgIHN0ZXBzLnB1c2goe1xuICAgICAgICAgIG5hbWU6ICdFeHRyYWN0JyxcbiAgICAgICAgICBhY3Rpb246IHByb3BzLnBsYXRmb3JtID09PSAnTGludXgnID8gJ0V4ZWN1dGVCYXNoJyA6ICdFeGVjdXRlUG93ZXJTaGVsbCcsXG4gICAgICAgICAgaW5wdXRzOiB7XG4gICAgICAgICAgICBjb21tYW5kczogdGhpcy5wcmVmaXhDb21tYW5kc1dpdGhFcnJvckhhbmRsaW5nKHByb3BzLnBsYXRmb3JtLCBleHRyYWN0Q29tbWFuZHMpLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cblxuICAgIGlmIChwcm9wcy5jb21tYW5kcy5sZW5ndGggPiAwKSB7XG4gICAgICBzdGVwcy5wdXNoKHtcbiAgICAgICAgbmFtZTogJ1J1bicsXG4gICAgICAgIGFjdGlvbjogcHJvcHMucGxhdGZvcm0gPT09ICdMaW51eCcgPyAnRXhlY3V0ZUJhc2gnIDogJ0V4ZWN1dGVQb3dlclNoZWxsJyxcbiAgICAgICAgaW5wdXRzOiB7XG4gICAgICAgICAgY29tbWFuZHM6IHRoaXMucHJlZml4Q29tbWFuZHNXaXRoRXJyb3JIYW5kbGluZyhwcm9wcy5wbGF0Zm9ybSwgcHJvcHMuY29tbWFuZHMpLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLnJlYm9vdCA/PyBmYWxzZSkge1xuICAgICAgc3RlcHMucHVzaCh7XG4gICAgICAgIG5hbWU6ICdSZWJvb3QnLFxuICAgICAgICBhY3Rpb246ICdSZWJvb3QnLFxuICAgICAgICBpbnB1dHM6IHt9LFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgY29uc3QgZGF0YSA9IHtcbiAgICAgIG5hbWU6IHByb3BzLmRpc3BsYXlOYW1lLFxuICAgICAgc2NoZW1hVmVyc2lvbjogJzEuMCcsXG4gICAgICBwaGFzZXM6IFtcbiAgICAgICAge1xuICAgICAgICAgIG5hbWU6ICdidWlsZCcsXG4gICAgICAgICAgc3RlcHMsXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH07XG5cbiAgICBjb25zdCBuYW1lID0gdW5pcXVlSW1hZ2VCdWlsZGVyTmFtZSh0aGlzKTtcbiAgICBjb25zdCBjb21wb25lbnQgPSBuZXcgaW1hZ2VidWlsZGVyLkNmbkNvbXBvbmVudCh0aGlzLCAnQ29tcG9uZW50Jywge1xuICAgICAgbmFtZTogbmFtZSxcbiAgICAgIGRlc2NyaXB0aW9uOiBwcm9wcy5kZXNjcmlwdGlvbixcbiAgICAgIHBsYXRmb3JtOiBwcm9wcy5wbGF0Zm9ybSxcbiAgICAgIHZlcnNpb246IHRoaXMuZ2VuZXJhdGVWZXJzaW9uKCdDb21wb25lbnQnLCBuYW1lLCB7XG4gICAgICAgIHBsYXRmb3JtOiBwcm9wcy5wbGF0Zm9ybSxcbiAgICAgICAgZGF0YSxcbiAgICAgICAgZGVzY3JpcHRpb246IHByb3BzLmRlc2NyaXB0aW9uLFxuICAgICAgfSksXG4gICAgICBkYXRhOiBKU09OLnN0cmluZ2lmeShkYXRhKSxcbiAgICB9KTtcblxuICAgIHRoaXMuYXJuID0gY29tcG9uZW50LmF0dHJBcm47XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHJlYWQgcGVybWlzc2lvbnMgdG8gdGhlIHByaW5jaXBhbCBvbiB0aGUgYXNzZXRzIGJ1Y2tldHMuXG4gICAqXG4gICAqIEBwYXJhbSBncmFudGVlXG4gICAqL1xuICBncmFudEFzc2V0c1JlYWQoZ3JhbnRlZTogaWFtLklHcmFudGFibGUpIHtcbiAgICBmb3IgKGNvbnN0IGFzc2V0IG9mIHRoaXMuYXNzZXRzKSB7XG4gICAgICBhc3NldC5ncmFudFJlYWQoZ3JhbnRlZSk7XG4gICAgfVxuICB9XG5cbiAgcHJlZml4Q29tbWFuZHNXaXRoRXJyb3JIYW5kbGluZyhwbGF0Zm9ybTogJ1dpbmRvd3MnIHwgJ0xpbnV4JywgY29tbWFuZHM6IHN0cmluZ1tdKSB7XG4gICAgaWYgKHBsYXRmb3JtID09ICdXaW5kb3dzJykge1xuICAgICAgcmV0dXJuIFtcbiAgICAgICAgJyRFcnJvckFjdGlvblByZWZlcmVuY2UgPSBcXCdTdG9wXFwnJyxcbiAgICAgICAgJyRQcm9ncmVzc1ByZWZlcmVuY2UgPSBcXCdTaWxlbnRseUNvbnRpbnVlXFwnJyxcbiAgICAgICAgJ1NldC1QU0RlYnVnIC1UcmFjZSAxJyxcbiAgICAgIF0uY29uY2F0KGNvbW1hbmRzKTtcbiAgICB9IGVsc2Uge1xuICAgICAgcmV0dXJuIFtcbiAgICAgICAgJ3NldCAtZXgnLFxuICAgICAgXS5jb25jYXQoY29tbWFuZHMpO1xuICAgIH1cbiAgfVxufVxuXG4vKipcbiAqIEBpbnRlcm5hbFxuICovXG5leHBvcnQgY2xhc3MgQXdzSW1hZ2VCdWlsZGVyUnVubmVySW1hZ2VCdWlsZGVyIGV4dGVuZHMgUnVubmVySW1hZ2VCdWlsZGVyQmFzZSB7XG4gIHByaXZhdGUgYm91bmREb2NrZXJJbWFnZT86IFJ1bm5lckltYWdlO1xuICBwcml2YXRlIGJvdW5kQW1pPzogUnVubmVyQW1pO1xuICBwcml2YXRlIHJlYWRvbmx5IG9zOiBPcztcbiAgcHJpdmF0ZSByZWFkb25seSBhcmNoaXRlY3R1cmU6IEFyY2hpdGVjdHVyZTtcbiAgcHJpdmF0ZSByZWFkb25seSBiYXNlSW1hZ2U6IHN0cmluZztcbiAgcHJpdmF0ZSByZWFkb25seSBiYXNlQW1pOiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgbG9nUmV0ZW50aW9uOiBSZXRlbnRpb25EYXlzO1xuICBwcml2YXRlIHJlYWRvbmx5IGxvZ1JlbW92YWxQb2xpY3k6IFJlbW92YWxQb2xpY3k7XG4gIHByaXZhdGUgcmVhZG9ubHkgdnBjOiBlYzIuSVZwYztcbiAgcHJpdmF0ZSByZWFkb25seSBzZWN1cml0eUdyb3VwczogZWMyLklTZWN1cml0eUdyb3VwW107XG4gIHByaXZhdGUgcmVhZG9ubHkgc3VibmV0U2VsZWN0aW9uOiBlYzIuU3VibmV0U2VsZWN0aW9uIHwgdW5kZWZpbmVkO1xuICBwcml2YXRlIHJlYWRvbmx5IHJlYnVpbGRJbnRlcnZhbDogY2RrLkR1cmF0aW9uO1xuICBwcml2YXRlIHJlYWRvbmx5IGJvdW5kQ29tcG9uZW50czogSW1hZ2VCdWlsZGVyQ29tcG9uZW50W10gPSBbXTtcbiAgcHJpdmF0ZSByZWFkb25seSBpbnN0YW5jZVR5cGU6IGVjMi5JbnN0YW5jZVR5cGU7XG4gIHByaXZhdGUgaW5mcmFzdHJ1Y3R1cmU6IGltYWdlYnVpbGRlci5DZm5JbmZyYXN0cnVjdHVyZUNvbmZpZ3VyYXRpb24gfCB1bmRlZmluZWQ7XG4gIHByaXZhdGUgcmVhZG9ubHkgcm9sZTogaWFtLlJvbGU7XG4gIHByaXZhdGUgcmVhZG9ubHkgZmFzdExhdW5jaE9wdGlvbnM/OiBGYXN0TGF1bmNoT3B0aW9ucztcbiAgcHVibGljIHJlYWRvbmx5IHN0b3JhZ2VTaXplPzogY2RrLlNpemU7XG4gIHByaXZhdGUgcmVhZG9ubHkgd2FpdE9uRGVwbG95OiBib29sZWFuO1xuICBwcml2YXRlIHJlYWRvbmx5IGRvY2tlclNldHVwQ29tbWFuZHM6IHN0cmluZ1tdO1xuICBwcml2YXRlIHJlYWRvbmx5IHRhZ3M6IHsgW2tleTogc3RyaW5nXTogc3RyaW5nIH07XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM/OiBSdW5uZXJJbWFnZUJ1aWxkZXJQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwgcHJvcHMpO1xuXG4gICAgaWYgKHByb3BzPy5jb2RlQnVpbGRPcHRpb25zKSB7XG4gICAgICBBbm5vdGF0aW9ucy5vZih0aGlzKS5hZGRXYXJuaW5nKCdjb2RlQnVpbGRPcHRpb25zIGFyZSBpZ25vcmVkIHdoZW4gdXNpbmcgQVdTIEltYWdlIEJ1aWxkZXIgdG8gYnVpbGQgcnVubmVyIGltYWdlcy4nKTtcbiAgICB9XG5cbiAgICB0aGlzLm9zID0gcHJvcHM/Lm9zID8/IE9zLkxJTlVYX1VCVU5UVTtcbiAgICB0aGlzLmFyY2hpdGVjdHVyZSA9IHByb3BzPy5hcmNoaXRlY3R1cmUgPz8gQXJjaGl0ZWN0dXJlLlg4Nl82NDtcbiAgICB0aGlzLnJlYnVpbGRJbnRlcnZhbCA9IHByb3BzPy5yZWJ1aWxkSW50ZXJ2YWwgPz8gRHVyYXRpb24uZGF5cyg3KTtcbiAgICB0aGlzLmxvZ1JldGVudGlvbiA9IHByb3BzPy5sb2dSZXRlbnRpb24gPz8gUmV0ZW50aW9uRGF5cy5PTkVfTU9OVEg7XG4gICAgdGhpcy5sb2dSZW1vdmFsUG9saWN5ID0gcHJvcHM/LmxvZ1JlbW92YWxQb2xpY3kgPz8gUmVtb3ZhbFBvbGljeS5ERVNUUk9ZO1xuICAgIHRoaXMudnBjID0gcHJvcHM/LnZwYyA/PyBlYzIuVnBjLmZyb21Mb29rdXAodGhpcywgJ1ZQQycsIHsgaXNEZWZhdWx0OiB0cnVlIH0pO1xuICAgIHRoaXMuc2VjdXJpdHlHcm91cHMgPSBwcm9wcz8uc2VjdXJpdHlHcm91cHMgPz8gW25ldyBlYzIuU2VjdXJpdHlHcm91cCh0aGlzLCAnU0cnLCB7IHZwYzogdGhpcy52cGMgfSldO1xuICAgIHRoaXMuc3VibmV0U2VsZWN0aW9uID0gcHJvcHM/LnN1Ym5ldFNlbGVjdGlvbjtcbiAgICB0aGlzLmJhc2VJbWFnZSA9IHByb3BzPy5iYXNlRG9ja2VySW1hZ2UgPz8gZGVmYXVsdEJhc2VEb2NrZXJJbWFnZSh0aGlzLm9zKTtcbiAgICB0aGlzLmJhc2VBbWkgPSBwcm9wcz8uYmFzZUFtaSA/PyBkZWZhdWx0QmFzZUFtaSh0aGlzLCB0aGlzLm9zLCB0aGlzLmFyY2hpdGVjdHVyZSk7XG4gICAgdGhpcy5pbnN0YW5jZVR5cGUgPSBwcm9wcz8uYXdzSW1hZ2VCdWlsZGVyT3B0aW9ucz8uaW5zdGFuY2VUeXBlID8/IGVjMi5JbnN0YW5jZVR5cGUub2YoZWMyLkluc3RhbmNlQ2xhc3MuTTZJLCBlYzIuSW5zdGFuY2VTaXplLkxBUkdFKTtcbiAgICB0aGlzLmZhc3RMYXVuY2hPcHRpb25zID0gcHJvcHM/LmF3c0ltYWdlQnVpbGRlck9wdGlvbnM/LmZhc3RMYXVuY2hPcHRpb25zO1xuICAgIHRoaXMuc3RvcmFnZVNpemUgPSBwcm9wcz8uYXdzSW1hZ2VCdWlsZGVyT3B0aW9ucz8uc3RvcmFnZVNpemU7XG4gICAgdGhpcy53YWl0T25EZXBsb3kgPSBwcm9wcz8ud2FpdE9uRGVwbG95ID8/IHRydWU7XG4gICAgdGhpcy5kb2NrZXJTZXR1cENvbW1hbmRzID0gcHJvcHM/LmRvY2tlclNldHVwQ29tbWFuZHMgPz8gW107XG5cbiAgICAvLyB0YWdzIGZvciBmaW5kaW5nIHJlc291cmNlc1xuICAgIHRoaXMudGFncyA9IHtcbiAgICAgICdHaXRIdWJSdW5uZXJzOlN0YWNrJzogY2RrLlN0YWNrLm9mKHRoaXMpLnN0YWNrTmFtZSxcbiAgICAgICdHaXRIdWJSdW5uZXJzOkJ1aWxkZXInOiB0aGlzLm5vZGUucGF0aCxcbiAgICB9O1xuXG4gICAgLy8gY29uZmlybSBpbnN0YW5jZSB0eXBlXG4gICAgaWYgKCF0aGlzLmFyY2hpdGVjdHVyZS5pbnN0YW5jZVR5cGVNYXRjaCh0aGlzLmluc3RhbmNlVHlwZSkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgQnVpbGRlciBhcmNoaXRlY3R1cmUgKCR7dGhpcy5hcmNoaXRlY3R1cmUubmFtZX0pIGRvZXNuJ3QgbWF0Y2ggc2VsZWN0ZWQgaW5zdGFuY2UgdHlwZSAoJHt0aGlzLmluc3RhbmNlVHlwZX0gLyAke3RoaXMuaW5zdGFuY2VUeXBlLmFyY2hpdGVjdHVyZX0pYCk7XG4gICAgfVxuXG4gICAgLy8gd2FybiBhZ2FpbnN0IGlzb2xhdGVkIG5ldHdvcmtzXG4gICAgaWYgKHByb3BzPy5zdWJuZXRTZWxlY3Rpb24/LnN1Ym5ldFR5cGUgPT0gZWMyLlN1Ym5ldFR5cGUuUFJJVkFURV9JU09MQVRFRCkge1xuICAgICAgQW5ub3RhdGlvbnMub2YodGhpcykuYWRkV2FybmluZygnUHJpdmF0ZSBpc29sYXRlZCBzdWJuZXRzIGNhbm5vdCBwdWxsIGZyb20gcHVibGljIEVDUiBhbmQgVlBDIGVuZHBvaW50IGlzIG5vdCBzdXBwb3J0ZWQgeWV0LiAnICtcbiAgICAgICAgJ1NlZSBodHRwczovL2dpdGh1Yi5jb20vYXdzL2NvbnRhaW5lcnMtcm9hZG1hcC9pc3N1ZXMvMTE2MCcpO1xuICAgIH1cblxuICAgIC8vIHJvbGUgdG8gYmUgdXNlZCBieSBBV1MgSW1hZ2UgQnVpbGRlclxuICAgIHRoaXMucm9sZSA9IG5ldyBpYW0uUm9sZSh0aGlzLCAnUm9sZScsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdlYzIuYW1hem9uYXdzLmNvbScpLFxuICAgIH0pO1xuICB9XG5cbiAgcHJpdmF0ZSBwbGF0Zm9ybSgpIHtcbiAgICBpZiAodGhpcy5vcy5pcyhPcy5XSU5ET1dTKSkge1xuICAgICAgcmV0dXJuICdXaW5kb3dzJztcbiAgICB9XG4gICAgaWYgKHRoaXMub3MuaXNJbihPcy5fQUxMX0xJTlVYX1ZFUlNJT05TKSkge1xuICAgICAgcmV0dXJuICdMaW51eCc7XG4gICAgfVxuICAgIHRocm93IG5ldyBFcnJvcihgT1MgJHt0aGlzLm9zLm5hbWV9IGlzIG5vdCBzdXBwb3J0ZWQgYnkgQVdTIEltYWdlIEJ1aWxkZXJgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDYWxsZWQgYnkgSVJ1bm5lclByb3ZpZGVyIHRvIGZpbmFsaXplIHNldHRpbmdzIGFuZCBjcmVhdGUgdGhlIGltYWdlIGJ1aWxkZXIuXG4gICAqL1xuICBiaW5kRG9ja2VySW1hZ2UoKTogUnVubmVySW1hZ2Uge1xuICAgIGlmICh0aGlzLmJvdW5kRG9ja2VySW1hZ2UpIHtcbiAgICAgIHJldHVybiB0aGlzLmJvdW5kRG9ja2VySW1hZ2U7XG4gICAgfVxuXG4gICAgLy8gY3JlYXRlIHJlcG9zaXRvcnkgdGhhdCBvbmx5IGtlZXBzIG9uZSB0YWdcbiAgICBjb25zdCByZXBvc2l0b3J5ID0gbmV3IGVjci5SZXBvc2l0b3J5KHRoaXMsICdSZXBvc2l0b3J5Jywge1xuICAgICAgaW1hZ2VTY2FuT25QdXNoOiB0cnVlLFxuICAgICAgaW1hZ2VUYWdNdXRhYmlsaXR5OiBUYWdNdXRhYmlsaXR5Lk1VVEFCTEUsXG4gICAgICByZW1vdmFsUG9saWN5OiBSZW1vdmFsUG9saWN5LkRFU1RST1ksXG4gICAgICBlbXB0eU9uRGVsZXRlOiB0cnVlLFxuICAgIH0pO1xuXG4gICAgY29uc3QgZGlzdCA9IG5ldyBpbWFnZWJ1aWxkZXIuQ2ZuRGlzdHJpYnV0aW9uQ29uZmlndXJhdGlvbih0aGlzLCAnRG9ja2VyIERpc3RyaWJ1dGlvbicsIHtcbiAgICAgIG5hbWU6IHVuaXF1ZUltYWdlQnVpbGRlck5hbWUodGhpcyksXG4gICAgICAvLyBkZXNjcmlwdGlvbjogdGhpcy5kZXNjcmlwdGlvbixcbiAgICAgIGRpc3RyaWJ1dGlvbnM6IFtcbiAgICAgICAge1xuICAgICAgICAgIHJlZ2lvbjogU3RhY2sub2YodGhpcykucmVnaW9uLFxuICAgICAgICAgIGNvbnRhaW5lckRpc3RyaWJ1dGlvbkNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICAgIENvbnRhaW5lclRhZ3M6IFsnbGF0ZXN0J10sXG4gICAgICAgICAgICBUYXJnZXRSZXBvc2l0b3J5OiB7XG4gICAgICAgICAgICAgIFNlcnZpY2U6ICdFQ1InLFxuICAgICAgICAgICAgICBSZXBvc2l0b3J5TmFtZTogcmVwb3NpdG9yeS5yZXBvc2l0b3J5TmFtZSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgICB0YWdzOiB0aGlzLnRhZ3MsXG4gICAgfSk7XG5cbiAgICBsZXQgZG9ja2VyZmlsZVRlbXBsYXRlID0gYEZST00ge3t7IGltYWdlYnVpbGRlcjpwYXJlbnRJbWFnZSB9fX1cbnt7eyBpbWFnZWJ1aWxkZXI6ZW52aXJvbm1lbnRzIH19fVxue3t7IGltYWdlYnVpbGRlcjpjb21wb25lbnRzIH19fWA7XG5cbiAgICBmb3IgKGNvbnN0IGMgb2YgdGhpcy5jb21wb25lbnRzKSB7XG4gICAgICBjb25zdCBjb21tYW5kcyA9IGMuZ2V0RG9ja2VyQ29tbWFuZHModGhpcy5vcywgdGhpcy5hcmNoaXRlY3R1cmUpO1xuICAgICAgaWYgKGNvbW1hbmRzLmxlbmd0aCA+IDApIHtcbiAgICAgICAgZG9ja2VyZmlsZVRlbXBsYXRlICs9ICdcXG4nICsgY29tbWFuZHMuam9pbignXFxuJykgKyAnXFxuJztcbiAgICAgIH1cbiAgICB9XG5cbiAgICBjb25zdCByZWNpcGUgPSBuZXcgQ29udGFpbmVyUmVjaXBlKHRoaXMsICdDb250YWluZXIgUmVjaXBlJywge1xuICAgICAgcGxhdGZvcm06IHRoaXMucGxhdGZvcm0oKSxcbiAgICAgIGNvbXBvbmVudHM6IHRoaXMuYmluZENvbXBvbmVudHMoKSxcbiAgICAgIHRhcmdldFJlcG9zaXRvcnk6IHJlcG9zaXRvcnksXG4gICAgICBkb2NrZXJmaWxlVGVtcGxhdGU6IGRvY2tlcmZpbGVUZW1wbGF0ZSxcbiAgICAgIHBhcmVudEltYWdlOiB0aGlzLmJhc2VJbWFnZSxcbiAgICAgIHRhZ3M6IHRoaXMudGFncyxcbiAgICB9KTtcblxuICAgIGNvbnN0IGxvZyA9IHRoaXMuY3JlYXRlTG9nKCdEb2NrZXIgTG9nJywgcmVjaXBlLm5hbWUpO1xuICAgIGNvbnN0IGluZnJhID0gdGhpcy5jcmVhdGVJbmZyYXN0cnVjdHVyZShbXG4gICAgICBpYW0uTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ0FtYXpvblNTTU1hbmFnZWRJbnN0YW5jZUNvcmUnKSxcbiAgICAgIGlhbS5NYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnRUMySW5zdGFuY2VQcm9maWxlRm9ySW1hZ2VCdWlsZGVyRUNSQ29udGFpbmVyQnVpbGRzJyksXG4gICAgXSk7XG5cbiAgICBpZiAodGhpcy53YWl0T25EZXBsb3kpIHtcbiAgICAgIHRoaXMuY3JlYXRlSW1hZ2UoaW5mcmEsIGRpc3QsIGxvZywgdW5kZWZpbmVkLCByZWNpcGUuYXJuKTtcbiAgICB9XG4gICAgdGhpcy5kb2NrZXJJbWFnZUNsZWFuZXIocmVjaXBlLCByZXBvc2l0b3J5KTtcblxuICAgIHRoaXMuY3JlYXRlUGlwZWxpbmUoaW5mcmEsIGRpc3QsIGxvZywgdW5kZWZpbmVkLCByZWNpcGUuYXJuKTtcblxuICAgIHRoaXMuYm91bmREb2NrZXJJbWFnZSA9IHtcbiAgICAgIGltYWdlUmVwb3NpdG9yeTogcmVwb3NpdG9yeSxcbiAgICAgIGltYWdlVGFnOiAnbGF0ZXN0JyxcbiAgICAgIG9zOiB0aGlzLm9zLFxuICAgICAgYXJjaGl0ZWN0dXJlOiB0aGlzLmFyY2hpdGVjdHVyZSxcbiAgICAgIGxvZ0dyb3VwOiBsb2csXG4gICAgICBydW5uZXJWZXJzaW9uOiBSdW5uZXJWZXJzaW9uLnNwZWNpZmljKCd1bmtub3duJyksXG4gICAgICAvLyBubyBkZXBlbmRhYmxlIGFzIENsb3VkRm9ybWF0aW9uIHdpbGwgZmFpbCB0byBnZXQgaW1hZ2UgQVJOIG9uY2UgdGhlIGltYWdlIGlzIGRlbGV0ZWQgKHdlIGRlbGV0ZSBvbGQgaW1hZ2VzIGRhaWx5KVxuICAgIH07XG5cbiAgICByZXR1cm4gdGhpcy5ib3VuZERvY2tlckltYWdlO1xuICB9XG5cbiAgcHJpdmF0ZSBkb2NrZXJJbWFnZUNsZWFuZXIocmVjaXBlOiBDb250YWluZXJSZWNpcGUsIHJlcG9zaXRvcnk6IGVjci5JUmVwb3NpdG9yeSkge1xuICAgIC8vIHRoaXMgaXMgaGVyZSB0byBwcm92aWRlIHNhZmUgdXBncmFkZSBmcm9tIG9sZCBjZGstZ2l0aHViLXJ1bm5lcnMgdmVyc2lvbnNcbiAgICAvLyB0aGlzIGxhbWJkYSB3YXMgdXNlZCBieSBhIGN1c3RvbSByZXNvdXJjZSB0byBkZWxldGUgYWxsIGltYWdlcyBidWlsZHMgb24gY2xlYW51cFxuICAgIC8vIGlmIHdlIHJlbW92ZSB0aGUgY3VzdG9tIHJlc291cmNlIGFuZCB0aGUgbGFtYmRhLCB0aGUgb2xkIGltYWdlcyB3aWxsIGJlIGRlbGV0ZWQgb24gdXBkYXRlXG4gICAgLy8ga2VlcGluZyB0aGUgbGFtYmRhIGJ1dCByZW1vdmluZyB0aGUgcGVybWlzc2lvbnMgd2lsbCBtYWtlIHN1cmUgdGhhdCBkZWxldGlvbiB3aWxsIGZhaWxcbiAgICBjb25zdCBvbGREZWxldGVyID0gc2luZ2xldG9uTGFtYmRhKEJ1aWxkSW1hZ2VGdW5jdGlvbiwgdGhpcywgJ2J1aWxkLWltYWdlJywge1xuICAgICAgZGVzY3JpcHRpb246ICdDdXN0b20gcmVzb3VyY2UgaGFuZGxlciB0aGF0IHRyaWdnZXJzIENvZGVCdWlsZCB0byBidWlsZCBydW5uZXIgaW1hZ2VzJyxcbiAgICAgIHRpbWVvdXQ6IGNkay5EdXJhdGlvbi5taW51dGVzKDMpLFxuICAgICAgbG9nR3JvdXA6IHNpbmdsZXRvbkxvZ0dyb3VwKHRoaXMsIFNpbmdsZXRvbkxvZ1R5cGUuUlVOTkVSX0lNQUdFX0JVSUxEKSxcbiAgICAgIGxvZ2dpbmdGb3JtYXQ6IGxhbWJkYS5Mb2dnaW5nRm9ybWF0LkpTT04sXG4gICAgfSk7XG4gICAgb2xkRGVsZXRlci5hZGRUb1JvbGVQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkRFTlksXG4gICAgICBhY3Rpb25zOiBbJ2ltYWdlYnVpbGRlcjpEZWxldGVJbWFnZSddLFxuICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICB9KSk7XG5cbiAgICAvLyBkZWxldGUgb2xkIHZlcnNpb24gb24gdXBkYXRlIGFuZCBvbiBzdGFjayBkZWxldGlvblxuICAgIHRoaXMuaW1hZ2VDbGVhbmVyKCdDb250YWluZXInLCByZWNpcGUubmFtZS50b0xvd2VyQ2FzZSgpLCByZWNpcGUudmVyc2lvbik7XG5cbiAgICAvLyBkZWxldGUgb2xkIGRvY2tlciBpbWFnZXMgKyBJQiByZXNvdXJjZXMgZGFpbHlcbiAgICBuZXcgaW1hZ2VidWlsZGVyLkNmbkxpZmVjeWNsZVBvbGljeSh0aGlzLCAnTGlmZWN5Y2xlIFBvbGljeSBEb2NrZXInLCB7XG4gICAgICBuYW1lOiB1bmlxdWVJbWFnZUJ1aWxkZXJOYW1lKHRoaXMpLFxuICAgICAgZGVzY3JpcHRpb246IGBEZWxldGUgb2xkIEdpdEh1YiBSdW5uZXIgRG9ja2VyIGltYWdlcyBmb3IgJHt0aGlzLm5vZGUucGF0aH1gLFxuICAgICAgZXhlY3V0aW9uUm9sZTogbmV3IGlhbS5Sb2xlKHRoaXMsICdMaWZlY3ljbGUgUG9saWN5IERvY2tlciBSb2xlJywge1xuICAgICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnaW1hZ2VidWlsZGVyLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgICAgaW5saW5lUG9saWNpZXM6IHtcbiAgICAgICAgICBpYjogbmV3IGlhbS5Qb2xpY3lEb2N1bWVudCh7XG4gICAgICAgICAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICAgICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgICAgICBhY3Rpb25zOiBbJ3RhZzpHZXRSZXNvdXJjZXMnLCAnaW1hZ2VidWlsZGVyOkRlbGV0ZUltYWdlJ10sXG4gICAgICAgICAgICAgICAgcmVzb3VyY2VzOiBbJyonXSwgLy8gSW1hZ2UgQnVpbGRlciBkb2Vzbid0IHN1cHBvcnQgc2NvcGluZyB0aGlzIDooXG4gICAgICAgICAgICAgIH0pLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9KSxcbiAgICAgICAgICBlY3I6IG5ldyBpYW0uUG9saWN5RG9jdW1lbnQoe1xuICAgICAgICAgICAgc3RhdGVtZW50czogW1xuICAgICAgICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICAgICAgYWN0aW9uczogWydlY3I6QmF0Y2hHZXRJbWFnZScsICdlY3I6QmF0Y2hEZWxldGVJbWFnZSddLFxuICAgICAgICAgICAgICAgIHJlc291cmNlczogW3JlcG9zaXRvcnkucmVwb3NpdG9yeUFybl0sXG4gICAgICAgICAgICAgIH0pLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9KSxcbiAgICAgICAgfSxcbiAgICAgIH0pLnJvbGVBcm4sXG4gICAgICBwb2xpY3lEZXRhaWxzOiBbe1xuICAgICAgICBhY3Rpb246IHtcbiAgICAgICAgICB0eXBlOiAnREVMRVRFJyxcbiAgICAgICAgICBpbmNsdWRlUmVzb3VyY2VzOiB7XG4gICAgICAgICAgICBjb250YWluZXJzOiB0cnVlLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICAgIGZpbHRlcjoge1xuICAgICAgICAgIHR5cGU6ICdDT1VOVCcsXG4gICAgICAgICAgdmFsdWU6IDIsXG4gICAgICAgIH0sXG4gICAgICB9XSxcbiAgICAgIHJlc291cmNlVHlwZTogJ0NPTlRBSU5FUl9JTUFHRScsXG4gICAgICByZXNvdXJjZVNlbGVjdGlvbjoge1xuICAgICAgICByZWNpcGVzOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgbmFtZTogcmVjaXBlLm5hbWUsXG4gICAgICAgICAgICBzZW1hbnRpY1ZlcnNpb246IHJlY2lwZS52ZXJzaW9uLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgIH0pO1xuICB9XG5cbiAgcHJvdGVjdGVkIGNyZWF0ZUxvZyhpZDogc3RyaW5nLCByZWNpcGVOYW1lOiBzdHJpbmcpOiBsb2dzLkxvZ0dyb3VwIHtcbiAgICByZXR1cm4gbmV3IGxvZ3MuTG9nR3JvdXAodGhpcywgaWQsIHtcbiAgICAgIGxvZ0dyb3VwTmFtZTogYC9hd3MvaW1hZ2VidWlsZGVyLyR7cmVjaXBlTmFtZX1gLFxuICAgICAgcmV0ZW50aW9uOiB0aGlzLmxvZ1JldGVudGlvbixcbiAgICAgIHJlbW92YWxQb2xpY3k6IHRoaXMubG9nUmVtb3ZhbFBvbGljeSxcbiAgICB9KTtcbiAgfVxuXG4gIHByb3RlY3RlZCBjcmVhdGVJbmZyYXN0cnVjdHVyZShtYW5hZ2VkUG9saWNpZXM6IGlhbS5JTWFuYWdlZFBvbGljeVtdKTogaW1hZ2VidWlsZGVyLkNmbkluZnJhc3RydWN0dXJlQ29uZmlndXJhdGlvbiB7XG4gICAgaWYgKHRoaXMuaW5mcmFzdHJ1Y3R1cmUpIHtcbiAgICAgIHJldHVybiB0aGlzLmluZnJhc3RydWN0dXJlO1xuICAgIH1cblxuICAgIGZvciAoY29uc3QgbWFuYWdlZFBvbGljeSBvZiBtYW5hZ2VkUG9saWNpZXMpIHtcbiAgICAgIHRoaXMucm9sZS5hZGRNYW5hZ2VkUG9saWN5KG1hbmFnZWRQb2xpY3kpO1xuICAgIH1cblxuICAgIGZvciAoY29uc3QgY29tcG9uZW50IG9mIHRoaXMuYm91bmRDb21wb25lbnRzKSB7XG4gICAgICBjb21wb25lbnQuZ3JhbnRBc3NldHNSZWFkKHRoaXMucm9sZSk7XG4gICAgfVxuXG4gICAgdGhpcy5pbmZyYXN0cnVjdHVyZSA9IG5ldyBpbWFnZWJ1aWxkZXIuQ2ZuSW5mcmFzdHJ1Y3R1cmVDb25maWd1cmF0aW9uKHRoaXMsICdJbmZyYXN0cnVjdHVyZScsIHtcbiAgICAgIG5hbWU6IHVuaXF1ZUltYWdlQnVpbGRlck5hbWUodGhpcyksXG4gICAgICAvLyBkZXNjcmlwdGlvbjogdGhpcy5kZXNjcmlwdGlvbixcbiAgICAgIHN1Ym5ldElkOiB0aGlzLnZwYz8uc2VsZWN0U3VibmV0cyh0aGlzLnN1Ym5ldFNlbGVjdGlvbikuc3VibmV0SWRzWzBdLFxuICAgICAgc2VjdXJpdHlHcm91cElkczogdGhpcy5zZWN1cml0eUdyb3Vwcz8ubWFwKHNnID0+IHNnLnNlY3VyaXR5R3JvdXBJZCksXG4gICAgICBpbnN0YW5jZVR5cGVzOiBbdGhpcy5pbnN0YW5jZVR5cGUudG9TdHJpbmcoKV0sXG4gICAgICBpbnN0YW5jZU1ldGFkYXRhT3B0aW9uczoge1xuICAgICAgICBodHRwVG9rZW5zOiAncmVxdWlyZWQnLFxuICAgICAgICAvLyBDb250YWluZXIgYnVpbGRzIHJlcXVpcmUgYSBtaW5pbXVtIG9mIHR3byBob3BzLlxuICAgICAgICBodHRwUHV0UmVzcG9uc2VIb3BMaW1pdDogMixcbiAgICAgIH0sXG4gICAgICBpbnN0YW5jZVByb2ZpbGVOYW1lOiBuZXcgaWFtLkNmbkluc3RhbmNlUHJvZmlsZSh0aGlzLCAnSW5zdGFuY2UgUHJvZmlsZScsIHtcbiAgICAgICAgcm9sZXM6IFtcbiAgICAgICAgICB0aGlzLnJvbGUucm9sZU5hbWUsXG4gICAgICAgIF0sXG4gICAgICB9KS5yZWYsXG4gICAgfSk7XG5cbiAgICByZXR1cm4gdGhpcy5pbmZyYXN0cnVjdHVyZTtcbiAgfVxuXG4gIHByb3RlY3RlZCBjcmVhdGVJbWFnZShpbmZyYTogaW1hZ2VidWlsZGVyLkNmbkluZnJhc3RydWN0dXJlQ29uZmlndXJhdGlvbiwgZGlzdDogaW1hZ2VidWlsZGVyLkNmbkRpc3RyaWJ1dGlvbkNvbmZpZ3VyYXRpb24sIGxvZzogbG9ncy5Mb2dHcm91cCxcbiAgICBpbWFnZVJlY2lwZUFybj86IHN0cmluZywgY29udGFpbmVyUmVjaXBlQXJuPzogc3RyaW5nKTogaW1hZ2VidWlsZGVyLkNmbkltYWdlIHtcbiAgICBjb25zdCBpbWFnZSA9IG5ldyBpbWFnZWJ1aWxkZXIuQ2ZuSW1hZ2UodGhpcywgdGhpcy5hbWlPckNvbnRhaW5lcklkKCdJbWFnZScsIGltYWdlUmVjaXBlQXJuLCBjb250YWluZXJSZWNpcGVBcm4pLCB7XG4gICAgICBpbmZyYXN0cnVjdHVyZUNvbmZpZ3VyYXRpb25Bcm46IGluZnJhLmF0dHJBcm4sXG4gICAgICBkaXN0cmlidXRpb25Db25maWd1cmF0aW9uQXJuOiBkaXN0LmF0dHJBcm4sXG4gICAgICBpbWFnZVJlY2lwZUFybixcbiAgICAgIGNvbnRhaW5lclJlY2lwZUFybixcbiAgICAgIGltYWdlVGVzdHNDb25maWd1cmF0aW9uOiB7XG4gICAgICAgIGltYWdlVGVzdHNFbmFibGVkOiBmYWxzZSxcbiAgICAgIH0sXG4gICAgICB0YWdzOiB0aGlzLnRhZ3MsXG4gICAgfSk7XG4gICAgaW1hZ2Uubm9kZS5hZGREZXBlbmRlbmN5KGluZnJhKTtcbiAgICBpbWFnZS5ub2RlLmFkZERlcGVuZGVuY3kobG9nKTtcblxuICAgIC8vIGRvIG5vdCBkZWxldGUgdGhlIGltYWdlIGFzIGl0IHdpbGwgYmUgZGVsZXRlZCBieSBpbWFnZUNsZWFuZXIoKS5cbiAgICAvLyBpZiB3ZSBkZWxldGUgaXQgaGVyZSwgaW1hZ2VDbGVhbmVyKCkgd29uJ3QgYmUgYWJsZSB0byBmaW5kIHRoZSBpbWFnZS5cbiAgICAvLyBpZiBpbWFnZUNsZWFuZXIoKSBjYW4ndCBmaW5kIHRoZSBpbWFnZSwgaXQgd29uJ3QgYmUgYWJsZSB0byBkZWxldGUgdGhlIGxpbmtlZCBBTUkvRG9ja2VyIGltYWdlLlxuICAgIC8vIHVzZSBSRVRBSU5fT05fVVBEQVRFX09SX0RFTEVURSwgc28gZXZlcnl0aGluZyBpcyBjbGVhbmVkIG9ubHkgb24gcm9sbGJhY2suXG4gICAgaW1hZ2UuYXBwbHlSZW1vdmFsUG9saWN5KFJlbW92YWxQb2xpY3kuUkVUQUlOX09OX1VQREFURV9PUl9ERUxFVEUpO1xuXG4gICAgcmV0dXJuIGltYWdlO1xuICB9XG5cbiAgcHJpdmF0ZSBhbWlPckNvbnRhaW5lcklkKGJhc2VJZDogc3RyaW5nLCBpbWFnZVJlY2lwZUFybj86IHN0cmluZywgY29udGFpbmVyUmVjaXBlQXJuPzogc3RyaW5nKSB7XG4gICAgaWYgKGltYWdlUmVjaXBlQXJuKSB7XG4gICAgICByZXR1cm4gYEFNSSAke2Jhc2VJZH1gO1xuICAgIH1cbiAgICBpZiAoY29udGFpbmVyUmVjaXBlQXJuKSB7XG4gICAgICByZXR1cm4gYERvY2tlciAke2Jhc2VJZH1gO1xuICAgIH1cbiAgICB0aHJvdyBuZXcgRXJyb3IoJ0VpdGhlciBpbWFnZVJlY2lwZUFybiBvciBjb250YWluZXJSZWNpcGVBcm4gbXVzdCBiZSBkZWZpbmVkJyk7XG4gIH1cblxuICBwcm90ZWN0ZWQgY3JlYXRlUGlwZWxpbmUoaW5mcmE6IGltYWdlYnVpbGRlci5DZm5JbmZyYXN0cnVjdHVyZUNvbmZpZ3VyYXRpb24sIGRpc3Q6IGltYWdlYnVpbGRlci5DZm5EaXN0cmlidXRpb25Db25maWd1cmF0aW9uLCBsb2c6IGxvZ3MuTG9nR3JvdXAsXG4gICAgaW1hZ2VSZWNpcGVBcm4/OiBzdHJpbmcsIGNvbnRhaW5lclJlY2lwZUFybj86IHN0cmluZyk6IGltYWdlYnVpbGRlci5DZm5JbWFnZVBpcGVsaW5lIHtcbiAgICAvLyBzZXQgc2NoZWR1bGVcbiAgICBsZXQgc2NoZWR1bGVPcHRpb25zOiBpbWFnZWJ1aWxkZXIuQ2ZuSW1hZ2VQaXBlbGluZS5TY2hlZHVsZVByb3BlcnR5IHwgdW5kZWZpbmVkO1xuICAgIGlmICh0aGlzLnJlYnVpbGRJbnRlcnZhbC50b0RheXMoKSA+IDApIHtcbiAgICAgIHNjaGVkdWxlT3B0aW9ucyA9IHtcbiAgICAgICAgc2NoZWR1bGVFeHByZXNzaW9uOiBldmVudHMuU2NoZWR1bGUucmF0ZSh0aGlzLnJlYnVpbGRJbnRlcnZhbCkuZXhwcmVzc2lvblN0cmluZyxcbiAgICAgICAgcGlwZWxpbmVFeGVjdXRpb25TdGFydENvbmRpdGlvbjogJ0VYUFJFU1NJT05fTUFUQ0hfT05MWScsXG4gICAgICB9O1xuICAgIH1cblxuICAgIC8vIGdlbmVyYXRlIHdvcmtmbG93cywgaWYgbmVlZGVkXG4gICAgbGV0IHdvcmtmbG93czogaW1hZ2VidWlsZGVyLkNmbkltYWdlUGlwZWxpbmUuV29ya2Zsb3dDb25maWd1cmF0aW9uUHJvcGVydHlbXSB8IHVuZGVmaW5lZDtcbiAgICBsZXQgZXhlY3V0aW9uUm9sZTogaWFtLklSb2xlIHwgdW5kZWZpbmVkO1xuICAgIGlmICh0aGlzLmRvY2tlclNldHVwQ29tbWFuZHMubGVuZ3RoID4gMCkge1xuICAgICAgd29ya2Zsb3dzID0gW3tcbiAgICAgICAgd29ya2Zsb3dBcm46IGdlbmVyYXRlQnVpbGRXb3JrZmxvd1dpdGhEb2NrZXJTZXR1cENvbW1hbmRzKHRoaXMsICdCdWlsZCcsIHRoaXMuZG9ja2VyU2V0dXBDb21tYW5kcykuYXJuLFxuICAgICAgfV07XG4gICAgICBleGVjdXRpb25Sb2xlID0gaWFtLlJvbGUuZnJvbVJvbGVBcm4odGhpcywgJ0ltYWdlIEJ1aWxkZXIgUm9sZScsIGNkay5TdGFjay5vZih0aGlzKS5mb3JtYXRBcm4oe1xuICAgICAgICBzZXJ2aWNlOiAnaWFtJyxcbiAgICAgICAgcmVnaW9uOiAnJyxcbiAgICAgICAgcmVzb3VyY2U6ICdyb2xlJyxcbiAgICAgICAgcmVzb3VyY2VOYW1lOiAnYXdzLXNlcnZpY2Utcm9sZS9pbWFnZWJ1aWxkZXIuYW1hem9uYXdzLmNvbS9BV1NTZXJ2aWNlUm9sZUZvckltYWdlQnVpbGRlcicsXG4gICAgICB9KSk7XG4gICAgfVxuXG4gICAgLy8gZ2VuZXJhdGUgcGlwZWxpbmVcbiAgICBjb25zdCBwaXBlbGluZSA9IG5ldyBpbWFnZWJ1aWxkZXIuQ2ZuSW1hZ2VQaXBlbGluZSh0aGlzLCB0aGlzLmFtaU9yQ29udGFpbmVySWQoJ1BpcGVsaW5lJywgaW1hZ2VSZWNpcGVBcm4sIGNvbnRhaW5lclJlY2lwZUFybiksIHtcbiAgICAgIG5hbWU6IHVuaXF1ZUltYWdlQnVpbGRlck5hbWUodGhpcyksXG4gICAgICAvLyBkZXNjcmlwdGlvbjogdGhpcy5kZXNjcmlwdGlvbixcbiAgICAgIGluZnJhc3RydWN0dXJlQ29uZmlndXJhdGlvbkFybjogaW5mcmEuYXR0ckFybixcbiAgICAgIGRpc3RyaWJ1dGlvbkNvbmZpZ3VyYXRpb25Bcm46IGRpc3QuYXR0ckFybixcbiAgICAgIGltYWdlUmVjaXBlQXJuLFxuICAgICAgY29udGFpbmVyUmVjaXBlQXJuLFxuICAgICAgc2NoZWR1bGU6IHNjaGVkdWxlT3B0aW9ucyxcbiAgICAgIGltYWdlVGVzdHNDb25maWd1cmF0aW9uOiB7XG4gICAgICAgIGltYWdlVGVzdHNFbmFibGVkOiBmYWxzZSxcbiAgICAgIH0sXG4gICAgICB3b3JrZmxvd3M6IHdvcmtmbG93cyxcbiAgICAgIGV4ZWN1dGlvblJvbGU6IGV4ZWN1dGlvblJvbGU/LnJvbGVBcm4sXG4gICAgICB0YWdzOiB0aGlzLnRhZ3MsXG4gICAgfSk7XG4gICAgcGlwZWxpbmUubm9kZS5hZGREZXBlbmRlbmN5KGluZnJhKTtcbiAgICBwaXBlbGluZS5ub2RlLmFkZERlcGVuZGVuY3kobG9nKTtcblxuICAgIHJldHVybiBwaXBlbGluZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgbmV0d29yayBjb25uZWN0aW9ucyBhc3NvY2lhdGVkIHdpdGggdGhpcyByZXNvdXJjZS5cbiAgICovXG4gIHB1YmxpYyBnZXQgY29ubmVjdGlvbnMoKTogZWMyLkNvbm5lY3Rpb25zIHtcbiAgICByZXR1cm4gbmV3IGVjMi5Db25uZWN0aW9ucyh7IHNlY3VyaXR5R3JvdXBzOiB0aGlzLnNlY3VyaXR5R3JvdXBzIH0pO1xuICB9XG5cbiAgcHVibGljIGdldCBncmFudFByaW5jaXBhbCgpOiBpYW0uSVByaW5jaXBhbCB7XG4gICAgcmV0dXJuIHRoaXMucm9sZTtcbiAgfVxuXG4gIGJpbmRBbWkoKTogUnVubmVyQW1pIHtcbiAgICBpZiAodGhpcy5ib3VuZEFtaSkge1xuICAgICAgcmV0dXJuIHRoaXMuYm91bmRBbWk7XG4gICAgfVxuXG4gICAgY29uc3QgbGF1bmNoVGVtcGxhdGUgPSBuZXcgZWMyLkxhdW5jaFRlbXBsYXRlKHRoaXMsICdMYXVuY2ggdGVtcGxhdGUnLCB7XG4gICAgICByZXF1aXJlSW1kc3YyOiB0cnVlLFxuICAgIH0pO1xuXG4gICAgY29uc3QgbGF1bmNoVGVtcGxhdGVDb25maWdzOiBpbWFnZWJ1aWxkZXIuQ2ZuRGlzdHJpYnV0aW9uQ29uZmlndXJhdGlvbi5MYXVuY2hUZW1wbGF0ZUNvbmZpZ3VyYXRpb25Qcm9wZXJ0eVtdID0gW3tcbiAgICAgIGxhdW5jaFRlbXBsYXRlSWQ6IGxhdW5jaFRlbXBsYXRlLmxhdW5jaFRlbXBsYXRlSWQsXG4gICAgICBzZXREZWZhdWx0VmVyc2lvbjogdHJ1ZSxcbiAgICB9XTtcbiAgICBjb25zdCBmYXN0TGF1bmNoQ29uZmlnczogaW1hZ2VidWlsZGVyLkNmbkRpc3RyaWJ1dGlvbkNvbmZpZ3VyYXRpb24uRmFzdExhdW5jaENvbmZpZ3VyYXRpb25Qcm9wZXJ0eVtdID0gW107XG5cbiAgICBpZiAodGhpcy5mYXN0TGF1bmNoT3B0aW9ucz8uZW5hYmxlZCA/PyBmYWxzZSkge1xuICAgICAgaWYgKCF0aGlzLm9zLmlzKE9zLldJTkRPV1MpKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignRmFzdCBsYXVuY2ggaXMgb25seSBzdXBwb3J0ZWQgZm9yIFdpbmRvd3MnKTtcbiAgICAgIH1cblxuICAgICAgLy8gY3JlYXRlIGEgc2VwYXJhdGUgbGF1bmNoIHRlbXBsYXRlIGZvciBmYXN0IGxhdW5jaCBzbzpcbiAgICAgIC8vICAtIHNldHRpbmdzIGRvbid0IGFmZmVjdCB0aGUgcnVubmVyc1xuICAgICAgLy8gIC0gZW5hYmxpbmcgZmFzdCBsYXVuY2ggb24gYW4gZXhpc3RpbmcgYnVpbGRlciB3b3JrcyAod2l0aG91dCBhIG5ldyBsYXVuY2ggdGVtcGxhdGUsIEVDMiBJbWFnZSBCdWlsZGVyIHdpbGwgdXNlIHRoZSBmaXJzdCB2ZXJzaW9uIG9mIHRoZSBsYXVuY2ggdGVtcGxhdGUsIHdoaWNoIGRvZXNuJ3QgaGF2ZSBpbnN0YW5jZSBvciBWUEMgY29uZmlnKVxuICAgICAgLy8gIC0gc2V0dGluZyB2cGMgKyBzdWJuZXQgb24gdGhlIG1haW4gbGF1bmNoIHRlbXBsYXRlIHdpbGwgY2F1c2UgUnVuSW5zdGFuY2VzIHRvIGZhaWxcbiAgICAgIC8vICAtIEVDMiBJbWFnZSBCdWlsZGVyIHNlZW1zIHRvIGdldCBjb25mdXNlZCB3aXRoIHdoaWNoIGxhdW5jaCB0ZW1wbGF0ZSB2ZXJzaW9uIHRvIGJhc2UgYW55IG5ldyB2ZXJzaW9uIG9uLCBzbyBhIG5ldyB0ZW1wbGF0ZSBpcyBhbHdheXMgYmVzdFxuICAgICAgY29uc3QgZmFzdExhdW5jaFRlbXBsYXRlID0gbmV3IGVjMi5DZm5MYXVuY2hUZW1wbGF0ZSh0aGlzLCAnRmFzdCBMYXVuY2ggVGVtcGxhdGUnLCB7XG4gICAgICAgIGxhdW5jaFRlbXBsYXRlRGF0YToge1xuICAgICAgICAgIG1ldGFkYXRhT3B0aW9uczoge1xuICAgICAgICAgICAgaHR0cFRva2VuczogJ3JlcXVpcmVkJyxcbiAgICAgICAgICB9LFxuICAgICAgICAgIGluc3RhbmNlVHlwZTogdGhpcy5pbnN0YW5jZVR5cGUudG9TdHJpbmcoKSxcbiAgICAgICAgICBuZXR3b3JrSW50ZXJmYWNlczogW3tcbiAgICAgICAgICAgIHN1Ym5ldElkOiB0aGlzLnZwYz8uc2VsZWN0U3VibmV0cyh0aGlzLnN1Ym5ldFNlbGVjdGlvbikuc3VibmV0SWRzWzBdLFxuICAgICAgICAgICAgZGV2aWNlSW5kZXg6IDAsXG4gICAgICAgICAgICBncm91cHM6IHRoaXMuc2VjdXJpdHlHcm91cHMubWFwKHNnID0+IHNnLnNlY3VyaXR5R3JvdXBJZCksXG4gICAgICAgICAgfV0sXG4gICAgICAgICAgdGFnU3BlY2lmaWNhdGlvbnM6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgcmVzb3VyY2VUeXBlOiAnaW5zdGFuY2UnLFxuICAgICAgICAgICAgICB0YWdzOiBbe1xuICAgICAgICAgICAgICAgIGtleTogJ05hbWUnLFxuICAgICAgICAgICAgICAgIHZhbHVlOiBgJHt0aGlzLm5vZGUucGF0aH0vRmFzdCBMYXVuY2ggSW5zdGFuY2VgLFxuICAgICAgICAgICAgICB9XSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIHJlc291cmNlVHlwZTogJ3ZvbHVtZScsXG4gICAgICAgICAgICAgIHRhZ3M6IFt7XG4gICAgICAgICAgICAgICAga2V5OiAnTmFtZScsXG4gICAgICAgICAgICAgICAgdmFsdWU6IGAke3RoaXMubm9kZS5wYXRofS9GYXN0IExhdW5jaCBJbnN0YW5jZWAsXG4gICAgICAgICAgICAgIH1dLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICB9LFxuICAgICAgICB0YWdTcGVjaWZpY2F0aW9uczogW3tcbiAgICAgICAgICByZXNvdXJjZVR5cGU6ICdsYXVuY2gtdGVtcGxhdGUnLFxuICAgICAgICAgIHRhZ3M6IFt7XG4gICAgICAgICAgICBrZXk6ICdOYW1lJyxcbiAgICAgICAgICAgIHZhbHVlOiBgJHt0aGlzLm5vZGUucGF0aH0vRmFzdCBMYXVuY2ggVGVtcGxhdGVgLFxuICAgICAgICAgIH1dLFxuICAgICAgICB9XSxcbiAgICAgIH0pO1xuXG4gICAgICBsYXVuY2hUZW1wbGF0ZUNvbmZpZ3MucHVzaCh7XG4gICAgICAgIGxhdW5jaFRlbXBsYXRlSWQ6IGZhc3RMYXVuY2hUZW1wbGF0ZS5hdHRyTGF1bmNoVGVtcGxhdGVJZCxcbiAgICAgICAgc2V0RGVmYXVsdFZlcnNpb246IHRydWUsXG4gICAgICB9KTtcbiAgICAgIGZhc3RMYXVuY2hDb25maWdzLnB1c2goe1xuICAgICAgICBlbmFibGVkOiB0cnVlLFxuICAgICAgICBsYXVuY2hUZW1wbGF0ZToge1xuICAgICAgICAgIGxhdW5jaFRlbXBsYXRlSWQ6IGZhc3RMYXVuY2hUZW1wbGF0ZS5hdHRyTGF1bmNoVGVtcGxhdGVJZCxcbiAgICAgICAgfSxcbiAgICAgICAgbWF4UGFyYWxsZWxMYXVuY2hlczogdGhpcy5mYXN0TGF1bmNoT3B0aW9ucz8ubWF4UGFyYWxsZWxMYXVuY2hlcyA/PyA2LFxuICAgICAgICBzbmFwc2hvdENvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICB0YXJnZXRSZXNvdXJjZUNvdW50OiB0aGlzLmZhc3RMYXVuY2hPcHRpb25zPy50YXJnZXRSZXNvdXJjZUNvdW50ID8/IDEsXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICBjb25zdCBzdGFja05hbWUgPSBjZGsuU3RhY2sub2YodGhpcykuc3RhY2tOYW1lO1xuICAgIGNvbnN0IGJ1aWxkZXJOYW1lID0gdGhpcy5ub2RlLnBhdGg7XG5cbiAgICBjb25zdCBkaXN0ID0gbmV3IGltYWdlYnVpbGRlci5DZm5EaXN0cmlidXRpb25Db25maWd1cmF0aW9uKHRoaXMsICdBTUkgRGlzdHJpYnV0aW9uJywge1xuICAgICAgbmFtZTogdW5pcXVlSW1hZ2VCdWlsZGVyTmFtZSh0aGlzKSxcbiAgICAgIC8vIGRlc2NyaXB0aW9uOiB0aGlzLmRlc2NyaXB0aW9uLFxuICAgICAgZGlzdHJpYnV0aW9uczogW1xuICAgICAgICB7XG4gICAgICAgICAgcmVnaW9uOiBTdGFjay5vZih0aGlzKS5yZWdpb24sXG4gICAgICAgICAgYW1pRGlzdHJpYnV0aW9uQ29uZmlndXJhdGlvbjoge1xuICAgICAgICAgICAgTmFtZTogYCR7Y2RrLk5hbWVzLnVuaXF1ZVJlc291cmNlTmFtZSh0aGlzLCB7XG4gICAgICAgICAgICAgIG1heExlbmd0aDogMTAwLFxuICAgICAgICAgICAgICBzZXBhcmF0b3I6ICctJyxcbiAgICAgICAgICAgICAgYWxsb3dlZFNwZWNpYWxDaGFyYWN0ZXJzOiAnXy0nLFxuICAgICAgICAgICAgfSl9LXt7IGltYWdlYnVpbGRlcjpidWlsZERhdGUgfX1gLFxuICAgICAgICAgICAgQW1pVGFnczoge1xuICAgICAgICAgICAgICAnTmFtZSc6IHRoaXMubm9kZS5pZCxcbiAgICAgICAgICAgICAgJ0dpdEh1YlJ1bm5lcnM6U3RhY2snOiBzdGFja05hbWUsXG4gICAgICAgICAgICAgICdHaXRIdWJSdW5uZXJzOkJ1aWxkZXInOiBidWlsZGVyTmFtZSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgICBsYXVuY2hUZW1wbGF0ZUNvbmZpZ3VyYXRpb25zOiBsYXVuY2hUZW1wbGF0ZUNvbmZpZ3MsXG4gICAgICAgICAgZmFzdExhdW5jaENvbmZpZ3VyYXRpb25zOiBmYXN0TGF1bmNoQ29uZmlncy5sZW5ndGggPiAwID8gZmFzdExhdW5jaENvbmZpZ3MgOiB1bmRlZmluZWQsXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgICAgdGFnczogdGhpcy50YWdzLFxuICAgIH0pO1xuXG4gICAgY29uc3QgcmVjaXBlID0gbmV3IEFtaVJlY2lwZSh0aGlzLCAnQW1pIFJlY2lwZScsIHtcbiAgICAgIHBsYXRmb3JtOiB0aGlzLnBsYXRmb3JtKCksXG4gICAgICBjb21wb25lbnRzOiB0aGlzLmJpbmRDb21wb25lbnRzKCksXG4gICAgICBhcmNoaXRlY3R1cmU6IHRoaXMuYXJjaGl0ZWN0dXJlLFxuICAgICAgYmFzZUFtaTogdGhpcy5iYXNlQW1pLFxuICAgICAgc3RvcmFnZVNpemU6IHRoaXMuc3RvcmFnZVNpemUsXG4gICAgICB0YWdzOiB0aGlzLnRhZ3MsXG4gICAgfSk7XG5cbiAgICBjb25zdCBsb2cgPSB0aGlzLmNyZWF0ZUxvZygnQW1pIExvZycsIHJlY2lwZS5uYW1lKTtcbiAgICBjb25zdCBpbmZyYSA9IHRoaXMuY3JlYXRlSW5mcmFzdHJ1Y3R1cmUoW1xuICAgICAgaWFtLk1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdBbWF6b25TU01NYW5hZ2VkSW5zdGFuY2VDb3JlJyksXG4gICAgICBpYW0uTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ0VDMkluc3RhbmNlUHJvZmlsZUZvckltYWdlQnVpbGRlcicpLFxuICAgIF0pO1xuICAgIGlmICh0aGlzLndhaXRPbkRlcGxveSkge1xuICAgICAgdGhpcy5jcmVhdGVJbWFnZShpbmZyYSwgZGlzdCwgbG9nLCByZWNpcGUuYXJuLCB1bmRlZmluZWQpO1xuICAgIH1cbiAgICB0aGlzLmNyZWF0ZVBpcGVsaW5lKGluZnJhLCBkaXN0LCBsb2csIHJlY2lwZS5hcm4sIHVuZGVmaW5lZCk7XG5cbiAgICB0aGlzLmJvdW5kQW1pID0ge1xuICAgICAgbGF1bmNoVGVtcGxhdGU6IGxhdW5jaFRlbXBsYXRlLFxuICAgICAgYXJjaGl0ZWN0dXJlOiB0aGlzLmFyY2hpdGVjdHVyZSxcbiAgICAgIG9zOiB0aGlzLm9zLFxuICAgICAgbG9nR3JvdXA6IGxvZyxcbiAgICAgIHJ1bm5lclZlcnNpb246IFJ1bm5lclZlcnNpb24uc3BlY2lmaWMoJ3Vua25vd24nKSxcbiAgICB9O1xuXG4gICAgdGhpcy5hbWlDbGVhbmVyKHJlY2lwZSwgc3RhY2tOYW1lLCBidWlsZGVyTmFtZSk7XG5cbiAgICByZXR1cm4gdGhpcy5ib3VuZEFtaTtcbiAgfVxuXG4gIHByaXZhdGUgYW1pQ2xlYW5lcihyZWNpcGU6IEFtaVJlY2lwZSwgc3RhY2tOYW1lOiBzdHJpbmcsIGJ1aWxkZXJOYW1lOiBzdHJpbmcpIHtcbiAgICAvLyB0aGlzIGlzIGhlcmUgdG8gcHJvdmlkZSBzYWZlIHVwZ3JhZGUgZnJvbSBvbGQgY2RrLWdpdGh1Yi1ydW5uZXJzIHZlcnNpb25zXG4gICAgLy8gdGhpcyBsYW1iZGEgd2FzIHVzZWQgYnkgYSBjdXN0b20gcmVzb3VyY2UgdG8gZGVsZXRlIGFsbCBhbWlzIHdoZW4gdGhlIGJ1aWxkZXIgd2FzIHJlbW92ZWRcbiAgICAvLyBpZiB3ZSByZW1vdmUgdGhlIGN1c3RvbSByZXNvdXJjZSwgcm9sZSBhbmQgbGFtYmRhLCBhbGwgYW1pcyB3aWxsIGJlIGRlbGV0ZWQgb24gdXBkYXRlXG4gICAgLy8ga2VlcGluZyB0aGUganVzdCByb2xlIGJ1dCByZW1vdmluZyB0aGUgcGVybWlzc2lvbnMgYWxvbmcgd2l0aCB0aGUgY3VzdG9tIHJlc291cmNlIHdpbGwgbWFrZSBzdXJlIHRoYXQgZGVsZXRpb24gd2lsbCBmYWlsXG4gICAgY29uc3Qgc3RhY2sgPSBjZGsuU3RhY2sub2YodGhpcyk7XG4gICAgaWYgKHN0YWNrLm5vZGUudHJ5RmluZENoaWxkKCdkZWxldGUtYW1pLWRjYzAzNmM4LTg3NmItNDUxZS1hMmMxLTU1MmY5ZTA2ZTllMScpID09IHVuZGVmaW5lZCkge1xuICAgICAgY29uc3Qgcm9sZSA9IG5ldyBpYW0uUm9sZShzdGFjaywgJ2RlbGV0ZS1hbWktZGNjMDM2YzgtODc2Yi00NTFlLWEyYzEtNTUyZjllMDZlOWUxJywge1xuICAgICAgICBkZXNjcmlwdGlvbjogJ0VtcHR5IHJvbGUgdG8gcHJldmVudCBkZWxldGlvbiBvZiBBTUlzIG9uIGNkay1naXRodWItcnVubmVycyB1cGdyYWRlJyxcbiAgICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2xhbWJkYS5hbWF6b25hd3MuY29tJyksXG4gICAgICAgIGlubGluZVBvbGljaWVzOiB7XG4gICAgICAgICAgZGVueTogbmV3IGlhbS5Qb2xpY3lEb2N1bWVudCh7XG4gICAgICAgICAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICAgICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgICAgICBhY3Rpb25zOiBbJ2VjMjpEZXJlZ2lzdGVySW1hZ2UnLCAnZWMyOkRlbGV0ZVNuYXBzaG90J10sXG4gICAgICAgICAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgICAgICAgICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuREVOWSxcbiAgICAgICAgICAgICAgfSksXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0pLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgICBjb25zdCBsMXJvbGU6IGlhbS5DZm5Sb2xlID0gcm9sZS5ub2RlLmRlZmF1bHRDaGlsZCBhcyBpYW0uQ2ZuUm9sZTtcbiAgICAgIGwxcm9sZS5vdmVycmlkZUxvZ2ljYWxJZCgnZGVsZXRlYW1pZGNjMDM2Yzg4NzZiNDUxZWEyYzE1NTJmOWUwNmU5ZTFTZXJ2aWNlUm9sZTFDQzU4QTZGJyk7XG4gICAgfVxuXG4gICAgLy8gZGVsZXRlIG9sZCB2ZXJzaW9uIG9uIHVwZGF0ZSBhbmQgb24gc3RhY2sgZGVsZXRpb25cbiAgICB0aGlzLmltYWdlQ2xlYW5lcignSW1hZ2UnLCByZWNpcGUubmFtZS50b0xvd2VyQ2FzZSgpLCByZWNpcGUudmVyc2lvbik7XG5cbiAgICAvLyBkZWxldGUgb2xkIEFNSXMgKyBJQiByZXNvdXJjZXMgZGFpbHlcbiAgICBuZXcgaW1hZ2VidWlsZGVyLkNmbkxpZmVjeWNsZVBvbGljeSh0aGlzLCAnTGlmZWN5Y2xlIFBvbGljeSBBTUknLCB7XG4gICAgICBuYW1lOiB1bmlxdWVJbWFnZUJ1aWxkZXJOYW1lKHRoaXMpLFxuICAgICAgZGVzY3JpcHRpb246IGBEZWxldGUgb2xkIEdpdEh1YiBSdW5uZXIgQU1JcyBmb3IgJHt0aGlzLm5vZGUucGF0aH1gLFxuICAgICAgZXhlY3V0aW9uUm9sZTogbmV3IGlhbS5Sb2xlKHRoaXMsICdMaWZlY3ljbGUgUG9saWN5IEFNSSBSb2xlJywge1xuICAgICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnaW1hZ2VidWlsZGVyLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgICAgaW5saW5lUG9saWNpZXM6IHtcbiAgICAgICAgICBpYjogbmV3IGlhbS5Qb2xpY3lEb2N1bWVudCh7XG4gICAgICAgICAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICAgICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgICAgICBhY3Rpb25zOiBbJ3RhZzpHZXRSZXNvdXJjZXMnLCAnaW1hZ2VidWlsZGVyOkRlbGV0ZUltYWdlJ10sXG4gICAgICAgICAgICAgICAgcmVzb3VyY2VzOiBbJyonXSwgLy8gSW1hZ2UgQnVpbGRlciBkb2Vzbid0IHN1cHBvcnQgc2NvcGluZyB0aGlzIDooXG4gICAgICAgICAgICAgIH0pLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9KSxcbiAgICAgICAgICBhbWk6IG5ldyBpYW0uUG9saWN5RG9jdW1lbnQoe1xuICAgICAgICAgICAgc3RhdGVtZW50czogW1xuICAgICAgICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICAgICAgYWN0aW9uczogWydlYzI6RGVzY3JpYmVJbWFnZXMnLCAnZWMyOkRlc2NyaWJlSW1hZ2VBdHRyaWJ1dGUnXSxcbiAgICAgICAgICAgICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgICAgICAgICAgICB9KSxcbiAgICAgICAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgICAgIGFjdGlvbnM6IFsnZWMyOkRlcmVnaXN0ZXJJbWFnZScsICdlYzI6RGVsZXRlU25hcHNob3QnXSxcbiAgICAgICAgICAgICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgICAgICAgICAgICAgIGNvbmRpdGlvbnM6IHtcbiAgICAgICAgICAgICAgICAgIFN0cmluZ0VxdWFsczoge1xuICAgICAgICAgICAgICAgICAgICAnYXdzOlJlc291cmNlVGFnL0dpdEh1YlJ1bm5lcnM6U3RhY2snOiBzdGFja05hbWUsXG4gICAgICAgICAgICAgICAgICAgICdhd3M6UmVzb3VyY2VUYWcvR2l0SHViUnVubmVyczpCdWlsZGVyJzogYnVpbGRlck5hbWUsXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIH0pLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9KSxcbiAgICAgICAgfSxcbiAgICAgIH0pLnJvbGVBcm4sXG4gICAgICBwb2xpY3lEZXRhaWxzOiBbe1xuICAgICAgICBhY3Rpb246IHtcbiAgICAgICAgICB0eXBlOiAnREVMRVRFJyxcbiAgICAgICAgICBpbmNsdWRlUmVzb3VyY2VzOiB7XG4gICAgICAgICAgICBhbWlzOiB0cnVlLFxuICAgICAgICAgICAgc25hcHNob3RzOiB0cnVlLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICAgIGZpbHRlcjoge1xuICAgICAgICAgIHR5cGU6ICdDT1VOVCcsXG4gICAgICAgICAgdmFsdWU6IDIsXG4gICAgICAgIH0sXG4gICAgICB9XSxcbiAgICAgIHJlc291cmNlVHlwZTogJ0FNSV9JTUFHRScsXG4gICAgICByZXNvdXJjZVNlbGVjdGlvbjoge1xuICAgICAgICByZWNpcGVzOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgbmFtZTogcmVjaXBlLm5hbWUsXG4gICAgICAgICAgICBzZW1hbnRpY1ZlcnNpb246IHJlY2lwZS52ZXJzaW9uLCAvLyBkb2NzIHNheSBpdCdzIG9wdGlvbmFsLCBidXQgaXQncyBub3RcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9KTtcbiAgfVxuXG4gIHByaXZhdGUgYmluZENvbXBvbmVudHMoKTogSW1hZ2VCdWlsZGVyQ29tcG9uZW50W10ge1xuICAgIGlmICh0aGlzLmJvdW5kQ29tcG9uZW50cy5sZW5ndGggPT0gMCkge1xuICAgICAgdGhpcy5ib3VuZENvbXBvbmVudHMucHVzaCguLi50aGlzLmNvbXBvbmVudHMubWFwKChjLCBpKSA9PiBjLl9hc0F3c0ltYWdlQnVpbGRlckNvbXBvbmVudCh0aGlzLCBgQ29tcG9uZW50ICR7aX0gJHtjLm5hbWV9YCwgdGhpcy5vcywgdGhpcy5hcmNoaXRlY3R1cmUpKSk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHRoaXMuYm91bmRDb21wb25lbnRzO1xuICB9XG5cbiAgcHJpdmF0ZSBpbWFnZUNsZWFuZXIodHlwZTogJ0NvbnRhaW5lcicgfCAnSW1hZ2UnLCByZWNpcGVOYW1lOiBzdHJpbmcsIHZlcnNpb246IHN0cmluZykge1xuICAgIGNvbnN0IGNsZWFuZXJGdW5jdGlvbiA9IHNpbmdsZXRvbkxhbWJkYShEZWxldGVSZXNvdXJjZXNGdW5jdGlvbiwgdGhpcywgJ2F3cy1pbWFnZS1idWlsZGVyLWRlbGV0ZS1yZXNvdXJjZXMnLCB7XG4gICAgICBkZXNjcmlwdGlvbjogJ0N1c3RvbSByZXNvdXJjZSBoYW5kbGVyIHRoYXQgZGVsZXRlcyByZXNvdXJjZXMgb2Ygb2xkIHZlcnNpb25zIG9mIEVDMiBJbWFnZSBCdWlsZGVyIGltYWdlcycsXG4gICAgICBpbml0aWFsUG9saWN5OiBbXG4gICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgICAnaW1hZ2VidWlsZGVyOkxpc3RJbWFnZUJ1aWxkVmVyc2lvbnMnLFxuICAgICAgICAgICAgJ2ltYWdlYnVpbGRlcjpEZWxldGVJbWFnZScsXG4gICAgICAgICAgXSxcbiAgICAgICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgICAgICB9KSxcbiAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgIGFjdGlvbnM6IFsnZWMyOkRlc2NyaWJlSW1hZ2VzJ10sXG4gICAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgICAgfSksXG4gICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICBhY3Rpb25zOiBbJ2VjMjpEZXJlZ2lzdGVySW1hZ2UnLCAnZWMyOkRlbGV0ZVNuYXBzaG90J10sXG4gICAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgICAgICBjb25kaXRpb25zOiB7XG4gICAgICAgICAgICBTdHJpbmdFcXVhbHM6IHtcbiAgICAgICAgICAgICAgJ2F3czpSZXNvdXJjZVRhZy9HaXRIdWJSdW5uZXJzOlN0YWNrJzogY2RrLlN0YWNrLm9mKHRoaXMpLnN0YWNrTmFtZSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSksXG4gICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICBhY3Rpb25zOiBbJ2VjcjpCYXRjaERlbGV0ZUltYWdlJ10sXG4gICAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgICAgfSksXG4gICAgICBdLFxuICAgICAgbG9nR3JvdXA6IHNpbmdsZXRvbkxvZ0dyb3VwKHRoaXMsIFNpbmdsZXRvbkxvZ1R5cGUuUlVOTkVSX0lNQUdFX0JVSUxEKSxcbiAgICAgIGxvZ2dpbmdGb3JtYXQ6IGxhbWJkYS5Mb2dnaW5nRm9ybWF0LkpTT04sXG4gICAgICB0aW1lb3V0OiBjZGsuRHVyYXRpb24ubWludXRlcygxMCksXG4gICAgfSk7XG5cbiAgICBuZXcgQ3VzdG9tUmVzb3VyY2UodGhpcywgYCR7dHlwZX0gQ2xlYW5lcmAsIHtcbiAgICAgIHNlcnZpY2VUb2tlbjogY2xlYW5lckZ1bmN0aW9uLmZ1bmN0aW9uQXJuLFxuICAgICAgcmVzb3VyY2VUeXBlOiAnQ3VzdG9tOjpJbWFnZUJ1aWxkZXItRGVsZXRlLVJlc291cmNlcycsXG4gICAgICBwcm9wZXJ0aWVzOiA8RGVsZXRlUmVzb3VyY2VzUHJvcHM+e1xuICAgICAgICBJbWFnZVZlcnNpb25Bcm46IGNkay5TdGFjay5vZih0aGlzKS5mb3JtYXRBcm4oe1xuICAgICAgICAgIHNlcnZpY2U6ICdpbWFnZWJ1aWxkZXInLFxuICAgICAgICAgIHJlc291cmNlOiAnaW1hZ2UnLFxuICAgICAgICAgIHJlc291cmNlTmFtZTogYCR7cmVjaXBlTmFtZX0vJHt2ZXJzaW9ufWAsXG4gICAgICAgICAgYXJuRm9ybWF0OiBjZGsuQXJuRm9ybWF0LlNMQVNIX1JFU09VUkNFX05BTUUsXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9KTtcbiAgfVxufVxuXG4vKipcbiAqIEBpbnRlcm5hbFxuICovXG5leHBvcnQgY2xhc3MgQXdzSW1hZ2VCdWlsZGVyRmFpbGVkQnVpbGROb3RpZmllciBpbXBsZW1lbnRzIGNkay5JQXNwZWN0IHtcbiAgcHVibGljIHN0YXRpYyBjcmVhdGVGaWx0ZXJpbmdUb3BpYyhzY29wZTogQ29uc3RydWN0LCB0YXJnZXRUb3BpYzogc25zLlRvcGljKTogc25zLklUb3BpYyB7XG4gICAgY29uc3QgdG9waWMgPSBuZXcgc25zLlRvcGljKHNjb3BlLCAnSW1hZ2UgQnVpbGRlciBCdWlsZHMnKTtcbiAgICBjb25zdCBmaWx0ZXIgPSBuZXcgRmlsdGVyRmFpbGVkQnVpbGRzRnVuY3Rpb24oc2NvcGUsICdJbWFnZSBCdWlsZGVyIEJ1aWxkcyBGaWx0ZXInLCB7XG4gICAgICBsb2dHcm91cDogc2luZ2xldG9uTG9nR3JvdXAoc2NvcGUsIFNpbmdsZXRvbkxvZ1R5cGUuUlVOTkVSX0lNQUdFX0JVSUxEKSxcbiAgICAgIGxvZ2dpbmdGb3JtYXQ6IGxhbWJkYS5Mb2dnaW5nRm9ybWF0LkpTT04sXG4gICAgICBlbnZpcm9ubWVudDoge1xuICAgICAgICBUQVJHRVRfVE9QSUNfQVJOOiB0YXJnZXRUb3BpYy50b3BpY0FybixcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICB0b3BpYy5hZGRTdWJzY3JpcHRpb24obmV3IHN1YnMuTGFtYmRhU3Vic2NyaXB0aW9uKGZpbHRlcikpO1xuICAgIHRhcmdldFRvcGljLmdyYW50UHVibGlzaChmaWx0ZXIpO1xuXG4gICAgcmV0dXJuIHRvcGljO1xuICB9XG5cbiAgY29uc3RydWN0b3IocHJpdmF0ZSB0b3BpYzogc25zLklUb3BpYykge1xuICB9XG5cbiAgcHVibGljIHZpc2l0KG5vZGU6IElDb25zdHJ1Y3QpOiB2b2lkIHtcbiAgICBpZiAobm9kZSBpbnN0YW5jZW9mIEF3c0ltYWdlQnVpbGRlclJ1bm5lckltYWdlQnVpbGRlcikge1xuICAgICAgY29uc3QgYnVpbGRlciA9IG5vZGUgYXMgQXdzSW1hZ2VCdWlsZGVyUnVubmVySW1hZ2VCdWlsZGVyO1xuICAgICAgY29uc3QgaW5mcmFOb2RlID0gYnVpbGRlci5ub2RlLnRyeUZpbmRDaGlsZCgnSW5mcmFzdHJ1Y3R1cmUnKTtcbiAgICAgIGlmIChpbmZyYU5vZGUpIHtcbiAgICAgICAgY29uc3QgaW5mcmEgPSBpbmZyYU5vZGUgYXMgaW1hZ2VidWlsZGVyLkNmbkluZnJhc3RydWN0dXJlQ29uZmlndXJhdGlvbjtcbiAgICAgICAgaW5mcmEuc25zVG9waWNBcm4gPSB0aGlzLnRvcGljLnRvcGljQXJuO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgY2RrLkFubm90YXRpb25zLm9mKGJ1aWxkZXIpLmFkZFdhcm5pbmcoJ1VudXNlZCBidWlsZGVyIGNhbm5vdCBnZXQgbm90aWZpY2F0aW9ucyBvZiBmYWlsZWQgYnVpbGRzJyk7XG4gICAgICB9XG4gICAgfVxuICB9XG59XG4iXX0=