import unittest

from pyvalidator.is_mobile_number import is_mobile_number
from . import print_test_ok

fixtures = [
    {
        'locale': 'am-AM',
        'valid': [
          '+37410324123',
          '+37422298765',
          '+37431276521',
          '022698763',
          '37491987654',
          '+37494567890',
        ],
        'invalid': [
          '12345',
          '+37411498855',
          '+37411498123',
          '05614988556',
          '',
          '37456789000',
        ],
    },
    {
        'locale': 'ar-AE',
        'valid': [
          '+971502674453',
          '+971521247658',
          '+971541255684',
          '+971555454458',
          '+971561498855',
          '+971585215778',
          '971585215778',
          '0585215778',
          '585215778',
        ],
        'invalid': [
          '12345',
          '+971511498855',
          '+9715614988556',
          '+9745614988556',
          '',
          '+9639626626262',
          '+963332210972',
          '0114152198',
          '962796477263',
        ],
    },
    {
        'locale': 'ar-BH',
        'valid': [
          '+97335078110',
          '+97339534385',
          '+97366331055',
          '+97333146000',
          '97335078110',
          '35078110',
          '66331055',
        ],
        'invalid': [
          '12345',
          '+973350781101',
          '+97379534385',
          '+973035078110',
          '',
          '+9639626626262',
          '+963332210972',
          '0114152198',
          '962796477263',
          '035078110',
          '16331055',
          'hello',
          '+9733507811a',
        ],
      },
      {
        'locale': 'ar-EG',
        'valid': [
          '+201004513789',
          '+201111453489',
          '+201221204610',
          '+201144621154',
          '+201200124304',
          '+201011201564',
          '+201124679001',
          '+201064790156',
          '+201274652177',
          '+201280134679',
          '+201090124576',
          '+201583728900',
          '201599495596',
          '201090124576',
          '01090124576',
          '01538920744',
          '1593075993',
          '1090124576',
        ],
        'invalid': [
          '+221004513789',
          '+201404513789',
          '12345',
          '',
          '+9639626626262',
          '+963332210972',
          '0114152198',
          '962796477263',
        ],
      },
      {
        'locale': 'ar-JO',
        'valid': [
          '0796477263',
          '0777866254',
          '0786725261',
          '+962796477263',
          '+962777866254',
          '+962786725261',
          '962796477263',
          '962777866254',
          '962786725261',
        ],
        'invalid': [
          '00962786725261',
          '00962796477263',
          '12345',
          '',
          '+9639626626262',
          '+963332210972',
          '0114152198',
        ],
      },
      {
        'locale': 'ar-KW',
        'valid': [
          '96550000000',
          '96560000000',
          '96590000000',
          '+96550000000',
          '+96550000220',
          '+96551111220',
        ],
        'invalid': [
          '+96570000220',
          '00962786725261',
          '00962796477263',
          '12345',
          '',
          '+9639626626262',
          '+963332210972',
          '0114152198',
        ],
      },
      {
        'locale': 'ar-LB',
        'valid': [
          '+96171234568',
          '+9613123456',
          '3456123',
          '3123456',
          '81978468',
          '77675798',
        ],
        'invalid': [
          '+961712345688888',
          '00912220000',
          '7767579888',
          '+0921110000',
          '+3123456888',
          '021222200000',
          '213333444444',
          '',
          '+212234',
          '+21',
          '02122333',
        ],
      },
      {
        'locale': 'ar-LY',
        'valid': [
          '912220000',
          '0923330000',
          '218945550000',
          '+218958880000',
          '212220000',
          '0212220000',
          '+218212220000',
        ],
        'invalid': [
          '9122220000',
          '00912220000',
          '09211110000',
          '+0921110000',
          '+2180921110000',
          '021222200000',
          '213333444444',
          '',
          '+212234',
          '+21',
          '02122333',
        ],
      },
      {
        'locale': 'ar-MA',
        'valid': [
          '0522714782',
          '0690851123',
          '0708186135',
          '+212522714782',
          '+212690851123',
          '+212708186135',
          '00212522714782',
          '00212690851123',
          '00212708186135',
        ],
        'invalid': [
          '522714782',
          '690851123',
          '708186135',
          '212522714782',
          '212690851123',
          '212708186135',
          '0212522714782',
          '0212690851123',
          '0212708186135',
          '',
          '12345',
          '0922714782',
          '+212190851123',
          '00212408186135',
        ],
      },
      {
        'locale': 'dz-BT',
        'valid': [
          '+97517374354',
          '+97517454971',
          '77324646',
          '016329712',
          '97517265559',
        ],
        'invalid': [
          '',
          '9898347255',
          '+96326626262',
          '963372',
          '0114152198',
        ],
      },
      {
        'locale': 'ar-OM',
        'valid': [
          '+96891212121',
          '0096899999999',
          '93112211',
          '99099009',
        ],
        'invalid': [
          '+96890212121',
          '0096890999999',
          '0090999999',
          '+9689021212',
          '',
          '+212234',
          '+21',
          '02122333',
        ],
      },
      {
        'locale': 'ar-SY',
        'valid': [
          '0944549710',
          '+963944549710',
          '956654379',
          '0944549710',
          '0962655597',
        ],
        'invalid': [
          '12345',
          '',
          '+9639626626262',
          '+963332210972',
          '0114152198',
        ],
      },
      {
        'locale': 'ar-SA',
        'valid': [
          '0556578654',
          '+966556578654',
          '966556578654',
          '596578654',
          '572655597',
        ],
        'invalid': [
          '12345',
          '',
          '+9665626626262',
          '+96633221097',
          '0114152198',
        ],
      },
      {
        'locale': 'ar-TN',
        'valid': [
          '23456789',
          '+21623456789',
          '21623456789',
        ],
        'invalid': [
          '12345',
          '75200123',
          '+216512345678',
          '13520459',
          '85479520',
        ],
      },
      {
        'locale': 'bg-BG',
        'valid': [
          '+359897123456',
          '+359898888888',
          '0897123123',
        ],
        'invalid': [
          '',
          '0898123',
          '+359212555666',
          '18001234567',
          '12125559999',
        ],
      },
      {
        'locale': 'bn-BD',
        'valid': [
          '+8801794626846',
          '01399098893',
          '8801671163269',
          '01717112029',
          '8801898765432',
          '+8801312345678',
          '01494676946',
        ],
        'invalid': [
          '',
          '0174626346',
          '017943563469',
          '18001234567',
          '0131234567',
        ],
      },
      {
        'locale': 'bs-BA',
        'valid': [
          '060123456',
          '061123456',
          '062123456',
          '063123456',
          '0641234567',
          '065123456',
          '066123456',
          '+38760123456',
          '+38761123456',
          '+38762123456',
          '+38763123456',
          '+387641234567',
          '+38765123456',
          '+38766123456',
          '0038760123456',
          '0038761123456',
          '0038762123456',
          '0038763123456',
          '00387641234567',
          '0038765123456',
          '0038766123456',
        ],
        'invalid': [
          '0601234567',
          '0611234567',
          '06212345',
          '06312345',
          '064123456',
          '0651234567',
          '06612345',
          '+3866123456',
          '+3856123456',
          '00038760123456',
          '038761123456',
        ],
      },
      {
        'locale': 'cs-CZ',
        'valid': [
          '+420 123 456 789',
          '+420 123456789',
          '+420123456789',
          '123 456 789',
          '123456789',
        ],
        'invalid': [
          '',
          '+42012345678',
          '+421 123 456 789',
          '+420 023456789',
          '+4201234567892',
        ],
      },
      {
        'locale': 'sk-SK',
        'valid': [
          '+421 123 456 789',
          '+421 123456789',
          '+421123456789',
          '123 456 789',
          '123456789',
        ],
        'invalid': [
          '',
          '+42112345678',
          '+422 123 456 789',
          '+421 023456789',
          '+4211234567892',
        ],
      },
      {
        'locale': 'de-DE',
        'valid': [
          '+4915123456789',
          '+4930405044550',
          '015123456789',
          '015123456789',
          '015623456789',
          '015623456789',
          '01601234567',
          '016012345678',
          '01621234567',
          '01631234567',
          '01701234567',
          '017612345678',
          '015345678910',
          '015412345678',
        ],
        'invalid': [
          '34412345678',
          '14412345678',
          '16212345678',
          '1761234567',
          '16412345678',
          '17012345678',
          '+4912345678910',
          '+49015123456789',
        ],
      },
      {
        'locale': 'de-AT',
        'valid': [
          '+436761234567',
          '06761234567',
          '00436123456789',
          '+436123456789',
          '01999',
          '+4372876',
          '06434908989562345',
        ],
        'invalid': [
          '167612345678',
          '1234',
          '064349089895623459',
        ],
      },
      {
        'locale': 'hu-HU',
        'valid': [
          '06301234567',
          '+36201234567',
          '06701234567',
        ],
        'invalid': [
          '1234',
          '06211234567',
          '+3620123456',
        ],
      },
      {
        'locale': 'mz-MZ',
        'valid': [
          '+258849229754',
          '258849229754',
          '849229754',
          '829229754',
          '839229754',
          '869229754',
          '859229754',
          '869229754',
          '879229754',
          '+258829229754',
          '+258839229754',
          '+258869229754',
          '+258859229754',
          '+258869229754',
          '+258879229754',
          '258829229754',
          '258839229754',
          '258869229754',
          '258859229754',
          '258869229754',
          '258879229754',
        ],
        'invalid': [
          '+248849229754',
          '158849229754',
          '249229754',
          '819229754',
          '899229754',
          '889229754',
          '89229754',
          '8619229754',
          '87922975411',
          '257829229754',
          '+255839229754',
          '+2258869229754',
          '+1258859229754',
          '+2588692297541',
          '+2588792519754',
          '25882922975411',
        ],
      },
      {
        'locale': 'pt-BR',
        'valid': [
          '+55 12 996551215',
          '+55 15 97661234',
          '+55 (12) 996551215',
          '+55 (15) 97661234',
          '55 (17) 96332-2155',
          '55 (17) 6332-2155',
          '55 15 976612345',
          '55 15 75661234',
          '+5512984567890',
          '+551283456789',
          '5512984567890',
          '551283456789',
          '015994569878',
          '01593456987',
          '022995678947',
          '02299567894',
          '(22)99567894',
          '(22)9956-7894',
          '(22) 99567894',
          '(22) 9956-7894',
          '(22)999567894',
          '(22)99956-7894',
          '(22) 999567894',
          '(22) 99956-7894',
          '(11) 94123-4567',
        ],
        'invalid': [
          '0819876543',
          '+55 15 7566123',
          '+017 123456789',
          '5501599623874',
          '+55012962308',
          '+55 015 1234-3214',
          '+55 11 91431-4567',
          '+55 (11) 91431-4567',
          '+551191431-4567',
          '5511914314567',
          '5511912345678',
          '(11) 91431-4567',
        ],
      },
      {
        'locale': 'zh-CN',
        'valid': [
          '13523333233',
          '13838389438',
          '14899230918',
          '14999230918',
          '15323456787',
          '15052052020',
          '16237108167',
          '008616238234822',
          '+8616238234822',
          '16565600001',
          '17269427292',
          '17469427292',
          '18199617480',
          '19151751717',
          '19651751717',
          '+8613238234822',
          '+8613487234567',
          '+8617823492338',
          '+8617823492338',
          '+8616637108167',
          '+8616637108167',
          '+8616712341234',
          '+8619912341234',
          '+8619812341234',
          '+8619712341234',
          '+8619612341234',
          '+8619512341234',
          '+8619312341234',
          '+8619212341234',
          '+8619112341234',
          '+8617269427292',
          '008618812341234',
          '008618812341234',
          '008617269427292',
          '92138389438',
          '+8692138389438',
          '008692138389438',
          '98199649964',
          '+8698099649964',
          '008698099649964',
        ],
        'invalid': [
          '12345',
          '',
          '12038389438',
          '12838389438',
          '013838389438',
          '+86-13838389438',
          '+08613811211114',
          '+008613811211114',
          '08613811211114',
          '0086-13811211114',
          '0086-138-1121-1114',
          'Vml2YW11cyBmZXJtZtesting123',
          '010-38238383',
        ],
      },
      {
        'locale': 'zh-TW',
        'valid': [
          '0987123456',
          '+886987123456',
          '886987123456',
          '+886-987123456',
          '886-987123456',
        ],
        'invalid': [
          '12345',
          '',
          'Vml2YW11cyBmZXJtZtesting123',
          '0-987123456',
        ],
      },
      {
        'locale': 'en-BM',
        'valid': [
          '+14417974653',
          '14413986653',
          '4415370973',
          '+14415005489',
        ],
        'invalid': [
          '85763287',
          '+14412020436',
          '+14412236546',
          '+14418245567',
          '+14416546789',
          '44087635627',
          '+4418970973',
          '',
          '+1441897465',
        ],
      },
      {
        'locale': 'en-ZA',
        'valid': [
          '0821231234',
          '+27821231234',
          '27821231234',
        ],
        'invalid': [
          '082123',
          '08212312345',
          '21821231234',
          '+21821231234',
          '+0821231234',
        ],
      },
      {
        'locale': 'en-AU',
        'valid': [
          '61404111222',
          '+61411222333',
          '0417123456',
        ],
        'invalid': [
          '082123',
          '08212312345',
          '21821231234',
          '+21821231234',
          '+0821231234',
          '04123456789',
        ],
      },
      {
        'locale': 'es-BO',
        'valid': [
          '+59175553635',
          '+59162223685',
          '+59179783890',
          '+59160081890',
          '79783890',
          '60081890',
        ],
        'invalid': [
          '082123',
          '08212312345',
          '21821231234',
          '+21821231234',
          '+59199783890',
        ],
      },
      {
        'locale': 'en-GG',
        'valid': [
          '+441481123456',
          '+441481789123',
          '441481123456',
          '441481789123',
        ],
        'invalid': [
          '999',
          '+441481123456789',
          '+447123456789',
        ],
      },
      {
        'locale': 'en-GH',
        'valid': [
          '0202345671',
          '0502345671',
          '0242345671',
          '0542345671',
          '0272345671',
          '0572345671',
          '0262345671',
          '0562345671',
          '0232345671',
          '0282345671',
          '+233202345671',
          '+233502345671',
          '+233242345671',
          '+233542345671',
          '+233272345671',
          '+233572345671',
          '+233262345671',
          '+233562345671',
          '+233232345671',
          '+233282345671',
          '+233592349493',
          '0550298219',
        ],
        'invalid': [
          '082123',
          '232345671',
          '0292345671',
          '+233292345671',
        ],
      },
      {
        'locale': 'en-HK',
        'valid': [
          '91234567',
          '9123-4567',
          '61234567',
          '51234567',
          '+85291234567',
          '+852-91234567',
          '+852-9123-4567',
          '+852 9123 4567',
          '9123 4567',
          '852-91234567',
        ],
        'invalid': [
          '999',
          '+852-912345678',
          '123456789',
          '+852-1234-56789',
        ],
      },
      {
        'locale': 'en-MO',
        'valid': [
          '61234567',
          '+85361234567',
          '+853-61234567',
          '+853-6123-4567',
          '+853 6123 4567',
          '6123 4567',
          '853-61234567',
        ],
        'invalid': [
          '999',
          '12345678',
          '612345678',
          '+853-12345678',
          '+853-22345678',
          '+853-82345678',
          '+853-612345678',
          '+853-1234-5678',
          '+853 1234 5678',
          '+853-6123-45678',
        ],
      },
      {
        'locale': 'en-IE',
        'valid': [
          '+353871234567',
          '353831234567',
          '353851234567',
          '353861234567',
          '353871234567',
          '353881234567',
          '353891234567',
          '0871234567',
          '0851234567',
        ],
        'invalid': [
          '999',
          '+353341234567',
          '+33589484858',
          '353841234567',
          '353811234567',
        ],
      },
      {
        'locale': 'en-KE',
        'valid': [
          '+254728590432',
          '+254733875610',
          '254728590234',
          '0733346543',
          '0700459022',
          '0110934567',
          '+254110456794',
          '254198452389',
        ],
        'invalid': [
          '999',
          '+25489032',
          '123456789',
          '+254800723845',
        ],
      },
      {
        'locale': 'en-MT',
        'valid': [
          '+35699000000',
          '+35679000000',
          '99000000',
        ],
        'invalid': [
          '356',
          '+35699000',
          '+35610000000',
        ],
      },
      {
        'locale': 'en-PH',
        'valid': [
          '+639275149120',
          '+639275142327',
          '+639003002023',
          '09275149116',
          '09194877624',
        ],
        'invalid': [
          '12112-13-345',
          '12345678901',
          'sx23YW11cyBmZxxXJt123123',
          '010-38238383',
          '966684123123-2590',
        ],
      },
      {
        'locale': 'en-UG',
        'valid': [
          '+256728590432',
          '+256733875610',
          '256728590234',
          '0773346543',
          '0700459022',
        ],
        'invalid': [
          '999',
          '+254728590432',
          '+25489032',
          '123456789',
          '+254800723845',
        ],
      },
      {
        'locale': 'en-RW',
        'valid': [
          '+250728590432',
          '+250733875610',
          '250738590234',
          '0753346543',
          '0780459022',
        ],
        'invalid': [
          '999',
          '+254728590432',
          '+25089032',
          '123456789',
          '+250800723845',
        ],
      },
      {
        'locale': 'en-TZ',
        'valid': [
          '+255728590432',
          '+255733875610',
          '255628590234',
          '0673346543',
          '0600459022',
        ],
        'invalid': [
          '999',
          '+254728590432',
          '+25589032',
          '123456789',
          '+255800723845',
        ],
      },
      {
        'locale': 'es-PE',
        'valid': [
          '+51912232764',
          '+51923464567',
          '+51968267382',
          '+51908792973',
          '974980472',
          '908792973',
          '+51974980472',
        ],
        'invalid': [
          '999',
          '+51812232764',
          '+5181223276499',
          '+25589032',
          '123456789',
        ],
      },
      {
        'locale': 'fr-FR',
        'valid': [
          '0612457898',
          '+33612457898',
          '33612457898',
          '0712457898',
          '+33712457898',
          '33712457898',
        ],
        'invalid': [
          '061245789',
          '06124578980',
          '0112457898',
          '0212457898',
          '0312457898',
          '0412457898',
          '0512457898',
          '0812457898',
          '0912457898',
          '+34612457898',
          '+336124578980',
          '+3361245789',
        ],
      },
      {
        'locale': 'fr-CA',
        'valid': ['19876543210', '8005552222', '+15673628910'],
        'invalid': [
          '564785',
          '0123456789',
          '1437439210',
          '+10345672645',
          '11435213543',
        ],
      },
      {
        'locale': 'fr-GF',
        'valid': [
          '0612457898',
          '+594612457898',
          '594612457898',
          '0712457898',
          '+594712457898',
          '594712457898',
        ],
        'invalid': [
          '061245789',
          '06124578980',
          '0112457898',
          '0212457898',
          '0312457898',
          '0412457898',
          '0512457898',
          '0812457898',
          '0912457898',
          '+54612457898',
          '+5946124578980',
          '+59461245789',
        ],
      },
      {
        'locale': 'fr-GP',
        'valid': [
          '0612457898',
          '+590612457898',
          '590612457898',
          '0712457898',
          '+590712457898',
          '590712457898',
        ],
        'invalid': [
          '061245789',
          '06124578980',
          '0112457898',
          '0212457898',
          '0312457898',
          '0412457898',
          '0512457898',
          '0812457898',
          '0912457898',
          '+594612457898',
          '+5906124578980',
          '+59061245789',
        ],
      },
      {
        'locale': 'fr-MQ',
        'valid': [
          '0612457898',
          '+596612457898',
          '596612457898',
          '0712457898',
          '+596712457898',
          '596712457898',
        ],
        'invalid': [
          '061245789',
          '06124578980',
          '0112457898',
          '0212457898',
          '0312457898',
          '0412457898',
          '0512457898',
          '0812457898',
          '0912457898',
          '+594612457898',
          '+5966124578980',
          '+59661245789',
        ],
      },
      {
        'locale': 'fr-RE',
        'valid': [
          '0612457898',
          '+262612457898',
          '262612457898',
          '0712457898',
          '+262712457898',
          '262712457898',
        ],
        'invalid': [
          '061245789',
          '06124578980',
          '0112457898',
          '0212457898',
          '0312457898',
          '0412457898',
          '0512457898',
          '0812457898',
          '0912457898',
          '+264612457898',
          '+2626124578980',
          '+26261245789',
        ],
      },
      {
        'locale': 'ka-GE',
        'valid': [
          '+99550001111',
          '+99551535213',
          '+995798526662',
          '798526662',
          '50001111',
          '798526662',
          '+995799766525',
        ],
        'invalid': [
          '+995500011118',
          '+9957997665250',
          '+995999766525',
          '20000000000',
          '68129485729',
          '6589394827',
          '298RI89572',
        ],
      },
      {
        'locale': 'el-GR',
        'valid': [
          '+306944848966',
          '6944848966',
          '306944848966',
        ],
        'invalid': [
          '2102323234',
          '+302646041461',
          '120000000',
          '20000000000',
          '68129485729',
          '6589394827',
          '298RI89572',
        ],
      },
      {
        'locale': 'en-GB',
        'valid': [
          '447789345856',
          '+447861235675',
          '07888814488',
        ],
        'invalid': [
          '67699567',
          '0773894868',
          '077389f8688',
          '+07888814488',
          '0152456999',
          '442073456754',
          '+443003434751',
          '05073456754',
          '08001123123',
        ],
      },
      {
        'locale': 'en-SG',
        'valid': [
          '32891278',
          '87654321',
          '98765432',
          '+6587654321',
          '+6598765432',
          '+6565241234',
        ],
        'invalid': [
          '332891231',
          '987654321',
          '876543219',
          '8765432',
          '9876543',
          '12345678',
          '+98765432',
          '+9876543212',
          '+15673628910',
          '19876543210',
          '8005552222',
        ],
      },
      {
        'locale': 'en-US',
        'valid': [
          '19876543210',
          '8005552222',
          '+15673628910',
          '+1(567)3628910',
          '+1(567)362-8910',
          '+1(567) 362-8910',
          '1(567)362-8910',
          '1(567)362 8910',
          '223-456-7890',
        ],
        'invalid': [
          '564785',
          '0123456789',
          '1437439210',
          '+10345672645',
          '11435213543',
          '1(067)362-8910',
          '1(167)362-8910',
          '+2(267)362-8910',
          '+3365520145',
        ],
      },
      {
        'locale': 'en-CA',
        'valid': ['19876543210', '8005552222', '+15673628910'],
        'invalid': [
          '564785',
          '0123456789',
          '1437439210',
          '+10345672645',
          '11435213543',
        ],
      },
      {
        'locale': 'en-ZM',
        'valid': [
          '0956684590',
          '0966684590',
          '0976684590',
          '+260956684590',
          '+260966684590',
          '+260976684590',
          '260976684590',
        ],
        'invalid': [
          '12345',
          '',
          'Vml2YW11cyBmZXJtZtesting123',
          '010-38238383',
          '966684590',
        ],
      },
      {
        'locale': ['en-ZW'],
        'valid': [
          '+263561890123',
          '+263715558041',
          '+263775551112',
          '+263775551695',
          '+263715556633',
        ],
        'invalid': [
          '12345',
          '',
          'Vml2YW11cyBmZXJtZtesting123',
          '+2631234567890',
          '+2641234567',
          '+263981234',
          '4736338855',
          '66338855',
        ],
      },
      {
        'locale': 'ru-RU',
        'valid': [
          '+79676338855',
          '79676338855',
          '89676338855',
          '9676338855',
        ],
        'invalid': [
          '12345',
          '',
          'Vml2YW11cyBmZXJtZtesting123',
          '010-38238383',
          '+9676338855',
          '19676338855',
          '6676338855',
          '+99676338855',
        ],
      },
      {
        'locale': 'si-LK',
        'valid': [
          '+94766661206',
          '94713114340',
          '0786642116',
          '078 7642116',
          '078-7642116',
          '0749994567',
        ],
        'invalid': [
          '9912349956789',
          '12345',
          '1678123456',
          '0731234567',
          '0797878674',
        ],
      },
      {
        'locale': 'sr-RS',
        'valid': [
          '0640133338',
          '063333133',
          '0668888878',
          '+381645678912',
          '+381611314000',
          '0655885010',
        ],
        'invalid': [
          '12345',
          '',
          'Vml2YW11cyBmZXJtZtesting123',
          '010-38238383',
          '+9676338855',
          '19676338855',
          '6676338855',
          '+99676338855',
        ],
      },
      {
        'locale': 'en-NZ',
        'valid': [
          '+6427987035',
          '642240512347',
          '0293981646',
          '029968425',
        ],
        'invalid': [
          '12345',
          '',
          'Vml2YW11cyBmZXJtZtesting123',
          '+642956696123566',
          '+02119620856',
          '+9676338855',
          '19676338855',
          '6676338855',
          '+99676338855',
        ],
      },
      {
        'locale': 'en-MU',
        'valid': [
          '+23012341234',
          '12341234',
          '012341234',
        ],
        'invalid': [
          '41234',
          '',
          '+230',
          '+2301',
          '+23012',
          '+230123',
          '+2301234',
          '+23012341',
          '+230123412',
          '+2301234123',
          '+230123412341',
          '+2301234123412',
          '+23012341234123',
        ],
      },
      {
        'locale': ['nb-NO', 'nn-NO'],
        'valid': [
          '+4796338855',
          '+4746338855',
          '4796338855',
          '4746338855',
          '46338855',
          '96338855',
        ],
        'invalid': [
          '12345',
          '',
          'Vml2YW11cyBmZXJtZtesting123',
          '+4676338855',
          '19676338855',
          '+4726338855',
          '4736338855',
          '66338855',
        ],
      },
      {
        'locale': ['ne-NP'],
        'valid': [
          '+9779817385479',
          '+9779717385478',
          '+9779862002615',
          '+9779853660020',
        ],
        'invalid': [
          '12345',
          '',
          'Vml2YW11cyBmZXJtZtesting123',
          '+97796123456789',
          '+9771234567',
          '+977981234',
          '4736338855',
          '66338855',
        ],
      },
      {
        'locale': 'vi-VN',
        'valid': [
          '0336012403',
          '+84586012403',
          '84981577798',
          '0708001240',
          '84813601243',
          '0523803765',
          '0863803732',
          '0883805866',
          '0892405867',
          '+84888696413',
          '0878123456',
          '84781234567',
          '0553803765',
        ],
        'invalid': [
          '12345',
          '',
          'Vml2YW11cyBmZXJtZtesting123',
          '010-38238383',
          '260976684590',
          '01678912345',
          '+841698765432',
          '841626543219',
          '0533803765',
          '08712345678',
          '+0321234567',
        ],
      },
      {
        'locale': 'es-AR',
        'valid': [
          '5491143214321',
          '+5491143214321',
          '+5492414321432',
          '5498418432143',
        ],
        'invalid': [
          '1143214321',
          '91143214321',
          '+91143214321',
          '549841004321432',
          '549 11 43214321',
          '549111543214321',
          '5714003425432',
          '549114a214321',
          '54 9 11 4321-4321',
        ],
      },
      {
        'locale': 'es-CO',
        'valid': [
          '+573003321235',
          '573003321235',
          '3003321235',
          '3213321235',
          '3103321235',
          '3243321235',
          '573011140876',
        ],
        'invalid': [
          '1234',
          '+57443875615',
          '57309875615',
          '57109834567',
          '5792434567',
          '5702345689',
          '5714003425432',
          '5703013347567',
          '069834567',
          '969834567',
          '579871235',
          '574321235',
          '5784321235',
          '5784321235',
          '9821235',
          '0698345',
          '3321235',
        ],
      },
      {
        'locale': 'es-CL',
        'valid': [
          '+56733875615',
          '56928590234',
          '0928590294',
          '0208590294',
        ],
        'invalid': [
          '1234',
          '+5633875615',
          '563875615',
          '56109834567',
          '56069834567',
        ],
      },
      {
        'locale': 'es-EC',
        'valid': [
          '+593987654321',
          '593987654321',
          '0987654321',
          '027332615',
          '+59323456789',
        ],
        'invalid': [
          '03321321',
          '+593387561',
          '59312345677',
          '02344635',
          '593123456789',
          '081234567',
          '+593912345678',
          '+593902345678',
          '+593287654321',
          '593287654321',
        ],
      },
      {
        'locale': 'es-CR',
        'valid': [
          '+50688888888',
          '+50665408090',
          '+50640895069',
          '25789563',
          '85789563',
        ],
        'invalid': [
          '+5081',
          '+5067777777',
          '+50188888888',
          '+50e987643254',
          '+506e4t4',
          '-50688888888',
          '50688888888',
          '12345678',
          '98765432',
          '01234567',
        ],
      },
      {
        'locale': 'es-CU',
        'valid': [
          '+5351234567',
          '005353216547',
          '51234567',
          '53214567',
        ],
        'invalid': [
          '1234',
          '+5341234567',
          '0041234567',
          '41234567',
          '11234567',
          '21234567',
          '31234567',
          '60303456',
          '71234567',
          '81234567',
          '91234567',
          '+5343216547',
          '+5332165498',
          '+53121234567',
          '',
          'abc',
          '+535123457',
        ],
      },
      {
        'locale': 'es-DO',
        'valid': [
          '+18096622563',
          '+18295614488',
          '+18495259567',
          '8492283478',
          '8092324576',
          '8292387713',
        ],
        'invalid': [
          '+18091',
          '+1849777777',
          '-18296643245',
          '+18086643245',
          '+18396643245',
          '8196643245',
          '+38492283478',
          '6492283478',
          '8192283478',
        ],
      },
      {
        'locale': 'es-HN',
        'valid': [
          '+50495551876',
          '+50488908787',
          '+50493456789',
          '+50489234567',
          '+50488987896',
          '+50497567389',
        ],
        'invalid': [
          '12345',
          '',
          'Vml2YW11cyBmZXJtZtesting123',
          '+34683456543',
          '65478932',
          '+50298787654',
          '+504989874',
        ],
      },
      {
        'locale': 'es-ES',
        'valid': [
          '+34654789321',
          '654789321',
          '+34714789321',
          '714789321',
          '+34744789321',
          '744789321',
        ],
        'invalid': [
          '12345',
          '',
          'Vml2YW11cyBmZXJtZtesting123',
          '+3465478932',
          '65478932',
          '+346547893210',
          '6547893210',
          '+3470478932',
          '7047893210',
          '+34854789321',
          '7547893219',
        ],
      },
      {
        'locale': 'es-MX',
        'valid': [
          '+52019654789321',
          '+52199654789321',
          '+5201965478932',
          '+5219654789321',
          '52019654789321',
          '52199654789321',
          '5201965478932',
          '5219654789321',
          '87654789321',
          '8654789321',
          '0187654789321',
          '18654789321',
        ],
        'invalid': [
          '12345',
          '',
          'Vml2YW11cyBmZXJtZtesting123',
          '+3465478932',
          '65478932',
          '+346547893210',
          '+34704789321',
          '704789321',
          '+34754789321',
        ],
      },
      {
        'locale': 'es-PA',
        'valid': [
          '+5076784565',
          '+5074321557',
          '5073331112',
          '+50723431212',
        ],
        'invalid': [
          '+50755555',
          '+207123456',
          '2001236542',
          '+507987643254',
          '+507jjjghtf',
        ],
      },
      {
        'locale': 'es-PY',
        'valid': [
          '+595991372649',
          '+595992847352',
          '+595993847593',
          '+595994857473',
          '+595995348532',
          '+595996435231',
          '+595981847362',
          '+595982435452',
          '+595983948502',
          '+595984342351',
          '+595985403481',
          '+595986384012',
          '+595971435231',
          '+595972103924',
          '+595973438542',
          '+595974425864',
          '+595975425843',
          '+595976342546',
          '+595961435234',
          '+595963425043',
        ],
        'invalid': [
          '12345',
          '',
          'Vml2YW11cyBmZXJtZtesting123',
          '65478932',
          '+59599384712',
          '+5959938471234',
          '+595547893218',
          '+591993546843',
        ],
      },
      {
        'locale': 'es-UY',
        'valid': [
          '+59899123456',
          '099123456',
          '+59894654321',
          '091111111',
        ],
        'invalid': [
          '54321',
          'montevideo',
          '',
          '+598099123456',
          '090883338',
          '099 999 999',
        ],
      },
      {
        'locale': 'es-VE',
        'valid': [
          '+582125457765',
          '+582125458053',
          '+584125458053',
        ],
        'invalid': [
          '+585129934395',
          '+58212993439',
          '',
        ],
      },
      {
        'locale': 'et-EE',
        'valid': [
          '+372 512 34 567',
          '372 512 34 567',
          '+37251234567',
          '51234567',
          '81234567',
          '+372842345678',
        ],
        'invalid': [
          '12345',
          '',
          'NotANumber',
          '+333 51234567',
          '61234567',
          '+51234567',
          '+372 539 57 4',
          '+372 900 1234',
          '12345678',
        ],
      },
      {
        'locale': 'pl-PL',
        'valid': [
          '+48512689767',
          '+48 56 376 87 47',
          '56 566 78 46',
          '657562855',
          '+48657562855',
          '+48 887472765',
          '+48 56 6572724',
          '+48 67 621 5461',
          '48 67 621 5461',
        ],
        'invalid': [
          '+48  67 621 5461',
          '+55657562855',
          '3454535',
          'teststring',
          '',
          '1800-88-8687',
          '+6019-5830837',
          '357562855',
        ],
      },
      {
        'locale': 'fa-IR',
        'valid': [
          '+989123456789',
          '989223456789',
          '09323456789',
          '09021456789',
          '+98-990-345-6789',
          '+98 938 345 6789',
          '0938 345 6789',
        ],
        'invalid': [
          '',
          '+989623456789',
          '+981123456789',
          '01234567890',
          '09423456789',
          '09823456789',
          '9123456789',
          '091234567890',
          '0912345678',
          '+98 912 3456 6789',
          '0912 345 678',
        ],
      },
      {
        'locale': 'fi-FI',
        'valid': [
          '+358505557171',
          '0455571',
          '0505557171',
          '358505557171',
          '04412345',
          '0457 123 45 67',
          '+358457 123 45 67',
          '+358 50 555 7171',
        ],
        'invalid': [
          '12345',
          '',
          '045557',
          '045555717112312332423423421',
          'Vml2YW11cyBmZXJtZtesting123',
          '010-38238383',
          '+3-585-0555-7171',
          '+9676338855',
          '19676338855',
          '6676338855',
          '+99676338855',
          '044123',
          '019123456789012345678901',
        ],
      },
      {
        'locale': 'fj-FJ',
        'valid': [
          '+6799898679',
          '6793788679',
          '+679 989 8679',
          '679 989 8679',
          '679 3456799',
          '679908 8909',
        ],
        'invalid': [
          '12345',
          '',
          '04555792',
          '902w99900030900000000099',
          '8uiuiuhhyy&GUU88d',
          '010-38238383',
          '19676338855',
          '679 9 89 8679',
          '6793 45679',
        ],
      },
      {
        'locale': 'ms-MY',
        'valid': [
          '+60128228789',
          '+60195830837',
          '+6019-5830837',
          '+6019-5830837',
          '+6010-4357675',
          '+60172012370',
          '0128737867',
          '0172012370',
          '01468987837',
          '01112347345',
          '016-2838768',
          '016 2838768',
        ],
        'invalid': [
          '12345',
          '601238788657',
          '088387675',
          '16-2838768',
          '032551433',
          '6088-387888',
          '088-261987',
          '1800-88-8687',
          '088-320000',
        ],
      },
      {
        'locale': 'fr-CM',
        'valid': [
          '+237677936141',
          '237623456789',
          '+237698124842',
          '237693029202',
        ],
        'invalid': [
          'NotANumber',
          '+(703)-572-2920',
          '+237 623 45 67 890',
          '+2379981247429',
        ],
      },
      {
        'locale': 'ko-KR',
        'valid': [
          '+82-010-1234-5678',
          '+82-10-1234-5678',
          '82-010-1234-5678',
          '82-10-1234-5678',
          '+82 10 1234 5678',
          '010-123-5678',
          '10-1234-5678',
          '+82 10 1234 5678',
          '011 1234 5678',
          '+820112345678',
          '01012345678',
          '+82 016 1234 5678',
          '82 19 1234 5678',
          '+82 010 12345678',
        ],
        'invalid': [
          'abcdefghi',
          '+82 10 1234 567',
          '+82 10o 1234 1234',
          '+82 101 1234 5678',
          '+82 10 12 5678',
          '+011 7766 1234',
          '011_7766_1234',
          '+820 11 7766 1234',
        ],
      },
      {
        'locale': 'ja-JP',
        'valid': [
          '09012345678',
          '08012345678',
          '07012345678',
          '06012345678',
          '090 1234 5678',
          '+8190-1234-5678',
          '+81 (0)90-1234-5678',
          '+819012345678',
          '+81-(0)90-1234-5678',
          '+81 90 1234 5678',
        ],
        'invalid': [
          '12345',
          '',
          '045555717112312332423423421',
          'Vml2YW11cyBmZXJtZtesting123',
          '+3-585-0555-7171',
          '0 1234 5689',
          '16 1234 5689',
          '03_1234_5689',
          '0312345678',
          '0721234567',
          '06 1234 5678',
          '072 123 4567',
          '0729 12 3456',
          '07296 1 2345',
          '072961 2345',
          '03-1234-5678',
          '+81312345678',
          '+816-1234-5678',
          '+81 090 1234 5678',
          '+8109012345678',
          '+81-090-1234-5678',
          '90 1234 5678',
        ],
      },
      {
        'locale': 'it-IT',
        'valid': [
          '370 3175423',
          '333202925',
          '+39 310 7688449',
          '+39 3339847632',
        ],
        'invalid': [
          '011 7387545',
          '12345',
          '+45 345 6782395',
        ],
      },
      {
        'locale': 'fr-BE',
        'valid': [
          '0470123456',
          '+32470123456',
          '32470123456',
          '0421234567',
          '+32421234567',
          '32421234567',
        ],
        'invalid': [
          '12345',
          '+3212345',
          '3212345',
          '04701234567',
          '+3204701234567',
          '3204701234567',
          '0212345678',
          '+320212345678',
          '320212345678',
          '021234567',
          '+3221234567',
          '3221234567',
        ],
      },
      {
        'locale': 'nl-BE',
        'valid': [
          '0470123456',
          '+32470123456',
          '32470123456',
          '0421234567',
          '+32421234567',
          '32421234567',
        ],
        'invalid': [
          '12345',
          '+3212345',
          '3212345',
          '04701234567',
          '+3204701234567',
          '3204701234567',
          '0212345678',
          '+320212345678',
          '320212345678',
          '021234567',
          '+3221234567',
          '3221234567',
        ],
      },
      {
        'locale': 'nl-NL',
        'valid': [
          '0670123456',
          '0612345678',
          '31612345678',
          '31670123456',
          '+31612345678',
          '+31670123456',
          '+31(0)612345678',
          '0031612345678',
          '0031(0)612345678',
        ],
        'invalid': [
          '12345',
          '+3112345',
          '3112345',
          '06701234567',
          '012345678',
          '+3104701234567',
          '3104701234567',
          '0212345678',
          '021234567',
          '+3121234567',
          '3121234567',
          '+310212345678',
          '310212345678',
        ],
      },
      {
        'locale': 'ro-RO',
        'valid': [
          '+40740123456',
          '+40 740123456',
          '+40740 123 456',
          '+40740.123.456',
          '+40740-123-456',
          '40740123456',
          '40 740123456',
          '40740 123 456',
          '40740.123.456',
          '40740-123-456',
          '0740123456',
          '0740/123456',
          '0740 123 456',
          '0740.123.456',
          '0740-123-456',
        ],
        'invalid': [
          '',
          'Vml2YW11cyBmZXJtZtesting123',
          '123456',
          '740123456',
          '+40640123456',
          '+40210123456',
        ],
      },
      {
        'locale': 'id-ID',
        'valid': [
          '0811 778 998',
          '0811 7785 9983',
          '0812 7784 9984',
          '0813 7782 9982',
          '0821 1234 1234',
          '0822 1234 1234',
          '0823 1234 1234',
          '0852 1234 6764',
          '0853 1234 6764',
          '0851 1234 6764',
          '0814 7782 9982',
          '0815 7782 9982',
          '0816 7782 9982',
          '0855 7782 9982',
          '0856 7782 9982',
          '0857 7782 9982',
          '0858 7782 9982',
          '0817 7785 9983',
          '0818 7784 9984',
          '0819 7782 9982',
          '0859 1234 1234',
          '0877 1234 1234',
          '0878 1234 1234',
          '0895 7785 9983',
          '0896 7784 9984',
          '0897 7782 9982',
          '0898 1234 1234',
          '0899 1234 1234',
          '0881 7785 9983',
          '0882 7784 9984',
          '0883 7782 9982',
          '0884 1234 1234',
          '0886 1234 1234',
          '0887 1234 1234',
          '0888 7785 9983',
          '0889 7784 9984',
          '0828 7784 9984',
          '0838 7784 9984',
          '0831 7784 9984',
          '0832 7784 9984',
          '0833 7784 9984',
          '089931236181900',
          '62811 778 998',
          '62811778998',
          '628993123618190',
          '62898 740123456',
          '62899 7401 2346',
          '+62811 778 998',
          '+62811778998',
          '+62812 9650 3508',
          '08197231819',
          '085361008008',
          '+62811787391',
        ],
        'invalid': [
          '0899312361819001',
          '0217123456',
          '622178878890',
          '6221 740123456',
          '0341 8123456',
          '0778 89800910',
          '0741 123456',
          '+6221740123456',
          '+65740 123 456',
          '',
          'ASDFGJKLmZXJtZtesting123',
          '123456',
          '740123456',
          '+65640123456',
          '+64210123456',
        ],
      },
      {
        'locale': 'lt-LT',
        'valid': [
          '+37051234567',
          '851234567',
        ],
        'invalid': [
          '+65740 123 456',
          '',
          'ASDFGJKLmZXJtZtesting123',
          '123456',
          '740123456',
          '+65640123456',
          '+64210123456',
        ],
      },
      {
        'locale': 'uk-UA',
        'valid': [
          '+380982345679',
          '380982345679',
          '80982345679',
          '0982345679',
        ],
        'invalid': [
          '+30982345679',
          '982345679',
          '+380 98 234 5679',
          '+380-98-234-5679',
          '',
          'ASDFGJKLmZXJtZtesting123',
          '123456',
          '740123456',
        ],
      },
      {
        'locale': 'uz-UZ',
        'valid': [
          '+998664835244',
          '998664835244',
          '664835244',
          '+998957124555',
          '998957124555',
          '957124555',
        ],
        'invalid': [
          '+998644835244',
          '998644835244',
          '644835244',
          '+99664835244',
          'ASDFGJKLmZXJtZtesting123',
          '123456789',
          '870123456',
          '',
          '+998',
          '998',
        ],
      },
      {
        'locale': 'da-DK',
        'valid': [
          '12345678',
          '12 34 56 78',
          '45 12345678',
          '4512345678',
          '45 12 34 56 78',
          '+45 12 34 56 78',
        ],
        'invalid': [
          '',
          '+45010203',
          'ASDFGJKLmZXJtZtesting123',
          '123456',
          '12 34 56',
          '123 123 12',
        ],
      },
      {
        'locale': 'sv-SE',
        'valid': [
          '+46701234567',
          '46701234567',
          '0721234567',
          '073-1234567',
          '0761-234567',
          '079-123 45 67',
        ],
        'invalid': [
          '12345',
          '+4670123456',
          '+46301234567',
          '+0731234567',
          '0731234 56',
          '+7312345678',
          '',
        ],
      },
      {
        'locale': 'fo-FO',
        'valid': [
          '123456',
          '12 34 56',
          '298 123456',
          '298123456',
          '298 12 34 56',
          '+298 12 34 56',
        ],
        'invalid': [
          '',
          '+4501020304',
          'ASDFGJKLmZXJtZtesting123',
          '12345678',
          '12 34 56 78',
        ],
      },
      {
        'locale': 'kl-GL',
        'valid': [
          '123456',
          '12 34 56',
          '299 123456',
          '299123456',
          '299 12 34 56',
          '+299 12 34 56',
        ],
        'invalid': [
          '',
          '+4501020304',
          'ASDFGJKLmZXJtZtesting123',
          '12345678',
          '12 34 56 78',
        ],
      },
      {
        'locale': 'kk-KZ',
        'valid': [
          '+77254716212',
          '77254716212',
          '87254716212',
          '7254716212',
        ],
        'invalid': [
          '12345',
          '',
          'ASDFGJKLmZXJtZtesting123',
          '010-38238383',
          '+9676338855',
          '19676338855',
          '6676338855',
          '+99676338855',
        ],
      },
      {
        'locale': 'be-BY',
        'valid': [
          '+375241234567',
          '+375251234567',
          '+375291234567',
          '+375331234567',
          '+375441234567',
          '375331234567',
        ],
        'invalid': [
          '12345',
          '',
          'ASDFGJKLmZXJtZtesting123',
          '010-38238383',
          '+9676338855',
          '19676338855',
          '6676338855',
          '+99676338855',
        ],
      },
      {
        'locale': 'th-TH',
        'valid': [
          '0912345678',
          '+66912345678',
          '66912345678',
        ],
        'invalid': [
          '99123456789',
          '12345',
          '67812345623',
          '081234567891',
        ],
      },
      {
        'locale': ['en-ZA', 'be-BY'],
        'valid': [
          '0821231234',
          '+27821231234',
          '27821231234',
          '+375241234567',
          '+375251234567',
          '+375291234567',
          '+375331234567',
          '+375441234567',
          '375331234567',
        ],
        'invalid': [
          '082123',
          '08212312345',
          '21821231234',
          '+21821231234',
          '+0821231234',
          '12345',
          '',
          'ASDFGJKLmZXJtZtesting123',
          '010-38238383',
          '+9676338855',
          '19676338855',
          '6676338855',
          '+99676338855',
        ],
      },
      {
        'locale': 'en-SL',
        'valid': [
          '+23274560591',
          '23274560591',
          '074560591',
        ],
        'invalid': [
          '0745605912',
          '12345',
          '232745605917',
          '0797878674',
          '23274560591 ',
        ],
      },
      {
        'locale': 'az-AZ',
        'valid': [
          '+994707007070',
          '0707007070',
          '+994502111111',
          '0505436743',
          '0554328772',
          '0993301022',
          '+994776007139',
        ],
        'invalid': [
          'wrong-number',
          '',
          '994707007070',
          '++9945005050',
          '556007070',
          '1234566',
          '+994778008080a',
        ],
      },
      {
        'locale': 'de-LU',
        'valid': [
          '601123456',
          '+352601123456',
        ],
        'invalid': [
          'NaN',
          '791234',
          '+352791234',
          '26791234',
          '+35226791234',
          '+112039812',
          '+352703123456',
          '1234',
        ],
      },
      {
        'locale': 'it-SM',
        'valid': [
          '612345',
          '05496123456',
          '+37861234567',
          '+390549612345678',
          '+37805496123456789',
        ],
        'invalid': [
          '61234567890',
          '6123',
          '1234567',
          '+49123456',
          'NotANumber',
        ],
      },
      {
        'locale': 'sq-AL',
        'valid': [
          '067123456',
          '+35567123456',
        ],
        'invalid': [
          '67123456',
          '06712345',
          '06712345678',
          '065123456',
          '057123456',
          'NotANumber',
        ],
      },
      {
        'locale': 'ca-AD',
        'valid': [
          '+376312345',
          '312345',
        ],
        'invalid': [
          '31234',
          '31234567',
          '512345',
          'NotANumber',
        ],
      },
      {
        'locale': 'pt-AO',
        'valid': [
          '+244911123432',
          '+244123091232',
        ],
        'invalid': [
          '+2449111234321',
          '31234',
          '31234567',
          '512345',
          'NotANumber',
        ],
      },
      {
        'locale': 'lv-LV',
        'valid': [
          '+37121234567',
          '37121234567',
        ],
        'invalid': [
          '+37201234567',
          '+3754321',
          '3712123456',
          '+371212345678',
          'NotANumber',
        ],
      },
      {
        'locale': 'en-PK',
        'valid': [
          '+923412877421',
          '+923001234567',
          '00923001234567',
          '923001234567',
          '03001234567',
        ],
        'invalid': [
          '+3001234567',
          '+933001234567',
          '+924001234567',
          '+92300123456720',
          '030012345672',
          '30012345673',
          '0030012345673',
          '3001234567',
        ],
      },
      {
        'locale': ['tg-TJ'],
        'valid': [
          '+992553388551',
          '+992553322551',
          '992553388551',
          '992553322551',
        ],
        'invalid': [
          '12345',
          '',
          'Vml2YW11cyBmZXJtZtesting123',
          '+995563388559',
          '+9955633559',
          '19676338855',
          '+992263388505',
          '9923633885',
          '99255363885',
          '66338855',
        ],
    },
]


class TestIsMobileNumber(unittest.TestCase):

    def test_valid_mobile_numbers(self):
        for fixture in fixtures:
            locale = fixture['locale']
            valid_numbers = fixture['valid']
            for number in valid_numbers:
                self.assertTrue(is_mobile_number(number, locale))
                print_test_ok(f'for locale {locale}')

    def test_invalid_mobile_numbers(self):
        for fixture in fixtures:
            locale = fixture['locale']
            invalid_numbers = fixture['invalid']
            for number in invalid_numbers:
                self.assertFalse(is_mobile_number(number, locale))
            print_test_ok(r'for locale {locale}')

    def test_valid_mobile_numbers_locale_any_explicite(self):
        valid_numbers = [
            '+992553322551',
            '0707007070',
            '+8190-1234-5678',
            '0670123456',
            '8005552222',
            '+37422298765'
        ]
        for number in valid_numbers:
            self.assertTrue(is_mobile_number(number, 'any'))
            print_test_ok(f', number: {number}')

    def test_valid_mobile_numbers_locale_any_implicit(self):
        valid_numbers = [
            '+992553322551',
            '0707007070',
            '+8190-1234-5678',
            '0670123456',
            '8005552222',
            '+37422298765'
        ]
        for number in valid_numbers:
            self.assertTrue(is_mobile_number(number))
            print_test_ok(f', number: {number}')

    def test_valid_mobile_numbers_strict_mode(self):
        valid_numbers = [
            '+992553322551',
            '+37422298765',
            '+380982345679',
            '+60128228789',
            '+852 9123 4567',
            '+33712457898',
            '+381638730746'
        ]
        for number in valid_numbers:
            self.assertTrue(is_mobile_number(number, 'any', { "strict_mode": True }))
            print_test_ok(f', number: {number}')

    def test_invalid_mobile_numbers_strict_mode(self):
        valid_numbers = [
            '0638730746',
            '37121234567',
            '0707007070',
            '46701234567',
            '0923330000'
        ]
        for number in valid_numbers:
            self.assertFalse(is_mobile_number(number, 'any', { "strict_mode": True }))
            print_test_ok(r', number: {number}')

    def test_throw_moboile_number_validation_for_bad_locale(self):
        self.assertRaises(Exception, is_mobile_number, ['+381638730746', 'BAD'])
        print_test_ok()
