"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.waitForAndAssertStackStatus = exports.waitForAndAssertImageAvailable = exports.waitForAndAssertInstanceStatus = exports.waitForAndAssertInstanceState = exports.waitForAndAssertResource = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const response_code_1 = require("../../../domain/response-code");
const stack_status_1 = require("../../../domain/stack-status");
const wait_for_and_assert_resource_1 = require("../../../patterns/automation/wait-for-and-assert-resource");
const simulate_pattern_1 = require("../../../utils/simulate-pattern");
const aws_service_1 = require("../../../domain/aws-service");
/**
 * Wait for and assert on an EC2 resource value
 */
function waitForAndAssertResource(props, simulationProps) {
    const component = new wait_for_and_assert_resource_1.WaitForAndAssertResource(new aws_cdk_lib_1.Stack(), 'waitForAndAssertResource', props);
    const result = simulate_pattern_1.simulatePattern(component, simulationProps, {});
    if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
        throw new Error(`Wait and assert for resource failes for ${JSON.stringify(props)}: ${result.stackTrace}`);
    }
}
exports.waitForAndAssertResource = waitForAndAssertResource;
/**
 * Wait for and assert on an EC2 instance state
 */
function waitForAndAssertInstanceState(props, simulationProps) {
    const { waitForState, assertState, instanceIds } = props;
    waitForAndAssertResource({
        service: aws_service_1.AwsService.EC2,
        pascalCaseApi: 'DescribeInstanceStatus',
        apiParams: {
            InstanceIds: instanceIds,
            IncludeAllInstances: true,
        },
        selector: '$.InstanceStatuses..InstanceState.Name',
        waitForValues: waitForState,
        desiredValues: [assertState],
    }, simulationProps);
}
exports.waitForAndAssertInstanceState = waitForAndAssertInstanceState;
/**
 * Wait for and assert on an EC2 instance status
 */
function waitForAndAssertInstanceStatus(props, simulationProps) {
    const { waitForStatus, assertStatus, instanceIds } = props;
    waitForAndAssertResource({
        service: aws_service_1.AwsService.EC2,
        pascalCaseApi: 'DescribeInstanceStatus',
        apiParams: {
            InstanceIds: instanceIds,
            IncludeAllInstances: true,
        },
        selector: '$.InstanceStatuses..InstanceStatus.Status',
        waitForValues: waitForStatus,
        desiredValues: [assertStatus],
    }, simulationProps);
}
exports.waitForAndAssertInstanceStatus = waitForAndAssertInstanceStatus;
/**
 * Wait for and assert on an EC2 image for available state
 */
function waitForAndAssertImageAvailable(props, simulationProps) {
    const { imageId } = props;
    waitForAndAssertResource({
        service: aws_service_1.AwsService.EC2,
        pascalCaseApi: 'DescribeImages',
        selector: '$.Images..State',
        apiParams: {
            ImageIds: [imageId],
        },
        waitForValues: ['available', 'invalid', 'deregistered', 'transient', 'failed', 'error'],
        desiredValues: ['available'],
    }, simulationProps);
}
exports.waitForAndAssertImageAvailable = waitForAndAssertImageAvailable;
/**
 * Wait for and assert on a stack's status
 */
function waitForAndAssertStackStatus(props, simulationProps) {
    const { waitForStatus, assertStatus, stackName } = props;
    waitForAndAssertResource({
        service: aws_service_1.AwsService.CLOUD_FORMATION,
        pascalCaseApi: 'DescribeStacks',
        apiParams: {
            StackName: stackName,
        },
        selector: '$.Stacks[0].StackStatus',
        waitForValues: waitForStatus.map(x => stack_status_1.StackStatus[x]),
        desiredValues: [stack_status_1.StackStatus[assertStatus]],
    }, simulationProps);
}
exports.waitForAndAssertStackStatus = waitForAndAssertStackStatus;
//# sourceMappingURL=data:application/json;base64,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