"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KongControlPlaneEKS = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
class KongControlPlaneEKS extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const rds_secrets = 'rds-secrets';
        const rds_secrets_provider_class = 'rds-secrets-provider-class';
        // const route_53_zone = aws_route53.HostedZone.fromLookup(this, 'MyZone', {
        //   domainName: props.endpoints.hostedZoneName,
        // });
        // const admin_cert = new aws_certificatemanager.Certificate(this, 'adminCert', {
        //   domainName: props.endpoints.adminDns,
        //   validation: aws_certificatemanager.CertificateValidation.fromDns(route_53_zone),
        // });
        // const cluster_cert = new aws_certificatemanager.Certificate(this, 'clusterCert', {
        //   domainName: props.endpoints.clusterDns,
        //   validation: aws_certificatemanager.CertificateValidation.fromDns(route_53_zone),
        // });
        // const telemetry_cert = new aws_certificatemanager.Certificate(this, 'telemetryCert', {
        //   domainName: props.endpoints.telemetryDns,
        //   validation: aws_certificatemanager.CertificateValidation.fromDns(route_53_zone),
        // });
        // const manager_cert = new aws_certificatemanager.Certificate(this, 'managerCert', {
        //   domainName: props.endpoints.managerDns,
        //   validation: aws_certificatemanager.CertificateValidation.fromDns(route_53_zone),
        // });
        const kong_namespace = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongNamespace', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'v1',
                    kind: 'Namespace',
                    metadata: { name: props.namespace },
                },
            ],
        });
        const kong_cp_service_account = new aws_cdk_lib_1.aws_eks.ServiceAccount(this, 'KongCpSA', {
            cluster: props.cluster,
            name: 'kong-cp-service-account',
            namespace: props.namespace,
        });
        const aws_region = aws_cdk_lib_1.Stack.of(this).region;
        const aws_account = aws_cdk_lib_1.Stack.of(this).account;
        kong_cp_service_account.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            resources: [
                `arn:aws:secretsmanager:${aws_region}:${aws_account}:secret:${props.license_secret_name}-??????`,
                `${props.rds.secret?.secretFullArn}`,
            ],
            actions: [
                'secretsmanager:GetSecretValue',
                'secretsmanager:DescribeSecret',
            ],
        }));
        kong_cp_service_account.node.addDependency(kong_namespace);
        const kong_certificate = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongCert', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'cert-manager.io/v1',
                    kind: 'Certificate',
                    metadata: {
                        name: props.cacertname,
                        namespace: props.namespace,
                    },
                    spec: {
                        commonName: props.endpoints.hostedZoneName,
                        dnsNames: [
                            props.endpoints.adminDns,
                            props.endpoints.clusterDns,
                            props.endpoints.managerDns,
                            props.endpoints.telemetryDns,
                        ],
                        duration: '2160h0m0s',
                        issuerRef: {
                            group: 'awspca.cert-manager.io',
                            kind: 'AWSPCAClusterIssuer',
                            name: props.clusterIssuerName,
                        },
                        renewBefore: '360h0m0s',
                        secretName: `${props.clusterIssuerName}-secret`,
                        usages: [
                            'server auth',
                            'client auth',
                        ],
                        privateKey: {
                            algorithm: 'RSA',
                            size: 2048,
                        },
                    },
                },
            ],
        });
        kong_certificate.node.addDependency(kong_namespace);
        // const secretNameParts = Fn.split('-', props.rds.secret?.secretName!);
        // const secretNameWithoutSuffix = Fn.join('-', [Fn.select(0, secretNameParts), Fn.select(1, secretNameParts)]);
        const secrets_crd_mount = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'SecretsMount', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'secrets-store.csi.x-k8s.io/v1',
                    kind: 'SecretProviderClass',
                    metadata: {
                        name: rds_secrets_provider_class,
                        namespace: props.namespace,
                    },
                    spec: {
                        provider: 'aws',
                        parameters: {
                            objects: `- objectName: "${props.rds.secret?.secretFullArn}"\n  jmesPath:\n    - path: username\n      objectAlias: dbusername\n    - path: password\n      objectAlias: dbpassword\n    - path: dbname\n      objectAlias: dbname\n    - path: host\n      objectAlias: dbhost\n`,
                        },
                        secretObjects: [
                            {
                                secretName: rds_secrets,
                                type: 'Opaque',
                                data: [
                                    {
                                        objectName: 'dbusername',
                                        key: 'dbusername',
                                    },
                                    {
                                        objectName: 'dbpassword',
                                        key: 'dbpassword',
                                    },
                                    {
                                        objectName: 'dbname',
                                        key: 'dbname',
                                    },
                                    {
                                        objectName: 'dbhost',
                                        key: 'dbhost',
                                    },
                                ],
                            },
                        ],
                    },
                },
                {
                    apiVersion: 'secrets-store.csi.x-k8s.io/v1',
                    kind: 'SecretProviderClass',
                    metadata: {
                        name: props.license_secret_name,
                        namespace: props.namespace,
                    },
                    spec: {
                        provider: 'aws',
                        secretObjects: [
                            {
                                secretName: props.license_secret_name,
                                type: 'opaque',
                                data: [
                                    {
                                        objectName: props.license_secret_name,
                                        key: 'license',
                                    },
                                ],
                            },
                        ],
                        parameters: {
                            objects: `- objectName: "${props.license_secret_name}"\n  objectType: "secretsmanager"\n`,
                        },
                    },
                },
            ],
        });
        secrets_crd_mount.node.addDependency(kong_namespace);
        const deploy_kong_cp_helm = new aws_cdk_lib_1.aws_eks.HelmChart(this, 'KongCpHelm', {
            ...props.HelmOptions,
            cluster: props.cluster,
            repository: 'https://charts.konghq.com',
            chart: 'kong',
            release: 'kong',
            namespace: props.namespace,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            wait: true,
            createNamespace: true,
            values: {
                ingressController: {
                    enabled: false,
                },
                // image: {
                //   repository: 'kong/kong-gateway',
                //   tag: '2.5.0.0-alpine',
                // },
                env: {
                    cluster_data_plane_purge_delay: 86400,
                    database: 'postgres',
                    role: 'control_plane',
                    cluster_mtls: 'pki',
                    cluster_cert: `/etc/secrets/${props.clusterIssuerName}-secret/tls.crt`,
                    cluster_cert_key: `/etc/secrets/${props.clusterIssuerName}-secret/tls.key`,
                    cluster_ca_cert: `/etc/secrets/${props.clusterIssuerName}-secret/ca.crt`,
                    admin_ssl_cert: `/etc/secrets/${props.clusterIssuerName}-secret/tls.crt`,
                    admin_ssl_cert_key: `/etc/secrets/${props.clusterIssuerName}-secret/tls.key`,
                    admin_gui_url: `https://${props.endpoints.managerDns}`,
                    admin_gui_ssl_cert: `/etc/secrets/${props.clusterIssuerName}-secret/tls.crt`,
                    admin_gui_ssl_cert_key: `/etc/secrets/${props.clusterIssuerName}-secret/tls.key`,
                    cluster_telemetry_endpoint: `${props.endpoints.telemetryDns}:443`,
                    pg_user: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbusername',
                            },
                        },
                    },
                    pg_password: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbpassword',
                            },
                        },
                    },
                    pg_database: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbname',
                            },
                        },
                    },
                    pg_host: {
                        valueFrom: {
                            secretKeyRef: {
                                name: rds_secrets,
                                key: 'dbhost',
                            },
                        },
                    },
                },
                cluster: {
                    enabled: true,
                    type: 'LoadBalancer',
                    tls: {
                        enabled: true,
                        servicePort: 443,
                    },
                    annotations: {
                        // 'service.beta.kubernetes.io/aws-load-balancer-proxy-protocol': '*',
                        'service.beta.kubernetes.io/aws-load-balancer-nlb-target-type': 'ip',
                        // 'service.beta.kubernetes.io/aws-load-balancer-ssl-cert': cluster_cert.certificateArn,
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'external',
                        'service.beta.kubernetes.io/aws-load-balancer-scheme': 'internet-facing',
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${props.cluster.clusterName}-eks-cluster`,
                        'external-dns.alpha.kubernetes.io/hostname': props.endpoints.clusterDns,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                clustertelemetry: {
                    enabled: true,
                    type: 'LoadBalancer',
                    tls: {
                        enabled: true,
                        servicePort: 443,
                    },
                    annotations: {
                        // 'service.beta.kubernetes.io/aws-load-balancer-proxy-protocol': '*',
                        'service.beta.kubernetes.io/aws-load-balancer-nlb-target-type': 'ip',
                        // 'service.beta.kubernetes.io/aws-load-balancer-ssl-cert': telemetry_cert.certificateArn,
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'external',
                        'service.beta.kubernetes.io/aws-load-balancer-scheme': 'internet-facing',
                        'external-dns.alpha.kubernetes.io/hostname': props.endpoints.telemetryDns,
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${props.cluster.clusterName}-eks-telemetry`,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                proxy: {
                    enabled: false,
                },
                admin: {
                    enabled: true,
                    type: 'LoadBalancer',
                    labels: {
                        'enable-metrics': true,
                    },
                    tls: {
                        servicePort: 443,
                    },
                    annotations: {
                        // 'service.beta.kubernetes.io/aws-load-balancer-proxy-protocol': '*',
                        'service.beta.kubernetes.io/aws-load-balancer-nlb-target-type': 'ip',
                        // 'service.beta.kubernetes.io/aws-load-balancer-ssl-cert': admin_cert.certificateArn,
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'external',
                        'service.beta.kubernetes.io/aws-load-balancer-scheme': 'internal',
                        'service.beta.kubernetes.io/aws-load-balancer-additional-resource-tags': 'Type=admin',
                        'external-dns.alpha.kubernetes.io/hostname': props.endpoints.adminDns,
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${props.cluster.clusterName}-eks-admin`,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                deployment: {
                    serviceAccount: {
                        create: false,
                        name: kong_cp_service_account.serviceAccountName,
                    },
                    userDefinedVolumes: [
                        {
                            name: props.license_secret_name,
                            csi: {
                                driver: 'secrets-store.csi.k8s.io',
                                readOnly: true,
                                volumeAttributes: {
                                    secretProviderClass: props.license_secret_name,
                                },
                            },
                        },
                        {
                            name: rds_secrets,
                            csi: {
                                driver: 'secrets-store.csi.k8s.io',
                                readOnly: true,
                                volumeAttributes: {
                                    secretProviderClass: rds_secrets_provider_class,
                                },
                            },
                        },
                    ],
                    userDefinedVolumeMounts: [
                        {
                            name: props.license_secret_name,
                            mountPath: '/mnt/license_secrets',
                            readOnly: true,
                        },
                        {
                            name: rds_secrets,
                            mountPath: '/mnt/rds_secrets',
                            readOnly: true,
                        },
                    ],
                },
                enterprise: {
                    enabled: true,
                    license_secret: props.license_secret_name,
                },
                serviceMonitor: {
                    enabled: true,
                },
                manager: {
                    enabled: true,
                    type: 'LoadBalancer',
                    tls: {
                        enabled: true,
                        servicePort: 443,
                    },
                    // http: {
                    //   enabled: false,
                    // },
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-nlb-target-type': 'ip',
                        // 'service.beta.kubernetes.io/aws-load-balancer-proxy-protocol': '*',
                        // 'service.beta.kubernetes.io/aws-load-balancer-ssl-cert': manager_cert.certificateArn,
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'external',
                        'service.beta.kubernetes.io/aws-load-balancer-scheme': 'internal',
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${props.cluster.clusterName}-eks-manager`,
                        'external-dns.alpha.kubernetes.io/hostname': props.endpoints.managerDns,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                postgresql: {
                    enabled: false,
                },
                secretVolumes: [
                    `${props.clusterIssuerName}-secret`,
                ],
            },
        });
        deploy_kong_cp_helm.node.addDependency(secrets_crd_mount, kong_namespace, kong_cp_service_account);
    }
}
exports.KongControlPlaneEKS = KongControlPlaneEKS;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi8uLi8uLi9zcmMvcmVzb3VyY2VzL2Vrcy9oZWxtLWNoYXJ0cy9rb25nL2luZGV4LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLDZDQUF5RTtBQUN6RSwyQ0FBdUM7QUF1QnZDLE1BQWEsbUJBQW9CLFNBQVEsc0JBQVM7SUFDaEQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFrQjtRQUMxRCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLE1BQU0sV0FBVyxHQUFHLGFBQWEsQ0FBQztRQUNsQyxNQUFNLDBCQUEwQixHQUFHLDRCQUE0QixDQUFDO1FBRWhFLDRFQUE0RTtRQUM1RSxnREFBZ0Q7UUFDaEQsTUFBTTtRQUNOLGlGQUFpRjtRQUNqRiwwQ0FBMEM7UUFDMUMscUZBQXFGO1FBQ3JGLE1BQU07UUFDTixxRkFBcUY7UUFDckYsNENBQTRDO1FBQzVDLHFGQUFxRjtRQUNyRixNQUFNO1FBQ04seUZBQXlGO1FBQ3pGLDhDQUE4QztRQUM5QyxxRkFBcUY7UUFDckYsTUFBTTtRQUNOLHFGQUFxRjtRQUNyRiw0Q0FBNEM7UUFDNUMscUZBQXFGO1FBQ3JGLE1BQU07UUFFTixNQUFNLGNBQWMsR0FBRyxJQUFJLHFCQUFPLENBQUMsa0JBQWtCLENBQUMsSUFBSSxFQUFFLGVBQWUsRUFBRTtZQUMzRSxPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU87WUFDdEIsU0FBUyxFQUFFLElBQUk7WUFDZixRQUFRLEVBQUU7Z0JBQ1I7b0JBQ0UsVUFBVSxFQUFFLElBQUk7b0JBQ2hCLElBQUksRUFBRSxXQUFXO29CQUNqQixRQUFRLEVBQUUsRUFBRSxJQUFJLEVBQUUsS0FBSyxDQUFDLFNBQVMsRUFBRTtpQkFDcEM7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILE1BQU0sdUJBQXVCLEdBQUcsSUFBSSxxQkFBTyxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQzNFLE9BQU8sRUFBRSxLQUFLLENBQUMsT0FBTztZQUN0QixJQUFJLEVBQUUseUJBQXlCO1lBQy9CLFNBQVMsRUFBRSxLQUFLLENBQUMsU0FBUztTQUMzQixDQUFDLENBQUM7UUFFSCxNQUFNLFVBQVUsR0FBRyxtQkFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxNQUFNLENBQUM7UUFDekMsTUFBTSxXQUFXLEdBQUcsbUJBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDO1FBRTNDLHVCQUF1QixDQUFDLG9CQUFvQixDQUFDLElBQUkscUJBQU8sQ0FBQyxlQUFlLENBQUM7WUFDdkUsU0FBUyxFQUFFO2dCQUNULDBCQUEwQixVQUFVLElBQUksV0FBVyxXQUFXLEtBQUssQ0FBQyxtQkFBbUIsU0FBUztnQkFDaEcsR0FBRyxLQUFLLENBQUMsR0FBRyxDQUFDLE1BQU0sRUFBRSxhQUFhLEVBQUU7YUFDckM7WUFDRCxPQUFPLEVBQUU7Z0JBQ1AsK0JBQStCO2dCQUMvQiwrQkFBK0I7YUFDaEM7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLHVCQUF1QixDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsY0FBYyxDQUFDLENBQUM7UUFFM0QsTUFBTSxnQkFBZ0IsR0FBRyxJQUFJLHFCQUFPLENBQUMsa0JBQWtCLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUN4RSxPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU87WUFDdEIsU0FBUyxFQUFFLElBQUk7WUFDZixRQUFRLEVBQUU7Z0JBQ1I7b0JBQ0UsVUFBVSxFQUFFLG9CQUFvQjtvQkFDaEMsSUFBSSxFQUFFLGFBQWE7b0JBQ25CLFFBQVEsRUFBRTt3QkFDUixJQUFJLEVBQUUsS0FBSyxDQUFDLFVBQVU7d0JBQ3RCLFNBQVMsRUFBRSxLQUFLLENBQUMsU0FBUztxQkFDM0I7b0JBQ0QsSUFBSSxFQUFFO3dCQUNKLFVBQVUsRUFBRSxLQUFLLENBQUMsU0FBUyxDQUFDLGNBQWM7d0JBQzFDLFFBQVEsRUFBRTs0QkFDUixLQUFLLENBQUMsU0FBUyxDQUFDLFFBQVE7NEJBQ3hCLEtBQUssQ0FBQyxTQUFTLENBQUMsVUFBVTs0QkFDMUIsS0FBSyxDQUFDLFNBQVMsQ0FBQyxVQUFVOzRCQUMxQixLQUFLLENBQUMsU0FBUyxDQUFDLFlBQVk7eUJBQzdCO3dCQUNELFFBQVEsRUFBRSxXQUFXO3dCQUNyQixTQUFTLEVBQUU7NEJBQ1QsS0FBSyxFQUFFLHdCQUF3Qjs0QkFDL0IsSUFBSSxFQUFFLHFCQUFxQjs0QkFDM0IsSUFBSSxFQUFFLEtBQUssQ0FBQyxpQkFBaUI7eUJBQzlCO3dCQUNELFdBQVcsRUFBRSxVQUFVO3dCQUN2QixVQUFVLEVBQUUsR0FBRyxLQUFLLENBQUMsaUJBQWlCLFNBQVM7d0JBQy9DLE1BQU0sRUFBRTs0QkFDTixhQUFhOzRCQUNiLGFBQWE7eUJBQ2Q7d0JBQ0QsVUFBVSxFQUFFOzRCQUNWLFNBQVMsRUFBRSxLQUFLOzRCQUNoQixJQUFJLEVBQUUsSUFBSTt5QkFDWDtxQkFDRjtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUVwRCx3RUFBd0U7UUFDeEUsZ0hBQWdIO1FBRWhILE1BQU0saUJBQWlCLEdBQUcsSUFBSSxxQkFBTyxDQUFDLGtCQUFrQixDQUFDLElBQUksRUFBRSxjQUFjLEVBQUU7WUFDN0UsT0FBTyxFQUFFLEtBQUssQ0FBQyxPQUFPO1lBQ3RCLFNBQVMsRUFBRSxJQUFJO1lBQ2YsUUFBUSxFQUFFO2dCQUNSO29CQUNFLFVBQVUsRUFBRSwrQkFBK0I7b0JBQzNDLElBQUksRUFBRSxxQkFBcUI7b0JBQzNCLFFBQVEsRUFBRTt3QkFDUixJQUFJLEVBQUUsMEJBQTBCO3dCQUNoQyxTQUFTLEVBQUUsS0FBSyxDQUFDLFNBQVM7cUJBQzNCO29CQUNELElBQUksRUFBRTt3QkFDSixRQUFRLEVBQUUsS0FBSzt3QkFDZixVQUFVLEVBQUU7NEJBQ1YsT0FBTyxFQUFFLGtCQUFrQixLQUFLLENBQUMsR0FBRyxDQUFDLE1BQU0sRUFBRSxhQUFhLHdOQUF3Tjt5QkFDblI7d0JBQ0QsYUFBYSxFQUFFOzRCQUNiO2dDQUNFLFVBQVUsRUFBRSxXQUFXO2dDQUN2QixJQUFJLEVBQUUsUUFBUTtnQ0FDZCxJQUFJLEVBQUU7b0NBQ0o7d0NBQ0UsVUFBVSxFQUFFLFlBQVk7d0NBQ3hCLEdBQUcsRUFBRSxZQUFZO3FDQUNsQjtvQ0FDRDt3Q0FDRSxVQUFVLEVBQUUsWUFBWTt3Q0FDeEIsR0FBRyxFQUFFLFlBQVk7cUNBQ2xCO29DQUNEO3dDQUNFLFVBQVUsRUFBRSxRQUFRO3dDQUNwQixHQUFHLEVBQUUsUUFBUTtxQ0FDZDtvQ0FDRDt3Q0FDRSxVQUFVLEVBQUUsUUFBUTt3Q0FDcEIsR0FBRyxFQUFFLFFBQVE7cUNBQ2Q7aUNBTUY7NkJBQ0Y7eUJBQ0Y7cUJBQ0Y7aUJBQ0Y7Z0JBQ0Q7b0JBQ0UsVUFBVSxFQUFFLCtCQUErQjtvQkFDM0MsSUFBSSxFQUFFLHFCQUFxQjtvQkFDM0IsUUFBUSxFQUFFO3dCQUNSLElBQUksRUFBRSxLQUFLLENBQUMsbUJBQW1CO3dCQUMvQixTQUFTLEVBQUUsS0FBSyxDQUFDLFNBQVM7cUJBQzNCO29CQUNELElBQUksRUFBRTt3QkFDSixRQUFRLEVBQUUsS0FBSzt3QkFDZixhQUFhLEVBQUU7NEJBQ2I7Z0NBQ0UsVUFBVSxFQUFFLEtBQUssQ0FBQyxtQkFBbUI7Z0NBQ3JDLElBQUksRUFBRSxRQUFRO2dDQUNkLElBQUksRUFBRTtvQ0FDSjt3Q0FDRSxVQUFVLEVBQUUsS0FBSyxDQUFDLG1CQUFtQjt3Q0FDckMsR0FBRyxFQUFFLFNBQVM7cUNBQ2Y7aUNBQ0Y7NkJBQ0Y7eUJBQ0Y7d0JBQ0QsVUFBVSxFQUFFOzRCQUNWLE9BQU8sRUFBRSxrQkFBa0IsS0FBSyxDQUFDLG1CQUFtQixxQ0FBcUM7eUJBTzFGO3FCQUVGO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFFSCxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBRXJELE1BQU0sbUJBQW1CLEdBQUcsSUFBSSxxQkFBTyxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsWUFBWSxFQUFFO1lBQ3BFLEdBQUcsS0FBSyxDQUFDLFdBQVc7WUFDcEIsT0FBTyxFQUFFLEtBQUssQ0FBQyxPQUFPO1lBQ3RCLFVBQVUsRUFBRSwyQkFBMkI7WUFDdkMsS0FBSyxFQUFFLE1BQU07WUFDYixPQUFPLEVBQUUsTUFBTTtZQUNmLFNBQVMsRUFBRSxLQUFLLENBQUMsU0FBUztZQUMxQixPQUFPLEVBQUUsc0JBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO1lBQzdCLElBQUksRUFBRSxJQUFJO1lBQ1YsZUFBZSxFQUFFLElBQUk7WUFDckIsTUFBTSxFQUFFO2dCQUNOLGlCQUFpQixFQUFFO29CQUNqQixPQUFPLEVBQUUsS0FBSztpQkFNZjtnQkFDRCxXQUFXO2dCQUNYLHFDQUFxQztnQkFDckMsMkJBQTJCO2dCQUMzQixLQUFLO2dCQUNMLEdBQUcsRUFBRTtvQkFDSCw4QkFBOEIsRUFBRSxLQUFLO29CQUNyQyxRQUFRLEVBQUUsVUFBVTtvQkFDcEIsSUFBSSxFQUFFLGVBQWU7b0JBQ3JCLFlBQVksRUFBRSxLQUFLO29CQUNuQixZQUFZLEVBQUUsZ0JBQWdCLEtBQUssQ0FBQyxpQkFBaUIsaUJBQWlCO29CQUN0RSxnQkFBZ0IsRUFBRSxnQkFBZ0IsS0FBSyxDQUFDLGlCQUFpQixpQkFBaUI7b0JBQzFFLGVBQWUsRUFBRSxnQkFBZ0IsS0FBSyxDQUFDLGlCQUFpQixnQkFBZ0I7b0JBQ3hFLGNBQWMsRUFBRSxnQkFBZ0IsS0FBSyxDQUFDLGlCQUFpQixpQkFBaUI7b0JBQ3hFLGtCQUFrQixFQUFFLGdCQUFnQixLQUFLLENBQUMsaUJBQWlCLGlCQUFpQjtvQkFDNUUsYUFBYSxFQUFFLFdBQVcsS0FBSyxDQUFDLFNBQVMsQ0FBQyxVQUFVLEVBQUU7b0JBQ3RELGtCQUFrQixFQUFFLGdCQUFnQixLQUFLLENBQUMsaUJBQWlCLGlCQUFpQjtvQkFDNUUsc0JBQXNCLEVBQUUsZ0JBQWdCLEtBQUssQ0FBQyxpQkFBaUIsaUJBQWlCO29CQUNoRiwwQkFBMEIsRUFBRSxHQUFHLEtBQUssQ0FBQyxTQUFTLENBQUMsWUFBWSxNQUFNO29CQUNqRSxPQUFPLEVBQUU7d0JBQ1AsU0FBUyxFQUFFOzRCQUNULFlBQVksRUFBRTtnQ0FDWixJQUFJLEVBQUUsV0FBVztnQ0FDakIsR0FBRyxFQUFFLFlBQVk7NkJBQ2xCO3lCQUNGO3FCQUNGO29CQUNELFdBQVcsRUFBRTt3QkFDWCxTQUFTLEVBQUU7NEJBQ1QsWUFBWSxFQUFFO2dDQUNaLElBQUksRUFBRSxXQUFXO2dDQUNqQixHQUFHLEVBQUUsWUFBWTs2QkFDbEI7eUJBQ0Y7cUJBQ0Y7b0JBQ0QsV0FBVyxFQUFFO3dCQUNYLFNBQVMsRUFBRTs0QkFDVCxZQUFZLEVBQUU7Z0NBQ1osSUFBSSxFQUFFLFdBQVc7Z0NBQ2pCLEdBQUcsRUFBRSxRQUFROzZCQUNkO3lCQUNGO3FCQUNGO29CQUNELE9BQU8sRUFBRTt3QkFDUCxTQUFTLEVBQUU7NEJBQ1QsWUFBWSxFQUFFO2dDQUNaLElBQUksRUFBRSxXQUFXO2dDQUNqQixHQUFHLEVBQUUsUUFBUTs2QkFDZDt5QkFDRjtxQkFDRjtpQkFTRjtnQkFDRCxPQUFPLEVBQUU7b0JBQ1AsT0FBTyxFQUFFLElBQUk7b0JBQ2IsSUFBSSxFQUFFLGNBQWM7b0JBQ3BCLEdBQUcsRUFBRTt3QkFDSCxPQUFPLEVBQUUsSUFBSTt3QkFDYixXQUFXLEVBQUUsR0FBRztxQkFDakI7b0JBQ0QsV0FBVyxFQUFFO3dCQUNYLHNFQUFzRTt3QkFDdEUsOERBQThELEVBQUUsSUFBSTt3QkFDcEUsd0ZBQXdGO3dCQUN4RixtREFBbUQsRUFBRSxVQUFVO3dCQUMvRCxxREFBcUQsRUFBRSxpQkFBaUI7d0JBQ3hFLG1EQUFtRCxFQUFFLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxXQUFXLGNBQWM7d0JBQy9GLDJDQUEyQyxFQUFFLEtBQUssQ0FBQyxTQUFTLENBQUMsVUFBVTt3QkFDdkUsZ0ZBQWdGLEVBQUUsSUFBSTtxQkFDdkY7aUJBQ0Y7Z0JBQ0QsZ0JBQWdCLEVBQUU7b0JBQ2hCLE9BQU8sRUFBRSxJQUFJO29CQUNiLElBQUksRUFBRSxjQUFjO29CQUNwQixHQUFHLEVBQUU7d0JBQ0gsT0FBTyxFQUFFLElBQUk7d0JBQ2IsV0FBVyxFQUFFLEdBQUc7cUJBQ2pCO29CQUNELFdBQVcsRUFBRTt3QkFDWCxzRUFBc0U7d0JBQ3RFLDhEQUE4RCxFQUFFLElBQUk7d0JBQ3BFLDBGQUEwRjt3QkFDMUYsbURBQW1ELEVBQUUsVUFBVTt3QkFDL0QscURBQXFELEVBQUUsaUJBQWlCO3dCQUN4RSwyQ0FBMkMsRUFBRSxLQUFLLENBQUMsU0FBUyxDQUFDLFlBQVk7d0JBQ3pFLG1EQUFtRCxFQUFFLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxXQUFXLGdCQUFnQjt3QkFDakcsZ0ZBQWdGLEVBQUUsSUFBSTtxQkFDdkY7aUJBQ0Y7Z0JBQ0QsS0FBSyxFQUFFO29CQUNMLE9BQU8sRUFBRSxLQUFLO2lCQUNmO2dCQUNELEtBQUssRUFBRTtvQkFDTCxPQUFPLEVBQUUsSUFBSTtvQkFDYixJQUFJLEVBQUUsY0FBYztvQkFDcEIsTUFBTSxFQUFFO3dCQUNOLGdCQUFnQixFQUFFLElBQUk7cUJBQ3ZCO29CQUNELEdBQUcsRUFBRTt3QkFDSCxXQUFXLEVBQUUsR0FBRztxQkFFakI7b0JBQ0QsV0FBVyxFQUFFO3dCQUNYLHNFQUFzRTt3QkFDdEUsOERBQThELEVBQUUsSUFBSTt3QkFDcEUsc0ZBQXNGO3dCQUN0RixtREFBbUQsRUFBRSxVQUFVO3dCQUMvRCxxREFBcUQsRUFBRSxVQUFVO3dCQUNqRSx1RUFBdUUsRUFBRSxZQUFZO3dCQUNyRiwyQ0FBMkMsRUFBRSxLQUFLLENBQUMsU0FBUyxDQUFDLFFBQVE7d0JBQ3JFLG1EQUFtRCxFQUFFLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxXQUFXLFlBQVk7d0JBQzdGLGdGQUFnRixFQUFFLElBQUk7cUJBQ3ZGO2lCQUNGO2dCQUNELFVBQVUsRUFDVjtvQkFDRSxjQUFjLEVBQUU7d0JBQ2QsTUFBTSxFQUFFLEtBQUs7d0JBQ2IsSUFBSSxFQUFFLHVCQUF1QixDQUFDLGtCQUFrQjtxQkFDakQ7b0JBQ0Qsa0JBQWtCLEVBQ2xCO3dCQUNFOzRCQUNFLElBQUksRUFBRSxLQUFLLENBQUMsbUJBQW1COzRCQUMvQixHQUFHLEVBQUU7Z0NBQ0gsTUFBTSxFQUFFLDBCQUEwQjtnQ0FDbEMsUUFBUSxFQUFFLElBQUk7Z0NBQ2QsZ0JBQWdCLEVBQUU7b0NBQ2hCLG1CQUFtQixFQUFFLEtBQUssQ0FBQyxtQkFBbUI7aUNBQy9DOzZCQUNGO3lCQUNGO3dCQUNEOzRCQUNFLElBQUksRUFBRSxXQUFXOzRCQUNqQixHQUFHLEVBQUU7Z0NBQ0gsTUFBTSxFQUFFLDBCQUEwQjtnQ0FDbEMsUUFBUSxFQUFFLElBQUk7Z0NBQ2QsZ0JBQWdCLEVBQUU7b0NBQ2hCLG1CQUFtQixFQUFFLDBCQUEwQjtpQ0FDaEQ7NkJBQ0Y7eUJBQ0Y7cUJBQ0Y7b0JBQ0QsdUJBQXVCLEVBQUU7d0JBQ3ZCOzRCQUNFLElBQUksRUFBRSxLQUFLLENBQUMsbUJBQW1COzRCQUMvQixTQUFTLEVBQUUsc0JBQXNCOzRCQUNqQyxRQUFRLEVBQUUsSUFBSTt5QkFDZjt3QkFDRDs0QkFDRSxJQUFJLEVBQUUsV0FBVzs0QkFDakIsU0FBUyxFQUFFLGtCQUFrQjs0QkFDN0IsUUFBUSxFQUFFLElBQUk7eUJBQ2Y7cUJBQ0Y7aUJBQ0Y7Z0JBQ0QsVUFBVSxFQUFFO29CQUNWLE9BQU8sRUFBRSxJQUFJO29CQUNiLGNBQWMsRUFBRSxLQUFLLENBQUMsbUJBQW1CO2lCQUMxQztnQkFDRCxjQUFjLEVBQUU7b0JBQ2QsT0FBTyxFQUFFLElBQUk7aUJBQ2Q7Z0JBQ0QsT0FBTyxFQUFFO29CQUNQLE9BQU8sRUFBRSxJQUFJO29CQUNiLElBQUksRUFBRSxjQUFjO29CQUNwQixHQUFHLEVBQUU7d0JBQ0gsT0FBTyxFQUFFLElBQUk7d0JBQ2IsV0FBVyxFQUFFLEdBQUc7cUJBQ2pCO29CQUNELFVBQVU7b0JBQ1Ysb0JBQW9CO29CQUNwQixLQUFLO29CQUNMLFdBQVcsRUFBRTt3QkFDWCw4REFBOEQsRUFBRSxJQUFJO3dCQUNwRSxzRUFBc0U7d0JBQ3RFLHdGQUF3Rjt3QkFDeEYsbURBQW1ELEVBQUUsVUFBVTt3QkFDL0QscURBQXFELEVBQUUsVUFBVTt3QkFDakUsbURBQW1ELEVBQUUsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLFdBQVcsY0FBYzt3QkFDL0YsMkNBQTJDLEVBQUUsS0FBSyxDQUFDLFNBQVMsQ0FBQyxVQUFVO3dCQUN2RSxnRkFBZ0YsRUFBRSxJQUFJO3FCQUN2RjtpQkFDRjtnQkFDRCxVQUFVLEVBQUU7b0JBQ1YsT0FBTyxFQUFFLEtBQUs7aUJBQ2Y7Z0JBQ0QsYUFBYSxFQUFFO29CQUNiLEdBQUcsS0FBSyxDQUFDLGlCQUFpQixTQUFTO2lCQUNwQzthQUNGO1NBRUYsQ0FBQyxDQUFDO1FBRUgsbUJBQW1CLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxpQkFBaUIsRUFBRSxjQUFjLEVBQUUsdUJBQXVCLENBQUMsQ0FBQztJQUNyRyxDQUFDO0NBQ0Y7QUE1WkQsa0RBNFpDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgRHVyYXRpb24sIGF3c19pYW0sIGF3c19la3MsIGF3c19yZHMsIFN0YWNrIH0gZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBEbnNQcm9wcyB9IGZyb20gJy4uLy4uL2ludGVyZmFjZXMvZG5zLXByb3BzJztcbi8vIGltcG9ydCB7IEtvbmdDb250cm9sUGxhbmUgfSBmcm9tICcuLi8uLi8uLi9Lb25nQ29udHJvbFBsYW5lJztcblxuLy8gZXhwb3J0IGludGVyZmFjZSBEbnNQcm9wcyB7XG4vLyAgIHJlYWRvbmx5IGFkbWluRG5zIDogc3RyaW5nO1xuLy8gICByZWFkb25seSBjbHVzdGVyRG5zIDogc3RyaW5nO1xuLy8gICByZWFkb25seSB0ZWxlbWV0cnlEbnMgOiBzdHJpbmc7XG4vLyAgIHJlYWRvbmx5IG1hbmFnZXJEbnMgOiBzdHJpbmc7XG4vLyB9XG5leHBvcnQgaW50ZXJmYWNlIEtvbmdDUFByb3BzIHtcbiAgcmVhZG9ubHkgY2x1c3RlciA6IGF3c19la3MuQ2x1c3RlcjtcbiAgcmVhZG9ubHkgcmRzOiBhd3NfcmRzLkRhdGFiYXNlSW5zdGFuY2U7XG4gIHJlYWRvbmx5IG5hbWVzcGFjZTogc3RyaW5nO1xuICAvLyByZWFkb25seSBub2RlZ3JvdXAgOiBhd3NfZWtzLk5vZGVncm91cDtcbiAgcmVhZG9ubHkgbGljZW5zZV9zZWNyZXRfbmFtZSA6IHN0cmluZztcbiAgcmVhZG9ubHkgY2FjZXJ0bmFtZSA6IFN0cmluZztcbiAgcmVhZG9ubHkgZW5kcG9pbnRzOiBEbnNQcm9wcztcbiAgLy8gcmVhZG9ubHkgaG9zdGVkWm9uZU5hbWU6IHN0cmluZztcbiAgcmVhZG9ubHkgY2x1c3Rlcklzc3Vlck5hbWU6IFN0cmluZztcbiAgcmVhZG9ubHkgSGVsbU9wdGlvbnM/OiBhd3NfZWtzLkhlbG1DaGFydE9wdGlvbnM7XG59XG5cbmV4cG9ydCBjbGFzcyBLb25nQ29udHJvbFBsYW5lRUtTIGV4dGVuZHMgQ29uc3RydWN0IHtcbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEtvbmdDUFByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGNvbnN0IHJkc19zZWNyZXRzID0gJ3Jkcy1zZWNyZXRzJztcbiAgICBjb25zdCByZHNfc2VjcmV0c19wcm92aWRlcl9jbGFzcyA9ICdyZHMtc2VjcmV0cy1wcm92aWRlci1jbGFzcyc7XG5cbiAgICAvLyBjb25zdCByb3V0ZV81M196b25lID0gYXdzX3JvdXRlNTMuSG9zdGVkWm9uZS5mcm9tTG9va3VwKHRoaXMsICdNeVpvbmUnLCB7XG4gICAgLy8gICBkb21haW5OYW1lOiBwcm9wcy5lbmRwb2ludHMuaG9zdGVkWm9uZU5hbWUsXG4gICAgLy8gfSk7XG4gICAgLy8gY29uc3QgYWRtaW5fY2VydCA9IG5ldyBhd3NfY2VydGlmaWNhdGVtYW5hZ2VyLkNlcnRpZmljYXRlKHRoaXMsICdhZG1pbkNlcnQnLCB7XG4gICAgLy8gICBkb21haW5OYW1lOiBwcm9wcy5lbmRwb2ludHMuYWRtaW5EbnMsXG4gICAgLy8gICB2YWxpZGF0aW9uOiBhd3NfY2VydGlmaWNhdGVtYW5hZ2VyLkNlcnRpZmljYXRlVmFsaWRhdGlvbi5mcm9tRG5zKHJvdXRlXzUzX3pvbmUpLFxuICAgIC8vIH0pO1xuICAgIC8vIGNvbnN0IGNsdXN0ZXJfY2VydCA9IG5ldyBhd3NfY2VydGlmaWNhdGVtYW5hZ2VyLkNlcnRpZmljYXRlKHRoaXMsICdjbHVzdGVyQ2VydCcsIHtcbiAgICAvLyAgIGRvbWFpbk5hbWU6IHByb3BzLmVuZHBvaW50cy5jbHVzdGVyRG5zLFxuICAgIC8vICAgdmFsaWRhdGlvbjogYXdzX2NlcnRpZmljYXRlbWFuYWdlci5DZXJ0aWZpY2F0ZVZhbGlkYXRpb24uZnJvbURucyhyb3V0ZV81M196b25lKSxcbiAgICAvLyB9KTtcbiAgICAvLyBjb25zdCB0ZWxlbWV0cnlfY2VydCA9IG5ldyBhd3NfY2VydGlmaWNhdGVtYW5hZ2VyLkNlcnRpZmljYXRlKHRoaXMsICd0ZWxlbWV0cnlDZXJ0Jywge1xuICAgIC8vICAgZG9tYWluTmFtZTogcHJvcHMuZW5kcG9pbnRzLnRlbGVtZXRyeURucyxcbiAgICAvLyAgIHZhbGlkYXRpb246IGF3c19jZXJ0aWZpY2F0ZW1hbmFnZXIuQ2VydGlmaWNhdGVWYWxpZGF0aW9uLmZyb21EbnMocm91dGVfNTNfem9uZSksXG4gICAgLy8gfSk7XG4gICAgLy8gY29uc3QgbWFuYWdlcl9jZXJ0ID0gbmV3IGF3c19jZXJ0aWZpY2F0ZW1hbmFnZXIuQ2VydGlmaWNhdGUodGhpcywgJ21hbmFnZXJDZXJ0Jywge1xuICAgIC8vICAgZG9tYWluTmFtZTogcHJvcHMuZW5kcG9pbnRzLm1hbmFnZXJEbnMsXG4gICAgLy8gICB2YWxpZGF0aW9uOiBhd3NfY2VydGlmaWNhdGVtYW5hZ2VyLkNlcnRpZmljYXRlVmFsaWRhdGlvbi5mcm9tRG5zKHJvdXRlXzUzX3pvbmUpLFxuICAgIC8vIH0pO1xuXG4gICAgY29uc3Qga29uZ19uYW1lc3BhY2UgPSBuZXcgYXdzX2Vrcy5LdWJlcm5ldGVzTWFuaWZlc3QodGhpcywgJ0tvbmdOYW1lc3BhY2UnLCB7XG4gICAgICBjbHVzdGVyOiBwcm9wcy5jbHVzdGVyLFxuICAgICAgb3ZlcndyaXRlOiB0cnVlLFxuICAgICAgbWFuaWZlc3Q6IFtcbiAgICAgICAge1xuICAgICAgICAgIGFwaVZlcnNpb246ICd2MScsXG4gICAgICAgICAga2luZDogJ05hbWVzcGFjZScsXG4gICAgICAgICAgbWV0YWRhdGE6IHsgbmFtZTogcHJvcHMubmFtZXNwYWNlIH0sXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH0pO1xuXG4gICAgY29uc3Qga29uZ19jcF9zZXJ2aWNlX2FjY291bnQgPSBuZXcgYXdzX2Vrcy5TZXJ2aWNlQWNjb3VudCh0aGlzLCAnS29uZ0NwU0EnLCB7XG4gICAgICBjbHVzdGVyOiBwcm9wcy5jbHVzdGVyLFxuICAgICAgbmFtZTogJ2tvbmctY3Atc2VydmljZS1hY2NvdW50JyxcbiAgICAgIG5hbWVzcGFjZTogcHJvcHMubmFtZXNwYWNlLFxuICAgIH0pO1xuXG4gICAgY29uc3QgYXdzX3JlZ2lvbiA9IFN0YWNrLm9mKHRoaXMpLnJlZ2lvbjtcbiAgICBjb25zdCBhd3NfYWNjb3VudCA9IFN0YWNrLm9mKHRoaXMpLmFjY291bnQ7XG5cbiAgICBrb25nX2NwX3NlcnZpY2VfYWNjb3VudC5hZGRUb1ByaW5jaXBhbFBvbGljeShuZXcgYXdzX2lhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgcmVzb3VyY2VzOiBbXG4gICAgICAgIGBhcm46YXdzOnNlY3JldHNtYW5hZ2VyOiR7YXdzX3JlZ2lvbn06JHthd3NfYWNjb3VudH06c2VjcmV0OiR7cHJvcHMubGljZW5zZV9zZWNyZXRfbmFtZX0tPz8/Pz8/YCxcbiAgICAgICAgYCR7cHJvcHMucmRzLnNlY3JldD8uc2VjcmV0RnVsbEFybn1gLFxuICAgICAgXSxcbiAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgJ3NlY3JldHNtYW5hZ2VyOkdldFNlY3JldFZhbHVlJyxcbiAgICAgICAgJ3NlY3JldHNtYW5hZ2VyOkRlc2NyaWJlU2VjcmV0JyxcbiAgICAgIF0sXG4gICAgfSkpO1xuXG4gICAga29uZ19jcF9zZXJ2aWNlX2FjY291bnQubm9kZS5hZGREZXBlbmRlbmN5KGtvbmdfbmFtZXNwYWNlKTtcblxuICAgIGNvbnN0IGtvbmdfY2VydGlmaWNhdGUgPSBuZXcgYXdzX2Vrcy5LdWJlcm5ldGVzTWFuaWZlc3QodGhpcywgJ0tvbmdDZXJ0Jywge1xuICAgICAgY2x1c3RlcjogcHJvcHMuY2x1c3RlcixcbiAgICAgIG92ZXJ3cml0ZTogdHJ1ZSxcbiAgICAgIG1hbmlmZXN0OiBbXG4gICAgICAgIHtcbiAgICAgICAgICBhcGlWZXJzaW9uOiAnY2VydC1tYW5hZ2VyLmlvL3YxJyxcbiAgICAgICAgICBraW5kOiAnQ2VydGlmaWNhdGUnLFxuICAgICAgICAgIG1ldGFkYXRhOiB7XG4gICAgICAgICAgICBuYW1lOiBwcm9wcy5jYWNlcnRuYW1lLFxuICAgICAgICAgICAgbmFtZXNwYWNlOiBwcm9wcy5uYW1lc3BhY2UsXG4gICAgICAgICAgfSxcbiAgICAgICAgICBzcGVjOiB7XG4gICAgICAgICAgICBjb21tb25OYW1lOiBwcm9wcy5lbmRwb2ludHMuaG9zdGVkWm9uZU5hbWUsXG4gICAgICAgICAgICBkbnNOYW1lczogW1xuICAgICAgICAgICAgICBwcm9wcy5lbmRwb2ludHMuYWRtaW5EbnMsXG4gICAgICAgICAgICAgIHByb3BzLmVuZHBvaW50cy5jbHVzdGVyRG5zLFxuICAgICAgICAgICAgICBwcm9wcy5lbmRwb2ludHMubWFuYWdlckRucyxcbiAgICAgICAgICAgICAgcHJvcHMuZW5kcG9pbnRzLnRlbGVtZXRyeURucyxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBkdXJhdGlvbjogJzIxNjBoMG0wcycsXG4gICAgICAgICAgICBpc3N1ZXJSZWY6IHtcbiAgICAgICAgICAgICAgZ3JvdXA6ICdhd3NwY2EuY2VydC1tYW5hZ2VyLmlvJyxcbiAgICAgICAgICAgICAga2luZDogJ0FXU1BDQUNsdXN0ZXJJc3N1ZXInLFxuICAgICAgICAgICAgICBuYW1lOiBwcm9wcy5jbHVzdGVySXNzdWVyTmFtZSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICByZW5ld0JlZm9yZTogJzM2MGgwbTBzJyxcbiAgICAgICAgICAgIHNlY3JldE5hbWU6IGAke3Byb3BzLmNsdXN0ZXJJc3N1ZXJOYW1lfS1zZWNyZXRgLFxuICAgICAgICAgICAgdXNhZ2VzOiBbXG4gICAgICAgICAgICAgICdzZXJ2ZXIgYXV0aCcsXG4gICAgICAgICAgICAgICdjbGllbnQgYXV0aCcsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgcHJpdmF0ZUtleToge1xuICAgICAgICAgICAgICBhbGdvcml0aG06ICdSU0EnLFxuICAgICAgICAgICAgICBzaXplOiAyMDQ4LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICB9KTtcblxuICAgIGtvbmdfY2VydGlmaWNhdGUubm9kZS5hZGREZXBlbmRlbmN5KGtvbmdfbmFtZXNwYWNlKTtcblxuICAgIC8vIGNvbnN0IHNlY3JldE5hbWVQYXJ0cyA9IEZuLnNwbGl0KCctJywgcHJvcHMucmRzLnNlY3JldD8uc2VjcmV0TmFtZSEpO1xuICAgIC8vIGNvbnN0IHNlY3JldE5hbWVXaXRob3V0U3VmZml4ID0gRm4uam9pbignLScsIFtGbi5zZWxlY3QoMCwgc2VjcmV0TmFtZVBhcnRzKSwgRm4uc2VsZWN0KDEsIHNlY3JldE5hbWVQYXJ0cyldKTtcblxuICAgIGNvbnN0IHNlY3JldHNfY3JkX21vdW50ID0gbmV3IGF3c19la3MuS3ViZXJuZXRlc01hbmlmZXN0KHRoaXMsICdTZWNyZXRzTW91bnQnLCB7XG4gICAgICBjbHVzdGVyOiBwcm9wcy5jbHVzdGVyLFxuICAgICAgb3ZlcndyaXRlOiB0cnVlLFxuICAgICAgbWFuaWZlc3Q6IFtcbiAgICAgICAge1xuICAgICAgICAgIGFwaVZlcnNpb246ICdzZWNyZXRzLXN0b3JlLmNzaS54LWs4cy5pby92MScsXG4gICAgICAgICAga2luZDogJ1NlY3JldFByb3ZpZGVyQ2xhc3MnLFxuICAgICAgICAgIG1ldGFkYXRhOiB7XG4gICAgICAgICAgICBuYW1lOiByZHNfc2VjcmV0c19wcm92aWRlcl9jbGFzcyxcbiAgICAgICAgICAgIG5hbWVzcGFjZTogcHJvcHMubmFtZXNwYWNlLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgc3BlYzoge1xuICAgICAgICAgICAgcHJvdmlkZXI6ICdhd3MnLFxuICAgICAgICAgICAgcGFyYW1ldGVyczoge1xuICAgICAgICAgICAgICBvYmplY3RzOiBgLSBvYmplY3ROYW1lOiBcIiR7cHJvcHMucmRzLnNlY3JldD8uc2VjcmV0RnVsbEFybn1cIlxcbiAgam1lc1BhdGg6XFxuICAgIC0gcGF0aDogdXNlcm5hbWVcXG4gICAgICBvYmplY3RBbGlhczogZGJ1c2VybmFtZVxcbiAgICAtIHBhdGg6IHBhc3N3b3JkXFxuICAgICAgb2JqZWN0QWxpYXM6IGRicGFzc3dvcmRcXG4gICAgLSBwYXRoOiBkYm5hbWVcXG4gICAgICBvYmplY3RBbGlhczogZGJuYW1lXFxuICAgIC0gcGF0aDogaG9zdFxcbiAgICAgIG9iamVjdEFsaWFzOiBkYmhvc3RcXG5gLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIHNlY3JldE9iamVjdHM6IFtcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIHNlY3JldE5hbWU6IHJkc19zZWNyZXRzLFxuICAgICAgICAgICAgICAgIHR5cGU6ICdPcGFxdWUnLFxuICAgICAgICAgICAgICAgIGRhdGE6IFtcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgb2JqZWN0TmFtZTogJ2RidXNlcm5hbWUnLFxuICAgICAgICAgICAgICAgICAgICBrZXk6ICdkYnVzZXJuYW1lJyxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIG9iamVjdE5hbWU6ICdkYnBhc3N3b3JkJyxcbiAgICAgICAgICAgICAgICAgICAga2V5OiAnZGJwYXNzd29yZCcsXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBvYmplY3ROYW1lOiAnZGJuYW1lJyxcbiAgICAgICAgICAgICAgICAgICAga2V5OiAnZGJuYW1lJyxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIG9iamVjdE5hbWU6ICdkYmhvc3QnLFxuICAgICAgICAgICAgICAgICAgICBrZXk6ICdkYmhvc3QnLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgIC8vIEtvbmcgZG9lcyBub3Qgc3VwcG9ydHMgcG9ydCBvYmZ1c2NhdGlvblxuICAgICAgICAgICAgICAgICAgLy8ge1xuICAgICAgICAgICAgICAgICAgLy8gICBvYmplY3ROYW1lOiAnZGJwb3J0JyxcbiAgICAgICAgICAgICAgICAgIC8vICAga2V5OiAnZGJwb3J0JyxcbiAgICAgICAgICAgICAgICAgIC8vIH0sXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIGFwaVZlcnNpb246ICdzZWNyZXRzLXN0b3JlLmNzaS54LWs4cy5pby92MScsXG4gICAgICAgICAga2luZDogJ1NlY3JldFByb3ZpZGVyQ2xhc3MnLFxuICAgICAgICAgIG1ldGFkYXRhOiB7XG4gICAgICAgICAgICBuYW1lOiBwcm9wcy5saWNlbnNlX3NlY3JldF9uYW1lLFxuICAgICAgICAgICAgbmFtZXNwYWNlOiBwcm9wcy5uYW1lc3BhY2UsXG4gICAgICAgICAgfSxcbiAgICAgICAgICBzcGVjOiB7XG4gICAgICAgICAgICBwcm92aWRlcjogJ2F3cycsXG4gICAgICAgICAgICBzZWNyZXRPYmplY3RzOiBbXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBzZWNyZXROYW1lOiBwcm9wcy5saWNlbnNlX3NlY3JldF9uYW1lLFxuICAgICAgICAgICAgICAgIHR5cGU6ICdvcGFxdWUnLFxuICAgICAgICAgICAgICAgIGRhdGE6IFtcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgb2JqZWN0TmFtZTogcHJvcHMubGljZW5zZV9zZWNyZXRfbmFtZSxcbiAgICAgICAgICAgICAgICAgICAga2V5OiAnbGljZW5zZScsXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgcGFyYW1ldGVyczoge1xuICAgICAgICAgICAgICBvYmplY3RzOiBgLSBvYmplY3ROYW1lOiBcIiR7cHJvcHMubGljZW5zZV9zZWNyZXRfbmFtZX1cIlxcbiAgb2JqZWN0VHlwZTogXCJzZWNyZXRzbWFuYWdlclwiXFxuYCxcbiAgICAgICAgICAgICAgLy8gW1xuICAgICAgICAgICAgICAvLyAgIHtcbiAgICAgICAgICAgICAgLy8gICAgIG9iamVjdE5hbWU6IHByb3BzLmxpY2Vuc2Vfc2VjcmV0X25hbWUsXG4gICAgICAgICAgICAgIC8vICAgICBvYmplY3RUeXBlOiAnc2VjcmV0c21hbmFnZXInLFxuICAgICAgICAgICAgICAvLyAgIH0sXG4gICAgICAgICAgICAgIC8vIF0sXG4gICAgICAgICAgICB9XG4gICAgICAgICAgICAsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgfSk7XG5cbiAgICBzZWNyZXRzX2NyZF9tb3VudC5ub2RlLmFkZERlcGVuZGVuY3koa29uZ19uYW1lc3BhY2UpO1xuXG4gICAgY29uc3QgZGVwbG95X2tvbmdfY3BfaGVsbSA9IG5ldyBhd3NfZWtzLkhlbG1DaGFydCh0aGlzLCAnS29uZ0NwSGVsbScsIHtcbiAgICAgIC4uLnByb3BzLkhlbG1PcHRpb25zLFxuICAgICAgY2x1c3RlcjogcHJvcHMuY2x1c3RlcixcbiAgICAgIHJlcG9zaXRvcnk6ICdodHRwczovL2NoYXJ0cy5rb25naHEuY29tJyxcbiAgICAgIGNoYXJ0OiAna29uZycsXG4gICAgICByZWxlYXNlOiAna29uZycsXG4gICAgICBuYW1lc3BhY2U6IHByb3BzLm5hbWVzcGFjZSxcbiAgICAgIHRpbWVvdXQ6IER1cmF0aW9uLm1pbnV0ZXMoMTUpLFxuICAgICAgd2FpdDogdHJ1ZSxcbiAgICAgIGNyZWF0ZU5hbWVzcGFjZTogdHJ1ZSxcbiAgICAgIHZhbHVlczoge1xuICAgICAgICBpbmdyZXNzQ29udHJvbGxlcjoge1xuICAgICAgICAgIGVuYWJsZWQ6IGZhbHNlLFxuICAgICAgICAgIC8vIGluc3RhbGxDUkRzOiBmYWxzZSxcbiAgICAgICAgICAvLyBpbWFnZToge1xuICAgICAgICAgIC8vICAgcmVwb3NpdG9yeTogJ2tvbmcva3ViZXJuZXRlcy1pbmdyZXNzLWNvbnRyb2xsZXInLFxuICAgICAgICAgIC8vICAgdGFnOiAnMS4zLjEtYWxwaW5lJyxcbiAgICAgICAgICAvLyB9LFxuICAgICAgICB9LFxuICAgICAgICAvLyBpbWFnZToge1xuICAgICAgICAvLyAgIHJlcG9zaXRvcnk6ICdrb25nL2tvbmctZ2F0ZXdheScsXG4gICAgICAgIC8vICAgdGFnOiAnMi41LjAuMC1hbHBpbmUnLFxuICAgICAgICAvLyB9LFxuICAgICAgICBlbnY6IHtcbiAgICAgICAgICBjbHVzdGVyX2RhdGFfcGxhbmVfcHVyZ2VfZGVsYXk6IDg2NDAwLFxuICAgICAgICAgIGRhdGFiYXNlOiAncG9zdGdyZXMnLFxuICAgICAgICAgIHJvbGU6ICdjb250cm9sX3BsYW5lJyxcbiAgICAgICAgICBjbHVzdGVyX210bHM6ICdwa2knLFxuICAgICAgICAgIGNsdXN0ZXJfY2VydDogYC9ldGMvc2VjcmV0cy8ke3Byb3BzLmNsdXN0ZXJJc3N1ZXJOYW1lfS1zZWNyZXQvdGxzLmNydGAsXG4gICAgICAgICAgY2x1c3Rlcl9jZXJ0X2tleTogYC9ldGMvc2VjcmV0cy8ke3Byb3BzLmNsdXN0ZXJJc3N1ZXJOYW1lfS1zZWNyZXQvdGxzLmtleWAsXG4gICAgICAgICAgY2x1c3Rlcl9jYV9jZXJ0OiBgL2V0Yy9zZWNyZXRzLyR7cHJvcHMuY2x1c3Rlcklzc3Vlck5hbWV9LXNlY3JldC9jYS5jcnRgLFxuICAgICAgICAgIGFkbWluX3NzbF9jZXJ0OiBgL2V0Yy9zZWNyZXRzLyR7cHJvcHMuY2x1c3Rlcklzc3Vlck5hbWV9LXNlY3JldC90bHMuY3J0YCxcbiAgICAgICAgICBhZG1pbl9zc2xfY2VydF9rZXk6IGAvZXRjL3NlY3JldHMvJHtwcm9wcy5jbHVzdGVySXNzdWVyTmFtZX0tc2VjcmV0L3Rscy5rZXlgLFxuICAgICAgICAgIGFkbWluX2d1aV91cmw6IGBodHRwczovLyR7cHJvcHMuZW5kcG9pbnRzLm1hbmFnZXJEbnN9YCxcbiAgICAgICAgICBhZG1pbl9ndWlfc3NsX2NlcnQ6IGAvZXRjL3NlY3JldHMvJHtwcm9wcy5jbHVzdGVySXNzdWVyTmFtZX0tc2VjcmV0L3Rscy5jcnRgLFxuICAgICAgICAgIGFkbWluX2d1aV9zc2xfY2VydF9rZXk6IGAvZXRjL3NlY3JldHMvJHtwcm9wcy5jbHVzdGVySXNzdWVyTmFtZX0tc2VjcmV0L3Rscy5rZXlgLFxuICAgICAgICAgIGNsdXN0ZXJfdGVsZW1ldHJ5X2VuZHBvaW50OiBgJHtwcm9wcy5lbmRwb2ludHMudGVsZW1ldHJ5RG5zfTo0NDNgLFxuICAgICAgICAgIHBnX3VzZXI6IHtcbiAgICAgICAgICAgIHZhbHVlRnJvbToge1xuICAgICAgICAgICAgICBzZWNyZXRLZXlSZWY6IHtcbiAgICAgICAgICAgICAgICBuYW1lOiByZHNfc2VjcmV0cyxcbiAgICAgICAgICAgICAgICBrZXk6ICdkYnVzZXJuYW1lJyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgICBwZ19wYXNzd29yZDoge1xuICAgICAgICAgICAgdmFsdWVGcm9tOiB7XG4gICAgICAgICAgICAgIHNlY3JldEtleVJlZjoge1xuICAgICAgICAgICAgICAgIG5hbWU6IHJkc19zZWNyZXRzLFxuICAgICAgICAgICAgICAgIGtleTogJ2RicGFzc3dvcmQnLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHBnX2RhdGFiYXNlOiB7XG4gICAgICAgICAgICB2YWx1ZUZyb206IHtcbiAgICAgICAgICAgICAgc2VjcmV0S2V5UmVmOiB7XG4gICAgICAgICAgICAgICAgbmFtZTogcmRzX3NlY3JldHMsXG4gICAgICAgICAgICAgICAga2V5OiAnZGJuYW1lJyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgICBwZ19ob3N0OiB7XG4gICAgICAgICAgICB2YWx1ZUZyb206IHtcbiAgICAgICAgICAgICAgc2VjcmV0S2V5UmVmOiB7XG4gICAgICAgICAgICAgICAgbmFtZTogcmRzX3NlY3JldHMsXG4gICAgICAgICAgICAgICAga2V5OiAnZGJob3N0JyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgICAvLyBwZ19wb3J0OiB7XG4gICAgICAgICAgLy8gICB2YWx1ZUZyb206IHtcbiAgICAgICAgICAvLyAgICAgc2VjcmV0S2V5UmVmOiB7XG4gICAgICAgICAgLy8gICAgICAgbmFtZTogcmRzX3NlY3JldHMsXG4gICAgICAgICAgLy8gICAgICAga2V5OiAnZGJwb3J0JyxcbiAgICAgICAgICAvLyAgICAgfSxcbiAgICAgICAgICAvLyAgIH0sXG4gICAgICAgICAgLy8gfSxcbiAgICAgICAgfSxcbiAgICAgICAgY2x1c3Rlcjoge1xuICAgICAgICAgIGVuYWJsZWQ6IHRydWUsXG4gICAgICAgICAgdHlwZTogJ0xvYWRCYWxhbmNlcicsXG4gICAgICAgICAgdGxzOiB7XG4gICAgICAgICAgICBlbmFibGVkOiB0cnVlLFxuICAgICAgICAgICAgc2VydmljZVBvcnQ6IDQ0MyxcbiAgICAgICAgICB9LFxuICAgICAgICAgIGFubm90YXRpb25zOiB7XG4gICAgICAgICAgICAvLyAnc2VydmljZS5iZXRhLmt1YmVybmV0ZXMuaW8vYXdzLWxvYWQtYmFsYW5jZXItcHJveHktcHJvdG9jb2wnOiAnKicsXG4gICAgICAgICAgICAnc2VydmljZS5iZXRhLmt1YmVybmV0ZXMuaW8vYXdzLWxvYWQtYmFsYW5jZXItbmxiLXRhcmdldC10eXBlJzogJ2lwJyxcbiAgICAgICAgICAgIC8vICdzZXJ2aWNlLmJldGEua3ViZXJuZXRlcy5pby9hd3MtbG9hZC1iYWxhbmNlci1zc2wtY2VydCc6IGNsdXN0ZXJfY2VydC5jZXJ0aWZpY2F0ZUFybixcbiAgICAgICAgICAgICdzZXJ2aWNlLmJldGEua3ViZXJuZXRlcy5pby9hd3MtbG9hZC1iYWxhbmNlci10eXBlJzogJ2V4dGVybmFsJyxcbiAgICAgICAgICAgICdzZXJ2aWNlLmJldGEua3ViZXJuZXRlcy5pby9hd3MtbG9hZC1iYWxhbmNlci1zY2hlbWUnOiAnaW50ZXJuZXQtZmFjaW5nJyxcbiAgICAgICAgICAgICdzZXJ2aWNlLmJldGEua3ViZXJuZXRlcy5pby9hd3MtbG9hZC1iYWxhbmNlci1uYW1lJzogYCR7cHJvcHMuY2x1c3Rlci5jbHVzdGVyTmFtZX0tZWtzLWNsdXN0ZXJgLFxuICAgICAgICAgICAgJ2V4dGVybmFsLWRucy5hbHBoYS5rdWJlcm5ldGVzLmlvL2hvc3RuYW1lJzogcHJvcHMuZW5kcG9pbnRzLmNsdXN0ZXJEbnMsXG4gICAgICAgICAgICAnc2VydmljZS5iZXRhLmt1YmVybmV0ZXMuaW8vYXdzLWxvYWQtYmFsYW5jZXItY3Jvc3Mtem9uZS1sb2FkLWJhbGFuY2luZy1lbmFibGVkJzogdHJ1ZSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgICBjbHVzdGVydGVsZW1ldHJ5OiB7XG4gICAgICAgICAgZW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgICB0eXBlOiAnTG9hZEJhbGFuY2VyJyxcbiAgICAgICAgICB0bHM6IHtcbiAgICAgICAgICAgIGVuYWJsZWQ6IHRydWUsXG4gICAgICAgICAgICBzZXJ2aWNlUG9ydDogNDQzLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgYW5ub3RhdGlvbnM6IHtcbiAgICAgICAgICAgIC8vICdzZXJ2aWNlLmJldGEua3ViZXJuZXRlcy5pby9hd3MtbG9hZC1iYWxhbmNlci1wcm94eS1wcm90b2NvbCc6ICcqJyxcbiAgICAgICAgICAgICdzZXJ2aWNlLmJldGEua3ViZXJuZXRlcy5pby9hd3MtbG9hZC1iYWxhbmNlci1ubGItdGFyZ2V0LXR5cGUnOiAnaXAnLFxuICAgICAgICAgICAgLy8gJ3NlcnZpY2UuYmV0YS5rdWJlcm5ldGVzLmlvL2F3cy1sb2FkLWJhbGFuY2VyLXNzbC1jZXJ0JzogdGVsZW1ldHJ5X2NlcnQuY2VydGlmaWNhdGVBcm4sXG4gICAgICAgICAgICAnc2VydmljZS5iZXRhLmt1YmVybmV0ZXMuaW8vYXdzLWxvYWQtYmFsYW5jZXItdHlwZSc6ICdleHRlcm5hbCcsXG4gICAgICAgICAgICAnc2VydmljZS5iZXRhLmt1YmVybmV0ZXMuaW8vYXdzLWxvYWQtYmFsYW5jZXItc2NoZW1lJzogJ2ludGVybmV0LWZhY2luZycsXG4gICAgICAgICAgICAnZXh0ZXJuYWwtZG5zLmFscGhhLmt1YmVybmV0ZXMuaW8vaG9zdG5hbWUnOiBwcm9wcy5lbmRwb2ludHMudGVsZW1ldHJ5RG5zLFxuICAgICAgICAgICAgJ3NlcnZpY2UuYmV0YS5rdWJlcm5ldGVzLmlvL2F3cy1sb2FkLWJhbGFuY2VyLW5hbWUnOiBgJHtwcm9wcy5jbHVzdGVyLmNsdXN0ZXJOYW1lfS1la3MtdGVsZW1ldHJ5YCxcbiAgICAgICAgICAgICdzZXJ2aWNlLmJldGEua3ViZXJuZXRlcy5pby9hd3MtbG9hZC1iYWxhbmNlci1jcm9zcy16b25lLWxvYWQtYmFsYW5jaW5nLWVuYWJsZWQnOiB0cnVlLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICAgIHByb3h5OiB7XG4gICAgICAgICAgZW5hYmxlZDogZmFsc2UsXG4gICAgICAgIH0sXG4gICAgICAgIGFkbWluOiB7XG4gICAgICAgICAgZW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgICB0eXBlOiAnTG9hZEJhbGFuY2VyJyxcbiAgICAgICAgICBsYWJlbHM6IHtcbiAgICAgICAgICAgICdlbmFibGUtbWV0cmljcyc6IHRydWUsXG4gICAgICAgICAgfSxcbiAgICAgICAgICB0bHM6IHtcbiAgICAgICAgICAgIHNlcnZpY2VQb3J0OiA0NDMsXG4gICAgICAgICAgICAvLyBvdmVycmlkZVNlcnZpY2VUYXJnZXRQb3J0OiA0NDMsXG4gICAgICAgICAgfSxcbiAgICAgICAgICBhbm5vdGF0aW9uczoge1xuICAgICAgICAgICAgLy8gJ3NlcnZpY2UuYmV0YS5rdWJlcm5ldGVzLmlvL2F3cy1sb2FkLWJhbGFuY2VyLXByb3h5LXByb3RvY29sJzogJyonLFxuICAgICAgICAgICAgJ3NlcnZpY2UuYmV0YS5rdWJlcm5ldGVzLmlvL2F3cy1sb2FkLWJhbGFuY2VyLW5sYi10YXJnZXQtdHlwZSc6ICdpcCcsXG4gICAgICAgICAgICAvLyAnc2VydmljZS5iZXRhLmt1YmVybmV0ZXMuaW8vYXdzLWxvYWQtYmFsYW5jZXItc3NsLWNlcnQnOiBhZG1pbl9jZXJ0LmNlcnRpZmljYXRlQXJuLFxuICAgICAgICAgICAgJ3NlcnZpY2UuYmV0YS5rdWJlcm5ldGVzLmlvL2F3cy1sb2FkLWJhbGFuY2VyLXR5cGUnOiAnZXh0ZXJuYWwnLFxuICAgICAgICAgICAgJ3NlcnZpY2UuYmV0YS5rdWJlcm5ldGVzLmlvL2F3cy1sb2FkLWJhbGFuY2VyLXNjaGVtZSc6ICdpbnRlcm5hbCcsXG4gICAgICAgICAgICAnc2VydmljZS5iZXRhLmt1YmVybmV0ZXMuaW8vYXdzLWxvYWQtYmFsYW5jZXItYWRkaXRpb25hbC1yZXNvdXJjZS10YWdzJzogJ1R5cGU9YWRtaW4nLFxuICAgICAgICAgICAgJ2V4dGVybmFsLWRucy5hbHBoYS5rdWJlcm5ldGVzLmlvL2hvc3RuYW1lJzogcHJvcHMuZW5kcG9pbnRzLmFkbWluRG5zLFxuICAgICAgICAgICAgJ3NlcnZpY2UuYmV0YS5rdWJlcm5ldGVzLmlvL2F3cy1sb2FkLWJhbGFuY2VyLW5hbWUnOiBgJHtwcm9wcy5jbHVzdGVyLmNsdXN0ZXJOYW1lfS1la3MtYWRtaW5gLFxuICAgICAgICAgICAgJ3NlcnZpY2UuYmV0YS5rdWJlcm5ldGVzLmlvL2F3cy1sb2FkLWJhbGFuY2VyLWNyb3NzLXpvbmUtbG9hZC1iYWxhbmNpbmctZW5hYmxlZCc6IHRydWUsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgICAgZGVwbG95bWVudDpcbiAgICAgICAge1xuICAgICAgICAgIHNlcnZpY2VBY2NvdW50OiB7XG4gICAgICAgICAgICBjcmVhdGU6IGZhbHNlLFxuICAgICAgICAgICAgbmFtZToga29uZ19jcF9zZXJ2aWNlX2FjY291bnQuc2VydmljZUFjY291bnROYW1lLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgdXNlckRlZmluZWRWb2x1bWVzOlxuICAgICAgICAgIFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgbmFtZTogcHJvcHMubGljZW5zZV9zZWNyZXRfbmFtZSxcbiAgICAgICAgICAgICAgY3NpOiB7XG4gICAgICAgICAgICAgICAgZHJpdmVyOiAnc2VjcmV0cy1zdG9yZS5jc2kuazhzLmlvJyxcbiAgICAgICAgICAgICAgICByZWFkT25seTogdHJ1ZSxcbiAgICAgICAgICAgICAgICB2b2x1bWVBdHRyaWJ1dGVzOiB7XG4gICAgICAgICAgICAgICAgICBzZWNyZXRQcm92aWRlckNsYXNzOiBwcm9wcy5saWNlbnNlX3NlY3JldF9uYW1lLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBuYW1lOiByZHNfc2VjcmV0cyxcbiAgICAgICAgICAgICAgY3NpOiB7XG4gICAgICAgICAgICAgICAgZHJpdmVyOiAnc2VjcmV0cy1zdG9yZS5jc2kuazhzLmlvJyxcbiAgICAgICAgICAgICAgICByZWFkT25seTogdHJ1ZSxcbiAgICAgICAgICAgICAgICB2b2x1bWVBdHRyaWJ1dGVzOiB7XG4gICAgICAgICAgICAgICAgICBzZWNyZXRQcm92aWRlckNsYXNzOiByZHNfc2VjcmV0c19wcm92aWRlcl9jbGFzcyxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICAgIHVzZXJEZWZpbmVkVm9sdW1lTW91bnRzOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIG5hbWU6IHByb3BzLmxpY2Vuc2Vfc2VjcmV0X25hbWUsXG4gICAgICAgICAgICAgIG1vdW50UGF0aDogJy9tbnQvbGljZW5zZV9zZWNyZXRzJyxcbiAgICAgICAgICAgICAgcmVhZE9ubHk6IHRydWUsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBuYW1lOiByZHNfc2VjcmV0cyxcbiAgICAgICAgICAgICAgbW91bnRQYXRoOiAnL21udC9yZHNfc2VjcmV0cycsXG4gICAgICAgICAgICAgIHJlYWRPbmx5OiB0cnVlLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICB9LFxuICAgICAgICBlbnRlcnByaXNlOiB7XG4gICAgICAgICAgZW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgICBsaWNlbnNlX3NlY3JldDogcHJvcHMubGljZW5zZV9zZWNyZXRfbmFtZSxcbiAgICAgICAgfSxcbiAgICAgICAgc2VydmljZU1vbml0b3I6IHtcbiAgICAgICAgICBlbmFibGVkOiB0cnVlLFxuICAgICAgICB9LFxuICAgICAgICBtYW5hZ2VyOiB7XG4gICAgICAgICAgZW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgICB0eXBlOiAnTG9hZEJhbGFuY2VyJyxcbiAgICAgICAgICB0bHM6IHtcbiAgICAgICAgICAgIGVuYWJsZWQ6IHRydWUsXG4gICAgICAgICAgICBzZXJ2aWNlUG9ydDogNDQzLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgLy8gaHR0cDoge1xuICAgICAgICAgIC8vICAgZW5hYmxlZDogZmFsc2UsXG4gICAgICAgICAgLy8gfSxcbiAgICAgICAgICBhbm5vdGF0aW9uczoge1xuICAgICAgICAgICAgJ3NlcnZpY2UuYmV0YS5rdWJlcm5ldGVzLmlvL2F3cy1sb2FkLWJhbGFuY2VyLW5sYi10YXJnZXQtdHlwZSc6ICdpcCcsXG4gICAgICAgICAgICAvLyAnc2VydmljZS5iZXRhLmt1YmVybmV0ZXMuaW8vYXdzLWxvYWQtYmFsYW5jZXItcHJveHktcHJvdG9jb2wnOiAnKicsXG4gICAgICAgICAgICAvLyAnc2VydmljZS5iZXRhLmt1YmVybmV0ZXMuaW8vYXdzLWxvYWQtYmFsYW5jZXItc3NsLWNlcnQnOiBtYW5hZ2VyX2NlcnQuY2VydGlmaWNhdGVBcm4sXG4gICAgICAgICAgICAnc2VydmljZS5iZXRhLmt1YmVybmV0ZXMuaW8vYXdzLWxvYWQtYmFsYW5jZXItdHlwZSc6ICdleHRlcm5hbCcsXG4gICAgICAgICAgICAnc2VydmljZS5iZXRhLmt1YmVybmV0ZXMuaW8vYXdzLWxvYWQtYmFsYW5jZXItc2NoZW1lJzogJ2ludGVybmFsJyxcbiAgICAgICAgICAgICdzZXJ2aWNlLmJldGEua3ViZXJuZXRlcy5pby9hd3MtbG9hZC1iYWxhbmNlci1uYW1lJzogYCR7cHJvcHMuY2x1c3Rlci5jbHVzdGVyTmFtZX0tZWtzLW1hbmFnZXJgLFxuICAgICAgICAgICAgJ2V4dGVybmFsLWRucy5hbHBoYS5rdWJlcm5ldGVzLmlvL2hvc3RuYW1lJzogcHJvcHMuZW5kcG9pbnRzLm1hbmFnZXJEbnMsXG4gICAgICAgICAgICAnc2VydmljZS5iZXRhLmt1YmVybmV0ZXMuaW8vYXdzLWxvYWQtYmFsYW5jZXItY3Jvc3Mtem9uZS1sb2FkLWJhbGFuY2luZy1lbmFibGVkJzogdHJ1ZSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgICBwb3N0Z3Jlc3FsOiB7XG4gICAgICAgICAgZW5hYmxlZDogZmFsc2UsXG4gICAgICAgIH0sXG4gICAgICAgIHNlY3JldFZvbHVtZXM6IFtcbiAgICAgICAgICBgJHtwcm9wcy5jbHVzdGVySXNzdWVyTmFtZX0tc2VjcmV0YCxcbiAgICAgICAgXSxcbiAgICAgIH0sXG5cbiAgICB9KTtcblxuICAgIGRlcGxveV9rb25nX2NwX2hlbG0ubm9kZS5hZGREZXBlbmRlbmN5KHNlY3JldHNfY3JkX21vdW50LCBrb25nX25hbWVzcGFjZSwga29uZ19jcF9zZXJ2aWNlX2FjY291bnQpO1xuICB9XG59Il19