"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const fs = require("fs");
const path = require("path");
const YAML = require("yaml");
const eks = require("../lib");
const kubectl_layer_1 = require("../lib/kubectl-layer");
const util_1 = require("./util");
// tslint:disable:max-line-length
const CLUSTER_VERSION = eks.KubernetesVersion.V1_16;
module.exports = {
    'a default cluster spans all subnets'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: false, defaultCapacity: 0, version: CLUSTER_VERSION });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EKS::Cluster', {
            ResourcesVpcConfig: {
                SubnetIds: [
                    { Ref: 'VPCPublicSubnet1SubnetB4246D30' },
                    { Ref: 'VPCPublicSubnet2Subnet74179F39' },
                    { Ref: 'VPCPrivateSubnet1Subnet8BCA10E0' },
                    { Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A' },
                ],
            },
        }));
        test.done();
    },
    'create custom cluster correctly in any aws region'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'stack', { env: { region: 'us-east-1' } });
        // WHEN
        const vpc = new ec2.Vpc(stack, 'VPC');
        new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: true, defaultCapacity: 0, version: CLUSTER_VERSION });
        const layer = kubectl_layer_1.KubectlLayer.getOrCreate(stack, {});
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-Cluster'));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Serverless::Application', {
            Location: {
                ApplicationId: 'arn:aws:serverlessrepo:us-east-1:903779448426:applications/lambda-layer-kubectl',
            },
        }));
        test.equal(layer.isChina(), false);
        test.done();
    },
    'create custom cluster correctly in any aws region in china'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'stack', { env: { region: 'cn-north-1' } });
        // WHEN
        const vpc = new ec2.Vpc(stack, 'VPC');
        new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: true, defaultCapacity: 0, version: CLUSTER_VERSION });
        new kubectl_layer_1.KubectlLayer(stack, 'NewLayer');
        const layer = kubectl_layer_1.KubectlLayer.getOrCreate(stack);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-Cluster'));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Serverless::Application', {
            Location: {
                ApplicationId: 'arn:aws-cn:serverlessrepo:cn-north-1:487369736442:applications/lambda-layer-kubectl',
            },
        }));
        test.equal(layer.isChina(), true);
        test.done();
    },
    'if "vpc" is not specified, vpc with default configuration will be created'(test) {
        // GIVEN
        const { stack } = util_1.testFixtureNoVpc();
        // WHEN
        new eks.Cluster(stack, 'cluster', { version: CLUSTER_VERSION });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC'));
        test.done();
    },
    'default capacity': {
        'x2 m5.large by default'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            // WHEN
            const cluster = new eks.Cluster(stack, 'cluster', { version: CLUSTER_VERSION });
            // THEN
            test.ok(cluster.defaultNodegroup);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EKS::Nodegroup', {
                InstanceTypes: [
                    'm5.large',
                ],
                ScalingConfig: {
                    DesiredSize: 2,
                    MaxSize: 2,
                    MinSize: 2,
                },
            }));
            test.done();
        },
        'quantity and type can be customized'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            // WHEN
            const cluster = new eks.Cluster(stack, 'cluster', {
                defaultCapacity: 10,
                defaultCapacityInstance: new ec2.InstanceType('m2.xlarge'),
                version: CLUSTER_VERSION,
            });
            // THEN
            test.ok(cluster.defaultNodegroup);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EKS::Nodegroup', {
                ScalingConfig: {
                    DesiredSize: 10,
                    MaxSize: 10,
                    MinSize: 10,
                },
            }));
            // expect(stack).to(haveResource('AWS::AutoScaling::LaunchConfiguration', { InstanceType: 'm2.xlarge' }));
            test.done();
        },
        'defaultCapacity=0 will not allocate at all'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            // WHEN
            const cluster = new eks.Cluster(stack, 'cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
            // THEN
            test.ok(!cluster.defaultCapacity);
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup'));
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::AutoScaling::LaunchConfiguration'));
            test.done();
        },
    },
    'creating a cluster tags the private VPC subnets'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: false, defaultCapacity: 0, version: CLUSTER_VERSION });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
            Tags: [
                { Key: 'aws-cdk:subnet-name', Value: 'Private' },
                { Key: 'aws-cdk:subnet-type', Value: 'Private' },
                { Key: 'kubernetes.io/role/internal-elb', Value: '1' },
                { Key: 'Name', Value: 'Stack/VPC/PrivateSubnet1' },
            ],
        }));
        test.done();
    },
    'creating a cluster tags the public VPC subnets'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        // WHEN
        new eks.Cluster(stack, 'Cluster', { vpc, kubectlEnabled: false, defaultCapacity: 0, version: CLUSTER_VERSION });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
            MapPublicIpOnLaunch: true,
            Tags: [
                { Key: 'aws-cdk:subnet-name', Value: 'Public' },
                { Key: 'aws-cdk:subnet-type', Value: 'Public' },
                { Key: 'kubernetes.io/role/elb', Value: '1' },
                { Key: 'Name', Value: 'Stack/VPC/PublicSubnet1' },
            ],
        }));
        test.done();
    },
    'adding capacity creates an ASG with tags'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc, kubectlEnabled: false,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // WHEN
        cluster.addCapacity('Default', {
            instanceType: new ec2.InstanceType('t2.medium'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup', {
            Tags: [
                {
                    Key: { 'Fn::Join': ['', ['kubernetes.io/cluster/', { Ref: 'ClusterEB0386A7' }]] },
                    PropagateAtLaunch: true,
                    Value: 'owned',
                },
                {
                    Key: 'Name',
                    PropagateAtLaunch: true,
                    Value: 'Stack/Cluster/Default',
                },
            ],
        }));
        test.done();
    },
    'create nodegroup with existing role'(test) {
        // GIVEN
        const { stack } = util_1.testFixtureNoVpc();
        // WHEN
        const cluster = new eks.Cluster(stack, 'cluster', {
            defaultCapacity: 10,
            defaultCapacityInstance: new ec2.InstanceType('m2.xlarge'),
            version: CLUSTER_VERSION,
        });
        const existingRole = new iam.Role(stack, 'ExistingRole', {
            assumedBy: new iam.AccountRootPrincipal(),
        });
        new eks.Nodegroup(stack, 'Nodegroup', {
            cluster,
            nodeRole: existingRole,
        });
        // THEN
        test.ok(cluster.defaultNodegroup);
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EKS::Nodegroup', {
            ScalingConfig: {
                DesiredSize: 10,
                MaxSize: 10,
                MinSize: 10,
            },
        }));
        test.done();
    },
    'adding bottlerocket capacity creates an ASG with tags'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            kubectlEnabled: false,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // WHEN
        cluster.addCapacity('Bottlerocket', {
            instanceType: new ec2.InstanceType('t2.medium'),
            machineImageType: eks.MachineImageType.BOTTLEROCKET,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup', {
            Tags: [
                {
                    Key: { 'Fn::Join': ['', ['kubernetes.io/cluster/', { Ref: 'ClusterEB0386A7' }]] },
                    PropagateAtLaunch: true,
                    Value: 'owned',
                },
                {
                    Key: 'Name',
                    PropagateAtLaunch: true,
                    Value: 'Stack/Cluster/Bottlerocket',
                },
            ],
        }));
        test.done();
    },
    'adding bottlerocket capacity with bootstrapOptions throws error'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            kubectlEnabled: false,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        test.throws(() => cluster.addCapacity('Bottlerocket', {
            instanceType: new ec2.InstanceType('t2.medium'),
            machineImageType: eks.MachineImageType.BOTTLEROCKET,
            bootstrapOptions: {},
        }), /bootstrapOptions is not supported for Bottlerocket/);
        test.done();
    },
    'exercise export/import'(test) {
        // GIVEN
        const { stack: stack1, vpc, app } = util_1.testFixture();
        const stack2 = new cdk.Stack(app, 'stack2', { env: { region: 'us-east-1' } });
        const cluster = new eks.Cluster(stack1, 'Cluster', {
            vpc,
            kubectlEnabled: false,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // WHEN
        const imported = eks.Cluster.fromClusterAttributes(stack2, 'Imported', {
            clusterArn: cluster.clusterArn,
            vpc: cluster.vpc,
            clusterEndpoint: cluster.clusterEndpoint,
            clusterName: cluster.clusterName,
            securityGroups: cluster.connections.securityGroups,
            clusterCertificateAuthorityData: cluster.clusterCertificateAuthorityData,
            clusterSecurityGroupId: cluster.clusterSecurityGroupId,
            clusterEncryptionConfigKeyArn: cluster.clusterEncryptionConfigKeyArn,
        });
        // this should cause an export/import
        new cdk.CfnOutput(stack2, 'ClusterARN', { value: imported.clusterArn });
        // THEN
        assert_1.expect(stack2).toMatch({
            Outputs: {
                ClusterARN: {
                    Value: {
                        'Fn::ImportValue': 'Stack:ExportsOutputFnGetAttClusterEB0386A7Arn2F2E3C3F',
                    },
                },
            },
        });
        test.done();
    },
    'disabled features when kubectl is disabled'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            kubectlEnabled: false,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        test.throws(() => cluster.awsAuth, /Cannot define aws-auth mappings if kubectl is disabled/);
        test.throws(() => cluster.addResource('foo', {}), /Unable to perform this operation since kubectl is not enabled for this cluster/);
        test.throws(() => cluster.addCapacity('boo', { instanceType: new ec2.InstanceType('r5d.24xlarge'), mapRole: true }), /Cannot map instance IAM role to RBAC if kubectl is disabled for the cluster/);
        test.throws(() => new eks.HelmChart(stack, 'MyChart', { cluster, chart: 'chart' }), /Unable to perform this operation since kubectl is not enabled for this cluster/);
        test.throws(() => cluster.openIdConnectProvider, /Cannot specify a OpenID Connect Provider if kubectl is disabled/);
        test.done();
    },
    'mastersRole can be used to map an IAM role to "system:masters" (required kubectl)'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const role = new iam.Role(stack, 'role', { assumedBy: new iam.AnyPrincipal() });
        // WHEN
        new eks.Cluster(stack, 'Cluster', {
            vpc,
            mastersRole: role,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource(eks.KubernetesResource.RESOURCE_TYPE, {
            Manifest: {
                'Fn::Join': [
                    '',
                    [
                        '[{"apiVersion":"v1","kind":"ConfigMap","metadata":{"name":"aws-auth","namespace":"kube-system"},"data":{"mapRoles":"[{\\"rolearn\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'roleC7B7E775',
                                'Arn',
                            ],
                        },
                        '\\",\\"username\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'roleC7B7E775',
                                'Arn',
                            ],
                        },
                        '\\",\\"groups\\":[\\"system:masters\\"]}]","mapUsers":"[]","mapAccounts":"[]"}}]',
                    ],
                ],
            },
        }));
        test.done();
    },
    'addResource can be used to apply k8s manifests on this cluster'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // WHEN
        cluster.addResource('manifest1', { foo: 123 });
        cluster.addResource('manifest2', { bar: 123 }, { boor: [1, 2, 3] });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource(eks.KubernetesResource.RESOURCE_TYPE, {
            Manifest: '[{"foo":123}]',
        }));
        assert_1.expect(stack).to(assert_1.haveResource(eks.KubernetesResource.RESOURCE_TYPE, {
            Manifest: '[{"bar":123},{"boor":[1,2,3]}]',
        }));
        test.done();
    },
    'kubectl resources can be created in a separate stack'(test) {
        // GIVEN
        const { stack, app } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'cluster', { version: CLUSTER_VERSION }); // cluster is under stack2
        // WHEN resource is under stack2
        const stack2 = new cdk.Stack(app, 'stack2', { env: { account: stack.account, region: stack.region } });
        new eks.KubernetesResource(stack2, 'myresource', {
            cluster,
            manifest: [{ foo: 'bar' }],
        });
        // THEN
        app.synth(); // no cyclic dependency (see https://github.com/aws/aws-cdk/issues/7231)
        // expect a single resource in the 2nd stack
        assert_1.expect(stack2).toMatch({
            Resources: {
                myresource49C6D325: {
                    Type: 'Custom::AWSCDK-EKS-KubernetesResource',
                    Properties: {
                        ServiceToken: {
                            'Fn::ImportValue': 'Stack:ExportsOutputFnGetAttawscdkawseksKubectlProviderNestedStackawscdkawseksKubectlProviderNestedStackResourceA7AEBA6BOutputsStackawscdkawseksKubectlProviderframeworkonEvent8897FD9BArn49BEF20C',
                        },
                        Manifest: '[{\"foo\":\"bar\"}]',
                        ClusterName: { 'Fn::ImportValue': 'Stack:ExportsOutputRefclusterC5B25D0D98D553F5' },
                        RoleArn: { 'Fn::ImportValue': 'Stack:ExportsOutputFnGetAttclusterCreationRole2B3B5002ArnF05122FC' },
                    },
                    UpdateReplacePolicy: 'Delete',
                    DeletionPolicy: 'Delete',
                },
            },
        });
        test.done();
    },
    'when kubectl is enabled (default) adding capacity will automatically map its IAM role'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // WHEN
        cluster.addCapacity('default', {
            instanceType: new ec2.InstanceType('t2.nano'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource(eks.KubernetesResource.RESOURCE_TYPE, {
            Manifest: {
                'Fn::Join': [
                    '',
                    [
                        '[{"apiVersion":"v1","kind":"ConfigMap","metadata":{"name":"aws-auth","namespace":"kube-system"},"data":{"mapRoles":"[{\\"rolearn\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'ClusterdefaultInstanceRoleF20A29CD',
                                'Arn',
                            ],
                        },
                        '\\",\\"username\\":\\"system:node:{{EC2PrivateDNSName}}\\",\\"groups\\":[\\"system:bootstrappers\\",\\"system:nodes\\"]}]","mapUsers":"[]","mapAccounts":"[]"}}]',
                    ],
                ],
            },
        }));
        test.done();
    },
    'addCapacity will *not* map the IAM role if mapRole is false'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // WHEN
        cluster.addCapacity('default', {
            instanceType: new ec2.InstanceType('t2.nano'),
            mapRole: false,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.not(assert_1.haveResource(eks.KubernetesResource.RESOURCE_TYPE)));
        test.done();
    },
    'addCapacity will *not* map the IAM role if kubectl is disabled'(test) {
        // GIVEN
        const { stack, vpc } = util_1.testFixture();
        const cluster = new eks.Cluster(stack, 'Cluster', {
            vpc,
            kubectlEnabled: false,
            defaultCapacity: 0,
            version: CLUSTER_VERSION,
        });
        // WHEN
        cluster.addCapacity('default', {
            instanceType: new ec2.InstanceType('t2.nano'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.not(assert_1.haveResource(eks.KubernetesResource.RESOURCE_TYPE)));
        test.done();
    },
    'outputs': {
        'aws eks update-kubeconfig is the only output synthesized by default'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.Cluster(stack, 'Cluster', { version: CLUSTER_VERSION });
            // THEN
            const assembly = app.synth();
            const template = assembly.getStackByName(stack.stackName).template;
            test.deepEqual(template.Outputs, {
                ClusterConfigCommand43AAE40F: { Value: { 'Fn::Join': ['', ['aws eks update-kubeconfig --name ', { Ref: 'Cluster9EE0221C' }, ' --region us-east-1']] } },
                ClusterGetTokenCommand06AE992E: { Value: { 'Fn::Join': ['', ['aws eks get-token --cluster-name ', { Ref: 'Cluster9EE0221C' }, ' --region us-east-1']] } },
            });
            test.done();
        },
        'if masters role is defined, it should be included in the config command'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            const mastersRole = new iam.Role(stack, 'masters', { assumedBy: new iam.AccountRootPrincipal() });
            new eks.Cluster(stack, 'Cluster', {
                mastersRole,
                version: CLUSTER_VERSION,
            });
            // THEN
            const assembly = app.synth();
            const template = assembly.getStackByName(stack.stackName).template;
            test.deepEqual(template.Outputs, {
                ClusterConfigCommand43AAE40F: { Value: { 'Fn::Join': ['', ['aws eks update-kubeconfig --name ', { Ref: 'Cluster9EE0221C' }, ' --region us-east-1 --role-arn ', { 'Fn::GetAtt': ['masters0D04F23D', 'Arn'] }]] } },
                ClusterGetTokenCommand06AE992E: { Value: { 'Fn::Join': ['', ['aws eks get-token --cluster-name ', { Ref: 'Cluster9EE0221C' }, ' --region us-east-1 --role-arn ', { 'Fn::GetAtt': ['masters0D04F23D', 'Arn'] }]] } },
            });
            test.done();
        },
        'if `outputConfigCommand=false` will disabled the output'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            const mastersRole = new iam.Role(stack, 'masters', { assumedBy: new iam.AccountRootPrincipal() });
            new eks.Cluster(stack, 'Cluster', {
                mastersRole,
                outputConfigCommand: false,
                version: CLUSTER_VERSION,
            });
            // THEN
            const assembly = app.synth();
            const template = assembly.getStackByName(stack.stackName).template;
            test.ok(!template.Outputs); // no outputs
            test.done();
        },
        '`outputClusterName` can be used to synthesize an output with the cluster name'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.Cluster(stack, 'Cluster', {
                outputConfigCommand: false,
                outputClusterName: true,
                version: CLUSTER_VERSION,
            });
            // THEN
            const assembly = app.synth();
            const template = assembly.getStackByName(stack.stackName).template;
            test.deepEqual(template.Outputs, {
                ClusterClusterNameEB26049E: { Value: { Ref: 'Cluster9EE0221C' } },
            });
            test.done();
        },
        '`outputMastersRoleArn` can be used to synthesize an output with the arn of the masters role if defined'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.Cluster(stack, 'Cluster', {
                outputConfigCommand: false,
                outputMastersRoleArn: true,
                mastersRole: new iam.Role(stack, 'masters', { assumedBy: new iam.AccountRootPrincipal() }),
                version: CLUSTER_VERSION,
            });
            // THEN
            const assembly = app.synth();
            const template = assembly.getStackByName(stack.stackName).template;
            test.deepEqual(template.Outputs, {
                ClusterMastersRoleArnB15964B1: { Value: { 'Fn::GetAtt': ['masters0D04F23D', 'Arn'] } },
            });
            test.done();
        },
        'boostrap user-data': {
            'rendered by default for ASGs'(test) {
                // GIVEN
                const { app, stack } = util_1.testFixtureNoVpc();
                const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
                // WHEN
                cluster.addCapacity('MyCapcity', { instanceType: new ec2.InstanceType('m3.xlargs') });
                // THEN
                const template = app.synth().getStackByName(stack.stackName).template;
                const userData = template.Resources.ClusterMyCapcityLaunchConfig58583345.Properties.UserData;
                test.deepEqual(userData, { 'Fn::Base64': { 'Fn::Join': ['', ['#!/bin/bash\nset -o xtrace\n/etc/eks/bootstrap.sh ', { Ref: 'Cluster9EE0221C' }, ' --kubelet-extra-args "--node-labels lifecycle=OnDemand" --use-max-pods true\n/opt/aws/bin/cfn-signal --exit-code $? --stack Stack --resource ClusterMyCapcityASGD4CD8B97 --region us-east-1']] } });
                test.done();
            },
            'not rendered if bootstrap is disabled'(test) {
                // GIVEN
                const { app, stack } = util_1.testFixtureNoVpc();
                const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
                // WHEN
                cluster.addCapacity('MyCapcity', {
                    instanceType: new ec2.InstanceType('m3.xlargs'),
                    bootstrapEnabled: false,
                });
                // THEN
                const template = app.synth().getStackByName(stack.stackName).template;
                const userData = template.Resources.ClusterMyCapcityLaunchConfig58583345.Properties.UserData;
                test.deepEqual(userData, { 'Fn::Base64': '#!/bin/bash' });
                test.done();
            },
            // cursory test for options: see test.user-data.ts for full suite
            'bootstrap options'(test) {
                // GIVEN
                const { app, stack } = util_1.testFixtureNoVpc();
                const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
                // WHEN
                cluster.addCapacity('MyCapcity', {
                    instanceType: new ec2.InstanceType('m3.xlargs'),
                    bootstrapOptions: {
                        kubeletExtraArgs: '--node-labels FOO=42',
                    },
                });
                // THEN
                const template = app.synth().getStackByName(stack.stackName).template;
                const userData = template.Resources.ClusterMyCapcityLaunchConfig58583345.Properties.UserData;
                test.deepEqual(userData, { 'Fn::Base64': { 'Fn::Join': ['', ['#!/bin/bash\nset -o xtrace\n/etc/eks/bootstrap.sh ', { Ref: 'Cluster9EE0221C' }, ' --kubelet-extra-args "--node-labels lifecycle=OnDemand  --node-labels FOO=42" --use-max-pods true\n/opt/aws/bin/cfn-signal --exit-code $? --stack Stack --resource ClusterMyCapcityASGD4CD8B97 --region us-east-1']] } });
                test.done();
            },
            'spot instances': {
                'nodes labeled an tainted accordingly'(test) {
                    // GIVEN
                    const { app, stack } = util_1.testFixtureNoVpc();
                    const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
                    // WHEN
                    cluster.addCapacity('MyCapcity', {
                        instanceType: new ec2.InstanceType('m3.xlargs'),
                        spotPrice: '0.01',
                    });
                    // THEN
                    const template = app.synth().getStackByName(stack.stackName).template;
                    const userData = template.Resources.ClusterMyCapcityLaunchConfig58583345.Properties.UserData;
                    test.deepEqual(userData, { 'Fn::Base64': { 'Fn::Join': ['', ['#!/bin/bash\nset -o xtrace\n/etc/eks/bootstrap.sh ', { Ref: 'Cluster9EE0221C' }, ' --kubelet-extra-args "--node-labels lifecycle=Ec2Spot --register-with-taints=spotInstance=true:PreferNoSchedule" --use-max-pods true\n/opt/aws/bin/cfn-signal --exit-code $? --stack Stack --resource ClusterMyCapcityASGD4CD8B97 --region us-east-1']] } });
                    test.done();
                },
                'if kubectl is enabled, the interrupt handler is added'(test) {
                    // GIVEN
                    const { stack } = util_1.testFixtureNoVpc();
                    const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
                    // WHEN
                    cluster.addCapacity('MyCapcity', {
                        instanceType: new ec2.InstanceType('m3.xlarge'),
                        spotPrice: '0.01',
                    });
                    // THEN
                    assert_1.expect(stack).to(assert_1.haveResource(eks.HelmChart.RESOURCE_TYPE, {
                        Release: 'stackclusterchartspotinterrupthandlerdec62e07',
                        Chart: 'aws-node-termination-handler',
                        Values: '{\"nodeSelector.lifecycle\":\"Ec2Spot\"}',
                        Namespace: 'kube-system',
                        Repository: 'https://aws.github.io/eks-charts',
                    }));
                    test.done();
                },
                'its possible to add two capacities with spot instances and only one stop handler will be installed'(test) {
                    // GIVEN
                    const { stack } = util_1.testFixtureNoVpc();
                    const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
                    // WHEN
                    cluster.addCapacity('Spot1', {
                        instanceType: new ec2.InstanceType('m3.xlarge'),
                        spotPrice: '0.01',
                    });
                    cluster.addCapacity('Spot2', {
                        instanceType: new ec2.InstanceType('m4.xlarge'),
                        spotPrice: '0.01',
                    });
                    // THEN
                    assert_1.expect(stack).to(assert_1.countResources(eks.HelmChart.RESOURCE_TYPE, 1));
                    test.done();
                },
                'if kubectl is disabled, interrupt handler is not added'(test) {
                    // GIVEN
                    const { stack } = util_1.testFixtureNoVpc();
                    const cluster = new eks.Cluster(stack, 'Cluster', {
                        defaultCapacity: 0,
                        kubectlEnabled: false,
                        version: CLUSTER_VERSION,
                    });
                    // WHEN
                    cluster.addCapacity('MyCapcity', {
                        instanceType: new ec2.InstanceType('m3.xlargs'),
                        spotPrice: '0.01',
                    });
                    // THEN
                    assert_1.expect(stack).notTo(assert_1.haveResource(eks.KubernetesResource.RESOURCE_TYPE));
                    test.done();
                },
            },
        },
        'if bootstrap is disabled cannot specify options'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
            // THEN
            test.throws(() => cluster.addCapacity('MyCapcity', {
                instanceType: new ec2.InstanceType('m3.xlargs'),
                bootstrapEnabled: false,
                bootstrapOptions: { awsApiRetryAttempts: 10 },
            }), /Cannot specify "bootstrapOptions" if "bootstrapEnabled" is false/);
            test.done();
        },
        'EksOptimizedImage() with no nodeType always uses STANDARD with LATEST_KUBERNETES_VERSION'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            const LATEST_KUBERNETES_VERSION = '1.14';
            // WHEN
            new eks.EksOptimizedImage().getImage(stack);
            // THEN
            const assembly = app.synth();
            const parameters = assembly.getStackByName(stack.stackName).template.Parameters;
            test.ok(Object.entries(parameters).some(([k, v]) => k.startsWith('SsmParameterValueawsserviceeksoptimizedami') &&
                v.Default.includes('/amazon-linux-2/')), 'EKS STANDARD AMI should be in ssm parameters');
            test.ok(Object.entries(parameters).some(([k, v]) => k.startsWith('SsmParameterValueawsserviceeksoptimizedami') &&
                v.Default.includes(LATEST_KUBERNETES_VERSION)), 'LATEST_KUBERNETES_VERSION should be in ssm parameters');
            test.done();
        },
        'EksOptimizedImage() with specific kubernetesVersion return correct AMI'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.EksOptimizedImage({ kubernetesVersion: '1.15' }).getImage(stack);
            // THEN
            const assembly = app.synth();
            const parameters = assembly.getStackByName(stack.stackName).template.Parameters;
            test.ok(Object.entries(parameters).some(([k, v]) => k.startsWith('SsmParameterValueawsserviceeksoptimizedami') &&
                v.Default.includes('/amazon-linux-2/')), 'EKS STANDARD AMI should be in ssm parameters');
            test.ok(Object.entries(parameters).some(([k, v]) => k.startsWith('SsmParameterValueawsserviceeksoptimizedami') &&
                v.Default.includes('/1.15/')), 'kubernetesVersion should be in ssm parameters');
            test.done();
        },
        'EKS-Optimized AMI with GPU support when addCapacity'(test) {
            // GIVEN
            const { app, stack } = util_1.testFixtureNoVpc();
            // WHEN
            new eks.Cluster(stack, 'cluster', {
                defaultCapacity: 0,
                version: CLUSTER_VERSION,
            }).addCapacity('GPUCapacity', {
                instanceType: new ec2.InstanceType('g4dn.xlarge'),
            });
            // THEN
            const assembly = app.synth();
            const parameters = assembly.getStackByName(stack.stackName).template.Parameters;
            test.ok(Object.entries(parameters).some(([k, v]) => k.startsWith('SsmParameterValueawsserviceeksoptimizedami') && v.Default.includes('amazon-linux-2-gpu')), 'EKS AMI with GPU should be in ssm parameters');
            test.done();
        },
        'when using custom resource a creation role & policy is defined'(test) {
            // GIVEN
            const { stack } = util_1.testFixture();
            // WHEN
            new eks.Cluster(stack, 'MyCluster', {
                clusterName: 'my-cluster-name',
                version: CLUSTER_VERSION,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-Cluster', {
                Config: {
                    name: 'my-cluster-name',
                    roleArn: { 'Fn::GetAtt': ['MyClusterRoleBA20FE72', 'Arn'] },
                    version: '1.16',
                    resourcesVpcConfig: {
                        securityGroupIds: [{ 'Fn::GetAtt': ['MyClusterControlPlaneSecurityGroup6B658F79', 'GroupId'] }],
                        subnetIds: [
                            { Ref: 'MyClusterDefaultVpcPublicSubnet1SubnetFAE5A9B6' },
                            { Ref: 'MyClusterDefaultVpcPublicSubnet2SubnetF6D028A0' },
                            { Ref: 'MyClusterDefaultVpcPrivateSubnet1SubnetE1D0DCDB' },
                            { Ref: 'MyClusterDefaultVpcPrivateSubnet2Subnet11FEA8D0' },
                        ],
                    },
                },
            }));
            // role can be assumed by 3 lambda handlers (2 for the cluster resource and 1 for the kubernetes resource)
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
                AssumeRolePolicyDocument: {
                    Statement: [
                        {
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                AWS: [
                                    {
                                        'Fn::GetAtt': [
                                            'awscdkawseksClusterResourceProviderNestedStackawscdkawseksClusterResourceProviderNestedStackResource9827C454',
                                            'Outputs.StackawscdkawseksClusterResourceProviderOnEventHandlerServiceRole3AEE0A43Arn',
                                        ],
                                    },
                                    {
                                        'Fn::GetAtt': [
                                            'awscdkawseksClusterResourceProviderNestedStackawscdkawseksClusterResourceProviderNestedStackResource9827C454',
                                            'Outputs.StackawscdkawseksClusterResourceProviderIsCompleteHandlerServiceRole8E7F1C11Arn',
                                        ],
                                    },
                                ],
                            },
                        },
                        {
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                AWS: {
                                    'Fn::GetAtt': [
                                        'awscdkawseksKubectlProviderNestedStackawscdkawseksKubectlProviderNestedStackResourceA7AEBA6B',
                                        'Outputs.StackawscdkawseksKubectlProviderHandlerServiceRole2C52B3ECArn',
                                    ],
                                },
                            },
                        },
                    ],
                    Version: '2012-10-17',
                },
            }));
            // policy allows creation role to pass the cluster role and to interact with the cluster (given we know the explicit cluster name)
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'iam:PassRole',
                            Effect: 'Allow',
                            Resource: {
                                'Fn::GetAtt': [
                                    'MyClusterRoleBA20FE72',
                                    'Arn',
                                ],
                            },
                        },
                        {
                            Action: [
                                'ec2:DescribeSubnets',
                                'ec2:DescribeRouteTables',
                            ],
                            Effect: 'Allow',
                            Resource: '*',
                        },
                        {
                            Action: [
                                'eks:CreateCluster',
                                'eks:DescribeCluster',
                                'eks:DescribeUpdate',
                                'eks:DeleteCluster',
                                'eks:UpdateClusterVersion',
                                'eks:UpdateClusterConfig',
                                'eks:CreateFargateProfile',
                                'eks:TagResource',
                                'eks:UntagResource',
                            ],
                            Effect: 'Allow',
                            Resource: [{
                                    'Fn::Join': [
                                        '',
                                        [
                                            'arn:',
                                            {
                                                Ref: 'AWS::Partition',
                                            },
                                            ':eks:us-east-1:',
                                            {
                                                Ref: 'AWS::AccountId',
                                            },
                                            ':cluster/my-cluster-name',
                                        ],
                                    ],
                                }, {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'arn:',
                                            {
                                                Ref: 'AWS::Partition',
                                            },
                                            ':eks:us-east-1:',
                                            {
                                                Ref: 'AWS::AccountId',
                                            },
                                            ':cluster/my-cluster-name/*',
                                        ],
                                    ],
                                }],
                        },
                        {
                            Action: [
                                'eks:DescribeFargateProfile',
                                'eks:DeleteFargateProfile',
                            ],
                            Effect: 'Allow',
                            Resource: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':eks:us-east-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':fargateprofile/my-cluster-name/*',
                                    ],
                                ],
                            },
                        },
                        {
                            Action: ['iam:GetRole', 'iam:listAttachedRolePolicies'],
                            Effect: 'Allow',
                            Resource: '*',
                        },
                        {
                            Action: 'iam:CreateServiceLinkedRole',
                            Effect: 'Allow',
                            Resource: '*',
                        },
                    ],
                    Version: '2012-10-17',
                },
            }));
            test.done();
        },
        'if an explicit cluster name is not provided, the creation role policy is wider (allows interacting with all clusters)'(test) {
            // GIVEN
            const { stack } = util_1.testFixture();
            // WHEN
            new eks.Cluster(stack, 'MyCluster', { version: CLUSTER_VERSION });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'iam:PassRole',
                            Effect: 'Allow',
                            Resource: {
                                'Fn::GetAtt': [
                                    'MyClusterRoleBA20FE72',
                                    'Arn',
                                ],
                            },
                        },
                        {
                            Action: [
                                'ec2:DescribeSubnets',
                                'ec2:DescribeRouteTables',
                            ],
                            Effect: 'Allow',
                            Resource: '*',
                        },
                        {
                            Action: [
                                'eks:CreateCluster',
                                'eks:DescribeCluster',
                                'eks:DescribeUpdate',
                                'eks:DeleteCluster',
                                'eks:UpdateClusterVersion',
                                'eks:UpdateClusterConfig',
                                'eks:CreateFargateProfile',
                                'eks:TagResource',
                                'eks:UntagResource',
                            ],
                            Effect: 'Allow',
                            Resource: ['*'],
                        },
                        {
                            Action: [
                                'eks:DescribeFargateProfile',
                                'eks:DeleteFargateProfile',
                            ],
                            Effect: 'Allow',
                            Resource: '*',
                        },
                        {
                            Action: ['iam:GetRole', 'iam:listAttachedRolePolicies'],
                            Effect: 'Allow',
                            Resource: '*',
                        },
                        {
                            Action: 'iam:CreateServiceLinkedRole',
                            Effect: 'Allow',
                            Resource: '*',
                        },
                    ],
                    Version: '2012-10-17',
                },
            }));
            test.done();
        },
        'if helm charts are used, its resource provider is allowed to assume the creation role'(test) {
            // GIVEN
            const { stack } = util_1.testFixture();
            const cluster = new eks.Cluster(stack, 'MyCluster', {
                clusterName: 'my-cluster-name',
                version: CLUSTER_VERSION,
            });
            // WHEN
            cluster.addChart('MyChart', {
                chart: 'foo',
            });
            // THEN
            // role can be assumed by 4 principals: two for the cluster resource, one
            // for kubernetes resource and one for the helm resource.
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
                AssumeRolePolicyDocument: {
                    Statement: [
                        {
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                AWS: [
                                    {
                                        'Fn::GetAtt': [
                                            'awscdkawseksClusterResourceProviderNestedStackawscdkawseksClusterResourceProviderNestedStackResource9827C454',
                                            'Outputs.StackawscdkawseksClusterResourceProviderOnEventHandlerServiceRole3AEE0A43Arn',
                                        ],
                                    },
                                    {
                                        'Fn::GetAtt': [
                                            'awscdkawseksClusterResourceProviderNestedStackawscdkawseksClusterResourceProviderNestedStackResource9827C454',
                                            'Outputs.StackawscdkawseksClusterResourceProviderIsCompleteHandlerServiceRole8E7F1C11Arn',
                                        ],
                                    },
                                ],
                            },
                        },
                        {
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                AWS: {
                                    'Fn::GetAtt': [
                                        'awscdkawseksKubectlProviderNestedStackawscdkawseksKubectlProviderNestedStackResourceA7AEBA6B',
                                        'Outputs.StackawscdkawseksKubectlProviderHandlerServiceRole2C52B3ECArn',
                                    ],
                                },
                            },
                        },
                    ],
                    Version: '2012-10-17',
                },
            }));
            test.done();
        },
        'coreDnsComputeType will patch the coreDNS configuration to use a "fargate" compute type and restore to "ec2" upon removal'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            new eks.Cluster(stack, 'MyCluster', {
                coreDnsComputeType: eks.CoreDnsComputeType.FARGATE,
                version: CLUSTER_VERSION,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-KubernetesPatch', {
                ResourceName: 'deployment/coredns',
                ResourceNamespace: 'kube-system',
                ApplyPatchJson: '{"spec":{"template":{"metadata":{"annotations":{"eks.amazonaws.com/compute-type":"fargate"}}}}}',
                RestorePatchJson: '{"spec":{"template":{"metadata":{"annotations":{"eks.amazonaws.com/compute-type":"ec2"}}}}}',
                ClusterName: {
                    Ref: 'MyCluster8AD82BF8',
                },
                RoleArn: {
                    'Fn::GetAtt': [
                        'MyClusterCreationRoleB5FA4FF3',
                        'Arn',
                    ],
                },
            }));
            test.done();
        },
        'if openIDConnectProvider a new OpenIDConnectProvider resource is created and exposed'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
            // WHEN
            const provider = cluster.openIdConnectProvider;
            // THEN
            test.equal(provider, cluster.openIdConnectProvider, 'openIdConnect provider is different and created more than once.');
            assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDKOpenIdConnectProvider', {
                ServiceToken: {
                    'Fn::GetAtt': [
                        'CustomAWSCDKOpenIdConnectProviderCustomResourceProviderHandlerF2C543E0',
                        'Arn',
                    ],
                },
                ClientIDList: [
                    'sts.amazonaws.com',
                ],
                ThumbprintList: [
                    '9e99a48a9960b14926bb7f3b02e22da2b0ab7280',
                ],
                Url: {
                    'Fn::GetAtt': [
                        'Cluster9EE0221C',
                        'OpenIdConnectIssuerUrl',
                    ],
                },
            }));
            test.done();
        },
        'inference instances are supported'(test) {
            // GIVEN
            const { stack } = util_1.testFixtureNoVpc();
            const cluster = new eks.Cluster(stack, 'Cluster', { defaultCapacity: 0, version: CLUSTER_VERSION });
            // WHEN
            cluster.addCapacity('InferenceInstances', {
                instanceType: new ec2.InstanceType('inf1.2xlarge'),
                minCapacity: 1,
            });
            const fileContents = fs.readFileSync(path.join(__dirname, '../lib', 'addons/neuron-device-plugin.yaml'), 'utf8');
            const sanitized = YAML.parse(fileContents);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource(eks.KubernetesResource.RESOURCE_TYPE, {
                Manifest: JSON.stringify([sanitized]),
            }));
            test.done();
        },
        'kubectl resources are always created after all fargate profiles'(test) {
            // GIVEN
            const { stack, app } = util_1.testFixture();
            const cluster = new eks.Cluster(stack, 'Cluster', { version: CLUSTER_VERSION });
            // WHEN
            cluster.addFargateProfile('profile1', { selectors: [{ namespace: 'profile1' }] });
            cluster.addResource('resource1', { foo: 123 });
            cluster.addFargateProfile('profile2', { selectors: [{ namespace: 'profile2' }] });
            new eks.HelmChart(stack, 'chart', { cluster, chart: 'mychart' });
            cluster.addFargateProfile('profile3', { selectors: [{ namespace: 'profile3' }] });
            new eks.KubernetesPatch(stack, 'patch1', {
                cluster,
                applyPatch: { foo: 123 },
                restorePatch: { bar: 123 },
                resourceName: 'foo/bar',
            });
            cluster.addFargateProfile('profile4', { selectors: [{ namespace: 'profile4' }] });
            // THEN
            const template = app.synth().getStackArtifact(stack.artifactId).template;
            const barrier = template.Resources.ClusterKubectlReadyBarrier200052AF;
            test.deepEqual(barrier.DependsOn, [
                'Clusterfargateprofileprofile1PodExecutionRoleE85F87B5',
                'Clusterfargateprofileprofile129AEA3C6',
                'Clusterfargateprofileprofile2PodExecutionRole22670AF8',
                'Clusterfargateprofileprofile233B9A117',
                'Clusterfargateprofileprofile3PodExecutionRole475C0D8F',
                'Clusterfargateprofileprofile3D06F3076',
                'Clusterfargateprofileprofile4PodExecutionRole086057FB',
                'Clusterfargateprofileprofile4A0E3BBE8',
                'ClusterCreationRoleDefaultPolicyE8BDFC7B',
                'ClusterCreationRole360249B6',
                'Cluster9EE0221C',
            ]);
            const kubectlResources = ['chartF2447AFC', 'patch1B964AC93', 'Clustermanifestresource10B1C9505', 'ClusterAwsAuthmanifestFE51F8AE'];
            // check that all kubectl resources depend on the barrier
            for (const r of kubectlResources) {
                test.deepEqual(template.Resources[r].DependsOn, ['ClusterKubectlReadyBarrier200052AF']);
            }
            test.done();
        },
        'kubectl provider role is trusted to assume cluster creation role'(test) {
            // GIVEN
            const { stack, app } = util_1.testFixture();
            const c1 = new eks.Cluster(stack, 'Cluster1', { version: CLUSTER_VERSION });
            const c2 = new eks.Cluster(stack, 'Cluster2', { version: CLUSTER_VERSION });
            // WHEN
            // activate kubectl provider
            c1.addResource('c1a', { foo: 123 });
            c1.addResource('c1b', { foo: 123 });
            c2.addResource('c2', { foo: 123 });
            // THEN
            const template = app.synth().getStackArtifact(stack.artifactId).template;
            const creationRoleToKubectlRole = {
                Cluster1CreationRoleA231BE8D: 'Outputs.StackawscdkawseksKubectlProviderHandlerServiceRole2C52B3ECArn',
                Cluster2CreationRole9254EAB6: 'Outputs.StackawscdkawseksKubectlProviderHandlerServiceRole2C52B3ECArn',
            };
            // verify that the kubectl role appears as the 2nd IAM trust policy statement
            for (const [creationRole, kubectlRole] of Object.entries(creationRoleToKubectlRole)) {
                const trustPolicy = template.Resources[creationRole].Properties.AssumeRolePolicyDocument.Statement;
                test.equal(trustPolicy.length, 2, 'expecting the creation role\'s trust policy to include two statements');
                test.deepEqual(trustPolicy[1].Principal.AWS['Fn::GetAtt'][1], kubectlRole);
            }
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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