"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const eks = require("../lib");
const hello = require("./hello-k8s");
const util_1 = require("./util");
class EksClusterStack extends util_1.TestStack {
    constructor(scope, id) {
        super(scope, id);
        // allow all account users to assume this role in order to admin the cluster
        const mastersRole = new iam.Role(this, 'AdminRole', {
            assumedBy: new iam.AccountRootPrincipal(),
        });
        // just need one nat gateway to simplify the test
        const vpc = new ec2.Vpc(this, 'Vpc', { maxAzs: 3, natGateways: 1 });
        // create the cluster with a default nodegroup capacity
        const cluster = new eks.Cluster(this, 'Cluster', {
            vpc,
            mastersRole,
            defaultCapacity: 2,
            version: eks.KubernetesVersion.V1_16,
        });
        // fargate profile for resources in the "default" namespace
        cluster.addFargateProfile('default', {
            selectors: [{ namespace: 'default' }],
        });
        // add some capacity to the cluster. The IAM instance role will
        // automatically be mapped via aws-auth to allow nodes to join the cluster.
        cluster.addCapacity('Nodes', {
            instanceType: new ec2.InstanceType('t2.medium'),
            minCapacity: 3,
        });
        // add bottlerocket nodes
        cluster.addCapacity('BottlerocketNodes', {
            instanceType: new ec2.InstanceType('t3.small'),
            minCapacity: 2,
            machineImageType: eks.MachineImageType.BOTTLEROCKET,
        });
        // spot instances (up to 10)
        cluster.addCapacity('spot', {
            spotPrice: '0.1094',
            instanceType: new ec2.InstanceType('t3.large'),
            maxCapacity: 10,
            bootstrapOptions: {
                kubeletExtraArgs: '--node-labels foo=bar,goo=far',
                awsApiRetryAttempts: 5,
            },
        });
        // inference instances
        cluster.addCapacity('InferenceInstances', {
            instanceType: new ec2.InstanceType('inf1.2xlarge'),
            minCapacity: 1,
        });
        // add a extra nodegroup
        cluster.addNodegroup('extra-ng', {
            instanceType: new ec2.InstanceType('t3.small'),
            minSize: 1,
            // reusing the default capacity nodegroup instance role when available
            nodeRole: cluster.defaultCapacity ? cluster.defaultCapacity.role : undefined,
        });
        // apply a kubernetes manifest
        cluster.addResource('HelloApp', ...hello.resources);
        // deploy the Kubernetes dashboard through a helm chart
        cluster.addChart('dashboard', {
            chart: 'kubernetes-dashboard',
            repository: 'https://kubernetes.github.io/dashboard/',
        });
        // deploy an nginx ingress in a namespace
        const nginxNamespace = cluster.addResource('nginx-namespace', {
            apiVersion: 'v1',
            kind: 'Namespace',
            metadata: {
                name: 'nginx',
            },
        });
        const nginxIngress = cluster.addChart('nginx-ingress', {
            chart: 'nginx-ingress',
            repository: 'https://helm.nginx.com/stable',
            namespace: 'nginx',
            wait: true,
            createNamespace: false,
            timeout: core_1.Duration.minutes(15),
        });
        // make sure namespace is deployed before the chart
        nginxIngress.node.addDependency(nginxNamespace);
        // add a service account connected to a IAM role
        cluster.addServiceAccount('MyServiceAccount');
        new core_1.CfnOutput(this, 'ClusterEndpoint', { value: cluster.clusterEndpoint });
        new core_1.CfnOutput(this, 'ClusterArn', { value: cluster.clusterArn });
        new core_1.CfnOutput(this, 'ClusterCertificateAuthorityData', { value: cluster.clusterCertificateAuthorityData });
        new core_1.CfnOutput(this, 'ClusterSecurityGroupId', { value: cluster.clusterSecurityGroupId });
        new core_1.CfnOutput(this, 'ClusterEncryptionConfigKeyArn', { value: cluster.clusterEncryptionConfigKeyArn });
        new core_1.CfnOutput(this, 'ClusterName', { value: cluster.clusterName });
    }
}
const app = new core_1.App();
// since the EKS optimized AMI is hard-coded here based on the region,
// we need to actually pass in a specific region.
new EksClusterStack(app, 'aws-cdk-eks-cluster-test');
app.synth();
//# sourceMappingURL=data:application/json;base64,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