"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsAuth = void 0;
const core_1 = require("@aws-cdk/core");
const k8s_resource_1 = require("./k8s-resource");
/**
 * Manages mapping between IAM users and roles to Kubernetes RBAC configuration.
 *
 * @see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
 */
class AwsAuth extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.roleMappings = new Array();
        this.userMappings = new Array();
        this.accounts = new Array();
        this.stack = core_1.Stack.of(this);
        new k8s_resource_1.KubernetesResource(this, 'manifest', {
            cluster: props.cluster,
            manifest: [
                {
                    apiVersion: 'v1',
                    kind: 'ConfigMap',
                    metadata: {
                        name: 'aws-auth',
                        namespace: 'kube-system',
                    },
                    data: {
                        mapRoles: this.synthesizeMapRoles(),
                        mapUsers: this.synthesizeMapUsers(),
                        mapAccounts: this.synthesizeMapAccounts(),
                    },
                },
            ],
        });
    }
    /**
     * Adds the specified IAM role to the `system:masters` RBAC group, which means
     * that anyone that can assume it will be able to administer this Kubernetes system.
     *
     * @param role The IAM role to add
     * @param username Optional user (defaults to the role ARN)
     */
    addMastersRole(role, username) {
        this.addRoleMapping(role, {
            username,
            groups: ['system:masters'],
        });
    }
    /**
     * Adds a mapping between an IAM role to a Kubernetes user and groups.
     *
     * @param role The IAM role to map
     * @param mapping Mapping to k8s user name and groups
     */
    addRoleMapping(role, mapping) {
        this.roleMappings.push({ role, mapping });
    }
    /**
     * Adds a mapping between an IAM user to a Kubernetes user and groups.
     *
     * @param user The IAM user to map
     * @param mapping Mapping to k8s user name and groups
     */
    addUserMapping(user, mapping) {
        this.userMappings.push({ user, mapping });
    }
    /**
     * Additional AWS account to add to the aws-auth configmap.
     * @param accountId account number
     */
    addAccount(accountId) {
        this.accounts.push(accountId);
    }
    synthesizeMapRoles() {
        return core_1.Lazy.anyValue({
            produce: () => this.stack.toJsonString(this.roleMappings.map(m => {
                var _a;
                return ({
                    rolearn: m.role.roleArn,
                    username: (_a = m.mapping.username) !== null && _a !== void 0 ? _a : m.role.roleArn,
                    groups: m.mapping.groups,
                });
            })),
        });
    }
    synthesizeMapUsers() {
        return core_1.Lazy.anyValue({
            produce: () => this.stack.toJsonString(this.userMappings.map(m => {
                var _a;
                return ({
                    userarn: m.user.userArn,
                    username: (_a = m.mapping.username) !== null && _a !== void 0 ? _a : m.user.userArn,
                    groups: m.mapping.groups,
                });
            })),
        });
    }
    synthesizeMapAccounts() {
        return core_1.Lazy.anyValue({
            produce: () => this.stack.toJsonString(this.accounts),
        });
    }
}
exports.AwsAuth = AwsAuth;
//# sourceMappingURL=data:application/json;base64,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