"""
This module provides utilities for representing and restructuring RFC 2822 and
MIME email messages for storage and transport.
"""
import os
from email.message import EmailMessage, Message, MIMEPart
from email.parser import BytesParser, Parser
from email.policy import EmailPolicy
from email.policy import default as _default
from pathlib import Path
from typing import Any, Optional, Sequence, Union

from pragmail.clients import _ResponseData
from pragmail.utils import sanitize

FILE_EXTENTION = ".txt"


class TransportUtils:
    """Class containing methods for handling message objects."""

    @staticmethod
    def data_as_bytes(message: _ResponseData) -> bytes:
        """Bring out bytes-like object from response data.

        Args:
            message (_ResponseData): IMAP FETCH Command response.
                `_ResponseData` is generated by `imaplib.IMAP4` methods for
                communicating with the mail server.

        Raises:
            TypeError: Raised if message is not equal to _ResponseData.

        Returns:
            bytes: Email message as bytes-like object.
        """
        data: Union[bytes, None] = None
        if isinstance(message, list):
            dat = message[0]
            if isinstance(dat, bytes):
                data = dat
            elif isinstance(dat, tuple):
                data = dat[1]
        if data:
            return data
        raise TypeError("message must be of type _ResponseData.")

    @staticmethod
    def read_message(
        message: Union[bytes, str, _ResponseData],
        headersonly: bool = False,
        _class: type[Union[EmailMessage, MIMEPart]] = EmailMessage,
        policy: EmailPolicy = _default,
    ) -> Union[EmailMessage, Message, MIMEPart, None]:
        """Parse message object. This is identical to
        `pragmail.utils.read_message()` except that it supports sequence of
        bytes-like objects instead of file-like objects.

        Args:
            message (Union[bytes, str, _ResponseData]):
                The message instance.
            headersonly (bool, optional):
                Whether to stop parsing after reading the headers or not.
                Defaults to False.
            _class (type[Union[EmailMessage, MIMEPart]], optional):
                No-argument callable. Defaults to EmailMessage.
            policy (EmailPolicy, optional): Policy class with the approrate
                policy methods. Defaults to `email.policy.default`.

        Raises:
            TypeError: If message instance is not either a string or a
                bytes-like object.

        Returns:
            Union[EmailMessage, Message, MIMEPart]: Parsed message instance.
        """
        tutil = TransportUtils
        parser: Union[BytesParser, Parser]
        msg: Union[EmailMessage, Message, MIMEPart]

        if isinstance(message, list):
            message = tutil.data_as_bytes(message)

        if isinstance(message, bytes):
            parser = BytesParser(_class=_class, policy=policy)
            msg = parser.parsebytes(text=message, headersonly=headersonly)
        elif isinstance(message, str):
            parser = Parser(_class=_class, policy=policy)
            msg = parser.parsestr(text=message, headersonly=headersonly)
        else:
            raise TypeError(
                (
                    "message must be of type string, bytes, or "
                    "a sequence of response from a mail server."
                )
            )

        if isinstance(msg, (EmailMessage, Message, MIMEPart)):
            return msg
        return None  # pragma: no cover

    @staticmethod
    def xtract_attachments(
        message: Union[EmailMessage, MIMEPart],
        decode: bool = False,
    ) -> dict[str, dict]:
        """Extract message attachment.

        Args:
            message (Union[EmailMessage, MIMEPart]): The message instance.
            decode (bool, optional): Whether the message should be decoded or
                not. Defaults to False.

        Returns:
            dict[str, dict]: A dictionary containing the content-type, file
                name, and attachment data.
        """
        msg_attm: dict[str, dict] = {}

        for idx, attachment in enumerate(message.iter_attachments()):
            attm_title = f"attachment_{idx}"
            ctype = attachment.get_content_type()
            filename = sanitize(attachment.get_filename())
            buffer = attachment.get_payload(decode=decode)

            msg_attm[attm_title] = {
                "ctype": ctype,
                "filename": filename,
                "buffer": buffer,
            }

        return msg_attm

    @staticmethod
    def xtract_headers(message: Message) -> list[tuple[str, Any]]:
        """Retrieve the message's header fields and values.

        Args:
            message (Message): The message object, which must be an instance
                of `email.message.Message`.

        Returns:
            list[tuple[str, Any]]: Header fields and values sorted according
                to the original message.
        """
        return message.items()

    @staticmethod
    def xtract_payload(
        message: Union[EmailMessage, MIMEPart],
        preferencelist: Sequence[str] = ("related", "html", "plain"),
    ) -> Union[Message, None]:
        """Extract message content.

        Args:
            message (Union[EmailMessage, MIMEPart]): The message instance.
            preferencelist (Sequence[str], optional): The content-type to
                extract. Defaults to ("related", "html", "plain").

        Returns:
            Union[Message, None]: The message payload or None if it doesn't
                exist.
        """
        return message.get_body(preferencelist=preferencelist)

    @staticmethod
    def create_file(
        filename: Union[Path, str],
        content: Optional[Union[str, bytes]] = None,
    ) -> None:
        """Create a file with the specified name and write its content.

        Args:
            filename (Union[Path, str]): The file name.
            content (Optional[Union[str, bytes]], optional): The file content.
                Defaults to None.
        """
        if isinstance(filename, Path):
            fpath = filename
        else:
            fpath = Path(filename)

        if not fpath.suffixes:
            fpath = fpath.with_suffix(FILE_EXTENTION)

        fpath.touch()
        if content is not None:
            if isinstance(content, str):
                fpath.write_text(content)
            else:
                fpath.write_bytes(content)

    @staticmethod
    def create_directory(
        dirname: Union[Path, str],
        parents: bool = True,
        exist_ok: bool = True,
    ) -> None:
        """Create a new directory in a given path.

        Args:
            dirname (Union[Path, str]): path segements a.k.a path-like string.
            parents (bool, optional): Create parents if missing. Defaults to
                True.
            exist_ok (bool, optional): Ignore `FileExistsError`. Defaults to
                True.
        """
        if isinstance(dirname, Path):
            directory = dirname
        else:
            directory = Path(dirname)

        directory.mkdir(parents=parents, exist_ok=exist_ok)

    @staticmethod
    def write_to_file(filname: str, data: Any, mode: str = "w"):
        """Write binary or text.

        Args:
            filname (str): The file name. May be path segements.
            data (Any): The file data to write.
            mode (str, optional): The mode in which the file is opened.
                Defaults to "w".
        """
        with open(filname, mode=mode) as file:
            file.write(data)

    @staticmethod
    def save_attachments(
        attachments: dict[str, dict],
        dirpath: Union[Path, str] = ".",
    ) -> None:
        """Extract and save files attached to the email message. The files
        will be stored inside a subdirectory on the same path as the message
        file.

        Args:
            attachments (dict[str, dict]): Must be or similar to the
                dictionary created by
                `transports.MessageStructure.xtract_attachments()`.
            dirpath (Union[Path, str]): Path where the file will be saved.
                Defaults to ".".
        """
        for item in attachments.items():
            ctype = item[1].get("ctype")
            filename = item[1].get("filename")
            buffer = item[1].get("buffer")

            if ctype and filename and buffer:
                fpath = os.path.join(dirpath, filename)
                tpt = TransportUtils

                try:
                    tpt.write_to_file(fpath, buffer)
                except TypeError:  # pragma: no cover
                    tpt.write_to_file(fpath, buffer, "wb")


def save_to_disk(
    message: Union[bytes, str, list[Union[bytes, tuple[bytes, bytes]]]],
    filename: str,
    _class: type[Union[EmailMessage, MIMEPart]] = EmailMessage,
) -> None:
    """Disassemble and restructure message instance as a txt file. Attachments
    are saved on the same path —in a subdirectory. The message file and its
    subdirectory name will be the same.

    Args:
        message (Union[bytes, str]):
            The message object to be saved.
        filename (str):
            It can also be a path.
        _class (type[Union[EmailMessage, MIMEPart]], optional):
            No-argument class from `email.message`. Defaults to
            EmailMessage.
    """
    tpt = TransportUtils
    msg = tpt.read_message(message, _class=_class)

    if isinstance(msg, (EmailMessage, MIMEPart)):
        load = tpt.xtract_payload(msg)
        attm = tpt.xtract_attachments(msg)

    # Create a directory for the message attachments to be saved. If there are
    # no media file attached to the email message, create a directory anyway.
    fpath = Path(filename)
    dpath = fpath.with_suffix("")
    attm_dpath = os.path.join(dpath, dpath)

    tpt.create_directory(dpath)
    tpt.save_attachments(attm, attm_dpath)

    if load:
        tpt.create_file(filename, content=load.as_string())


if __name__ == "__main__":
    pass
