"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerDeploymentGroup = exports.InstanceTagSet = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const codedeploy_generated_1 = require("../codedeploy.generated");
const utils_1 = require("../utils");
const application_1 = require("./application");
const deployment_config_1 = require("./deployment-config");
const load_balancer_1 = require("./load-balancer");
/**
 * Represents a reference to a CodeDeploy EC2/on-premise Deployment Group.
 *
 * If you're managing the Deployment Group alongside the rest of your CDK resources,
 * use the {@link ServerDeploymentGroup} class.
 *
 * If you want to reference an already existing Deployment Group,
 * or one defined in a different CDK Stack,
 * use the {@link #import} method.
 */
class ServerDeploymentGroupBase extends cdk.Resource {
    constructor(scope, id, deploymentConfig, props) {
        super(scope, id, props);
        this.deploymentConfig = deploymentConfig || deployment_config_1.ServerDeploymentConfig.ONE_AT_A_TIME;
    }
}
class ImportedServerDeploymentGroup extends ServerDeploymentGroupBase {
    constructor(scope, id, props) {
        super(scope, id, props.deploymentConfig);
        this.role = undefined;
        this.autoScalingGroups = undefined;
        this.application = props.application;
        this.deploymentGroupName = props.deploymentGroupName;
        this.deploymentGroupArn = utils_1.arnForDeploymentGroup(props.application.applicationName, props.deploymentGroupName);
    }
}
/**
 * Represents a set of instance tag groups.
 *
 * An instance will match a set if it matches all of the groups in the set -
 * in other words, sets follow 'and' semantics.
 * You can have a maximum of 3 tag groups inside a set.
 *
 * @stability stable
 */
class InstanceTagSet {
    /**
     * @stability stable
     */
    constructor(...instanceTagGroups) {
        if (instanceTagGroups.length > 3) {
            throw new Error('An instance tag set can have a maximum of 3 instance tag groups, ' +
                `but ${instanceTagGroups.length} were provided`);
        }
        this._instanceTagGroups = instanceTagGroups;
    }
    /**
     * @stability stable
     */
    get instanceTagGroups() {
        return this._instanceTagGroups.slice();
    }
}
exports.InstanceTagSet = InstanceTagSet;
/**
 * A CodeDeploy Deployment Group that deploys to EC2/on-premise instances.
 *
 * @stability stable
 * @resource AWS::CodeDeploy::DeploymentGroup
 */
class ServerDeploymentGroup extends ServerDeploymentGroupBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        super(scope, id, props.deploymentConfig, {
            physicalName: props.deploymentGroupName,
        });
        this.application = props.application || new application_1.ServerApplication(this, 'Application');
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('codedeploy.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSCodeDeployRole')],
        });
        this._autoScalingGroups = props.autoScalingGroups || [];
        this.installAgent = props.installAgent === undefined ? true : props.installAgent;
        this.codeDeployBucket = s3.Bucket.fromBucketName(this, 'Bucket', `aws-codedeploy-${cdk.Stack.of(this).region}`);
        for (const asg of this._autoScalingGroups) {
            this.addCodeDeployAgentInstallUserData(asg);
        }
        this.alarms = props.alarms || [];
        const resource = new codedeploy_generated_1.CfnDeploymentGroup(this, 'Resource', {
            applicationName: this.application.applicationName,
            deploymentGroupName: this.physicalName,
            serviceRoleArn: this.role.roleArn,
            deploymentConfigName: props.deploymentConfig &&
                props.deploymentConfig.deploymentConfigName,
            autoScalingGroups: cdk.Lazy.list({ produce: () => this._autoScalingGroups.map(asg => asg.autoScalingGroupName) }, { omitEmpty: true }),
            loadBalancerInfo: this.loadBalancerInfo(props.loadBalancer),
            deploymentStyle: props.loadBalancer === undefined
                ? undefined
                : {
                    deploymentOption: 'WITH_TRAFFIC_CONTROL',
                },
            ec2TagSet: this.ec2TagSet(props.ec2InstanceTags),
            onPremisesTagSet: this.onPremiseTagSet(props.onPremiseInstanceTags),
            alarmConfiguration: cdk.Lazy.any({ produce: () => utils_1.renderAlarmConfiguration(this.alarms, props.ignorePollAlarmsFailure) }),
            autoRollbackConfiguration: cdk.Lazy.any({ produce: () => utils_1.renderAutoRollbackConfiguration(this.alarms, props.autoRollback) }),
        });
        this.deploymentGroupName = this.getResourceNameAttribute(resource.ref);
        this.deploymentGroupArn = this.getResourceArnAttribute(utils_1.arnForDeploymentGroup(this.application.applicationName, resource.ref), {
            service: 'codedeploy',
            resource: 'deploymentgroup',
            resourceName: `${this.application.applicationName}/${this.physicalName}`,
            sep: ':',
        });
    }
    /**
     * Import an EC2/on-premise Deployment Group defined either outside the CDK app, or in a different region.
     *
     * @param scope the parent Construct for this new Construct.
     * @param id the logical ID of this new Construct.
     * @param attrs the properties of the referenced Deployment Group.
     * @returns a Construct representing a reference to an existing Deployment Group
     * @stability stable
     */
    static fromServerDeploymentGroupAttributes(scope, id, attrs) {
        return new ImportedServerDeploymentGroup(scope, id, attrs);
    }
    /**
     * Adds an additional auto-scaling group to this Deployment Group.
     *
     * @param asg the auto-scaling group to add to this Deployment Group.
     * @stability stable
     */
    addAutoScalingGroup(asg) {
        this._autoScalingGroups.push(asg);
        this.addCodeDeployAgentInstallUserData(asg);
    }
    /**
     * Associates an additional alarm with this Deployment Group.
     *
     * @param alarm the alarm to associate with this Deployment Group.
     * @stability stable
     */
    addAlarm(alarm) {
        this.alarms.push(alarm);
    }
    /**
     * @stability stable
     */
    get autoScalingGroups() {
        return this._autoScalingGroups.slice();
    }
    addCodeDeployAgentInstallUserData(asg) {
        if (!this.installAgent) {
            return;
        }
        this.codeDeployBucket.grantRead(asg, 'latest/*');
        switch (asg.osType) {
            case ec2.OperatingSystemType.LINUX:
                asg.addUserData('PKG_CMD=`which yum 2>/dev/null`', 'if [ -z "$PKG_CMD" ]; then', 'PKG_CMD=apt-get', 'else', 'PKG=CMD=yum', 'fi', '$PKG_CMD update -y', '$PKG_CMD install -y ruby2.0', 'if [ $? -ne 0 ]; then', '$PKG_CMD install -y ruby', 'fi', '$PKG_CMD install -y awscli', 'TMP_DIR=`mktemp -d`', 'cd $TMP_DIR', `aws s3 cp s3://aws-codedeploy-${cdk.Stack.of(this).region}/latest/install . --region ${cdk.Stack.of(this).region}`, 'chmod +x ./install', './install auto', 'rm -fr $TMP_DIR');
                break;
            case ec2.OperatingSystemType.WINDOWS:
                asg.addUserData('Set-Variable -Name TEMPDIR -Value (New-TemporaryFile).DirectoryName', `aws s3 cp s3://aws-codedeploy-${cdk.Stack.of(this).region}/latest/codedeploy-agent.msi $TEMPDIR\\codedeploy-agent.msi`, 'cd $TEMPDIR', '.\\codedeploy-agent.msi /quiet /l c:\\temp\\host-agent-install-log.txt');
                break;
        }
    }
    loadBalancerInfo(loadBalancer) {
        if (!loadBalancer) {
            return undefined;
        }
        switch (loadBalancer.generation) {
            case load_balancer_1.LoadBalancerGeneration.FIRST:
                return {
                    elbInfoList: [
                        { name: loadBalancer.name },
                    ],
                };
            case load_balancer_1.LoadBalancerGeneration.SECOND:
                return {
                    targetGroupInfoList: [
                        { name: loadBalancer.name },
                    ],
                };
        }
    }
    ec2TagSet(tagSet) {
        if (!tagSet || tagSet.instanceTagGroups.length === 0) {
            return undefined;
        }
        return {
            ec2TagSetList: tagSet.instanceTagGroups.map(tagGroup => {
                return {
                    ec2TagGroup: this.tagGroup2TagsArray(tagGroup),
                };
            }),
        };
    }
    onPremiseTagSet(tagSet) {
        if (!tagSet || tagSet.instanceTagGroups.length === 0) {
            return undefined;
        }
        return {
            onPremisesTagSetList: tagSet.instanceTagGroups.map(tagGroup => {
                return {
                    onPremisesTagGroup: this.tagGroup2TagsArray(tagGroup),
                };
            }),
        };
    }
    tagGroup2TagsArray(tagGroup) {
        const tagsInGroup = new Array();
        for (const tagKey in tagGroup) {
            if (tagGroup.hasOwnProperty(tagKey)) {
                const tagValues = tagGroup[tagKey];
                if (tagKey.length > 0) {
                    if (tagValues.length > 0) {
                        for (const tagValue of tagValues) {
                            tagsInGroup.push({
                                key: tagKey,
                                value: tagValue,
                                type: 'KEY_AND_VALUE',
                            });
                        }
                    }
                    else {
                        tagsInGroup.push({
                            key: tagKey,
                            type: 'KEY_ONLY',
                        });
                    }
                }
                else {
                    if (tagValues.length > 0) {
                        for (const tagValue of tagValues) {
                            tagsInGroup.push({
                                value: tagValue,
                                type: 'VALUE_ONLY',
                            });
                        }
                    }
                    else {
                        throw new Error('Cannot specify both an empty key and no values for an instance tag filter');
                    }
                }
            }
        }
        return tagsInGroup;
    }
}
exports.ServerDeploymentGroup = ServerDeploymentGroup;
//# sourceMappingURL=data:application/json;base64,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