"use strict";
const assert_1 = require("@aws-cdk/assert");
const cdk = require("@aws-cdk/core");
const lambda = require("../lib");
module.exports = {
    'can import a Lambda version by ARN'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const version = lambda.Version.fromVersionArn(stack, 'Version', 'arn:aws:lambda:region:account-id:function:function-name:version');
        new cdk.CfnOutput(stack, 'ARN', { value: version.functionArn });
        new cdk.CfnOutput(stack, 'Name', { value: version.functionName });
        // THEN
        assert_1.expect(stack).toMatch({
            Outputs: {
                ARN: {
                    Value: 'arn:aws:lambda:region:account-id:function:function-name:version',
                },
                Name: {
                    Value: 'function-name:version',
                },
            },
        });
        test.done();
    },
    'create a version with event invoke config'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'Fn', {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromInline('foo'),
        });
        // WHEN
        new lambda.Version(stack, 'Version', {
            lambda: fn,
            maxEventAge: cdk.Duration.hours(1),
            retryAttempts: 0,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventInvokeConfig', {
            FunctionName: {
                Ref: 'Fn9270CBC0',
            },
            Qualifier: {
                'Fn::GetAtt': [
                    'Version6A868472',
                    'Version',
                ],
            },
            MaximumEventAgeInSeconds: 3600,
            MaximumRetryAttempts: 0,
        }));
        test.done();
    },
    'throws when calling configureAsyncInvoke on already configured version'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'Fn', {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromInline('foo'),
        });
        const version = new lambda.Version(stack, 'Version', {
            lambda: fn,
            maxEventAge: cdk.Duration.hours(1),
            retryAttempts: 0,
        });
        // THEN
        test.throws(() => version.configureAsyncInvoke({ retryAttempts: 1 }), /An EventInvokeConfig has already been configured/);
        test.done();
    },
    'event invoke config on imported versions'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const version1 = lambda.Version.fromVersionArn(stack, 'Version1', 'arn:aws:lambda:region:account-id:function:function-name:version1');
        const version2 = lambda.Version.fromVersionArn(stack, 'Version2', 'arn:aws:lambda:region:account-id:function:function-name:version2');
        // WHEN
        version1.configureAsyncInvoke({
            retryAttempts: 1,
        });
        version2.configureAsyncInvoke({
            retryAttempts: 0,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventInvokeConfig', {
            FunctionName: 'function-name',
            Qualifier: 'version1',
            MaximumRetryAttempts: 1,
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventInvokeConfig', {
            FunctionName: 'function-name',
            Qualifier: 'version2',
            MaximumRetryAttempts: 0,
        }));
        test.done();
    },
    'addAlias can be used to add an alias that points to a version'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'Fn', {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromInline('foo'),
        });
        const version = fn.currentVersion;
        // WHEN
        version.addAlias('foo');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Alias', {
            'FunctionName': {
                'Ref': 'Fn9270CBC0',
            },
            'FunctionVersion': {
                'Fn::GetAtt': [
                    'FnCurrentVersion17A89ABB19ed45993ff69fd011ae9fd4ab6e2005',
                    'Version',
                ],
            },
            'Name': 'foo',
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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