"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const function_1 = require("./function");
const function_base_1 = require("./function-base");
/**
 * A Lambda that will only ever be added to a stack once.
 *
 * This construct is a way to guarantee that the lambda function will be guaranteed to be part of the stack,
 * once and only once, irrespective of how many times the construct is declared to be part of the stack.
 * This is guaranteed as long as the `uuid` property and the optional `lambdaPurpose` property stay the same
 * whenever they're declared into the stack.
 *
 * @resource AWS::Lambda::Function
 */
class SingletonFunction extends function_base_1.FunctionBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.lambdaFunction = this.ensureLambda(props);
        this.permissionsNode = this.lambdaFunction.node;
        this.functionArn = this.lambdaFunction.functionArn;
        this.functionName = this.lambdaFunction.functionName;
        this.role = this.lambdaFunction.role;
        this.grantPrincipal = this.lambdaFunction.grantPrincipal;
        this.canCreatePermissions = true; // Doesn't matter, addPermission is overriden anyway
    }
    addPermission(name, permission) {
        return this.lambdaFunction.addPermission(name, permission);
    }
    ensureLambda(props) {
        const constructName = (props.lambdaPurpose || 'SingletonLambda') + slugify(props.uuid);
        const existing = cdk.Stack.of(this).node.tryFindChild(constructName);
        if (existing) {
            // Just assume this is true
            return existing;
        }
        return new function_1.Function(cdk.Stack.of(this), constructName, props);
    }
}
exports.SingletonFunction = SingletonFunction;
function slugify(x) {
    return x.replace(/[^a-zA-Z0-9]/g, '');
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2luZ2xldG9uLWxhbWJkYS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInNpbmdsZXRvbi1sYW1iZGEudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFDQSxxQ0FBcUM7QUFDckMseUNBQXVFO0FBQ3ZFLG1EQUEwRDtBQTJCMUQ7Ozs7Ozs7OztHQVNHO0FBQ0gsTUFBYSxpQkFBa0IsU0FBUSw0QkFBWTtJQVNqRCxZQUFZLEtBQW9CLEVBQUUsRUFBVSxFQUFFLEtBQTZCO1FBQ3pFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsSUFBSSxDQUFDLGNBQWMsR0FBRyxJQUFJLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQy9DLElBQUksQ0FBQyxlQUFlLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUM7UUFFaEQsSUFBSSxDQUFDLFdBQVcsR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLFdBQVcsQ0FBQztRQUNuRCxJQUFJLENBQUMsWUFBWSxHQUFHLElBQUksQ0FBQyxjQUFjLENBQUMsWUFBWSxDQUFDO1FBQ3JELElBQUksQ0FBQyxJQUFJLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUM7UUFDckMsSUFBSSxDQUFDLGNBQWMsR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLGNBQWMsQ0FBQztRQUV6RCxJQUFJLENBQUMsb0JBQW9CLEdBQUcsSUFBSSxDQUFDLENBQUMsb0RBQW9EO0lBQ3hGLENBQUM7SUFFTSxhQUFhLENBQUMsSUFBWSxFQUFFLFVBQXNCO1FBQ3ZELE9BQU8sSUFBSSxDQUFDLGNBQWMsQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLFVBQVUsQ0FBQyxDQUFDO0lBQzdELENBQUM7SUFFTyxZQUFZLENBQUMsS0FBNkI7UUFDaEQsTUFBTSxhQUFhLEdBQUcsQ0FBQyxLQUFLLENBQUMsYUFBYSxJQUFJLGlCQUFpQixDQUFDLEdBQUcsT0FBTyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN2RixNQUFNLFFBQVEsR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBQ3JFLElBQUksUUFBUSxFQUFFO1lBQ1osMkJBQTJCO1lBQzNCLE9BQU8sUUFBd0IsQ0FBQztTQUNqQztRQUVELE9BQU8sSUFBSSxtQkFBYyxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxFQUFFLGFBQWEsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUN0RSxDQUFDO0NBQ0Y7QUFyQ0QsOENBcUNDO0FBRUQsU0FBUyxPQUFPLENBQUMsQ0FBUztJQUN4QixPQUFPLENBQUMsQ0FBQyxPQUFPLENBQUMsZUFBZSxFQUFFLEVBQUUsQ0FBQyxDQUFDO0FBQ3hDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBpYW0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBGdW5jdGlvbiBhcyBMYW1iZGFGdW5jdGlvbiwgRnVuY3Rpb25Qcm9wcyB9IGZyb20gJy4vZnVuY3Rpb24nO1xuaW1wb3J0IHsgRnVuY3Rpb25CYXNlLCBJRnVuY3Rpb24gfSBmcm9tICcuL2Z1bmN0aW9uLWJhc2UnO1xuaW1wb3J0IHsgUGVybWlzc2lvbiB9IGZyb20gJy4vcGVybWlzc2lvbic7XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgYSBuZXdseSBjcmVhdGVkIHNpbmdsZXRvbiBMYW1iZGFcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBTaW5nbGV0b25GdW5jdGlvblByb3BzIGV4dGVuZHMgRnVuY3Rpb25Qcm9wcyB7XG4gIC8qKlxuICAgKiBBIHVuaXF1ZSBpZGVudGlmaWVyIHRvIGlkZW50aWZ5IHRoaXMgbGFtYmRhXG4gICAqXG4gICAqIFRoZSBpZGVudGlmaWVyIHNob3VsZCBiZSB1bmlxdWUgYWNyb3NzIGFsbCBjdXN0b20gcmVzb3VyY2UgcHJvdmlkZXJzLlxuICAgKiBXZSByZWNvbW1lbmQgZ2VuZXJhdGluZyBhIFVVSUQgcGVyIHByb3ZpZGVyLlxuICAgKi9cbiAgcmVhZG9ubHkgdXVpZDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBIGRlc2NyaXB0aXZlIG5hbWUgZm9yIHRoZSBwdXJwb3NlIG9mIHRoaXMgTGFtYmRhLlxuICAgKlxuICAgKiBJZiB0aGUgTGFtYmRhIGRvZXMgbm90IGhhdmUgYSBwaHlzaWNhbCBuYW1lLCB0aGlzIHN0cmluZyB3aWxsIGJlXG4gICAqIHJlZmxlY3RlZCBpdHMgZ2VuZXJhdGVkIG5hbWUuIFRoZSBjb21iaW5hdGlvbiBvZiBsYW1iZGFQdXJwb3NlXG4gICAqIGFuZCB1dWlkIG11c3QgYmUgdW5pcXVlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBTaW5nbGV0b25MYW1iZGFcbiAgICovXG4gIHJlYWRvbmx5IGxhbWJkYVB1cnBvc2U/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogQSBMYW1iZGEgdGhhdCB3aWxsIG9ubHkgZXZlciBiZSBhZGRlZCB0byBhIHN0YWNrIG9uY2UuXG4gKlxuICogVGhpcyBjb25zdHJ1Y3QgaXMgYSB3YXkgdG8gZ3VhcmFudGVlIHRoYXQgdGhlIGxhbWJkYSBmdW5jdGlvbiB3aWxsIGJlIGd1YXJhbnRlZWQgdG8gYmUgcGFydCBvZiB0aGUgc3RhY2ssXG4gKiBvbmNlIGFuZCBvbmx5IG9uY2UsIGlycmVzcGVjdGl2ZSBvZiBob3cgbWFueSB0aW1lcyB0aGUgY29uc3RydWN0IGlzIGRlY2xhcmVkIHRvIGJlIHBhcnQgb2YgdGhlIHN0YWNrLlxuICogVGhpcyBpcyBndWFyYW50ZWVkIGFzIGxvbmcgYXMgdGhlIGB1dWlkYCBwcm9wZXJ0eSBhbmQgdGhlIG9wdGlvbmFsIGBsYW1iZGFQdXJwb3NlYCBwcm9wZXJ0eSBzdGF5IHRoZSBzYW1lXG4gKiB3aGVuZXZlciB0aGV5J3JlIGRlY2xhcmVkIGludG8gdGhlIHN0YWNrLlxuICpcbiAqIEByZXNvdXJjZSBBV1M6OkxhbWJkYTo6RnVuY3Rpb25cbiAqL1xuZXhwb3J0IGNsYXNzIFNpbmdsZXRvbkZ1bmN0aW9uIGV4dGVuZHMgRnVuY3Rpb25CYXNlIHtcbiAgcHVibGljIHJlYWRvbmx5IGdyYW50UHJpbmNpcGFsOiBpYW0uSVByaW5jaXBhbDtcbiAgcHVibGljIHJlYWRvbmx5IGZ1bmN0aW9uTmFtZTogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgZnVuY3Rpb25Bcm46IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IHJvbGU/OiBpYW0uSVJvbGU7XG4gIHB1YmxpYyByZWFkb25seSBwZXJtaXNzaW9uc05vZGU6IGNkay5Db25zdHJ1Y3ROb2RlO1xuICBwcm90ZWN0ZWQgcmVhZG9ubHkgY2FuQ3JlYXRlUGVybWlzc2lvbnM6IGJvb2xlYW47XG4gIHByaXZhdGUgbGFtYmRhRnVuY3Rpb246IElGdW5jdGlvbjtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogY2RrLkNvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFNpbmdsZXRvbkZ1bmN0aW9uUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgdGhpcy5sYW1iZGFGdW5jdGlvbiA9IHRoaXMuZW5zdXJlTGFtYmRhKHByb3BzKTtcbiAgICB0aGlzLnBlcm1pc3Npb25zTm9kZSA9IHRoaXMubGFtYmRhRnVuY3Rpb24ubm9kZTtcblxuICAgIHRoaXMuZnVuY3Rpb25Bcm4gPSB0aGlzLmxhbWJkYUZ1bmN0aW9uLmZ1bmN0aW9uQXJuO1xuICAgIHRoaXMuZnVuY3Rpb25OYW1lID0gdGhpcy5sYW1iZGFGdW5jdGlvbi5mdW5jdGlvbk5hbWU7XG4gICAgdGhpcy5yb2xlID0gdGhpcy5sYW1iZGFGdW5jdGlvbi5yb2xlO1xuICAgIHRoaXMuZ3JhbnRQcmluY2lwYWwgPSB0aGlzLmxhbWJkYUZ1bmN0aW9uLmdyYW50UHJpbmNpcGFsO1xuXG4gICAgdGhpcy5jYW5DcmVhdGVQZXJtaXNzaW9ucyA9IHRydWU7IC8vIERvZXNuJ3QgbWF0dGVyLCBhZGRQZXJtaXNzaW9uIGlzIG92ZXJyaWRlbiBhbnl3YXlcbiAgfVxuXG4gIHB1YmxpYyBhZGRQZXJtaXNzaW9uKG5hbWU6IHN0cmluZywgcGVybWlzc2lvbjogUGVybWlzc2lvbikge1xuICAgIHJldHVybiB0aGlzLmxhbWJkYUZ1bmN0aW9uLmFkZFBlcm1pc3Npb24obmFtZSwgcGVybWlzc2lvbik7XG4gIH1cblxuICBwcml2YXRlIGVuc3VyZUxhbWJkYShwcm9wczogU2luZ2xldG9uRnVuY3Rpb25Qcm9wcyk6IElGdW5jdGlvbiB7XG4gICAgY29uc3QgY29uc3RydWN0TmFtZSA9IChwcm9wcy5sYW1iZGFQdXJwb3NlIHx8ICdTaW5nbGV0b25MYW1iZGEnKSArIHNsdWdpZnkocHJvcHMudXVpZCk7XG4gICAgY29uc3QgZXhpc3RpbmcgPSBjZGsuU3RhY2sub2YodGhpcykubm9kZS50cnlGaW5kQ2hpbGQoY29uc3RydWN0TmFtZSk7XG4gICAgaWYgKGV4aXN0aW5nKSB7XG4gICAgICAvLyBKdXN0IGFzc3VtZSB0aGlzIGlzIHRydWVcbiAgICAgIHJldHVybiBleGlzdGluZyBhcyBGdW5jdGlvbkJhc2U7XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBMYW1iZGFGdW5jdGlvbihjZGsuU3RhY2sub2YodGhpcyksIGNvbnN0cnVjdE5hbWUsIHByb3BzKTtcbiAgfVxufVxuXG5mdW5jdGlvbiBzbHVnaWZ5KHg6IHN0cmluZyk6IHN0cmluZyB7XG4gIHJldHVybiB4LnJlcGxhY2UoL1teYS16QS1aMC05XS9nLCAnJyk7XG59XG4iXX0=