"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const ecr = require("../lib");
module.exports = {
    'construct repository'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new ecr.Repository(stack, 'Repo');
        // THEN
        assert_1.expect(stack).toMatch({
            Resources: {
                Repo02AC86CF: {
                    Type: "AWS::ECR::Repository",
                    DeletionPolicy: "Retain",
                    UpdateReplacePolicy: "Retain",
                }
            }
        });
        test.done();
    },
    'tag-based lifecycle policy'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const repo = new ecr.Repository(stack, 'Repo');
        // WHEN
        repo.addLifecycleRule({ tagPrefixList: ['abc'], maxImageCount: 1 });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            LifecyclePolicy: {
                // tslint:disable-next-line:max-line-length
                LifecyclePolicyText: "{\"rules\":[{\"rulePriority\":1,\"selection\":{\"tagStatus\":\"tagged\",\"tagPrefixList\":[\"abc\"],\"countType\":\"imageCountMoreThan\",\"countNumber\":1},\"action\":{\"type\":\"expire\"}}]}"
            }
        }));
        test.done();
    },
    'add day-based lifecycle policy'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const repo = new ecr.Repository(stack, 'Repo');
        repo.addLifecycleRule({
            maxImageAge: cdk.Duration.days(5),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            LifecyclePolicy: {
                // tslint:disable-next-line:max-line-length
                LifecyclePolicyText: "{\"rules\":[{\"rulePriority\":1,\"selection\":{\"tagStatus\":\"any\",\"countType\":\"sinceImagePushed\",\"countNumber\":5,\"countUnit\":\"days\"},\"action\":{\"type\":\"expire\"}}]}",
            }
        }));
        test.done();
    },
    'add count-based lifecycle policy'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const repo = new ecr.Repository(stack, 'Repo');
        // WHEN
        repo.addLifecycleRule({
            maxImageCount: 5,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            LifecyclePolicy: {
                // tslint:disable-next-line:max-line-length
                LifecyclePolicyText: "{\"rules\":[{\"rulePriority\":1,\"selection\":{\"tagStatus\":\"any\",\"countType\":\"imageCountMoreThan\",\"countNumber\":5},\"action\":{\"type\":\"expire\"}}]}",
            }
        }));
        test.done();
    },
    'mixing numbered and unnumbered rules'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const repo = new ecr.Repository(stack, 'Repo');
        // WHEN
        repo.addLifecycleRule({ tagStatus: ecr.TagStatus.TAGGED, tagPrefixList: ['a'], maxImageCount: 5 });
        repo.addLifecycleRule({ rulePriority: 10, tagStatus: ecr.TagStatus.TAGGED, tagPrefixList: ['b'], maxImageCount: 5 });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            LifecyclePolicy: {
                // tslint:disable-next-line:max-line-length
                LifecyclePolicyText: "{\"rules\":[{\"rulePriority\":10,\"selection\":{\"tagStatus\":\"tagged\",\"tagPrefixList\":[\"b\"],\"countType\":\"imageCountMoreThan\",\"countNumber\":5},\"action\":{\"type\":\"expire\"}},{\"rulePriority\":11,\"selection\":{\"tagStatus\":\"tagged\",\"tagPrefixList\":[\"a\"],\"countType\":\"imageCountMoreThan\",\"countNumber\":5},\"action\":{\"type\":\"expire\"}}]}"
            }
        }));
        test.done();
    },
    'tagstatus Any is automatically sorted to the back'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const repo = new ecr.Repository(stack, 'Repo');
        // WHEN
        repo.addLifecycleRule({ maxImageCount: 5 });
        repo.addLifecycleRule({ tagStatus: ecr.TagStatus.TAGGED, tagPrefixList: ['important'], maxImageCount: 999 });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            LifecyclePolicy: {
                // tslint:disable-next-line:max-line-length
                LifecyclePolicyText: "{\"rules\":[{\"rulePriority\":1,\"selection\":{\"tagStatus\":\"tagged\",\"tagPrefixList\":[\"important\"],\"countType\":\"imageCountMoreThan\",\"countNumber\":999},\"action\":{\"type\":\"expire\"}},{\"rulePriority\":2,\"selection\":{\"tagStatus\":\"any\",\"countType\":\"imageCountMoreThan\",\"countNumber\":5},\"action\":{\"type\":\"expire\"}}]}"
            }
        }));
        test.done();
    },
    'lifecycle rules can be added upon initialization'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new ecr.Repository(stack, 'Repo', {
            lifecycleRules: [
                { maxImageCount: 3 }
            ]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            "LifecyclePolicy": {
                // tslint:disable-next-line:max-line-length
                "LifecyclePolicyText": "{\"rules\":[{\"rulePriority\":1,\"selection\":{\"tagStatus\":\"any\",\"countType\":\"imageCountMoreThan\",\"countNumber\":3},\"action\":{\"type\":\"expire\"}}]}"
            }
        }));
        test.done();
    },
    'calculate repository URI'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const repo = new ecr.Repository(stack, 'Repo');
        // WHEN
        const uri = repo.repositoryUri;
        // THEN
        const arnSplit = { 'Fn::Split': [':', { 'Fn::GetAtt': ['Repo02AC86CF', 'Arn'] }] };
        test.deepEqual(stack.resolve(uri), {
            'Fn::Join': ['', [
                    { 'Fn::Select': [4, arnSplit] },
                    '.dkr.ecr.',
                    { 'Fn::Select': [3, arnSplit] },
                    '.',
                    { Ref: 'AWS::URLSuffix' },
                    '/',
                    { Ref: 'Repo02AC86CF' }
                ]]
        });
        test.done();
    },
    'import with concrete arn'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const repo2 = ecr.Repository.fromRepositoryArn(stack, 'repo', 'arn:aws:ecr:us-east-1:585695036304:repository/foo/bar/foo/fooo');
        // THEN
        test.deepEqual(stack.resolve(repo2.repositoryArn), 'arn:aws:ecr:us-east-1:585695036304:repository/foo/bar/foo/fooo');
        test.deepEqual(stack.resolve(repo2.repositoryName), 'foo/bar/foo/fooo');
        test.done();
    },
    'fails if importing with token arn and no name'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN/THEN
        test.throws(() => ecr.Repository.fromRepositoryArn(stack, 'arn', cdk.Fn.getAtt('Boom', 'Boom').toString()), /\"repositoryArn\" is a late-bound value, and therefore \"repositoryName\" is required\. Use \`fromRepositoryAttributes\` instead/);
        test.done();
    },
    'import with token arn and repository name (see awslabs/aws-cdk#1232)'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const repo = ecr.Repository.fromRepositoryAttributes(stack, 'Repo', {
            repositoryArn: cdk.Fn.getAtt('Boom', 'Arn').toString(),
            repositoryName: cdk.Fn.getAtt('Boom', 'Name').toString()
        });
        // THEN
        test.deepEqual(stack.resolve(repo.repositoryArn), { 'Fn::GetAtt': ['Boom', 'Arn'] });
        test.deepEqual(stack.resolve(repo.repositoryName), { 'Fn::GetAtt': ['Boom', 'Name'] });
        test.done();
    },
    'import only with a repository name (arn is deduced)'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const repo = ecr.Repository.fromRepositoryName(stack, 'just-name', 'my-repo');
        // THEN
        test.deepEqual(stack.resolve(repo.repositoryArn), {
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':ecr:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':repository/my-repo'
                ]
            ]
        });
        test.deepEqual(stack.resolve(repo.repositoryName), 'my-repo');
        test.done();
    },
    'arnForLocalRepository can be used to render an ARN for a local repository'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const repoName = cdk.Fn.getAtt('Boom', 'Name').toString();
        // WHEN
        const repo = ecr.Repository.fromRepositoryAttributes(stack, 'Repo', {
            repositoryArn: ecr.Repository.arnForLocalRepository(repoName, stack),
            repositoryName: repoName
        });
        // THEN
        test.deepEqual(stack.resolve(repo.repositoryName), { 'Fn::GetAtt': ['Boom', 'Name'] });
        test.deepEqual(stack.resolve(repo.repositoryArn), {
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':ecr:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':repository/',
                    { 'Fn::GetAtt': ['Boom', 'Name'] }
                ]]
        });
        test.done();
    },
    'resource policy'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const repo = new ecr.Repository(stack, 'Repo');
        // WHEN
        repo.addToResourcePolicy(new iam.PolicyStatement({ actions: ['*'] }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
            RepositoryPolicyText: {
                Statement: [
                    {
                        Action: "*",
                        Effect: "Allow"
                    }
                ],
                Version: "2012-10-17"
            },
        }));
        test.done();
    },
    'events': {
        'onImagePushed without imageTag creates the correct event'(test) {
            const stack = new cdk.Stack();
            const repo = new ecr.Repository(stack, 'Repo');
            repo.onCloudTrailImagePushed('EventRule', {
                target: {
                    bind: () => ({ arn: 'ARN', id: '' })
                }
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                "EventPattern": {
                    "source": [
                        "aws.ecr",
                    ],
                    "detail": {
                        "eventName": [
                            "PutImage",
                        ],
                        "requestParameters": {
                            "repositoryName": [
                                {
                                    "Ref": "Repo02AC86CF"
                                }
                            ],
                        },
                    },
                },
                "State": "ENABLED",
            }));
            test.done();
        },
        'onImageScanCompleted without imageTags creates the correct event'(test) {
            const stack = new cdk.Stack();
            const repo = new ecr.Repository(stack, 'Repo');
            repo.onImageScanCompleted('EventRule', {
                target: {
                    bind: () => ({ arn: 'ARN', id: '' })
                }
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                "EventPattern": {
                    "source": [
                        "aws.ecr",
                    ],
                    "detail": {
                        "repository-name": [
                            {
                                "Ref": "Repo02AC86CF"
                            }
                        ],
                        "scan-status": [
                            "COMPLETE"
                        ]
                    }
                },
                "State": "ENABLED",
            }));
            test.done();
        },
        'onImageScanCompleted with one imageTag creates the correct event'(test) {
            const stack = new cdk.Stack();
            const repo = new ecr.Repository(stack, 'Repo');
            repo.onImageScanCompleted('EventRule', {
                imageTags: ['some-tag'],
                target: {
                    bind: () => ({ arn: 'ARN', id: '' })
                }
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                "EventPattern": {
                    "source": [
                        "aws.ecr",
                    ],
                    "detail": {
                        "repository-name": [
                            {
                                "Ref": "Repo02AC86CF"
                            }
                        ],
                        "image-tags": [
                            "some-tag"
                        ],
                        "scan-status": [
                            "COMPLETE"
                        ]
                    }
                },
                "State": "ENABLED",
            }));
            test.done();
        },
        'onImageScanCompleted with multiple imageTags creates the correct event'(test) {
            const stack = new cdk.Stack();
            const repo = new ecr.Repository(stack, 'Repo');
            repo.onImageScanCompleted('EventRule', {
                imageTags: ['tag1', 'tag2', 'tag3'],
                target: {
                    bind: () => ({ arn: 'ARN', id: '' })
                }
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                "EventPattern": {
                    "source": [
                        "aws.ecr",
                    ],
                    "detail": {
                        "repository-name": [
                            {
                                "Ref": "Repo02AC86CF"
                            }
                        ],
                        "image-tags": [
                            "tag1",
                            "tag2",
                            "tag3"
                        ],
                        "scan-status": [
                            "COMPLETE"
                        ]
                    }
                },
                "State": "ENABLED",
            }));
            test.done();
        },
        'removal policy is "Retain" by default'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            new ecr.Repository(stack, 'Repo');
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
                "Type": "AWS::ECR::Repository",
                "DeletionPolicy": "Retain"
            }, assert_1.ResourcePart.CompleteDefinition));
            test.done();
        },
        '"Delete" removal policy can be set explicitly'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            new ecr.Repository(stack, 'Repo', {
                removalPolicy: cdk.RemovalPolicy.DESTROY
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
                "Type": "AWS::ECR::Repository",
                "DeletionPolicy": "Delete"
            }, assert_1.ResourcePart.CompleteDefinition));
            test.done();
        },
        'grant adds appropriate resource-*'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const repo = new ecr.Repository(stack, 'TestHarnessRepo');
            // WHEN
            repo.grantPull(new iam.AnyPrincipal());
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
                "RepositoryPolicyText": {
                    "Statement": [
                        {
                            "Action": [
                                "ecr:BatchCheckLayerAvailability",
                                "ecr:GetDownloadUrlForLayer",
                                "ecr:BatchGetImage"
                            ],
                            "Effect": "Allow",
                            "Principal": "*",
                        }
                    ],
                    "Version": "2012-10-17"
                }
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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